/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iot1clickdevices;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iot1clickdevices.internal.Iot1ClickDevicesServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iot1clickdevices.model.ClaimDevicesByClaimCodeRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.ClaimDevicesByClaimCodeResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.DescribeDeviceRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.DescribeDeviceResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.FinalizeDeviceClaimRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.FinalizeDeviceClaimResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.ForbiddenException;
import software.amazon.awssdk.services.iot1clickdevices.model.GetDeviceMethodsRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.GetDeviceMethodsResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.InitiateDeviceClaimRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.InitiateDeviceClaimResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.InternalFailureException;
import software.amazon.awssdk.services.iot1clickdevices.model.InvalidRequestException;
import software.amazon.awssdk.services.iot1clickdevices.model.InvokeDeviceMethodRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.InvokeDeviceMethodResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.Iot1ClickDevicesException;
import software.amazon.awssdk.services.iot1clickdevices.model.ListDeviceEventsRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.ListDeviceEventsResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.ListDevicesRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.ListDevicesResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.PreconditionFailedException;
import software.amazon.awssdk.services.iot1clickdevices.model.RangeNotSatisfiableException;
import software.amazon.awssdk.services.iot1clickdevices.model.ResourceConflictException;
import software.amazon.awssdk.services.iot1clickdevices.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iot1clickdevices.model.TagResourceRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.TagResourceResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.UnclaimDeviceRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.UnclaimDeviceResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.UntagResourceRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.UntagResourceResponse;
import software.amazon.awssdk.services.iot1clickdevices.model.UpdateDeviceStateRequest;
import software.amazon.awssdk.services.iot1clickdevices.model.UpdateDeviceStateResponse;
import software.amazon.awssdk.services.iot1clickdevices.transform.ClaimDevicesByClaimCodeRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.DescribeDeviceRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.FinalizeDeviceClaimRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.GetDeviceMethodsRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.InitiateDeviceClaimRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.InvokeDeviceMethodRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.ListDeviceEventsRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.ListDevicesRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.UnclaimDeviceRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.iot1clickdevices.transform.UpdateDeviceStateRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link Iot1ClickDevicesClient}.
 *
 * @see Iot1ClickDevicesClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIot1ClickDevicesClient implements Iot1ClickDevicesClient {
    private static final Logger log = Logger.loggerFor(DefaultIot1ClickDevicesClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final Iot1ClickDevicesServiceClientConfiguration serviceClientConfiguration;

    protected DefaultIot1ClickDevicesClient(Iot1ClickDevicesServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds device(s) to your account (i.e., claim one or more devices) if and only if you received a claim code with
     * the device(s).
     * </p>
     *
     * @param claimDevicesByClaimCodeRequest
     * @return Result of the ClaimDevicesByClaimCode operation returned by the service.
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws ForbiddenException
     *         403 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.ClaimDevicesByClaimCode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/ClaimDevicesByClaimCode"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ClaimDevicesByClaimCodeResponse claimDevicesByClaimCode(ClaimDevicesByClaimCodeRequest claimDevicesByClaimCodeRequest)
            throws InvalidRequestException, InternalFailureException, ForbiddenException, AwsServiceException,
            SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ClaimDevicesByClaimCodeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ClaimDevicesByClaimCodeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(claimDevicesByClaimCodeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, claimDevicesByClaimCodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ClaimDevicesByClaimCode");

            return clientHandler
                    .execute(new ClientExecutionParams<ClaimDevicesByClaimCodeRequest, ClaimDevicesByClaimCodeResponse>()
                            .withOperationName("ClaimDevicesByClaimCode").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(claimDevicesByClaimCodeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ClaimDevicesByClaimCodeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Given a device ID, returns a DescribeDeviceResponse object describing the details of the device.
     * </p>
     *
     * @param describeDeviceRequest
     * @return Result of the DescribeDevice operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.DescribeDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/DescribeDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeDeviceResponse describeDevice(DescribeDeviceRequest describeDeviceRequest) throws ResourceNotFoundException,
            InvalidRequestException, InternalFailureException, AwsServiceException, SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDeviceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDevice");

            return clientHandler.execute(new ClientExecutionParams<DescribeDeviceRequest, DescribeDeviceResponse>()
                    .withOperationName("DescribeDevice").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeDeviceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDeviceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Given a device ID, finalizes the claim request for the associated device.
     * </p>
     * <note>
     * <p>
     * Claiming a device consists of initiating a claim, then publishing a device event, and finalizing the claim. For a
     * device of type button, a device event can be published by simply clicking the device.
     * </p>
     * </note>
     *
     * @param finalizeDeviceClaimRequest
     * @return Result of the FinalizeDeviceClaim operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws PreconditionFailedException
     *         412 response
     * @throws ResourceConflictException
     *         409 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.FinalizeDeviceClaim
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/FinalizeDeviceClaim" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public FinalizeDeviceClaimResponse finalizeDeviceClaim(FinalizeDeviceClaimRequest finalizeDeviceClaimRequest)
            throws ResourceNotFoundException, InvalidRequestException, InternalFailureException, PreconditionFailedException,
            ResourceConflictException, AwsServiceException, SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<FinalizeDeviceClaimResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, FinalizeDeviceClaimResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(finalizeDeviceClaimRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, finalizeDeviceClaimRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "FinalizeDeviceClaim");

            return clientHandler.execute(new ClientExecutionParams<FinalizeDeviceClaimRequest, FinalizeDeviceClaimResponse>()
                    .withOperationName("FinalizeDeviceClaim").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(finalizeDeviceClaimRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new FinalizeDeviceClaimRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Given a device ID, returns the invokable methods associated with the device.
     * </p>
     *
     * @param getDeviceMethodsRequest
     * @return Result of the GetDeviceMethods operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.GetDeviceMethods
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/GetDeviceMethods" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDeviceMethodsResponse getDeviceMethods(GetDeviceMethodsRequest getDeviceMethodsRequest)
            throws ResourceNotFoundException, InvalidRequestException, InternalFailureException, AwsServiceException,
            SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDeviceMethodsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDeviceMethodsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDeviceMethodsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeviceMethodsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeviceMethods");

            return clientHandler.execute(new ClientExecutionParams<GetDeviceMethodsRequest, GetDeviceMethodsResponse>()
                    .withOperationName("GetDeviceMethods").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getDeviceMethodsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDeviceMethodsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Given a device ID, initiates a claim request for the associated device.
     * </p>
     * <note>
     * <p>
     * Claiming a device consists of initiating a claim, then publishing a device event, and finalizing the claim. For a
     * device of type button, a device event can be published by simply clicking the device.
     * </p>
     * </note>
     *
     * @param initiateDeviceClaimRequest
     * @return Result of the InitiateDeviceClaim operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws ResourceConflictException
     *         409 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.InitiateDeviceClaim
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/InitiateDeviceClaim" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public InitiateDeviceClaimResponse initiateDeviceClaim(InitiateDeviceClaimRequest initiateDeviceClaimRequest)
            throws ResourceNotFoundException, InvalidRequestException, InternalFailureException, ResourceConflictException,
            AwsServiceException, SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InitiateDeviceClaimResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, InitiateDeviceClaimResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(initiateDeviceClaimRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initiateDeviceClaimRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitiateDeviceClaim");

            return clientHandler.execute(new ClientExecutionParams<InitiateDeviceClaimRequest, InitiateDeviceClaimResponse>()
                    .withOperationName("InitiateDeviceClaim").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(initiateDeviceClaimRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InitiateDeviceClaimRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Given a device ID, issues a request to invoke a named device method (with possible parameters). See the
     * "Example POST" code snippet below.
     * </p>
     *
     * @param invokeDeviceMethodRequest
     * @return Result of the InvokeDeviceMethod operation returned by the service.
     * @throws InvalidRequestException
     *         400 response
     * @throws PreconditionFailedException
     *         412 response
     * @throws InternalFailureException
     *         500 response
     * @throws ResourceNotFoundException
     *         404 response
     * @throws RangeNotSatisfiableException
     *         416 response
     * @throws ResourceConflictException
     *         409 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.InvokeDeviceMethod
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/InvokeDeviceMethod" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public InvokeDeviceMethodResponse invokeDeviceMethod(InvokeDeviceMethodRequest invokeDeviceMethodRequest)
            throws InvalidRequestException, PreconditionFailedException, InternalFailureException, ResourceNotFoundException,
            RangeNotSatisfiableException, ResourceConflictException, AwsServiceException, SdkClientException,
            Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InvokeDeviceMethodResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, InvokeDeviceMethodResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeDeviceMethodRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeDeviceMethodRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeDeviceMethod");

            return clientHandler.execute(new ClientExecutionParams<InvokeDeviceMethodRequest, InvokeDeviceMethodResponse>()
                    .withOperationName("InvokeDeviceMethod").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(invokeDeviceMethodRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InvokeDeviceMethodRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Using a device ID, returns a DeviceEventsResponse object containing an array of events for the device.
     * </p>
     *
     * @param listDeviceEventsRequest
     * @return Result of the ListDeviceEvents operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws RangeNotSatisfiableException
     *         416 response
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.ListDeviceEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/ListDeviceEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDeviceEventsResponse listDeviceEvents(ListDeviceEventsRequest listDeviceEventsRequest)
            throws ResourceNotFoundException, RangeNotSatisfiableException, InvalidRequestException, InternalFailureException,
            AwsServiceException, SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDeviceEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDeviceEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeviceEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeviceEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeviceEvents");

            return clientHandler.execute(new ClientExecutionParams<ListDeviceEventsRequest, ListDeviceEventsResponse>()
                    .withOperationName("ListDeviceEvents").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDeviceEventsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDeviceEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the 1-Click compatible devices associated with your AWS account.
     * </p>
     *
     * @param listDevicesRequest
     * @return Result of the ListDevices operation returned by the service.
     * @throws RangeNotSatisfiableException
     *         416 response
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.ListDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/ListDevices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDevicesResponse listDevices(ListDevicesRequest listDevicesRequest) throws RangeNotSatisfiableException,
            InvalidRequestException, InternalFailureException, AwsServiceException, SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDevicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDevicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDevicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDevices");

            return clientHandler.execute(new ClientExecutionParams<ListDevicesRequest, ListDevicesResponse>()
                    .withOperationName("ListDevices").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listDevicesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDevicesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags associated with the specified resource ARN.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws InternalFailureException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalFailureException, AwsServiceException, SdkClientException,
            Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or updates the tags associated with the resource ARN. See <a
     * href="https://docs.aws.amazon.com/iot-1-click/latest/developerguide/1click-appendix.html#1click-limits">AWS IoT
     * 1-Click Service Limits</a> for the maximum number of tags allowed per resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InvalidRequestException, InternalFailureException, AwsServiceException, SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates a device from your AWS account using its device ID.
     * </p>
     *
     * @param unclaimDeviceRequest
     * @return Result of the UnclaimDevice operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.UnclaimDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/UnclaimDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UnclaimDeviceResponse unclaimDevice(UnclaimDeviceRequest unclaimDeviceRequest) throws ResourceNotFoundException,
            InvalidRequestException, InternalFailureException, AwsServiceException, SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UnclaimDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UnclaimDeviceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(unclaimDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unclaimDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnclaimDevice");

            return clientHandler.execute(new ClientExecutionParams<UnclaimDeviceRequest, UnclaimDeviceResponse>()
                    .withOperationName("UnclaimDevice").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(unclaimDeviceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UnclaimDeviceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Using tag keys, deletes the tags (key/value pairs) associated with the specified resource ARN.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InvalidRequestException, InternalFailureException, AwsServiceException, SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Using a Boolean value (true or false), this operation enables or disables the device given a device ID.
     * </p>
     *
     * @param updateDeviceStateRequest
     * @return Result of the UpdateDeviceState operation returned by the service.
     * @throws ResourceNotFoundException
     *         404 response
     * @throws InvalidRequestException
     *         400 response
     * @throws InternalFailureException
     *         500 response
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Iot1ClickDevicesException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Iot1ClickDevicesClient.UpdateDeviceState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/devices-2018-05-14/UpdateDeviceState" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateDeviceStateResponse updateDeviceState(UpdateDeviceStateRequest updateDeviceStateRequest)
            throws ResourceNotFoundException, InvalidRequestException, InternalFailureException, AwsServiceException,
            SdkClientException, Iot1ClickDevicesException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDeviceStateResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDeviceStateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDeviceStateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDeviceStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT 1Click Devices Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDeviceState");

            return clientHandler.execute(new ClientExecutionParams<UpdateDeviceStateRequest, UpdateDeviceStateResponse>()
                    .withOperationName("UpdateDeviceState").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateDeviceStateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDeviceStateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        Iot1ClickDevicesServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = Iot1ClickDevicesServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Iot1ClickDevicesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PreconditionFailedException")
                                .exceptionBuilderSupplier(PreconditionFailedException::builder).httpStatusCode(412).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RangeNotSatisfiableException")
                                .exceptionBuilderSupplier(RangeNotSatisfiableException::builder).httpStatusCode(416).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceConflictException")
                                .exceptionBuilderSupplier(ResourceConflictException::builder).httpStatusCode(409).build());
    }

    @Override
    public final Iot1ClickDevicesServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
