/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotanalytics.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration information for delivery of dataset contents to Amazon Simple Storage Service (Amazon S3).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3DestinationConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3DestinationConfiguration.Builder, S3DestinationConfiguration> {
    private static final SdkField<String> BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("bucket")
            .getter(getter(S3DestinationConfiguration::bucket)).setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucket").build()).build();

    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("key")
            .getter(getter(S3DestinationConfiguration::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("key").build()).build();

    private static final SdkField<GlueConfiguration> GLUE_CONFIGURATION_FIELD = SdkField
            .<GlueConfiguration> builder(MarshallingType.SDK_POJO).memberName("glueConfiguration")
            .getter(getter(S3DestinationConfiguration::glueConfiguration)).setter(setter(Builder::glueConfiguration))
            .constructor(GlueConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("glueConfiguration").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(S3DestinationConfiguration::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_FIELD, KEY_FIELD,
            GLUE_CONFIGURATION_FIELD, ROLE_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucket;

    private final String key;

    private final GlueConfiguration glueConfiguration;

    private final String roleArn;

    private S3DestinationConfiguration(BuilderImpl builder) {
        this.bucket = builder.bucket;
        this.key = builder.key;
        this.glueConfiguration = builder.glueConfiguration;
        this.roleArn = builder.roleArn;
    }

    /**
     * <p>
     * The name of the S3 bucket to which dataset contents are delivered.
     * </p>
     * 
     * @return The name of the S3 bucket to which dataset contents are delivered.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * <p>
     * The key of the dataset contents object in an S3 bucket. Each object has a key that is a unique identifier. Each
     * object has exactly one key.
     * </p>
     * <p>
     * You can create a unique key with the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use <code>!{iotanalytics:scheduleTime}</code> to insert the time of a scheduled SQL query run.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use <code>!{iotanalytics:versionId}</code> to insert a unique hash that identifies a dataset content.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use <code>!{iotanalytics:creationTime}</code> to insert the creation time of a dataset content.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following example creates a unique key for a CSV file:
     * <code>dataset/mydataset/!{iotanalytics:scheduleTime}/!{iotanalytics:versionId}.csv</code>
     * </p>
     * <note>
     * <p>
     * If you don't use <code>!{iotanalytics:versionId}</code> to specify the key, you might get duplicate keys. For
     * example, you might have two dataset contents with the same <code>scheduleTime</code> but different
     * <code>versionId</code>s. This means that one dataset content overwrites the other.
     * </p>
     * </note>
     * 
     * @return The key of the dataset contents object in an S3 bucket. Each object has a key that is a unique
     *         identifier. Each object has exactly one key.</p>
     *         <p>
     *         You can create a unique key with the following options:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Use <code>!{iotanalytics:scheduleTime}</code> to insert the time of a scheduled SQL query run.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Use <code>!{iotanalytics:versionId}</code> to insert a unique hash that identifies a dataset content.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Use <code>!{iotanalytics:creationTime}</code> to insert the creation time of a dataset content.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The following example creates a unique key for a CSV file:
     *         <code>dataset/mydataset/!{iotanalytics:scheduleTime}/!{iotanalytics:versionId}.csv</code>
     *         </p>
     *         <note>
     *         <p>
     *         If you don't use <code>!{iotanalytics:versionId}</code> to specify the key, you might get duplicate keys.
     *         For example, you might have two dataset contents with the same <code>scheduleTime</code> but different
     *         <code>versionId</code>s. This means that one dataset content overwrites the other.
     *         </p>
     */
    public final String key() {
        return key;
    }

    /**
     * <p>
     * Configuration information for coordination with Glue, a fully managed extract, transform and load (ETL) service.
     * </p>
     * 
     * @return Configuration information for coordination with Glue, a fully managed extract, transform and load (ETL)
     *         service.
     */
    public final GlueConfiguration glueConfiguration() {
        return glueConfiguration;
    }

    /**
     * <p>
     * The ARN of the role that grants IoT Analytics permission to interact with your Amazon S3 and Glue resources.
     * </p>
     * 
     * @return The ARN of the role that grants IoT Analytics permission to interact with your Amazon S3 and Glue
     *         resources.
     */
    public final String roleArn() {
        return roleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(glueConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3DestinationConfiguration)) {
            return false;
        }
        S3DestinationConfiguration other = (S3DestinationConfiguration) obj;
        return Objects.equals(bucket(), other.bucket()) && Objects.equals(key(), other.key())
                && Objects.equals(glueConfiguration(), other.glueConfiguration()) && Objects.equals(roleArn(), other.roleArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3DestinationConfiguration").add("Bucket", bucket()).add("Key", key())
                .add("GlueConfiguration", glueConfiguration()).add("RoleArn", roleArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "key":
            return Optional.ofNullable(clazz.cast(key()));
        case "glueConfiguration":
            return Optional.ofNullable(clazz.cast(glueConfiguration()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3DestinationConfiguration, T> g) {
        return obj -> g.apply((S3DestinationConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3DestinationConfiguration> {
        /**
         * <p>
         * The name of the S3 bucket to which dataset contents are delivered.
         * </p>
         * 
         * @param bucket
         *        The name of the S3 bucket to which dataset contents are delivered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * The key of the dataset contents object in an S3 bucket. Each object has a key that is a unique identifier.
         * Each object has exactly one key.
         * </p>
         * <p>
         * You can create a unique key with the following options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Use <code>!{iotanalytics:scheduleTime}</code> to insert the time of a scheduled SQL query run.
         * </p>
         * </li>
         * <li>
         * <p>
         * Use <code>!{iotanalytics:versionId}</code> to insert a unique hash that identifies a dataset content.
         * </p>
         * </li>
         * <li>
         * <p>
         * Use <code>!{iotanalytics:creationTime}</code> to insert the creation time of a dataset content.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The following example creates a unique key for a CSV file:
         * <code>dataset/mydataset/!{iotanalytics:scheduleTime}/!{iotanalytics:versionId}.csv</code>
         * </p>
         * <note>
         * <p>
         * If you don't use <code>!{iotanalytics:versionId}</code> to specify the key, you might get duplicate keys. For
         * example, you might have two dataset contents with the same <code>scheduleTime</code> but different
         * <code>versionId</code>s. This means that one dataset content overwrites the other.
         * </p>
         * </note>
         * 
         * @param key
         *        The key of the dataset contents object in an S3 bucket. Each object has a key that is a unique
         *        identifier. Each object has exactly one key.</p>
         *        <p>
         *        You can create a unique key with the following options:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Use <code>!{iotanalytics:scheduleTime}</code> to insert the time of a scheduled SQL query run.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Use <code>!{iotanalytics:versionId}</code> to insert a unique hash that identifies a dataset content.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Use <code>!{iotanalytics:creationTime}</code> to insert the creation time of a dataset content.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The following example creates a unique key for a CSV file:
         *        <code>dataset/mydataset/!{iotanalytics:scheduleTime}/!{iotanalytics:versionId}.csv</code>
         *        </p>
         *        <note>
         *        <p>
         *        If you don't use <code>!{iotanalytics:versionId}</code> to specify the key, you might get duplicate
         *        keys. For example, you might have two dataset contents with the same <code>scheduleTime</code> but
         *        different <code>versionId</code>s. This means that one dataset content overwrites the other.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * Configuration information for coordination with Glue, a fully managed extract, transform and load (ETL)
         * service.
         * </p>
         * 
         * @param glueConfiguration
         *        Configuration information for coordination with Glue, a fully managed extract, transform and load
         *        (ETL) service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder glueConfiguration(GlueConfiguration glueConfiguration);

        /**
         * <p>
         * Configuration information for coordination with Glue, a fully managed extract, transform and load (ETL)
         * service.
         * </p>
         * This is a convenience method that creates an instance of the {@link GlueConfiguration.Builder} avoiding the
         * need to create one manually via {@link GlueConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GlueConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #glueConfiguration(GlueConfiguration)}.
         * 
         * @param glueConfiguration
         *        a consumer that will call methods on {@link GlueConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #glueConfiguration(GlueConfiguration)
         */
        default Builder glueConfiguration(Consumer<GlueConfiguration.Builder> glueConfiguration) {
            return glueConfiguration(GlueConfiguration.builder().applyMutation(glueConfiguration).build());
        }

        /**
         * <p>
         * The ARN of the role that grants IoT Analytics permission to interact with your Amazon S3 and Glue resources.
         * </p>
         * 
         * @param roleArn
         *        The ARN of the role that grants IoT Analytics permission to interact with your Amazon S3 and Glue
         *        resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);
    }

    static final class BuilderImpl implements Builder {
        private String bucket;

        private String key;

        private GlueConfiguration glueConfiguration;

        private String roleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(S3DestinationConfiguration model) {
            bucket(model.bucket);
            key(model.key);
            glueConfiguration(model.glueConfiguration);
            roleArn(model.roleArn);
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final GlueConfiguration.Builder getGlueConfiguration() {
            return glueConfiguration != null ? glueConfiguration.toBuilder() : null;
        }

        public final void setGlueConfiguration(GlueConfiguration.BuilderImpl glueConfiguration) {
            this.glueConfiguration = glueConfiguration != null ? glueConfiguration.build() : null;
        }

        @Override
        public final Builder glueConfiguration(GlueConfiguration glueConfiguration) {
            this.glueConfiguration = glueConfiguration;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        @Override
        public S3DestinationConfiguration build() {
            return new S3DestinationConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
