/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotanalytics.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a data store.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Datastore implements SdkPojo, Serializable, ToCopyableBuilder<Datastore.Builder, Datastore> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Datastore::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<DatastoreStorage> STORAGE_FIELD = SdkField.<DatastoreStorage> builder(MarshallingType.SDK_POJO)
            .memberName("storage").getter(getter(Datastore::storage)).setter(setter(Builder::storage))
            .constructor(DatastoreStorage::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("storage").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Datastore::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(Datastore::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<RetentionPeriod> RETENTION_PERIOD_FIELD = SdkField
            .<RetentionPeriod> builder(MarshallingType.SDK_POJO).memberName("retentionPeriod")
            .getter(getter(Datastore::retentionPeriod)).setter(setter(Builder::retentionPeriod))
            .constructor(RetentionPeriod::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("retentionPeriod").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTime").getter(getter(Datastore::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdateTime").getter(getter(Datastore::lastUpdateTime)).setter(setter(Builder::lastUpdateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdateTime").build()).build();

    private static final SdkField<Instant> LAST_MESSAGE_ARRIVAL_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastMessageArrivalTime").getter(getter(Datastore::lastMessageArrivalTime))
            .setter(setter(Builder::lastMessageArrivalTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastMessageArrivalTime").build())
            .build();

    private static final SdkField<FileFormatConfiguration> FILE_FORMAT_CONFIGURATION_FIELD = SdkField
            .<FileFormatConfiguration> builder(MarshallingType.SDK_POJO).memberName("fileFormatConfiguration")
            .getter(getter(Datastore::fileFormatConfiguration)).setter(setter(Builder::fileFormatConfiguration))
            .constructor(FileFormatConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fileFormatConfiguration").build())
            .build();

    private static final SdkField<DatastorePartitions> DATASTORE_PARTITIONS_FIELD = SdkField
            .<DatastorePartitions> builder(MarshallingType.SDK_POJO).memberName("datastorePartitions")
            .getter(getter(Datastore::datastorePartitions)).setter(setter(Builder::datastorePartitions))
            .constructor(DatastorePartitions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("datastorePartitions").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, STORAGE_FIELD,
            ARN_FIELD, STATUS_FIELD, RETENTION_PERIOD_FIELD, CREATION_TIME_FIELD, LAST_UPDATE_TIME_FIELD,
            LAST_MESSAGE_ARRIVAL_TIME_FIELD, FILE_FORMAT_CONFIGURATION_FIELD, DATASTORE_PARTITIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final DatastoreStorage storage;

    private final String arn;

    private final String status;

    private final RetentionPeriod retentionPeriod;

    private final Instant creationTime;

    private final Instant lastUpdateTime;

    private final Instant lastMessageArrivalTime;

    private final FileFormatConfiguration fileFormatConfiguration;

    private final DatastorePartitions datastorePartitions;

    private Datastore(BuilderImpl builder) {
        this.name = builder.name;
        this.storage = builder.storage;
        this.arn = builder.arn;
        this.status = builder.status;
        this.retentionPeriod = builder.retentionPeriod;
        this.creationTime = builder.creationTime;
        this.lastUpdateTime = builder.lastUpdateTime;
        this.lastMessageArrivalTime = builder.lastMessageArrivalTime;
        this.fileFormatConfiguration = builder.fileFormatConfiguration;
        this.datastorePartitions = builder.datastorePartitions;
    }

    /**
     * <p>
     * The name of the data store.
     * </p>
     * 
     * @return The name of the data store.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Where data in a data store is stored.. You can choose <code>serviceManagedS3</code> storage,
     * <code>customerManagedS3</code> storage, or <code>iotSiteWiseMultiLayerStorage</code> storage. The default is
     * <code>serviceManagedS3</code>. You can't change the choice of Amazon S3 storage after your data store is created.
     * </p>
     * 
     * @return Where data in a data store is stored.. You can choose <code>serviceManagedS3</code> storage,
     *         <code>customerManagedS3</code> storage, or <code>iotSiteWiseMultiLayerStorage</code> storage. The default
     *         is <code>serviceManagedS3</code>. You can't change the choice of Amazon S3 storage after your data store
     *         is created.
     */
    public final DatastoreStorage storage() {
        return storage;
    }

    /**
     * <p>
     * The ARN of the data store.
     * </p>
     * 
     * @return The ARN of the data store.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The status of a data store:
     * </p>
     * <dl>
     * <dt>CREATING</dt>
     * <dd>
     * <p>
     * The data store is being created.
     * </p>
     * </dd>
     * <dt>ACTIVE</dt>
     * <dd>
     * <p>
     * The data store has been created and can be used.
     * </p>
     * </dd>
     * <dt>DELETING</dt>
     * <dd>
     * <p>
     * The data store is being deleted.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DatastoreStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a data store:</p>
     *         <dl>
     *         <dt>CREATING</dt>
     *         <dd>
     *         <p>
     *         The data store is being created.
     *         </p>
     *         </dd>
     *         <dt>ACTIVE</dt>
     *         <dd>
     *         <p>
     *         The data store has been created and can be used.
     *         </p>
     *         </dd>
     *         <dt>DELETING</dt>
     *         <dd>
     *         <p>
     *         The data store is being deleted.
     *         </p>
     *         </dd>
     * @see DatastoreStatus
     */
    public final DatastoreStatus status() {
        return DatastoreStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of a data store:
     * </p>
     * <dl>
     * <dt>CREATING</dt>
     * <dd>
     * <p>
     * The data store is being created.
     * </p>
     * </dd>
     * <dt>ACTIVE</dt>
     * <dd>
     * <p>
     * The data store has been created and can be used.
     * </p>
     * </dd>
     * <dt>DELETING</dt>
     * <dd>
     * <p>
     * The data store is being deleted.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DatastoreStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a data store:</p>
     *         <dl>
     *         <dt>CREATING</dt>
     *         <dd>
     *         <p>
     *         The data store is being created.
     *         </p>
     *         </dd>
     *         <dt>ACTIVE</dt>
     *         <dd>
     *         <p>
     *         The data store has been created and can be used.
     *         </p>
     *         </dd>
     *         <dt>DELETING</dt>
     *         <dd>
     *         <p>
     *         The data store is being deleted.
     *         </p>
     *         </dd>
     * @see DatastoreStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * How long, in days, message data is kept for the data store. When <code>customerManagedS3</code> storage is
     * selected, this parameter is ignored.
     * </p>
     * 
     * @return How long, in days, message data is kept for the data store. When <code>customerManagedS3</code> storage
     *         is selected, this parameter is ignored.
     */
    public final RetentionPeriod retentionPeriod() {
        return retentionPeriod;
    }

    /**
     * <p>
     * When the data store was created.
     * </p>
     * 
     * @return When the data store was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The last time the data store was updated.
     * </p>
     * 
     * @return The last time the data store was updated.
     */
    public final Instant lastUpdateTime() {
        return lastUpdateTime;
    }

    /**
     * <p>
     * The last time when a new message arrived in the data store.
     * </p>
     * <p>
     * IoT Analytics updates this value at most once per minute for Amazon Simple Storage Service one data store. Hence,
     * the <code>lastMessageArrivalTime</code> value is an approximation.
     * </p>
     * <p>
     * This feature only applies to messages that arrived in the data store after October 23, 2020.
     * </p>
     * 
     * @return The last time when a new message arrived in the data store.</p>
     *         <p>
     *         IoT Analytics updates this value at most once per minute for Amazon Simple Storage Service one data
     *         store. Hence, the <code>lastMessageArrivalTime</code> value is an approximation.
     *         </p>
     *         <p>
     *         This feature only applies to messages that arrived in the data store after October 23, 2020.
     */
    public final Instant lastMessageArrivalTime() {
        return lastMessageArrivalTime;
    }

    /**
     * <p>
     * Contains the configuration information of file formats. IoT Analytics data stores support JSON and <a
     * href="https://parquet.apache.org/">Parquet</a>.
     * </p>
     * <p>
     * The default file format is JSON. You can specify only one format.
     * </p>
     * <p>
     * You can't change the file format after you create the data store.
     * </p>
     * 
     * @return Contains the configuration information of file formats. IoT Analytics data stores support JSON and <a
     *         href="https://parquet.apache.org/">Parquet</a>.</p>
     *         <p>
     *         The default file format is JSON. You can specify only one format.
     *         </p>
     *         <p>
     *         You can't change the file format after you create the data store.
     */
    public final FileFormatConfiguration fileFormatConfiguration() {
        return fileFormatConfiguration;
    }

    /**
     * <p>
     * Contains information about the partition dimensions in a data store.
     * </p>
     * 
     * @return Contains information about the partition dimensions in a data store.
     */
    public final DatastorePartitions datastorePartitions() {
        return datastorePartitions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(storage());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(retentionPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastMessageArrivalTime());
        hashCode = 31 * hashCode + Objects.hashCode(fileFormatConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(datastorePartitions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Datastore)) {
            return false;
        }
        Datastore other = (Datastore) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(storage(), other.storage())
                && Objects.equals(arn(), other.arn()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(retentionPeriod(), other.retentionPeriod())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdateTime(), other.lastUpdateTime())
                && Objects.equals(lastMessageArrivalTime(), other.lastMessageArrivalTime())
                && Objects.equals(fileFormatConfiguration(), other.fileFormatConfiguration())
                && Objects.equals(datastorePartitions(), other.datastorePartitions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Datastore").add("Name", name()).add("Storage", storage()).add("Arn", arn())
                .add("Status", statusAsString()).add("RetentionPeriod", retentionPeriod()).add("CreationTime", creationTime())
                .add("LastUpdateTime", lastUpdateTime()).add("LastMessageArrivalTime", lastMessageArrivalTime())
                .add("FileFormatConfiguration", fileFormatConfiguration()).add("DatastorePartitions", datastorePartitions())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "storage":
            return Optional.ofNullable(clazz.cast(storage()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "retentionPeriod":
            return Optional.ofNullable(clazz.cast(retentionPeriod()));
        case "creationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "lastUpdateTime":
            return Optional.ofNullable(clazz.cast(lastUpdateTime()));
        case "lastMessageArrivalTime":
            return Optional.ofNullable(clazz.cast(lastMessageArrivalTime()));
        case "fileFormatConfiguration":
            return Optional.ofNullable(clazz.cast(fileFormatConfiguration()));
        case "datastorePartitions":
            return Optional.ofNullable(clazz.cast(datastorePartitions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("storage", STORAGE_FIELD);
        map.put("arn", ARN_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("retentionPeriod", RETENTION_PERIOD_FIELD);
        map.put("creationTime", CREATION_TIME_FIELD);
        map.put("lastUpdateTime", LAST_UPDATE_TIME_FIELD);
        map.put("lastMessageArrivalTime", LAST_MESSAGE_ARRIVAL_TIME_FIELD);
        map.put("fileFormatConfiguration", FILE_FORMAT_CONFIGURATION_FIELD);
        map.put("datastorePartitions", DATASTORE_PARTITIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Datastore, T> g) {
        return obj -> g.apply((Datastore) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Datastore> {
        /**
         * <p>
         * The name of the data store.
         * </p>
         * 
         * @param name
         *        The name of the data store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Where data in a data store is stored.. You can choose <code>serviceManagedS3</code> storage,
         * <code>customerManagedS3</code> storage, or <code>iotSiteWiseMultiLayerStorage</code> storage. The default is
         * <code>serviceManagedS3</code>. You can't change the choice of Amazon S3 storage after your data store is
         * created.
         * </p>
         * 
         * @param storage
         *        Where data in a data store is stored.. You can choose <code>serviceManagedS3</code> storage,
         *        <code>customerManagedS3</code> storage, or <code>iotSiteWiseMultiLayerStorage</code> storage. The
         *        default is <code>serviceManagedS3</code>. You can't change the choice of Amazon S3 storage after your
         *        data store is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storage(DatastoreStorage storage);

        /**
         * <p>
         * Where data in a data store is stored.. You can choose <code>serviceManagedS3</code> storage,
         * <code>customerManagedS3</code> storage, or <code>iotSiteWiseMultiLayerStorage</code> storage. The default is
         * <code>serviceManagedS3</code>. You can't change the choice of Amazon S3 storage after your data store is
         * created.
         * </p>
         * This is a convenience method that creates an instance of the {@link DatastoreStorage.Builder} avoiding the
         * need to create one manually via {@link DatastoreStorage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DatastoreStorage.Builder#build()} is called immediately and its
         * result is passed to {@link #storage(DatastoreStorage)}.
         * 
         * @param storage
         *        a consumer that will call methods on {@link DatastoreStorage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #storage(DatastoreStorage)
         */
        default Builder storage(Consumer<DatastoreStorage.Builder> storage) {
            return storage(DatastoreStorage.builder().applyMutation(storage).build());
        }

        /**
         * <p>
         * The ARN of the data store.
         * </p>
         * 
         * @param arn
         *        The ARN of the data store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The status of a data store:
         * </p>
         * <dl>
         * <dt>CREATING</dt>
         * <dd>
         * <p>
         * The data store is being created.
         * </p>
         * </dd>
         * <dt>ACTIVE</dt>
         * <dd>
         * <p>
         * The data store has been created and can be used.
         * </p>
         * </dd>
         * <dt>DELETING</dt>
         * <dd>
         * <p>
         * The data store is being deleted.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param status
         *        The status of a data store:</p>
         *        <dl>
         *        <dt>CREATING</dt>
         *        <dd>
         *        <p>
         *        The data store is being created.
         *        </p>
         *        </dd>
         *        <dt>ACTIVE</dt>
         *        <dd>
         *        <p>
         *        The data store has been created and can be used.
         *        </p>
         *        </dd>
         *        <dt>DELETING</dt>
         *        <dd>
         *        <p>
         *        The data store is being deleted.
         *        </p>
         *        </dd>
         * @see DatastoreStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DatastoreStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of a data store:
         * </p>
         * <dl>
         * <dt>CREATING</dt>
         * <dd>
         * <p>
         * The data store is being created.
         * </p>
         * </dd>
         * <dt>ACTIVE</dt>
         * <dd>
         * <p>
         * The data store has been created and can be used.
         * </p>
         * </dd>
         * <dt>DELETING</dt>
         * <dd>
         * <p>
         * The data store is being deleted.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param status
         *        The status of a data store:</p>
         *        <dl>
         *        <dt>CREATING</dt>
         *        <dd>
         *        <p>
         *        The data store is being created.
         *        </p>
         *        </dd>
         *        <dt>ACTIVE</dt>
         *        <dd>
         *        <p>
         *        The data store has been created and can be used.
         *        </p>
         *        </dd>
         *        <dt>DELETING</dt>
         *        <dd>
         *        <p>
         *        The data store is being deleted.
         *        </p>
         *        </dd>
         * @see DatastoreStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DatastoreStatus
         */
        Builder status(DatastoreStatus status);

        /**
         * <p>
         * How long, in days, message data is kept for the data store. When <code>customerManagedS3</code> storage is
         * selected, this parameter is ignored.
         * </p>
         * 
         * @param retentionPeriod
         *        How long, in days, message data is kept for the data store. When <code>customerManagedS3</code>
         *        storage is selected, this parameter is ignored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retentionPeriod(RetentionPeriod retentionPeriod);

        /**
         * <p>
         * How long, in days, message data is kept for the data store. When <code>customerManagedS3</code> storage is
         * selected, this parameter is ignored.
         * </p>
         * This is a convenience method that creates an instance of the {@link RetentionPeriod.Builder} avoiding the
         * need to create one manually via {@link RetentionPeriod#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RetentionPeriod.Builder#build()} is called immediately and its
         * result is passed to {@link #retentionPeriod(RetentionPeriod)}.
         * 
         * @param retentionPeriod
         *        a consumer that will call methods on {@link RetentionPeriod.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #retentionPeriod(RetentionPeriod)
         */
        default Builder retentionPeriod(Consumer<RetentionPeriod.Builder> retentionPeriod) {
            return retentionPeriod(RetentionPeriod.builder().applyMutation(retentionPeriod).build());
        }

        /**
         * <p>
         * When the data store was created.
         * </p>
         * 
         * @param creationTime
         *        When the data store was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The last time the data store was updated.
         * </p>
         * 
         * @param lastUpdateTime
         *        The last time the data store was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTime(Instant lastUpdateTime);

        /**
         * <p>
         * The last time when a new message arrived in the data store.
         * </p>
         * <p>
         * IoT Analytics updates this value at most once per minute for Amazon Simple Storage Service one data store.
         * Hence, the <code>lastMessageArrivalTime</code> value is an approximation.
         * </p>
         * <p>
         * This feature only applies to messages that arrived in the data store after October 23, 2020.
         * </p>
         * 
         * @param lastMessageArrivalTime
         *        The last time when a new message arrived in the data store.</p>
         *        <p>
         *        IoT Analytics updates this value at most once per minute for Amazon Simple Storage Service one data
         *        store. Hence, the <code>lastMessageArrivalTime</code> value is an approximation.
         *        </p>
         *        <p>
         *        This feature only applies to messages that arrived in the data store after October 23, 2020.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastMessageArrivalTime(Instant lastMessageArrivalTime);

        /**
         * <p>
         * Contains the configuration information of file formats. IoT Analytics data stores support JSON and <a
         * href="https://parquet.apache.org/">Parquet</a>.
         * </p>
         * <p>
         * The default file format is JSON. You can specify only one format.
         * </p>
         * <p>
         * You can't change the file format after you create the data store.
         * </p>
         * 
         * @param fileFormatConfiguration
         *        Contains the configuration information of file formats. IoT Analytics data stores support JSON and <a
         *        href="https://parquet.apache.org/">Parquet</a>.</p>
         *        <p>
         *        The default file format is JSON. You can specify only one format.
         *        </p>
         *        <p>
         *        You can't change the file format after you create the data store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileFormatConfiguration(FileFormatConfiguration fileFormatConfiguration);

        /**
         * <p>
         * Contains the configuration information of file formats. IoT Analytics data stores support JSON and <a
         * href="https://parquet.apache.org/">Parquet</a>.
         * </p>
         * <p>
         * The default file format is JSON. You can specify only one format.
         * </p>
         * <p>
         * You can't change the file format after you create the data store.
         * </p>
         * This is a convenience method that creates an instance of the {@link FileFormatConfiguration.Builder} avoiding
         * the need to create one manually via {@link FileFormatConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FileFormatConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #fileFormatConfiguration(FileFormatConfiguration)}.
         * 
         * @param fileFormatConfiguration
         *        a consumer that will call methods on {@link FileFormatConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fileFormatConfiguration(FileFormatConfiguration)
         */
        default Builder fileFormatConfiguration(Consumer<FileFormatConfiguration.Builder> fileFormatConfiguration) {
            return fileFormatConfiguration(FileFormatConfiguration.builder().applyMutation(fileFormatConfiguration).build());
        }

        /**
         * <p>
         * Contains information about the partition dimensions in a data store.
         * </p>
         * 
         * @param datastorePartitions
         *        Contains information about the partition dimensions in a data store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datastorePartitions(DatastorePartitions datastorePartitions);

        /**
         * <p>
         * Contains information about the partition dimensions in a data store.
         * </p>
         * This is a convenience method that creates an instance of the {@link DatastorePartitions.Builder} avoiding the
         * need to create one manually via {@link DatastorePartitions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DatastorePartitions.Builder#build()} is called immediately and
         * its result is passed to {@link #datastorePartitions(DatastorePartitions)}.
         * 
         * @param datastorePartitions
         *        a consumer that will call methods on {@link DatastorePartitions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #datastorePartitions(DatastorePartitions)
         */
        default Builder datastorePartitions(Consumer<DatastorePartitions.Builder> datastorePartitions) {
            return datastorePartitions(DatastorePartitions.builder().applyMutation(datastorePartitions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private DatastoreStorage storage;

        private String arn;

        private String status;

        private RetentionPeriod retentionPeriod;

        private Instant creationTime;

        private Instant lastUpdateTime;

        private Instant lastMessageArrivalTime;

        private FileFormatConfiguration fileFormatConfiguration;

        private DatastorePartitions datastorePartitions;

        private BuilderImpl() {
        }

        private BuilderImpl(Datastore model) {
            name(model.name);
            storage(model.storage);
            arn(model.arn);
            status(model.status);
            retentionPeriod(model.retentionPeriod);
            creationTime(model.creationTime);
            lastUpdateTime(model.lastUpdateTime);
            lastMessageArrivalTime(model.lastMessageArrivalTime);
            fileFormatConfiguration(model.fileFormatConfiguration);
            datastorePartitions(model.datastorePartitions);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final DatastoreStorage.Builder getStorage() {
            return storage != null ? storage.toBuilder() : null;
        }

        public final void setStorage(DatastoreStorage.BuilderImpl storage) {
            this.storage = storage != null ? storage.build() : null;
        }

        @Override
        public final Builder storage(DatastoreStorage storage) {
            this.storage = storage;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(DatastoreStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final RetentionPeriod.Builder getRetentionPeriod() {
            return retentionPeriod != null ? retentionPeriod.toBuilder() : null;
        }

        public final void setRetentionPeriod(RetentionPeriod.BuilderImpl retentionPeriod) {
            this.retentionPeriod = retentionPeriod != null ? retentionPeriod.build() : null;
        }

        @Override
        public final Builder retentionPeriod(RetentionPeriod retentionPeriod) {
            this.retentionPeriod = retentionPeriod;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getLastUpdateTime() {
            return lastUpdateTime;
        }

        public final void setLastUpdateTime(Instant lastUpdateTime) {
            this.lastUpdateTime = lastUpdateTime;
        }

        @Override
        public final Builder lastUpdateTime(Instant lastUpdateTime) {
            this.lastUpdateTime = lastUpdateTime;
            return this;
        }

        public final Instant getLastMessageArrivalTime() {
            return lastMessageArrivalTime;
        }

        public final void setLastMessageArrivalTime(Instant lastMessageArrivalTime) {
            this.lastMessageArrivalTime = lastMessageArrivalTime;
        }

        @Override
        public final Builder lastMessageArrivalTime(Instant lastMessageArrivalTime) {
            this.lastMessageArrivalTime = lastMessageArrivalTime;
            return this;
        }

        public final FileFormatConfiguration.Builder getFileFormatConfiguration() {
            return fileFormatConfiguration != null ? fileFormatConfiguration.toBuilder() : null;
        }

        public final void setFileFormatConfiguration(FileFormatConfiguration.BuilderImpl fileFormatConfiguration) {
            this.fileFormatConfiguration = fileFormatConfiguration != null ? fileFormatConfiguration.build() : null;
        }

        @Override
        public final Builder fileFormatConfiguration(FileFormatConfiguration fileFormatConfiguration) {
            this.fileFormatConfiguration = fileFormatConfiguration;
            return this;
        }

        public final DatastorePartitions.Builder getDatastorePartitions() {
            return datastorePartitions != null ? datastorePartitions.toBuilder() : null;
        }

        public final void setDatastorePartitions(DatastorePartitions.BuilderImpl datastorePartitions) {
            this.datastorePartitions = datastorePartitions != null ? datastorePartitions.build() : null;
        }

        @Override
        public final Builder datastorePartitions(DatastorePartitions datastorePartitions) {
            this.datastorePartitions = datastorePartitions;
            return this;
        }

        @Override
        public Datastore build() {
            return new Datastore(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
