/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotanalytics.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class RunPipelineActivityRequest extends IoTAnalyticsRequest implements
        ToCopyableBuilder<RunPipelineActivityRequest.Builder, RunPipelineActivityRequest> {
    private static final SdkField<PipelineActivity> PIPELINE_ACTIVITY_FIELD = SdkField
            .<PipelineActivity> builder(MarshallingType.SDK_POJO).memberName("pipelineActivity")
            .getter(getter(RunPipelineActivityRequest::pipelineActivity)).setter(setter(Builder::pipelineActivity))
            .constructor(PipelineActivity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pipelineActivity").build()).build();

    private static final SdkField<List<SdkBytes>> PAYLOADS_FIELD = SdkField
            .<List<SdkBytes>> builder(MarshallingType.LIST)
            .memberName("payloads")
            .getter(getter(RunPipelineActivityRequest::payloads))
            .setter(setter(Builder::payloads))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("payloads").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PIPELINE_ACTIVITY_FIELD,
            PAYLOADS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final PipelineActivity pipelineActivity;

    private final List<SdkBytes> payloads;

    private RunPipelineActivityRequest(BuilderImpl builder) {
        super(builder);
        this.pipelineActivity = builder.pipelineActivity;
        this.payloads = builder.payloads;
    }

    /**
     * <p>
     * The pipeline activity that is run. This must not be a channel activity or a data store activity because these
     * activities are used in a pipeline only to load the original message and to store the (possibly) transformed
     * message. If a Lambda activity is specified, only short-running Lambda functions (those with a timeout of less
     * than 30 seconds or less) can be used.
     * </p>
     * 
     * @return The pipeline activity that is run. This must not be a channel activity or a data store activity because
     *         these activities are used in a pipeline only to load the original message and to store the (possibly)
     *         transformed message. If a Lambda activity is specified, only short-running Lambda functions (those with a
     *         timeout of less than 30 seconds or less) can be used.
     */
    public final PipelineActivity pipelineActivity() {
        return pipelineActivity;
    }

    /**
     * For responses, this returns true if the service returned a value for the Payloads property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPayloads() {
        return payloads != null && !(payloads instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The sample message payloads on which the pipeline activity is run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPayloads} method.
     * </p>
     * 
     * @return The sample message payloads on which the pipeline activity is run.
     */
    public final List<SdkBytes> payloads() {
        return payloads;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(pipelineActivity());
        hashCode = 31 * hashCode + Objects.hashCode(hasPayloads() ? payloads() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RunPipelineActivityRequest)) {
            return false;
        }
        RunPipelineActivityRequest other = (RunPipelineActivityRequest) obj;
        return Objects.equals(pipelineActivity(), other.pipelineActivity()) && hasPayloads() == other.hasPayloads()
                && Objects.equals(payloads(), other.payloads());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RunPipelineActivityRequest").add("PipelineActivity", pipelineActivity())
                .add("Payloads", hasPayloads() ? payloads() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "pipelineActivity":
            return Optional.ofNullable(clazz.cast(pipelineActivity()));
        case "payloads":
            return Optional.ofNullable(clazz.cast(payloads()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("pipelineActivity", PIPELINE_ACTIVITY_FIELD);
        map.put("payloads", PAYLOADS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RunPipelineActivityRequest, T> g) {
        return obj -> g.apply((RunPipelineActivityRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends IoTAnalyticsRequest.Builder, SdkPojo, CopyableBuilder<Builder, RunPipelineActivityRequest> {
        /**
         * <p>
         * The pipeline activity that is run. This must not be a channel activity or a data store activity because these
         * activities are used in a pipeline only to load the original message and to store the (possibly) transformed
         * message. If a Lambda activity is specified, only short-running Lambda functions (those with a timeout of less
         * than 30 seconds or less) can be used.
         * </p>
         * 
         * @param pipelineActivity
         *        The pipeline activity that is run. This must not be a channel activity or a data store activity
         *        because these activities are used in a pipeline only to load the original message and to store the
         *        (possibly) transformed message. If a Lambda activity is specified, only short-running Lambda functions
         *        (those with a timeout of less than 30 seconds or less) can be used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineActivity(PipelineActivity pipelineActivity);

        /**
         * <p>
         * The pipeline activity that is run. This must not be a channel activity or a data store activity because these
         * activities are used in a pipeline only to load the original message and to store the (possibly) transformed
         * message. If a Lambda activity is specified, only short-running Lambda functions (those with a timeout of less
         * than 30 seconds or less) can be used.
         * </p>
         * This is a convenience method that creates an instance of the {@link PipelineActivity.Builder} avoiding the
         * need to create one manually via {@link PipelineActivity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PipelineActivity.Builder#build()} is called immediately and its
         * result is passed to {@link #pipelineActivity(PipelineActivity)}.
         * 
         * @param pipelineActivity
         *        a consumer that will call methods on {@link PipelineActivity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #pipelineActivity(PipelineActivity)
         */
        default Builder pipelineActivity(Consumer<PipelineActivity.Builder> pipelineActivity) {
            return pipelineActivity(PipelineActivity.builder().applyMutation(pipelineActivity).build());
        }

        /**
         * <p>
         * The sample message payloads on which the pipeline activity is run.
         * </p>
         * 
         * @param payloads
         *        The sample message payloads on which the pipeline activity is run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payloads(Collection<SdkBytes> payloads);

        /**
         * <p>
         * The sample message payloads on which the pipeline activity is run.
         * </p>
         * 
         * @param payloads
         *        The sample message payloads on which the pipeline activity is run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payloads(SdkBytes... payloads);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends IoTAnalyticsRequest.BuilderImpl implements Builder {
        private PipelineActivity pipelineActivity;

        private List<SdkBytes> payloads = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RunPipelineActivityRequest model) {
            super(model);
            pipelineActivity(model.pipelineActivity);
            payloads(model.payloads);
        }

        public final PipelineActivity.Builder getPipelineActivity() {
            return pipelineActivity != null ? pipelineActivity.toBuilder() : null;
        }

        public final void setPipelineActivity(PipelineActivity.BuilderImpl pipelineActivity) {
            this.pipelineActivity = pipelineActivity != null ? pipelineActivity.build() : null;
        }

        @Override
        public final Builder pipelineActivity(PipelineActivity pipelineActivity) {
            this.pipelineActivity = pipelineActivity;
            return this;
        }

        public final List<ByteBuffer> getPayloads() {
            if (payloads instanceof SdkAutoConstructList) {
                return null;
            }
            return payloads == null ? null : payloads.stream().map(SdkBytes::asByteBuffer).collect(Collectors.toList());
        }

        public final void setPayloads(Collection<ByteBuffer> payloads) {
            payloads(payloads == null ? null : payloads.stream().map(SdkBytes::fromByteBuffer).collect(Collectors.toList()));
        }

        @Override
        public final Builder payloads(Collection<SdkBytes> payloads) {
            this.payloads = MessagePayloadsCopier.copy(payloads);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder payloads(SdkBytes... payloads) {
            payloads(Arrays.asList(payloads));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public RunPipelineActivityRequest build() {
            return new RunPipelineActivityRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
