/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotanalytics;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.iotanalytics.internal.IoTAnalyticsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iotanalytics.model.BatchPutMessageRequest;
import software.amazon.awssdk.services.iotanalytics.model.BatchPutMessageResponse;
import software.amazon.awssdk.services.iotanalytics.model.CancelPipelineReprocessingRequest;
import software.amazon.awssdk.services.iotanalytics.model.CancelPipelineReprocessingResponse;
import software.amazon.awssdk.services.iotanalytics.model.CreateChannelRequest;
import software.amazon.awssdk.services.iotanalytics.model.CreateChannelResponse;
import software.amazon.awssdk.services.iotanalytics.model.CreateDatasetContentRequest;
import software.amazon.awssdk.services.iotanalytics.model.CreateDatasetContentResponse;
import software.amazon.awssdk.services.iotanalytics.model.CreateDatasetRequest;
import software.amazon.awssdk.services.iotanalytics.model.CreateDatasetResponse;
import software.amazon.awssdk.services.iotanalytics.model.CreateDatastoreRequest;
import software.amazon.awssdk.services.iotanalytics.model.CreateDatastoreResponse;
import software.amazon.awssdk.services.iotanalytics.model.CreatePipelineRequest;
import software.amazon.awssdk.services.iotanalytics.model.CreatePipelineResponse;
import software.amazon.awssdk.services.iotanalytics.model.DeleteChannelRequest;
import software.amazon.awssdk.services.iotanalytics.model.DeleteChannelResponse;
import software.amazon.awssdk.services.iotanalytics.model.DeleteDatasetContentRequest;
import software.amazon.awssdk.services.iotanalytics.model.DeleteDatasetContentResponse;
import software.amazon.awssdk.services.iotanalytics.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.iotanalytics.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.iotanalytics.model.DeleteDatastoreRequest;
import software.amazon.awssdk.services.iotanalytics.model.DeleteDatastoreResponse;
import software.amazon.awssdk.services.iotanalytics.model.DeletePipelineRequest;
import software.amazon.awssdk.services.iotanalytics.model.DeletePipelineResponse;
import software.amazon.awssdk.services.iotanalytics.model.DescribeChannelRequest;
import software.amazon.awssdk.services.iotanalytics.model.DescribeChannelResponse;
import software.amazon.awssdk.services.iotanalytics.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.iotanalytics.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.iotanalytics.model.DescribeDatastoreRequest;
import software.amazon.awssdk.services.iotanalytics.model.DescribeDatastoreResponse;
import software.amazon.awssdk.services.iotanalytics.model.DescribeLoggingOptionsRequest;
import software.amazon.awssdk.services.iotanalytics.model.DescribeLoggingOptionsResponse;
import software.amazon.awssdk.services.iotanalytics.model.DescribePipelineRequest;
import software.amazon.awssdk.services.iotanalytics.model.DescribePipelineResponse;
import software.amazon.awssdk.services.iotanalytics.model.GetDatasetContentRequest;
import software.amazon.awssdk.services.iotanalytics.model.GetDatasetContentResponse;
import software.amazon.awssdk.services.iotanalytics.model.InternalFailureException;
import software.amazon.awssdk.services.iotanalytics.model.InvalidRequestException;
import software.amazon.awssdk.services.iotanalytics.model.IoTAnalyticsException;
import software.amazon.awssdk.services.iotanalytics.model.LimitExceededException;
import software.amazon.awssdk.services.iotanalytics.model.ListChannelsRequest;
import software.amazon.awssdk.services.iotanalytics.model.ListChannelsResponse;
import software.amazon.awssdk.services.iotanalytics.model.ListDatasetContentsRequest;
import software.amazon.awssdk.services.iotanalytics.model.ListDatasetContentsResponse;
import software.amazon.awssdk.services.iotanalytics.model.ListDatasetsRequest;
import software.amazon.awssdk.services.iotanalytics.model.ListDatasetsResponse;
import software.amazon.awssdk.services.iotanalytics.model.ListDatastoresRequest;
import software.amazon.awssdk.services.iotanalytics.model.ListDatastoresResponse;
import software.amazon.awssdk.services.iotanalytics.model.ListPipelinesRequest;
import software.amazon.awssdk.services.iotanalytics.model.ListPipelinesResponse;
import software.amazon.awssdk.services.iotanalytics.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.iotanalytics.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.iotanalytics.model.PutLoggingOptionsRequest;
import software.amazon.awssdk.services.iotanalytics.model.PutLoggingOptionsResponse;
import software.amazon.awssdk.services.iotanalytics.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.iotanalytics.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotanalytics.model.RunPipelineActivityRequest;
import software.amazon.awssdk.services.iotanalytics.model.RunPipelineActivityResponse;
import software.amazon.awssdk.services.iotanalytics.model.SampleChannelDataRequest;
import software.amazon.awssdk.services.iotanalytics.model.SampleChannelDataResponse;
import software.amazon.awssdk.services.iotanalytics.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotanalytics.model.StartPipelineReprocessingRequest;
import software.amazon.awssdk.services.iotanalytics.model.StartPipelineReprocessingResponse;
import software.amazon.awssdk.services.iotanalytics.model.TagResourceRequest;
import software.amazon.awssdk.services.iotanalytics.model.TagResourceResponse;
import software.amazon.awssdk.services.iotanalytics.model.ThrottlingException;
import software.amazon.awssdk.services.iotanalytics.model.UntagResourceRequest;
import software.amazon.awssdk.services.iotanalytics.model.UntagResourceResponse;
import software.amazon.awssdk.services.iotanalytics.model.UpdateChannelRequest;
import software.amazon.awssdk.services.iotanalytics.model.UpdateChannelResponse;
import software.amazon.awssdk.services.iotanalytics.model.UpdateDatasetRequest;
import software.amazon.awssdk.services.iotanalytics.model.UpdateDatasetResponse;
import software.amazon.awssdk.services.iotanalytics.model.UpdateDatastoreRequest;
import software.amazon.awssdk.services.iotanalytics.model.UpdateDatastoreResponse;
import software.amazon.awssdk.services.iotanalytics.model.UpdatePipelineRequest;
import software.amazon.awssdk.services.iotanalytics.model.UpdatePipelineResponse;
import software.amazon.awssdk.services.iotanalytics.transform.BatchPutMessageRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.CancelPipelineReprocessingRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.CreateChannelRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.CreateDatasetContentRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.CreateDatasetRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.CreateDatastoreRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.CreatePipelineRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.DeleteChannelRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.DeleteDatasetContentRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.DeleteDatastoreRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.DeletePipelineRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.DescribeChannelRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.DescribeDatastoreRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.DescribeLoggingOptionsRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.DescribePipelineRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.GetDatasetContentRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.ListDatasetContentsRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.ListDatasetsRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.ListDatastoresRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.ListPipelinesRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.PutLoggingOptionsRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.RunPipelineActivityRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.SampleChannelDataRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.StartPipelineReprocessingRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.UpdateChannelRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.UpdateDatasetRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.UpdateDatastoreRequestMarshaller;
import software.amazon.awssdk.services.iotanalytics.transform.UpdatePipelineRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IoTAnalyticsAsyncClient}.
 *
 * @see IoTAnalyticsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIoTAnalyticsAsyncClient implements IoTAnalyticsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIoTAnalyticsAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIoTAnalyticsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Sends messages to a channel.
     * </p>
     *
     * @param batchPutMessageRequest
     * @return A Java Future containing the result of the BatchPutMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.BatchPutMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/BatchPutMessage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchPutMessageResponse> batchPutMessage(BatchPutMessageRequest batchPutMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchPutMessageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchPutMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchPutMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchPutMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchPutMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchPutMessageRequest, BatchPutMessageResponse>()
                            .withOperationName("BatchPutMessage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new BatchPutMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(batchPutMessageRequest));
            CompletableFuture<BatchPutMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels the reprocessing of data through the pipeline.
     * </p>
     *
     * @param cancelPipelineReprocessingRequest
     * @return A Java Future containing the result of the CancelPipelineReprocessing operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.CancelPipelineReprocessing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/CancelPipelineReprocessing"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelPipelineReprocessingResponse> cancelPipelineReprocessing(
            CancelPipelineReprocessingRequest cancelPipelineReprocessingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelPipelineReprocessingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelPipelineReprocessingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelPipelineReprocessing");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelPipelineReprocessingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelPipelineReprocessingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelPipelineReprocessingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelPipelineReprocessingRequest, CancelPipelineReprocessingResponse>()
                            .withOperationName("CancelPipelineReprocessing").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelPipelineReprocessingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelPipelineReprocessingRequest));
            CompletableFuture<CancelPipelineReprocessingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to create a channel. A channel collects data from an MQTT topic and archives the raw, unprocessed messages
     * before publishing the data to a pipeline.
     * </p>
     *
     * @param createChannelRequest
     * @return A Java Future containing the result of the CreateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceAlreadyExistsException A resource with the same name already exists.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The command caused an internal limit to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/CreateChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelResponse> createChannel(CreateChannelRequest createChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelRequest, CreateChannelResponse>()
                            .withOperationName("CreateChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createChannelRequest));
            CompletableFuture<CreateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to create a dataset. A dataset stores data retrieved from a data store by applying a
     * <code>queryAction</code> (a SQL query) or a <code>containerAction</code> (executing a containerized application).
     * This operation creates the skeleton of a dataset. The dataset can be populated manually by calling
     * <code>CreateDatasetContent</code> or automatically according to a trigger you specify.
     * </p>
     *
     * @param createDatasetRequest
     * @return A Java Future containing the result of the CreateDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceAlreadyExistsException A resource with the same name already exists.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The command caused an internal limit to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.CreateDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/CreateDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetResponse> createDataset(CreateDatasetRequest createDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDatasetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetRequest, CreateDatasetResponse>()
                            .withOperationName("CreateDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDatasetRequest));
            CompletableFuture<CreateDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates the content of a dataset by applying a <code>queryAction</code> (a SQL query) or a
     * <code>containerAction</code> (executing a containerized application).
     * </p>
     *
     * @param createDatasetContentRequest
     * @return A Java Future containing the result of the CreateDatasetContent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.CreateDatasetContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/CreateDatasetContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetContentResponse> createDatasetContent(
            CreateDatasetContentRequest createDatasetContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDatasetContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatasetContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetContentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDatasetContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetContentRequest, CreateDatasetContentResponse>()
                            .withOperationName("CreateDatasetContent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDatasetContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDatasetContentRequest));
            CompletableFuture<CreateDatasetContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a data store, which is a repository for messages.
     * </p>
     *
     * @param createDatastoreRequest
     * @return A Java Future containing the result of the CreateDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceAlreadyExistsException A resource with the same name already exists.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The command caused an internal limit to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.CreateDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/CreateDatastore" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatastoreResponse> createDatastore(CreateDatastoreRequest createDatastoreRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDatastoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDatastore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDatastoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatastoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatastoreRequest, CreateDatastoreResponse>()
                            .withOperationName("CreateDatastore").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDatastoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDatastoreRequest));
            CompletableFuture<CreateDatastoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a pipeline. A pipeline consumes messages from a channel and allows you to process the messages before
     * storing them in a data store. You must specify both a <code>channel</code> and a <code>datastore</code> activity
     * and, optionally, as many as 23 additional activities in the <code>pipelineActivities</code> array.
     * </p>
     *
     * @param createPipelineRequest
     * @return A Java Future containing the result of the CreatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceAlreadyExistsException A resource with the same name already exists.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The command caused an internal limit to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.CreatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/CreatePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePipelineResponse> createPipeline(CreatePipelineRequest createPipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePipelineRequest, CreatePipelineResponse>()
                            .withOperationName("CreatePipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPipelineRequest));
            CompletableFuture<CreatePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified channel.
     * </p>
     *
     * @param deleteChannelRequest
     * @return A Java Future containing the result of the DeleteChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/DeleteChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelResponse> deleteChannel(DeleteChannelRequest deleteChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelRequest, DeleteChannelResponse>()
                            .withOperationName("DeleteChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteChannelRequest));
            CompletableFuture<DeleteChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified dataset.
     * </p>
     * <p>
     * You do not have to delete the content of the dataset before you perform this operation.
     * </p>
     *
     * @param deleteDatasetRequest
     * @return A Java Future containing the result of the DeleteDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.DeleteDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/DeleteDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetResponse> deleteDataset(DeleteDatasetRequest deleteDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDatasetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                            .withOperationName("DeleteDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDatasetRequest));
            CompletableFuture<DeleteDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the content of the specified dataset.
     * </p>
     *
     * @param deleteDatasetContentRequest
     * @return A Java Future containing the result of the DeleteDatasetContent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.DeleteDatasetContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/DeleteDatasetContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetContentResponse> deleteDatasetContent(
            DeleteDatasetContentRequest deleteDatasetContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDatasetContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatasetContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetContentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDatasetContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetContentRequest, DeleteDatasetContentResponse>()
                            .withOperationName("DeleteDatasetContent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDatasetContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDatasetContentRequest));
            CompletableFuture<DeleteDatasetContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified data store.
     * </p>
     *
     * @param deleteDatastoreRequest
     * @return A Java Future containing the result of the DeleteDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.DeleteDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/DeleteDatastore" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatastoreResponse> deleteDatastore(DeleteDatastoreRequest deleteDatastoreRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDatastoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDatastore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDatastoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatastoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatastoreRequest, DeleteDatastoreResponse>()
                            .withOperationName("DeleteDatastore").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDatastoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDatastoreRequest));
            CompletableFuture<DeleteDatastoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified pipeline.
     * </p>
     *
     * @param deletePipelineRequest
     * @return A Java Future containing the result of the DeletePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.DeletePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/DeletePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePipelineResponse> deletePipeline(DeletePipelineRequest deletePipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePipelineRequest, DeletePipelineResponse>()
                            .withOperationName("DeletePipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeletePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePipelineRequest));
            CompletableFuture<DeletePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a channel.
     * </p>
     *
     * @param describeChannelRequest
     * @return A Java Future containing the result of the DescribeChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.DescribeChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/DescribeChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelResponse> describeChannel(DescribeChannelRequest describeChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeChannelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelRequest, DescribeChannelResponse>()
                            .withOperationName("DescribeChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeChannelRequest));
            CompletableFuture<DescribeChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a dataset.
     * </p>
     *
     * @param describeDatasetRequest
     * @return A Java Future containing the result of the DescribeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.DescribeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/DescribeDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetResponse> describeDataset(DescribeDatasetRequest describeDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDatasetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                            .withOperationName("DescribeDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDatasetRequest));
            CompletableFuture<DescribeDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a data store.
     * </p>
     *
     * @param describeDatastoreRequest
     * @return A Java Future containing the result of the DescribeDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.DescribeDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/DescribeDatastore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatastoreResponse> describeDatastore(DescribeDatastoreRequest describeDatastoreRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDatastoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDatastore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatastoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatastoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatastoreRequest, DescribeDatastoreResponse>()
                            .withOperationName("DescribeDatastore").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDatastoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDatastoreRequest));
            CompletableFuture<DescribeDatastoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the current settings of the IoT Analytics logging options.
     * </p>
     *
     * @param describeLoggingOptionsRequest
     * @return A Java Future containing the result of the DescribeLoggingOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.DescribeLoggingOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/DescribeLoggingOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLoggingOptionsResponse> describeLoggingOptions(
            DescribeLoggingOptionsRequest describeLoggingOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLoggingOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLoggingOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLoggingOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLoggingOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLoggingOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLoggingOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLoggingOptionsRequest, DescribeLoggingOptionsResponse>()
                            .withOperationName("DescribeLoggingOptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeLoggingOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeLoggingOptionsRequest));
            CompletableFuture<DescribeLoggingOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a pipeline.
     * </p>
     *
     * @param describePipelineRequest
     * @return A Java Future containing the result of the DescribePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.DescribePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/DescribePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePipelineResponse> describePipeline(DescribePipelineRequest describePipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describePipelineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePipelineRequest, DescribePipelineResponse>()
                            .withOperationName("DescribePipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describePipelineRequest));
            CompletableFuture<DescribePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the contents of a dataset as presigned URIs.
     * </p>
     *
     * @param getDatasetContentRequest
     * @return A Java Future containing the result of the GetDatasetContent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.GetDatasetContent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/GetDatasetContent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDatasetContentResponse> getDatasetContent(GetDatasetContentRequest getDatasetContentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDatasetContentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDatasetContentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDatasetContent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDatasetContentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDatasetContentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDatasetContentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDatasetContentRequest, GetDatasetContentResponse>()
                            .withOperationName("GetDatasetContent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDatasetContentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDatasetContentRequest));
            CompletableFuture<GetDatasetContentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of channels.
     * </p>
     *
     * @param listChannelsRequest
     * @return A Java Future containing the result of the ListChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelsResponse> listChannels(ListChannelsRequest listChannelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listChannelsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                            .withOperationName("ListChannels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listChannelsRequest));
            CompletableFuture<ListChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists information about dataset contents that have been created.
     * </p>
     *
     * @param listDatasetContentsRequest
     * @return A Java Future containing the result of the ListDatasetContents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.ListDatasetContents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/ListDatasetContents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetContentsResponse> listDatasetContents(
            ListDatasetContentsRequest listDatasetContentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDatasetContentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetContentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasetContents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetContentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDatasetContentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetContentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetContentsRequest, ListDatasetContentsResponse>()
                            .withOperationName("ListDatasetContents").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDatasetContentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDatasetContentsRequest));
            CompletableFuture<ListDatasetContentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about datasets.
     * </p>
     *
     * @param listDatasetsRequest
     * @return A Java Future containing the result of the ListDatasets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.ListDatasets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/ListDatasets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetsResponse> listDatasets(ListDatasetsRequest listDatasetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDatasetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDatasetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetsRequest, ListDatasetsResponse>()
                            .withOperationName("ListDatasets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDatasetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDatasetsRequest));
            CompletableFuture<ListDatasetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of data stores.
     * </p>
     *
     * @param listDatastoresRequest
     * @return A Java Future containing the result of the ListDatastores operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.ListDatastores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/ListDatastores" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatastoresResponse> listDatastores(ListDatastoresRequest listDatastoresRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDatastoresRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatastoresRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatastores");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatastoresResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDatastoresResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatastoresResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatastoresRequest, ListDatastoresResponse>()
                            .withOperationName("ListDatastores").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDatastoresRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDatastoresRequest));
            CompletableFuture<ListDatastoresResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of pipelines.
     * </p>
     *
     * @param listPipelinesRequest
     * @return A Java Future containing the result of the ListPipelines operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.ListPipelines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/ListPipelines" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPipelinesResponse> listPipelines(ListPipelinesRequest listPipelinesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPipelinesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipelinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipelines");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPipelinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPipelinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPipelinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPipelinesRequest, ListPipelinesResponse>()
                            .withOperationName("ListPipelines").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListPipelinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPipelinesRequest));
            CompletableFuture<ListPipelinesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags (metadata) that you have assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The command caused an internal limit to be exceeded.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets or updates the IoT Analytics logging options.
     * </p>
     * <p>
     * If you update the value of any <code>loggingOptions</code> field, it takes up to one minute for the change to
     * take effect. Also, if you change the policy attached to the role you specified in the <code>roleArn</code> field
     * (for example, to correct an invalid policy), it takes up to five minutes for that change to take effect.
     * </p>
     *
     * @param putLoggingOptionsRequest
     * @return A Java Future containing the result of the PutLoggingOptions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.PutLoggingOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/PutLoggingOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutLoggingOptionsResponse> putLoggingOptions(PutLoggingOptionsRequest putLoggingOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putLoggingOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putLoggingOptionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutLoggingOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutLoggingOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutLoggingOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutLoggingOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutLoggingOptionsRequest, PutLoggingOptionsResponse>()
                            .withOperationName("PutLoggingOptions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutLoggingOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putLoggingOptionsRequest));
            CompletableFuture<PutLoggingOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Simulates the results of running a pipeline activity on a message payload.
     * </p>
     *
     * @param runPipelineActivityRequest
     * @return A Java Future containing the result of the RunPipelineActivity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.RunPipelineActivity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/RunPipelineActivity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RunPipelineActivityResponse> runPipelineActivity(
            RunPipelineActivityRequest runPipelineActivityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(runPipelineActivityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, runPipelineActivityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RunPipelineActivity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RunPipelineActivityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RunPipelineActivityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RunPipelineActivityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RunPipelineActivityRequest, RunPipelineActivityResponse>()
                            .withOperationName("RunPipelineActivity").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RunPipelineActivityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(runPipelineActivityRequest));
            CompletableFuture<RunPipelineActivityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a sample of messages from the specified channel ingested during the specified timeframe. Up to 10
     * messages can be retrieved.
     * </p>
     *
     * @param sampleChannelDataRequest
     * @return A Java Future containing the result of the SampleChannelData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.SampleChannelData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/SampleChannelData"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SampleChannelDataResponse> sampleChannelData(SampleChannelDataRequest sampleChannelDataRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sampleChannelDataRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sampleChannelDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SampleChannelData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SampleChannelDataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SampleChannelDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SampleChannelDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SampleChannelDataRequest, SampleChannelDataResponse>()
                            .withOperationName("SampleChannelData").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SampleChannelDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sampleChannelDataRequest));
            CompletableFuture<SampleChannelDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the reprocessing of raw message data through the pipeline.
     * </p>
     *
     * @param startPipelineReprocessingRequest
     * @return A Java Future containing the result of the StartPipelineReprocessing operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>ResourceAlreadyExistsException A resource with the same name already exists.</li>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.StartPipelineReprocessing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/StartPipelineReprocessing"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartPipelineReprocessingResponse> startPipelineReprocessing(
            StartPipelineReprocessingRequest startPipelineReprocessingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startPipelineReprocessingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startPipelineReprocessingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartPipelineReprocessing");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartPipelineReprocessingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartPipelineReprocessingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartPipelineReprocessingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartPipelineReprocessingRequest, StartPipelineReprocessingResponse>()
                            .withOperationName("StartPipelineReprocessing").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartPipelineReprocessingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startPipelineReprocessingRequest));
            CompletableFuture<StartPipelineReprocessingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata that can be used to manage a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The command caused an internal limit to be exceeded.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the given tags (metadata) from the resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The command caused an internal limit to be exceeded.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to update the settings of a channel.
     * </p>
     *
     * @param updateChannelRequest
     * @return A Java Future containing the result of the UpdateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/UpdateChannel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelResponse> updateChannel(UpdateChannelRequest updateChannelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateChannelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelRequest, UpdateChannelResponse>()
                            .withOperationName("UpdateChannel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateChannelRequest));
            CompletableFuture<UpdateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the settings of a dataset.
     * </p>
     *
     * @param updateDatasetRequest
     * @return A Java Future containing the result of the UpdateDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.UpdateDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/UpdateDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatasetResponse> updateDataset(UpdateDatasetRequest updateDatasetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDatasetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatasetRequest, UpdateDatasetResponse>()
                            .withOperationName("UpdateDataset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDatasetRequest));
            CompletableFuture<UpdateDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used to update the settings of a data store.
     * </p>
     *
     * @param updateDatastoreRequest
     * @return A Java Future containing the result of the UpdateDatastore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.UpdateDatastore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/UpdateDatastore" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatastoreResponse> updateDatastore(UpdateDatastoreRequest updateDatastoreRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDatastoreRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatastoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatastore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatastoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDatastoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDatastoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatastoreRequest, UpdateDatastoreResponse>()
                            .withOperationName("UpdateDatastore").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDatastoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDatastoreRequest));
            CompletableFuture<UpdateDatastoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the settings of a pipeline. You must specify both a <code>channel</code> and a <code>datastore</code>
     * activity and, optionally, as many as 23 additional activities in the <code>pipelineActivities</code> array.
     * </p>
     *
     * @param updatePipelineRequest
     * @return A Java Future containing the result of the UpdatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request was not valid.</li>
     *         <li>ResourceNotFoundException A resource with the specified name could not be found.</li>
     *         <li>InternalFailureException There was an internal failure.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The command caused an internal limit to be exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IoTAnalyticsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IoTAnalyticsAsyncClient.UpdatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotanalytics-2017-11-27/UpdatePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePipelineResponse> updatePipeline(UpdatePipelineRequest updatePipelineRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePipelineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTAnalytics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePipelineRequest, UpdatePipelineResponse>()
                            .withOperationName("UpdatePipeline").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePipelineRequest));
            CompletableFuture<UpdatePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final IoTAnalyticsServiceClientConfiguration serviceClientConfiguration() {
        return new IoTAnalyticsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IoTAnalyticsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        IoTAnalyticsServiceClientConfigurationBuilder serviceConfigBuilder = new IoTAnalyticsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
