/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotdataplane;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotdataplane.model.ConflictException;
import software.amazon.awssdk.services.iotdataplane.model.DeleteThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.DeleteThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.model.GetThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.GetThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.model.InternalFailureException;
import software.amazon.awssdk.services.iotdataplane.model.InvalidRequestException;
import software.amazon.awssdk.services.iotdataplane.model.IotDataPlaneException;
import software.amazon.awssdk.services.iotdataplane.model.ListNamedShadowsForThingRequest;
import software.amazon.awssdk.services.iotdataplane.model.ListNamedShadowsForThingResponse;
import software.amazon.awssdk.services.iotdataplane.model.MethodNotAllowedException;
import software.amazon.awssdk.services.iotdataplane.model.PublishRequest;
import software.amazon.awssdk.services.iotdataplane.model.PublishResponse;
import software.amazon.awssdk.services.iotdataplane.model.RequestEntityTooLargeException;
import software.amazon.awssdk.services.iotdataplane.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotdataplane.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotdataplane.model.ThrottlingException;
import software.amazon.awssdk.services.iotdataplane.model.UnauthorizedException;
import software.amazon.awssdk.services.iotdataplane.model.UnsupportedDocumentEncodingException;
import software.amazon.awssdk.services.iotdataplane.model.UpdateThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.UpdateThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.transform.DeleteThingShadowRequestMarshaller;
import software.amazon.awssdk.services.iotdataplane.transform.GetThingShadowRequestMarshaller;
import software.amazon.awssdk.services.iotdataplane.transform.ListNamedShadowsForThingRequestMarshaller;
import software.amazon.awssdk.services.iotdataplane.transform.PublishRequestMarshaller;
import software.amazon.awssdk.services.iotdataplane.transform.UpdateThingShadowRequestMarshaller;

/**
 * Internal implementation of {@link IotDataPlaneClient}.
 *
 * @see IotDataPlaneClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotDataPlaneClient implements IotDataPlaneClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotDataPlaneClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the shadow for the specified thing.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_DeleteThingShadow.html">DeleteThingShadow</a> in
     * the AWS IoT Developer Guide.
     * </p>
     *
     * @param deleteThingShadowRequest
     *        The input for the DeleteThingShadow operation.
     * @return Result of the DeleteThingShadow operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws UnsupportedDocumentEncodingException
     *         The document encoding is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.DeleteThingShadow
     */
    @Override
    public DeleteThingShadowResponse deleteThingShadow(DeleteThingShadowRequest deleteThingShadowRequest)
            throws ResourceNotFoundException, InvalidRequestException, ThrottlingException, UnauthorizedException,
            ServiceUnavailableException, InternalFailureException, MethodNotAllowedException,
            UnsupportedDocumentEncodingException, AwsServiceException, SdkClientException, IotDataPlaneException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<DeleteThingShadowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteThingShadowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteThingShadowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteThingShadow");

            return clientHandler.execute(new ClientExecutionParams<DeleteThingShadowRequest, DeleteThingShadowResponse>()
                    .withOperationName("DeleteThingShadow").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteThingShadowRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteThingShadowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the shadow for the specified thing.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_GetThingShadow.html">GetThingShadow</a> in the AWS
     * IoT Developer Guide.
     * </p>
     *
     * @param getThingShadowRequest
     *        The input for the GetThingShadow operation.
     * @return Result of the GetThingShadow operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws UnsupportedDocumentEncodingException
     *         The document encoding is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.GetThingShadow
     */
    @Override
    public GetThingShadowResponse getThingShadow(GetThingShadowRequest getThingShadowRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, UnauthorizedException, ServiceUnavailableException,
            InternalFailureException, MethodNotAllowedException, UnsupportedDocumentEncodingException, AwsServiceException,
            SdkClientException, IotDataPlaneException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetThingShadowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetThingShadowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getThingShadowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetThingShadow");

            return clientHandler.execute(new ClientExecutionParams<GetThingShadowRequest, GetThingShadowResponse>()
                    .withOperationName("GetThingShadow").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getThingShadowRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetThingShadowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the shadows for the specified thing.
     * </p>
     *
     * @param listNamedShadowsForThingRequest
     * @return Result of the ListNamedShadowsForThing operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.ListNamedShadowsForThing
     */
    @Override
    public ListNamedShadowsForThingResponse listNamedShadowsForThing(
            ListNamedShadowsForThingRequest listNamedShadowsForThingRequest) throws ResourceNotFoundException,
            InvalidRequestException, ThrottlingException, UnauthorizedException, ServiceUnavailableException,
            InternalFailureException, MethodNotAllowedException, AwsServiceException, SdkClientException, IotDataPlaneException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNamedShadowsForThingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNamedShadowsForThingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNamedShadowsForThingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNamedShadowsForThing");

            return clientHandler
                    .execute(new ClientExecutionParams<ListNamedShadowsForThingRequest, ListNamedShadowsForThingResponse>()
                            .withOperationName("ListNamedShadowsForThing").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listNamedShadowsForThingRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListNamedShadowsForThingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Publishes state information.
     * </p>
     * <p>
     * For more information, see <a href="http://docs.aws.amazon.com/iot/latest/developerguide/protocols.html#http">HTTP
     * Protocol</a> in the AWS IoT Developer Guide.
     * </p>
     *
     * @param publishRequest
     *        The input for the Publish operation.
     * @return Result of the Publish operation returned by the service.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.Publish
     */
    @Override
    public PublishResponse publish(PublishRequest publishRequest) throws InternalFailureException, InvalidRequestException,
            UnauthorizedException, MethodNotAllowedException, AwsServiceException, SdkClientException, IotDataPlaneException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PublishResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PublishResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, publishRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Publish");

            return clientHandler.execute(new ClientExecutionParams<PublishRequest, PublishResponse>()
                    .withOperationName("Publish").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(publishRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new PublishRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the shadow for the specified thing.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_UpdateThingShadow.html">UpdateThingShadow</a> in
     * the AWS IoT Developer Guide.
     * </p>
     *
     * @param updateThingShadowRequest
     *        The input for the UpdateThingShadow operation.
     * @return Result of the UpdateThingShadow operation returned by the service.
     * @throws ConflictException
     *         The specified version does not match the version of the document.
     * @throws RequestEntityTooLargeException
     *         The payload exceeds the maximum size allowed.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws UnsupportedDocumentEncodingException
     *         The document encoding is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.UpdateThingShadow
     */
    @Override
    public UpdateThingShadowResponse updateThingShadow(UpdateThingShadowRequest updateThingShadowRequest)
            throws ConflictException, RequestEntityTooLargeException, InvalidRequestException, ThrottlingException,
            UnauthorizedException, ServiceUnavailableException, InternalFailureException, MethodNotAllowedException,
            UnsupportedDocumentEncodingException, AwsServiceException, SdkClientException, IotDataPlaneException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<UpdateThingShadowResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateThingShadowResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateThingShadowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateThingShadow");

            return clientHandler.execute(new ClientExecutionParams<UpdateThingShadowRequest, UpdateThingShadowResponse>()
                    .withOperationName("UpdateThingShadow").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateThingShadowRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateThingShadowRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotDataPlaneException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedDocumentEncodingException")
                                .exceptionBuilderSupplier(UnsupportedDocumentEncodingException::builder).httpStatusCode(415)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MethodNotAllowedException")
                                .exceptionBuilderSupplier(MethodNotAllowedException::builder).httpStatusCode(405).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestEntityTooLargeException")
                                .exceptionBuilderSupplier(RequestEntityTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
