/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotdataplane;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.iotdataplane.model.ConflictException;
import software.amazon.awssdk.services.iotdataplane.model.DeleteThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.DeleteThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.model.GetRetainedMessageRequest;
import software.amazon.awssdk.services.iotdataplane.model.GetRetainedMessageResponse;
import software.amazon.awssdk.services.iotdataplane.model.GetThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.GetThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.model.InternalFailureException;
import software.amazon.awssdk.services.iotdataplane.model.InvalidRequestException;
import software.amazon.awssdk.services.iotdataplane.model.IotDataPlaneException;
import software.amazon.awssdk.services.iotdataplane.model.ListNamedShadowsForThingRequest;
import software.amazon.awssdk.services.iotdataplane.model.ListNamedShadowsForThingResponse;
import software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest;
import software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesResponse;
import software.amazon.awssdk.services.iotdataplane.model.MethodNotAllowedException;
import software.amazon.awssdk.services.iotdataplane.model.PublishRequest;
import software.amazon.awssdk.services.iotdataplane.model.PublishResponse;
import software.amazon.awssdk.services.iotdataplane.model.RequestEntityTooLargeException;
import software.amazon.awssdk.services.iotdataplane.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotdataplane.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotdataplane.model.ThrottlingException;
import software.amazon.awssdk.services.iotdataplane.model.UnauthorizedException;
import software.amazon.awssdk.services.iotdataplane.model.UnsupportedDocumentEncodingException;
import software.amazon.awssdk.services.iotdataplane.model.UpdateThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.UpdateThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesIterable;

/**
 * Service client for accessing AWS IoT Data Plane. This can be created using the static {@link #builder()} method.
 *
 * <fullname>IoT data</fullname>
 * <p>
 * IoT data enables secure, bi-directional communication between Internet-connected things (such as sensors, actuators,
 * embedded devices, or smart appliances) and the Amazon Web Services cloud. It implements a broker for applications and
 * things to publish messages over HTTP (Publish) and retrieve, update, and delete shadows. A shadow is a persistent
 * representation of your things and their state in the Amazon Web Services cloud.
 * </p>
 * <p>
 * Find the endpoint address for actions in IoT data by running this CLI command:
 * </p>
 * <p>
 * <code>aws iot describe-endpoint --endpoint-type iot:Data-ATS</code>
 * </p>
 * <p>
 * The service name used by <a href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Amazon Web
 * ServicesSignature Version 4</a> to sign requests is: <i>iotdevicegateway</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface IotDataPlaneClient extends AwsClient {
    String SERVICE_NAME = "iotdata";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "data-ats.iot";

    /**
     * <p>
     * Deletes the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >DeleteThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_DeleteThingShadow.html">DeleteThingShadow</a> in
     * the IoT Developer Guide.
     * </p>
     *
     * @param deleteThingShadowRequest
     *        The input for the DeleteThingShadow operation.
     * @return Result of the DeleteThingShadow operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws UnsupportedDocumentEncodingException
     *         The document encoding is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.DeleteThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/DeleteThingShadow" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteThingShadowResponse deleteThingShadow(DeleteThingShadowRequest deleteThingShadowRequest)
            throws ResourceNotFoundException, InvalidRequestException, ThrottlingException, UnauthorizedException,
            ServiceUnavailableException, InternalFailureException, MethodNotAllowedException,
            UnsupportedDocumentEncodingException, AwsServiceException, SdkClientException, IotDataPlaneException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >DeleteThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_DeleteThingShadow.html">DeleteThingShadow</a> in
     * the IoT Developer Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteThingShadowRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteThingShadowRequest#builder()}
     * </p>
     *
     * @param deleteThingShadowRequest
     *        A {@link Consumer} that will call methods on {@link DeleteThingShadowRequest.Builder} to create a request.
     *        The input for the DeleteThingShadow operation.
     * @return Result of the DeleteThingShadow operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws UnsupportedDocumentEncodingException
     *         The document encoding is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.DeleteThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/DeleteThingShadow" target="_top">AWS
     *      API Documentation</a>
     */
    default DeleteThingShadowResponse deleteThingShadow(Consumer<DeleteThingShadowRequest.Builder> deleteThingShadowRequest)
            throws ResourceNotFoundException, InvalidRequestException, ThrottlingException, UnauthorizedException,
            ServiceUnavailableException, InternalFailureException, MethodNotAllowedException,
            UnsupportedDocumentEncodingException, AwsServiceException, SdkClientException, IotDataPlaneException {
        return deleteThingShadow(DeleteThingShadowRequest.builder().applyMutation(deleteThingShadowRequest).build());
    }

    /**
     * <p>
     * Gets the details of a single retained message for the specified topic.
     * </p>
     * <p>
     * This action returns the message payload of the retained message, which can incur messaging costs. To list only
     * the topic names of the retained messages, call <a
     * href="/iot/latest/developerguide/API_iotdata_ListRetainedMessages.html">ListRetainedMessages</a>.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleethubfordevicemanagement.html#awsiotfleethubfordevicemanagement-actions-as-permissions"
     * >GetRetainedMessage</a> action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     *
     * @param getRetainedMessageRequest
     *        The input for the GetRetainedMessage operation.
     * @return Result of the GetRetainedMessage operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.GetRetainedMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/GetRetainedMessage" target="_top">AWS
     *      API Documentation</a>
     */
    default GetRetainedMessageResponse getRetainedMessage(GetRetainedMessageRequest getRetainedMessageRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, UnauthorizedException,
            ServiceUnavailableException, InternalFailureException, MethodNotAllowedException, AwsServiceException,
            SdkClientException, IotDataPlaneException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the details of a single retained message for the specified topic.
     * </p>
     * <p>
     * This action returns the message payload of the retained message, which can incur messaging costs. To list only
     * the topic names of the retained messages, call <a
     * href="/iot/latest/developerguide/API_iotdata_ListRetainedMessages.html">ListRetainedMessages</a>.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleethubfordevicemanagement.html#awsiotfleethubfordevicemanagement-actions-as-permissions"
     * >GetRetainedMessage</a> action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRetainedMessageRequest.Builder} avoiding the
     * need to create one manually via {@link GetRetainedMessageRequest#builder()}
     * </p>
     *
     * @param getRetainedMessageRequest
     *        A {@link Consumer} that will call methods on {@link GetRetainedMessageRequest.Builder} to create a
     *        request. The input for the GetRetainedMessage operation.
     * @return Result of the GetRetainedMessage operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.GetRetainedMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/GetRetainedMessage" target="_top">AWS
     *      API Documentation</a>
     */
    default GetRetainedMessageResponse getRetainedMessage(Consumer<GetRetainedMessageRequest.Builder> getRetainedMessageRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, UnauthorizedException,
            ServiceUnavailableException, InternalFailureException, MethodNotAllowedException, AwsServiceException,
            SdkClientException, IotDataPlaneException {
        return getRetainedMessage(GetRetainedMessageRequest.builder().applyMutation(getRetainedMessageRequest).build());
    }

    /**
     * <p>
     * Gets the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_GetThingShadow.html">GetThingShadow</a> in the IoT
     * Developer Guide.
     * </p>
     *
     * @param getThingShadowRequest
     *        The input for the GetThingShadow operation.
     * @return Result of the GetThingShadow operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws UnsupportedDocumentEncodingException
     *         The document encoding is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.GetThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/GetThingShadow" target="_top">AWS API
     *      Documentation</a>
     */
    default GetThingShadowResponse getThingShadow(GetThingShadowRequest getThingShadowRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, UnauthorizedException, ServiceUnavailableException,
            InternalFailureException, MethodNotAllowedException, UnsupportedDocumentEncodingException, AwsServiceException,
            SdkClientException, IotDataPlaneException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_GetThingShadow.html">GetThingShadow</a> in the IoT
     * Developer Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetThingShadowRequest.Builder} avoiding the need to
     * create one manually via {@link GetThingShadowRequest#builder()}
     * </p>
     *
     * @param getThingShadowRequest
     *        A {@link Consumer} that will call methods on {@link GetThingShadowRequest.Builder} to create a request.
     *        The input for the GetThingShadow operation.
     * @return Result of the GetThingShadow operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws UnsupportedDocumentEncodingException
     *         The document encoding is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.GetThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/GetThingShadow" target="_top">AWS API
     *      Documentation</a>
     */
    default GetThingShadowResponse getThingShadow(Consumer<GetThingShadowRequest.Builder> getThingShadowRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, UnauthorizedException,
            ServiceUnavailableException, InternalFailureException, MethodNotAllowedException,
            UnsupportedDocumentEncodingException, AwsServiceException, SdkClientException, IotDataPlaneException {
        return getThingShadow(GetThingShadowRequest.builder().applyMutation(getThingShadowRequest).build());
    }

    /**
     * <p>
     * Lists the shadows for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListNamedShadowsForThing</a> action.
     * </p>
     *
     * @param listNamedShadowsForThingRequest
     * @return Result of the ListNamedShadowsForThing operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.ListNamedShadowsForThing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListNamedShadowsForThing"
     *      target="_top">AWS API Documentation</a>
     */
    default ListNamedShadowsForThingResponse listNamedShadowsForThing(
            ListNamedShadowsForThingRequest listNamedShadowsForThingRequest) throws ResourceNotFoundException,
            InvalidRequestException, ThrottlingException, UnauthorizedException, ServiceUnavailableException,
            InternalFailureException, MethodNotAllowedException, AwsServiceException, SdkClientException, IotDataPlaneException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the shadows for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListNamedShadowsForThing</a> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListNamedShadowsForThingRequest.Builder} avoiding
     * the need to create one manually via {@link ListNamedShadowsForThingRequest#builder()}
     * </p>
     *
     * @param listNamedShadowsForThingRequest
     *        A {@link Consumer} that will call methods on {@link ListNamedShadowsForThingRequest.Builder} to create a
     *        request.
     * @return Result of the ListNamedShadowsForThing operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.ListNamedShadowsForThing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListNamedShadowsForThing"
     *      target="_top">AWS API Documentation</a>
     */
    default ListNamedShadowsForThingResponse listNamedShadowsForThing(
            Consumer<ListNamedShadowsForThingRequest.Builder> listNamedShadowsForThingRequest) throws ResourceNotFoundException,
            InvalidRequestException, ThrottlingException, UnauthorizedException, ServiceUnavailableException,
            InternalFailureException, MethodNotAllowedException, AwsServiceException, SdkClientException, IotDataPlaneException {
        return listNamedShadowsForThing(ListNamedShadowsForThingRequest.builder().applyMutation(listNamedShadowsForThingRequest)
                .build());
    }

    /**
     * <p>
     * Lists summary information about the retained messages stored for the account.
     * </p>
     * <p>
     * This action returns only the topic names of the retained messages. It doesn't return any message payloads.
     * Although this action doesn't return a message payload, it can still incur messaging costs.
     * </p>
     * <p>
     * To get the message payload of a retained message, call <a
     * href="https://docs.aws.amazon.com/iot/latest/developerguide/API_iotdata_GetRetainedMessage.html"
     * >GetRetainedMessage</a> with the topic name of the retained message.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleethubfordevicemanagement.html#awsiotfleethubfordevicemanagement-actions-as-permissions"
     * >ListRetainedMessages</a> action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     *
     * @param listRetainedMessagesRequest
     * @return Result of the ListRetainedMessages operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.ListRetainedMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListRetainedMessages" target="_top">AWS
     *      API Documentation</a>
     */
    default ListRetainedMessagesResponse listRetainedMessages(ListRetainedMessagesRequest listRetainedMessagesRequest)
            throws InvalidRequestException, ThrottlingException, UnauthorizedException, ServiceUnavailableException,
            InternalFailureException, MethodNotAllowedException, AwsServiceException, SdkClientException, IotDataPlaneException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists summary information about the retained messages stored for the account.
     * </p>
     * <p>
     * This action returns only the topic names of the retained messages. It doesn't return any message payloads.
     * Although this action doesn't return a message payload, it can still incur messaging costs.
     * </p>
     * <p>
     * To get the message payload of a retained message, call <a
     * href="https://docs.aws.amazon.com/iot/latest/developerguide/API_iotdata_GetRetainedMessage.html"
     * >GetRetainedMessage</a> with the topic name of the retained message.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleethubfordevicemanagement.html#awsiotfleethubfordevicemanagement-actions-as-permissions"
     * >ListRetainedMessages</a> action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRetainedMessagesRequest.Builder} avoiding the
     * need to create one manually via {@link ListRetainedMessagesRequest#builder()}
     * </p>
     *
     * @param listRetainedMessagesRequest
     *        A {@link Consumer} that will call methods on {@link ListRetainedMessagesRequest.Builder} to create a
     *        request.
     * @return Result of the ListRetainedMessages operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.ListRetainedMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListRetainedMessages" target="_top">AWS
     *      API Documentation</a>
     */
    default ListRetainedMessagesResponse listRetainedMessages(
            Consumer<ListRetainedMessagesRequest.Builder> listRetainedMessagesRequest) throws InvalidRequestException,
            ThrottlingException, UnauthorizedException, ServiceUnavailableException, InternalFailureException,
            MethodNotAllowedException, AwsServiceException, SdkClientException, IotDataPlaneException {
        return listRetainedMessages(ListRetainedMessagesRequest.builder().applyMutation(listRetainedMessagesRequest).build());
    }

    /**
     * <p>
     * Lists summary information about the retained messages stored for the account.
     * </p>
     * <p>
     * This action returns only the topic names of the retained messages. It doesn't return any message payloads.
     * Although this action doesn't return a message payload, it can still incur messaging costs.
     * </p>
     * <p>
     * To get the message payload of a retained message, call <a
     * href="https://docs.aws.amazon.com/iot/latest/developerguide/API_iotdata_GetRetainedMessage.html"
     * >GetRetainedMessage</a> with the topic name of the retained message.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleethubfordevicemanagement.html#awsiotfleethubfordevicemanagement-actions-as-permissions"
     * >ListRetainedMessages</a> action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRetainedMessages(software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesIterable responses = client.listRetainedMessagesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesIterable responses = client
     *             .listRetainedMessagesPaginator(request);
     *     for (software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesIterable responses = client.listRetainedMessagesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRetainedMessages(software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRetainedMessagesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.ListRetainedMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListRetainedMessages" target="_top">AWS
     *      API Documentation</a>
     */
    default ListRetainedMessagesIterable listRetainedMessagesPaginator(ListRetainedMessagesRequest listRetainedMessagesRequest)
            throws InvalidRequestException, ThrottlingException, UnauthorizedException, ServiceUnavailableException,
            InternalFailureException, MethodNotAllowedException, AwsServiceException, SdkClientException, IotDataPlaneException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists summary information about the retained messages stored for the account.
     * </p>
     * <p>
     * This action returns only the topic names of the retained messages. It doesn't return any message payloads.
     * Although this action doesn't return a message payload, it can still incur messaging costs.
     * </p>
     * <p>
     * To get the message payload of a retained message, call <a
     * href="https://docs.aws.amazon.com/iot/latest/developerguide/API_iotdata_GetRetainedMessage.html"
     * >GetRetainedMessage</a> with the topic name of the retained message.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleethubfordevicemanagement.html#awsiotfleethubfordevicemanagement-actions-as-permissions"
     * >ListRetainedMessages</a> action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRetainedMessages(software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesIterable responses = client.listRetainedMessagesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesIterable responses = client
     *             .listRetainedMessagesPaginator(request);
     *     for (software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdataplane.paginators.ListRetainedMessagesIterable responses = client.listRetainedMessagesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRetainedMessages(software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRetainedMessagesRequest.Builder} avoiding the
     * need to create one manually via {@link ListRetainedMessagesRequest#builder()}
     * </p>
     *
     * @param listRetainedMessagesRequest
     *        A {@link Consumer} that will call methods on {@link ListRetainedMessagesRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.ListRetainedMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListRetainedMessages" target="_top">AWS
     *      API Documentation</a>
     */
    default ListRetainedMessagesIterable listRetainedMessagesPaginator(
            Consumer<ListRetainedMessagesRequest.Builder> listRetainedMessagesRequest) throws InvalidRequestException,
            ThrottlingException, UnauthorizedException, ServiceUnavailableException, InternalFailureException,
            MethodNotAllowedException, AwsServiceException, SdkClientException, IotDataPlaneException {
        return listRetainedMessagesPaginator(ListRetainedMessagesRequest.builder().applyMutation(listRetainedMessagesRequest)
                .build());
    }

    /**
     * <p>
     * Publishes an MQTT message.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >Publish</a> action.
     * </p>
     * <p>
     * For more information about MQTT messages, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/mqtt.html">MQTT Protocol</a> in the IoT Developer
     * Guide.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     *
     * @param publishRequest
     *        The input for the Publish operation.
     * @return Result of the Publish operation returned by the service.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.Publish
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/Publish" target="_top">AWS API
     *      Documentation</a>
     */
    default PublishResponse publish(PublishRequest publishRequest) throws InternalFailureException, InvalidRequestException,
            UnauthorizedException, MethodNotAllowedException, ThrottlingException, AwsServiceException, SdkClientException,
            IotDataPlaneException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Publishes an MQTT message.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >Publish</a> action.
     * </p>
     * <p>
     * For more information about MQTT messages, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/mqtt.html">MQTT Protocol</a> in the IoT Developer
     * Guide.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PublishRequest.Builder} avoiding the need to create
     * one manually via {@link PublishRequest#builder()}
     * </p>
     *
     * @param publishRequest
     *        A {@link Consumer} that will call methods on {@link PublishRequest.Builder} to create a request. The input
     *        for the Publish operation.
     * @return Result of the Publish operation returned by the service.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.Publish
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/Publish" target="_top">AWS API
     *      Documentation</a>
     */
    default PublishResponse publish(Consumer<PublishRequest.Builder> publishRequest) throws InternalFailureException,
            InvalidRequestException, UnauthorizedException, MethodNotAllowedException, ThrottlingException, AwsServiceException,
            SdkClientException, IotDataPlaneException {
        return publish(PublishRequest.builder().applyMutation(publishRequest).build());
    }

    /**
     * <p>
     * Updates the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >UpdateThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_UpdateThingShadow.html">UpdateThingShadow</a> in
     * the IoT Developer Guide.
     * </p>
     *
     * @param updateThingShadowRequest
     *        The input for the UpdateThingShadow operation.
     * @return Result of the UpdateThingShadow operation returned by the service.
     * @throws ConflictException
     *         The specified version does not match the version of the document.
     * @throws RequestEntityTooLargeException
     *         The payload exceeds the maximum size allowed.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws UnsupportedDocumentEncodingException
     *         The document encoding is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.UpdateThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/UpdateThingShadow" target="_top">AWS
     *      API Documentation</a>
     */
    default UpdateThingShadowResponse updateThingShadow(UpdateThingShadowRequest updateThingShadowRequest)
            throws ConflictException, RequestEntityTooLargeException, InvalidRequestException, ThrottlingException,
            UnauthorizedException, ServiceUnavailableException, InternalFailureException, MethodNotAllowedException,
            UnsupportedDocumentEncodingException, AwsServiceException, SdkClientException, IotDataPlaneException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >UpdateThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_UpdateThingShadow.html">UpdateThingShadow</a> in
     * the IoT Developer Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateThingShadowRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateThingShadowRequest#builder()}
     * </p>
     *
     * @param updateThingShadowRequest
     *        A {@link Consumer} that will call methods on {@link UpdateThingShadowRequest.Builder} to create a request.
     *        The input for the UpdateThingShadow operation.
     * @return Result of the UpdateThingShadow operation returned by the service.
     * @throws ConflictException
     *         The specified version does not match the version of the document.
     * @throws RequestEntityTooLargeException
     *         The payload exceeds the maximum size allowed.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws UnauthorizedException
     *         You are not authorized to perform this operation.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws MethodNotAllowedException
     *         The specified combination of HTTP verb and URI is not supported.
     * @throws UnsupportedDocumentEncodingException
     *         The document encoding is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDataPlaneClient.UpdateThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/UpdateThingShadow" target="_top">AWS
     *      API Documentation</a>
     */
    default UpdateThingShadowResponse updateThingShadow(Consumer<UpdateThingShadowRequest.Builder> updateThingShadowRequest)
            throws ConflictException, RequestEntityTooLargeException, InvalidRequestException, ThrottlingException,
            UnauthorizedException, ServiceUnavailableException, InternalFailureException, MethodNotAllowedException,
            UnsupportedDocumentEncodingException, AwsServiceException, SdkClientException, IotDataPlaneException {
        return updateThingShadow(UpdateThingShadowRequest.builder().applyMutation(updateThingShadowRequest).build());
    }

    /**
     * Create a {@link IotDataPlaneClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static IotDataPlaneClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link IotDataPlaneClient}.
     */
    static IotDataPlaneClientBuilder builder() {
        return new DefaultIotDataPlaneClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default IotDataPlaneServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
