/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotdataplane;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.iotdataplane.internal.IotDataPlaneServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iotdataplane.model.ConflictException;
import software.amazon.awssdk.services.iotdataplane.model.DeleteThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.DeleteThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.model.GetRetainedMessageRequest;
import software.amazon.awssdk.services.iotdataplane.model.GetRetainedMessageResponse;
import software.amazon.awssdk.services.iotdataplane.model.GetThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.GetThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.model.InternalFailureException;
import software.amazon.awssdk.services.iotdataplane.model.InvalidRequestException;
import software.amazon.awssdk.services.iotdataplane.model.IotDataPlaneException;
import software.amazon.awssdk.services.iotdataplane.model.ListNamedShadowsForThingRequest;
import software.amazon.awssdk.services.iotdataplane.model.ListNamedShadowsForThingResponse;
import software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesRequest;
import software.amazon.awssdk.services.iotdataplane.model.ListRetainedMessagesResponse;
import software.amazon.awssdk.services.iotdataplane.model.MethodNotAllowedException;
import software.amazon.awssdk.services.iotdataplane.model.PublishRequest;
import software.amazon.awssdk.services.iotdataplane.model.PublishResponse;
import software.amazon.awssdk.services.iotdataplane.model.RequestEntityTooLargeException;
import software.amazon.awssdk.services.iotdataplane.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotdataplane.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotdataplane.model.ThrottlingException;
import software.amazon.awssdk.services.iotdataplane.model.UnauthorizedException;
import software.amazon.awssdk.services.iotdataplane.model.UnsupportedDocumentEncodingException;
import software.amazon.awssdk.services.iotdataplane.model.UpdateThingShadowRequest;
import software.amazon.awssdk.services.iotdataplane.model.UpdateThingShadowResponse;
import software.amazon.awssdk.services.iotdataplane.transform.DeleteThingShadowRequestMarshaller;
import software.amazon.awssdk.services.iotdataplane.transform.GetRetainedMessageRequestMarshaller;
import software.amazon.awssdk.services.iotdataplane.transform.GetThingShadowRequestMarshaller;
import software.amazon.awssdk.services.iotdataplane.transform.ListNamedShadowsForThingRequestMarshaller;
import software.amazon.awssdk.services.iotdataplane.transform.ListRetainedMessagesRequestMarshaller;
import software.amazon.awssdk.services.iotdataplane.transform.PublishRequestMarshaller;
import software.amazon.awssdk.services.iotdataplane.transform.UpdateThingShadowRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IotDataPlaneAsyncClient}.
 *
 * @see IotDataPlaneAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotDataPlaneAsyncClient implements IotDataPlaneAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIotDataPlaneAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotDataPlaneAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >DeleteThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_DeleteThingShadow.html">DeleteThingShadow</a> in
     * the IoT Developer Guide.
     * </p>
     *
     * @param deleteThingShadowRequest
     *        The input for the DeleteThingShadow operation.
     * @return A Java Future containing the result of the DeleteThingShadow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>UnsupportedDocumentEncodingException The document encoding is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.DeleteThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/DeleteThingShadow" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteThingShadowResponse> deleteThingShadow(DeleteThingShadowRequest deleteThingShadowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteThingShadowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteThingShadowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteThingShadow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<DeleteThingShadowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteThingShadowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteThingShadowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteThingShadowRequest, DeleteThingShadowResponse>()
                            .withOperationName("DeleteThingShadow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteThingShadowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteThingShadowRequest));
            CompletableFuture<DeleteThingShadowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the details of a single retained message for the specified topic.
     * </p>
     * <p>
     * This action returns the message payload of the retained message, which can incur messaging costs. To list only
     * the topic names of the retained messages, call <a
     * href="https://docs.aws.amazon.com/iot/latest/apireference/API_iotdata_ListRetainedMessages.html"
     * >ListRetainedMessages</a>.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleethubfordevicemanagement.html#awsiotfleethubfordevicemanagement-actions-as-permissions"
     * >GetRetainedMessage</a> action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     *
     * @param getRetainedMessageRequest
     *        The input for the GetRetainedMessage operation.
     * @return A Java Future containing the result of the GetRetainedMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.GetRetainedMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/GetRetainedMessage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRetainedMessageResponse> getRetainedMessage(GetRetainedMessageRequest getRetainedMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRetainedMessageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRetainedMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRetainedMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRetainedMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRetainedMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRetainedMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRetainedMessageRequest, GetRetainedMessageResponse>()
                            .withOperationName("GetRetainedMessage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRetainedMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRetainedMessageRequest));
            CompletableFuture<GetRetainedMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_GetThingShadow.html">GetThingShadow</a> in the IoT
     * Developer Guide.
     * </p>
     *
     * @param getThingShadowRequest
     *        The input for the GetThingShadow operation.
     * @return A Java Future containing the result of the GetThingShadow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>UnsupportedDocumentEncodingException The document encoding is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.GetThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/GetThingShadow" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetThingShadowResponse> getThingShadow(GetThingShadowRequest getThingShadowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getThingShadowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getThingShadowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetThingShadow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetThingShadowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetThingShadowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetThingShadowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetThingShadowRequest, GetThingShadowResponse>()
                            .withOperationName("GetThingShadow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetThingShadowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getThingShadowRequest));
            CompletableFuture<GetThingShadowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the shadows for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListNamedShadowsForThing</a> action.
     * </p>
     *
     * @param listNamedShadowsForThingRequest
     * @return A Java Future containing the result of the ListNamedShadowsForThing operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.ListNamedShadowsForThing
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListNamedShadowsForThing"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNamedShadowsForThingResponse> listNamedShadowsForThing(
            ListNamedShadowsForThingRequest listNamedShadowsForThingRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listNamedShadowsForThingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNamedShadowsForThingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNamedShadowsForThing");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNamedShadowsForThingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNamedShadowsForThingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNamedShadowsForThingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNamedShadowsForThingRequest, ListNamedShadowsForThingResponse>()
                            .withOperationName("ListNamedShadowsForThing").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListNamedShadowsForThingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listNamedShadowsForThingRequest));
            CompletableFuture<ListNamedShadowsForThingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists summary information about the retained messages stored for the account.
     * </p>
     * <p>
     * This action returns only the topic names of the retained messages. It doesn't return any message payloads.
     * Although this action doesn't return a message payload, it can still incur messaging costs.
     * </p>
     * <p>
     * To get the message payload of a retained message, call <a
     * href="https://docs.aws.amazon.com/iot/latest/apireference/API_iotdata_GetRetainedMessage.html"
     * >GetRetainedMessage</a> with the topic name of the retained message.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleethubfordevicemanagement.html#awsiotfleethubfordevicemanagement-actions-as-permissions"
     * >ListRetainedMessages</a> action.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     *
     * @param listRetainedMessagesRequest
     * @return A Java Future containing the result of the ListRetainedMessages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.ListRetainedMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/ListRetainedMessages" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRetainedMessagesResponse> listRetainedMessages(
            ListRetainedMessagesRequest listRetainedMessagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRetainedMessagesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRetainedMessagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRetainedMessages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRetainedMessagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRetainedMessagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRetainedMessagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRetainedMessagesRequest, ListRetainedMessagesResponse>()
                            .withOperationName("ListRetainedMessages").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRetainedMessagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRetainedMessagesRequest));
            CompletableFuture<ListRetainedMessagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Publishes an MQTT message.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >Publish</a> action.
     * </p>
     * <p>
     * For more information about MQTT messages, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/mqtt.html">MQTT Protocol</a> in the IoT Developer
     * Guide.
     * </p>
     * <p>
     * For more information about messaging costs, see <a
     * href="http://aws.amazon.com/iot-core/pricing/#Messaging">Amazon Web Services IoT Core pricing - Messaging</a>.
     * </p>
     *
     * @param publishRequest
     *        The input for the Publish operation.
     * @return A Java Future containing the result of the Publish operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.Publish
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/Publish" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PublishResponse> publish(PublishRequest publishRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(publishRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, publishRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Publish");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PublishResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PublishResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PublishResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PublishRequest, PublishResponse>().withOperationName("Publish")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new PublishRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(publishRequest));
            CompletableFuture<PublishResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the shadow for the specified thing.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >UpdateThingShadow</a> action.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/iot/latest/developerguide/API_UpdateThingShadow.html">UpdateThingShadow</a> in
     * the IoT Developer Guide.
     * </p>
     *
     * @param updateThingShadowRequest
     *        The input for the UpdateThingShadow operation.
     * @return A Java Future containing the result of the UpdateThingShadow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException The specified version does not match the version of the document.</li>
     *         <li>RequestEntityTooLargeException The payload exceeds the maximum size allowed.</li>
     *         <li>InvalidRequestException The request is not valid.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>UnauthorizedException You are not authorized to perform this operation.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>InternalFailureException An unexpected error has occurred.</li>
     *         <li>MethodNotAllowedException The specified combination of HTTP verb and URI is not supported.</li>
     *         <li>UnsupportedDocumentEncodingException The document encoding is not supported.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotDataPlaneAsyncClient.UpdateThingShadow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-data-2015-05-28/UpdateThingShadow" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateThingShadowResponse> updateThingShadow(UpdateThingShadowRequest updateThingShadowRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateThingShadowRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateThingShadowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateThingShadow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<UpdateThingShadowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateThingShadowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateThingShadowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateThingShadowRequest, UpdateThingShadowResponse>()
                            .withOperationName("UpdateThingShadow").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateThingShadowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateThingShadowRequest));
            CompletableFuture<UpdateThingShadowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final IotDataPlaneServiceClientConfiguration serviceClientConfiguration() {
        return new IotDataPlaneServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotDataPlaneException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestEntityTooLargeException")
                                .exceptionBuilderSupplier(RequestEntityTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedDocumentEncodingException")
                                .exceptionBuilderSupplier(UnsupportedDocumentEncodingException::builder).httpStatusCode(415)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MethodNotAllowedException")
                                .exceptionBuilderSupplier(MethodNotAllowedException::builder).httpStatusCode(405).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        IotDataPlaneServiceClientConfigurationBuilder serviceConfigBuilder = new IotDataPlaneServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
