/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotdeviceadvisor;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ConflictException;
import software.amazon.awssdk.services.iotdeviceadvisor.model.CreateSuiteDefinitionRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.CreateSuiteDefinitionResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.DeleteSuiteDefinitionRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.DeleteSuiteDefinitionResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetEndpointRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetEndpointResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteDefinitionRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteDefinitionResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteRunReportRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteRunReportResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteRunRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.GetSuiteRunResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.InternalServerException;
import software.amazon.awssdk.services.iotdeviceadvisor.model.IotDeviceAdvisorException;
import software.amazon.awssdk.services.iotdeviceadvisor.model.IotDeviceAdvisorRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteDefinitionsRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteDefinitionsResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteRunsRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteRunsResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotdeviceadvisor.model.StartSuiteRunRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.StartSuiteRunResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.StopSuiteRunRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.StopSuiteRunResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.TagResourceRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.TagResourceResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.UntagResourceRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.UntagResourceResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.UpdateSuiteDefinitionRequest;
import software.amazon.awssdk.services.iotdeviceadvisor.model.UpdateSuiteDefinitionResponse;
import software.amazon.awssdk.services.iotdeviceadvisor.model.ValidationException;
import software.amazon.awssdk.services.iotdeviceadvisor.paginators.ListSuiteDefinitionsIterable;
import software.amazon.awssdk.services.iotdeviceadvisor.paginators.ListSuiteRunsIterable;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.CreateSuiteDefinitionRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.DeleteSuiteDefinitionRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.GetEndpointRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.GetSuiteDefinitionRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.GetSuiteRunReportRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.GetSuiteRunRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.ListSuiteDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.ListSuiteRunsRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.StartSuiteRunRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.StopSuiteRunRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotdeviceadvisor.transform.UpdateSuiteDefinitionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IotDeviceAdvisorClient}.
 *
 * @see IotDeviceAdvisorClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotDeviceAdvisorClient implements IotDeviceAdvisorClient {
    private static final Logger log = Logger.loggerFor(DefaultIotDeviceAdvisorClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotDeviceAdvisorClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a Device Advisor test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >CreateSuiteDefinition</a> action.
     * </p>
     *
     * @param createSuiteDefinitionRequest
     * @return Result of the CreateSuiteDefinition operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.CreateSuiteDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/CreateSuiteDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSuiteDefinitionResponse createSuiteDefinition(CreateSuiteDefinitionRequest createSuiteDefinitionRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSuiteDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateSuiteDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSuiteDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSuiteDefinition");

            return clientHandler.execute(new ClientExecutionParams<CreateSuiteDefinitionRequest, CreateSuiteDefinitionResponse>()
                    .withOperationName("CreateSuiteDefinition").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSuiteDefinitionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSuiteDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Device Advisor test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >DeleteSuiteDefinition</a> action.
     * </p>
     *
     * @param deleteSuiteDefinitionRequest
     * @return Result of the DeleteSuiteDefinition operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.DeleteSuiteDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/DeleteSuiteDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSuiteDefinitionResponse deleteSuiteDefinition(DeleteSuiteDefinitionRequest deleteSuiteDefinitionRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSuiteDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteSuiteDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSuiteDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSuiteDefinition");

            return clientHandler.execute(new ClientExecutionParams<DeleteSuiteDefinitionRequest, DeleteSuiteDefinitionResponse>()
                    .withOperationName("DeleteSuiteDefinition").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSuiteDefinitionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSuiteDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Invokes the GetEndpoint operation.
     *
     * @param getEndpointRequest
     * @return Result of the GetEndpoint operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws ResourceNotFoundException
     *         Sends Resource Not Found Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.GetEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/GetEndpoint" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetEndpointResponse getEndpoint(GetEndpointRequest getEndpointRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEndpointResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEndpointResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEndpoint");

            return clientHandler.execute(new ClientExecutionParams<GetEndpointRequest, GetEndpointResponse>()
                    .withOperationName("GetEndpoint").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getEndpointRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEndpointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a Device Advisor test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetSuiteDefinition</a> action.
     * </p>
     *
     * @param getSuiteDefinitionRequest
     * @return Result of the GetSuiteDefinition operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws ResourceNotFoundException
     *         Sends Resource Not Found Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.GetSuiteDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/GetSuiteDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSuiteDefinitionResponse getSuiteDefinition(GetSuiteDefinitionRequest getSuiteDefinitionRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSuiteDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSuiteDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSuiteDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSuiteDefinition");

            return clientHandler.execute(new ClientExecutionParams<GetSuiteDefinitionRequest, GetSuiteDefinitionResponse>()
                    .withOperationName("GetSuiteDefinition").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSuiteDefinitionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSuiteDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a Device Advisor test suite run.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetSuiteRun</a> action.
     * </p>
     *
     * @param getSuiteRunRequest
     * @return Result of the GetSuiteRun operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws ResourceNotFoundException
     *         Sends Resource Not Found Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.GetSuiteRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/GetSuiteRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSuiteRunResponse getSuiteRun(GetSuiteRunRequest getSuiteRunRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSuiteRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSuiteRunResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSuiteRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSuiteRun");

            return clientHandler.execute(new ClientExecutionParams<GetSuiteRunRequest, GetSuiteRunResponse>()
                    .withOperationName("GetSuiteRun").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSuiteRunRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSuiteRunRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a report download link for a successful Device Advisor qualifying test suite run.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >GetSuiteRunReport</a> action.
     * </p>
     *
     * @param getSuiteRunReportRequest
     * @return Result of the GetSuiteRunReport operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws ResourceNotFoundException
     *         Sends Resource Not Found Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.GetSuiteRunReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/GetSuiteRunReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSuiteRunReportResponse getSuiteRunReport(GetSuiteRunReportRequest getSuiteRunReportRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSuiteRunReportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSuiteRunReportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSuiteRunReportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSuiteRunReport");

            return clientHandler.execute(new ClientExecutionParams<GetSuiteRunReportRequest, GetSuiteRunReportResponse>()
                    .withOperationName("GetSuiteRunReport").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getSuiteRunReportRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSuiteRunReportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Device Advisor test suites you have created.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListSuiteDefinitions</a> action.
     * </p>
     *
     * @param listSuiteDefinitionsRequest
     * @return Result of the ListSuiteDefinitions operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.ListSuiteDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/ListSuiteDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSuiteDefinitionsResponse listSuiteDefinitions(ListSuiteDefinitionsRequest listSuiteDefinitionsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSuiteDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSuiteDefinitionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSuiteDefinitionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSuiteDefinitions");

            return clientHandler.execute(new ClientExecutionParams<ListSuiteDefinitionsRequest, ListSuiteDefinitionsResponse>()
                    .withOperationName("ListSuiteDefinitions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSuiteDefinitionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSuiteDefinitionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Device Advisor test suites you have created.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListSuiteDefinitions</a> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSuiteDefinitions(software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteDefinitionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdeviceadvisor.paginators.ListSuiteDefinitionsIterable responses = client.listSuiteDefinitionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.iotdeviceadvisor.paginators.ListSuiteDefinitionsIterable responses = client
     *             .listSuiteDefinitionsPaginator(request);
     *     for (software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteDefinitionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdeviceadvisor.paginators.ListSuiteDefinitionsIterable responses = client.listSuiteDefinitionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSuiteDefinitions(software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSuiteDefinitionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.ListSuiteDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/ListSuiteDefinitions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSuiteDefinitionsIterable listSuiteDefinitionsPaginator(ListSuiteDefinitionsRequest listSuiteDefinitionsRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            IotDeviceAdvisorException {
        return new ListSuiteDefinitionsIterable(this, applyPaginatorUserAgent(listSuiteDefinitionsRequest));
    }

    /**
     * <p>
     * Lists the runs of the specified Device Advisor test suite. You can list all runs of the test suite, or the runs
     * of a specific version of the test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListSuiteRuns</a> action.
     * </p>
     *
     * @param listSuiteRunsRequest
     * @return Result of the ListSuiteRuns operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.ListSuiteRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/ListSuiteRuns"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSuiteRunsResponse listSuiteRuns(ListSuiteRunsRequest listSuiteRunsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSuiteRunsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSuiteRunsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSuiteRunsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSuiteRuns");

            return clientHandler.execute(new ClientExecutionParams<ListSuiteRunsRequest, ListSuiteRunsResponse>()
                    .withOperationName("ListSuiteRuns").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSuiteRunsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSuiteRunsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the runs of the specified Device Advisor test suite. You can list all runs of the test suite, or the runs
     * of a specific version of the test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListSuiteRuns</a> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSuiteRuns(software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteRunsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdeviceadvisor.paginators.ListSuiteRunsIterable responses = client.listSuiteRunsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.iotdeviceadvisor.paginators.ListSuiteRunsIterable responses = client
     *             .listSuiteRunsPaginator(request);
     *     for (software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteRunsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotdeviceadvisor.paginators.ListSuiteRunsIterable responses = client.listSuiteRunsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSuiteRuns(software.amazon.awssdk.services.iotdeviceadvisor.model.ListSuiteRunsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSuiteRunsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.ListSuiteRuns
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/ListSuiteRuns"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSuiteRunsIterable listSuiteRunsPaginator(ListSuiteRunsRequest listSuiteRunsRequest) throws ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, IotDeviceAdvisorException {
        return new ListSuiteRunsIterable(this, applyPaginatorUserAgent(listSuiteRunsRequest));
    }

    /**
     * <p>
     * Lists the tags attached to an IoT Device Advisor resource.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >ListTagsForResource</a> action.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws ResourceNotFoundException
     *         Sends Resource Not Found Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a Device Advisor test suite run.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >StartSuiteRun</a> action.
     * </p>
     *
     * @param startSuiteRunRequest
     * @return Result of the StartSuiteRun operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws ConflictException
     *         Sends Conflict Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.StartSuiteRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/StartSuiteRun"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartSuiteRunResponse startSuiteRun(StartSuiteRunRequest startSuiteRunRequest) throws ValidationException,
            InternalServerException, ConflictException, AwsServiceException, SdkClientException, IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartSuiteRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartSuiteRunResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startSuiteRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartSuiteRun");

            return clientHandler.execute(new ClientExecutionParams<StartSuiteRunRequest, StartSuiteRunResponse>()
                    .withOperationName("StartSuiteRun").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startSuiteRunRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartSuiteRunRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a Device Advisor test suite run that is currently running.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >StopSuiteRun</a> action.
     * </p>
     *
     * @param stopSuiteRunRequest
     * @return Result of the StopSuiteRun operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws ResourceNotFoundException
     *         Sends Resource Not Found Exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.StopSuiteRun
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/StopSuiteRun" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopSuiteRunResponse stopSuiteRun(StopSuiteRunRequest stopSuiteRunRequest) throws ValidationException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException,
            IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopSuiteRunResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopSuiteRunResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopSuiteRunRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopSuiteRun");

            return clientHandler.execute(new ClientExecutionParams<StopSuiteRunRequest, StopSuiteRunResponse>()
                    .withOperationName("StopSuiteRun").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopSuiteRunRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopSuiteRunRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds to and modifies existing tags of an IoT Device Advisor resource.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >TagResource</a> action.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws ResourceNotFoundException
     *         Sends Resource Not Found Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from an IoT Device Advisor resource.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >UntagResource</a> action.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws ResourceNotFoundException
     *         Sends Resource Not Found Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a Device Advisor test suite.
     * </p>
     * <p>
     * Requires permission to access the <a href=
     * "https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiot.html#awsiot-actions-as-permissions"
     * >UpdateSuiteDefinition</a> action.
     * </p>
     *
     * @param updateSuiteDefinitionRequest
     * @return Result of the UpdateSuiteDefinition operation returned by the service.
     * @throws ValidationException
     *         Sends invalid request exception.
     * @throws InternalServerException
     *         Sends Internal Failure Exception.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotDeviceAdvisorException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotDeviceAdvisorClient.UpdateSuiteDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotdeviceadvisor-2020-09-18/UpdateSuiteDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateSuiteDefinitionResponse updateSuiteDefinition(UpdateSuiteDefinitionRequest updateSuiteDefinitionRequest)
            throws ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            IotDeviceAdvisorException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSuiteDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSuiteDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSuiteDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IotDeviceAdvisor");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSuiteDefinition");

            return clientHandler.execute(new ClientExecutionParams<UpdateSuiteDefinitionRequest, UpdateSuiteDefinitionResponse>()
                    .withOperationName("UpdateSuiteDefinition").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateSuiteDefinitionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSuiteDefinitionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotDeviceAdvisorException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends IotDeviceAdvisorRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
