/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotfleethub;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotfleethub.internal.IoTFleetHubServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iotfleethub.model.ConflictException;
import software.amazon.awssdk.services.iotfleethub.model.CreateApplicationRequest;
import software.amazon.awssdk.services.iotfleethub.model.CreateApplicationResponse;
import software.amazon.awssdk.services.iotfleethub.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.iotfleethub.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.iotfleethub.model.DescribeApplicationRequest;
import software.amazon.awssdk.services.iotfleethub.model.DescribeApplicationResponse;
import software.amazon.awssdk.services.iotfleethub.model.InternalFailureException;
import software.amazon.awssdk.services.iotfleethub.model.InvalidRequestException;
import software.amazon.awssdk.services.iotfleethub.model.IoTFleetHubException;
import software.amazon.awssdk.services.iotfleethub.model.LimitExceededException;
import software.amazon.awssdk.services.iotfleethub.model.ListApplicationsRequest;
import software.amazon.awssdk.services.iotfleethub.model.ListApplicationsResponse;
import software.amazon.awssdk.services.iotfleethub.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.iotfleethub.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.iotfleethub.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotfleethub.model.TagResourceRequest;
import software.amazon.awssdk.services.iotfleethub.model.TagResourceResponse;
import software.amazon.awssdk.services.iotfleethub.model.ThrottlingException;
import software.amazon.awssdk.services.iotfleethub.model.UntagResourceRequest;
import software.amazon.awssdk.services.iotfleethub.model.UntagResourceResponse;
import software.amazon.awssdk.services.iotfleethub.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.iotfleethub.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.iotfleethub.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.iotfleethub.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.iotfleethub.transform.DescribeApplicationRequestMarshaller;
import software.amazon.awssdk.services.iotfleethub.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.iotfleethub.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.iotfleethub.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotfleethub.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.iotfleethub.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IoTFleetHubClient}.
 *
 * @see IoTFleetHubClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIoTFleetHubClient implements IoTFleetHubClient {
    private static final Logger log = Logger.loggerFor(DefaultIoTFleetHubClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIoTFleetHubClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a Fleet Hub for AWS IoT Device Management web application.
     * </p>
     * <note>
     * <p>
     * Fleet Hub for AWS IoT Device Management is in public preview and is subject to change.
     * </p>
     * </note>
     *
     * @param createApplicationRequest
     * @return Result of the CreateApplication operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws LimitExceededException
     *         A limit has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTFleetHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTFleetHubClient.CreateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotfleethub-2020-11-03/CreateApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateApplicationResponse createApplication(CreateApplicationRequest createApplicationRequest)
            throws InvalidRequestException, InternalFailureException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, IoTFleetHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTFleetHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");

            return clientHandler.execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                    .withOperationName("CreateApplication").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a Fleet Hub for AWS IoT Device Management web application.
     * </p>
     * <note>
     * <p>
     * Fleet Hub for AWS IoT Device Management is in public preview and is subject to change.
     * </p>
     * </note>
     *
     * @param deleteApplicationRequest
     * @return Result of the DeleteApplication operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTFleetHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTFleetHubClient.DeleteApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotfleethub-2020-11-03/DeleteApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteApplicationResponse deleteApplication(DeleteApplicationRequest deleteApplicationRequest)
            throws InvalidRequestException, ResourceNotFoundException, InternalFailureException, ThrottlingException,
            AwsServiceException, SdkClientException, IoTFleetHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTFleetHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");

            return clientHandler.execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                    .withOperationName("DeleteApplication").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a Fleet Hub for AWS IoT Device Management web application.
     * </p>
     * <note>
     * <p>
     * Fleet Hub for AWS IoT Device Management is in public preview and is subject to change.
     * </p>
     * </note>
     *
     * @param describeApplicationRequest
     * @return Result of the DescribeApplication operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTFleetHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTFleetHubClient.DescribeApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotfleethub-2020-11-03/DescribeApplication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeApplicationResponse describeApplication(DescribeApplicationRequest describeApplicationRequest)
            throws InvalidRequestException, ResourceNotFoundException, InternalFailureException, ThrottlingException,
            AwsServiceException, SdkClientException, IoTFleetHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeApplicationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTFleetHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeApplication");

            return clientHandler.execute(new ClientExecutionParams<DescribeApplicationRequest, DescribeApplicationResponse>()
                    .withOperationName("DescribeApplication").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of Fleet Hub for AWS IoT Device Management web applications for the current account.
     * </p>
     * <note>
     * <p>
     * Fleet Hub for AWS IoT Device Management is in public preview and is subject to change.
     * </p>
     * </note>
     *
     * @param listApplicationsRequest
     * @return Result of the ListApplications operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTFleetHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTFleetHubClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotfleethub-2020-11-03/ListApplications" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListApplicationsResponse listApplications(ListApplicationsRequest listApplicationsRequest)
            throws InvalidRequestException, InternalFailureException, ThrottlingException, AwsServiceException,
            SdkClientException, IoTFleetHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListApplicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listApplicationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTFleetHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");

            return clientHandler.execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                    .withOperationName("ListApplications").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listApplicationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     * <note>
     * <p>
     * Fleet Hub for AWS IoT Device Management is in public preview and is subject to change.
     * </p>
     * </note>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTFleetHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTFleetHubClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotfleethub-2020-11-03/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalFailureException, InvalidRequestException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IoTFleetHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTFleetHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds to or modifies the tags of the specified resource. Tags are metadata which can be used to manage a resource.
     * </p>
     * <note>
     * <p>
     * Fleet Hub for AWS IoT Device Management is in public preview and is subject to change.
     * </p>
     * </note>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTFleetHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTFleetHubClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotfleethub-2020-11-03/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalFailureException,
            InvalidRequestException, ResourceNotFoundException, AwsServiceException, SdkClientException, IoTFleetHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTFleetHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags (metadata) from the resource.
     * </p>
     * <note>
     * <p>
     * Fleet Hub for AWS IoT Device Management is in public preview and is subject to change.
     * </p>
     * </note>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTFleetHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTFleetHubClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotfleethub-2020-11-03/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalFailureException,
            InvalidRequestException, ResourceNotFoundException, AwsServiceException, SdkClientException, IoTFleetHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTFleetHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates information about a Fleet Hub for a AWS IoT Device Management web application.
     * </p>
     * <note>
     * <p>
     * Fleet Hub for AWS IoT Device Management is in public preview and is subject to change.
     * </p>
     * </note>
     *
     * @param updateApplicationRequest
     * @return Result of the UpdateApplication operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws InternalFailureException
     *         An unexpected error has occurred.
     * @throws ConflictException
     *         The request conflicts with the current state of the resource.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IoTFleetHubException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IoTFleetHubClient.UpdateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iotfleethub-2020-11-03/UpdateApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateApplicationResponse updateApplication(UpdateApplicationRequest updateApplicationRequest)
            throws InvalidRequestException, ResourceNotFoundException, InternalFailureException, ConflictException,
            ThrottlingException, AwsServiceException, SdkClientException, IoTFleetHubException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateApplicationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoTFleetHub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplication");

            return clientHandler.execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                    .withOperationName("UpdateApplication").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        IoTFleetHubServiceClientConfigurationBuilder serviceConfigBuilder = new IoTFleetHubServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IoTFleetHubException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalFailureException")
                                .exceptionBuilderSupplier(InternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public final IoTFleetHubServiceClientConfiguration serviceClientConfiguration() {
        return new IoTFleetHubServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
