/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotjobsdataplane.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains data about the state of a job execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobExecutionState implements SdkPojo, Serializable,
        ToCopyableBuilder<JobExecutionState.Builder, JobExecutionState> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(JobExecutionState::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Map<String, String>> STATUS_DETAILS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("statusDetails")
            .getter(getter(JobExecutionState::statusDetails))
            .setter(setter(Builder::statusDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusDetails").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Long> VERSION_NUMBER_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("versionNumber").getter(getter(JobExecutionState::versionNumber)).setter(setter(Builder::versionNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("versionNumber").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            STATUS_DETAILS_FIELD, VERSION_NUMBER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String status;

    private final Map<String, String> statusDetails;

    private final Long versionNumber;

    private JobExecutionState(BuilderImpl builder) {
        this.status = builder.status;
        this.statusDetails = builder.statusDetails;
        this.versionNumber = builder.versionNumber;
    }

    /**
     * <p>
     * The status of the job execution. Can be one of: "QUEUED", "IN_PROGRESS", "FAILED", "SUCCESS", "CANCELED",
     * "REJECTED", or "REMOVED".
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link JobExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the job execution. Can be one of: "QUEUED", "IN_PROGRESS", "FAILED", "SUCCESS", "CANCELED",
     *         "REJECTED", or "REMOVED".
     * @see JobExecutionStatus
     */
    public JobExecutionStatus status() {
        return JobExecutionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the job execution. Can be one of: "QUEUED", "IN_PROGRESS", "FAILED", "SUCCESS", "CANCELED",
     * "REJECTED", or "REMOVED".
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link JobExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the job execution. Can be one of: "QUEUED", "IN_PROGRESS", "FAILED", "SUCCESS", "CANCELED",
     *         "REJECTED", or "REMOVED".
     * @see JobExecutionStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * Returns true if the StatusDetails property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasStatusDetails() {
        return statusDetails != null && !(statusDetails instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A collection of name/value pairs that describe the status of the job execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStatusDetails()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A collection of name/value pairs that describe the status of the job execution.
     */
    public Map<String, String> statusDetails() {
        return statusDetails;
    }

    /**
     * <p>
     * The version of the job execution. Job execution versions are incremented each time they are updated by a device.
     * </p>
     * 
     * @return The version of the job execution. Job execution versions are incremented each time they are updated by a
     *         device.
     */
    public Long versionNumber() {
        return versionNumber;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasStatusDetails() ? statusDetails() : null);
        hashCode = 31 * hashCode + Objects.hashCode(versionNumber());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobExecutionState)) {
            return false;
        }
        JobExecutionState other = (JobExecutionState) obj;
        return Objects.equals(statusAsString(), other.statusAsString()) && hasStatusDetails() == other.hasStatusDetails()
                && Objects.equals(statusDetails(), other.statusDetails())
                && Objects.equals(versionNumber(), other.versionNumber());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("JobExecutionState").add("Status", statusAsString())
                .add("StatusDetails", hasStatusDetails() ? statusDetails() : null).add("VersionNumber", versionNumber()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusDetails":
            return Optional.ofNullable(clazz.cast(statusDetails()));
        case "versionNumber":
            return Optional.ofNullable(clazz.cast(versionNumber()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobExecutionState, T> g) {
        return obj -> g.apply((JobExecutionState) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobExecutionState> {
        /**
         * <p>
         * The status of the job execution. Can be one of: "QUEUED", "IN_PROGRESS", "FAILED", "SUCCESS", "CANCELED",
         * "REJECTED", or "REMOVED".
         * </p>
         * 
         * @param status
         *        The status of the job execution. Can be one of: "QUEUED", "IN_PROGRESS", "FAILED", "SUCCESS",
         *        "CANCELED", "REJECTED", or "REMOVED".
         * @see JobExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobExecutionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the job execution. Can be one of: "QUEUED", "IN_PROGRESS", "FAILED", "SUCCESS", "CANCELED",
         * "REJECTED", or "REMOVED".
         * </p>
         * 
         * @param status
         *        The status of the job execution. Can be one of: "QUEUED", "IN_PROGRESS", "FAILED", "SUCCESS",
         *        "CANCELED", "REJECTED", or "REMOVED".
         * @see JobExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobExecutionStatus
         */
        Builder status(JobExecutionStatus status);

        /**
         * <p>
         * A collection of name/value pairs that describe the status of the job execution.
         * </p>
         * 
         * @param statusDetails
         *        A collection of name/value pairs that describe the status of the job execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(Map<String, String> statusDetails);

        /**
         * <p>
         * The version of the job execution. Job execution versions are incremented each time they are updated by a
         * device.
         * </p>
         * 
         * @param versionNumber
         *        The version of the job execution. Job execution versions are incremented each time they are updated by
         *        a device.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionNumber(Long versionNumber);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private Map<String, String> statusDetails = DefaultSdkAutoConstructMap.getInstance();

        private Long versionNumber;

        private BuilderImpl() {
        }

        private BuilderImpl(JobExecutionState model) {
            status(model.status);
            statusDetails(model.statusDetails);
            versionNumber(model.versionNumber);
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(JobExecutionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Map<String, String> getStatusDetails() {
            if (statusDetails instanceof SdkAutoConstructMap) {
                return null;
            }
            return statusDetails;
        }

        @Override
        public final Builder statusDetails(Map<String, String> statusDetails) {
            this.statusDetails = DetailsMapCopier.copy(statusDetails);
            return this;
        }

        public final void setStatusDetails(Map<String, String> statusDetails) {
            this.statusDetails = DetailsMapCopier.copy(statusDetails);
        }

        public final Long getVersionNumber() {
            return versionNumber;
        }

        @Override
        public final Builder versionNumber(Long versionNumber) {
            this.versionNumber = versionNumber;
            return this;
        }

        public final void setVersionNumber(Long versionNumber) {
            this.versionNumber = versionNumber;
        }

        @Override
        public JobExecutionState build() {
            return new JobExecutionState(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
