/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotjobsdataplane;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotjobsdataplane.model.CertificateValidationException;
import software.amazon.awssdk.services.iotjobsdataplane.model.DescribeJobExecutionRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.DescribeJobExecutionResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.GetPendingJobExecutionsRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.GetPendingJobExecutionsResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.InvalidRequestException;
import software.amazon.awssdk.services.iotjobsdataplane.model.InvalidStateTransitionException;
import software.amazon.awssdk.services.iotjobsdataplane.model.IotJobsDataPlaneException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotjobsdataplane.model.StartNextPendingJobExecutionRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.StartNextPendingJobExecutionResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.TerminalStateException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ThrottlingException;
import software.amazon.awssdk.services.iotjobsdataplane.model.UpdateJobExecutionRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.UpdateJobExecutionResponse;
import software.amazon.awssdk.services.iotjobsdataplane.transform.DescribeJobExecutionRequestMarshaller;
import software.amazon.awssdk.services.iotjobsdataplane.transform.GetPendingJobExecutionsRequestMarshaller;
import software.amazon.awssdk.services.iotjobsdataplane.transform.StartNextPendingJobExecutionRequestMarshaller;
import software.amazon.awssdk.services.iotjobsdataplane.transform.UpdateJobExecutionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IotJobsDataPlaneAsyncClient}.
 *
 * @see IotJobsDataPlaneAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotJobsDataPlaneAsyncClient implements IotJobsDataPlaneAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIotJobsDataPlaneAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotJobsDataPlaneAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Gets details of a job execution.
     * </p>
     *
     * @param describeJobExecutionRequest
     * @return A Java Future containing the result of the DescribeJobExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The contents of the request were invalid. For example, this code is returned
     *         when an UpdateJobExecution request contains invalid status details. The message contains details about
     *         the error.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>CertificateValidationException The certificate is invalid.</li>
     *         <li>TerminalStateException The job is in a terminal state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotJobsDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotJobsDataPlaneAsyncClient.DescribeJobExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/DescribeJobExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeJobExecutionResponse> describeJobExecution(
            DescribeJobExecutionRequest describeJobExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeJobExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeJobExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeJobExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeJobExecutionRequest, DescribeJobExecutionResponse>()
                            .withOperationName("DescribeJobExecution")
                            .withMarshaller(new DescribeJobExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeJobExecutionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeJobExecutionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeJobExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the list of all jobs for a thing that are not in a terminal status.
     * </p>
     *
     * @param getPendingJobExecutionsRequest
     * @return A Java Future containing the result of the GetPendingJobExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The contents of the request were invalid. For example, this code is returned
     *         when an UpdateJobExecution request contains invalid status details. The message contains details about
     *         the error.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>CertificateValidationException The certificate is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotJobsDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotJobsDataPlaneAsyncClient.GetPendingJobExecutions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/GetPendingJobExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPendingJobExecutionsResponse> getPendingJobExecutions(
            GetPendingJobExecutionsRequest getPendingJobExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPendingJobExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPendingJobExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPendingJobExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPendingJobExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPendingJobExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPendingJobExecutionsRequest, GetPendingJobExecutionsResponse>()
                            .withOperationName("GetPendingJobExecutions")
                            .withMarshaller(new GetPendingJobExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPendingJobExecutionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getPendingJobExecutionsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetPendingJobExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets and starts the next pending (status IN_PROGRESS or QUEUED) job execution for a thing.
     * </p>
     *
     * @param startNextPendingJobExecutionRequest
     * @return A Java Future containing the result of the StartNextPendingJobExecution operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The contents of the request were invalid. For example, this code is returned
     *         when an UpdateJobExecution request contains invalid status details. The message contains details about
     *         the error.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>CertificateValidationException The certificate is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotJobsDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotJobsDataPlaneAsyncClient.StartNextPendingJobExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/StartNextPendingJobExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartNextPendingJobExecutionResponse> startNextPendingJobExecution(
            StartNextPendingJobExecutionRequest startNextPendingJobExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startNextPendingJobExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartNextPendingJobExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartNextPendingJobExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartNextPendingJobExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartNextPendingJobExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartNextPendingJobExecutionRequest, StartNextPendingJobExecutionResponse>()
                            .withOperationName("StartNextPendingJobExecution")
                            .withMarshaller(new StartNextPendingJobExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startNextPendingJobExecutionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = startNextPendingJobExecutionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<StartNextPendingJobExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status of a job execution.
     * </p>
     *
     * @param updateJobExecutionRequest
     * @return A Java Future containing the result of the UpdateJobExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException The contents of the request were invalid. For example, this code is returned
     *         when an UpdateJobExecution request contains invalid status details. The message contains details about
     *         the error.</li>
     *         <li>ResourceNotFoundException The specified resource does not exist.</li>
     *         <li>ThrottlingException The rate exceeds the limit.</li>
     *         <li>ServiceUnavailableException The service is temporarily unavailable.</li>
     *         <li>CertificateValidationException The certificate is invalid.</li>
     *         <li>InvalidStateTransitionException An update attempted to change the job execution to a state that is
     *         invalid because of the job execution's current state (for example, an attempt to change a request in
     *         state SUCCESS to state IN_PROGRESS). In this case, the body of the error message also contains the
     *         executionState field.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotJobsDataPlaneException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample IotJobsDataPlaneAsyncClient.UpdateJobExecution
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/UpdateJobExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobExecutionResponse> updateJobExecution(UpdateJobExecutionRequest updateJobExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJobExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateJobExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateJobExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateJobExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobExecutionRequest, UpdateJobExecutionResponse>()
                            .withOperationName("UpdateJobExecution")
                            .withMarshaller(new UpdateJobExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateJobExecutionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateJobExecutionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateJobExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotJobsDataPlaneException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                                .exceptionBuilderSupplier(InvalidStateTransitionException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TerminalStateException")
                                .exceptionBuilderSupplier(TerminalStateException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CertificateValidationException")
                                .exceptionBuilderSupplier(CertificateValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
