/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotjobsdataplane;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.iotjobsdataplane.internal.IotJobsDataPlaneServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.iotjobsdataplane.model.CertificateValidationException;
import software.amazon.awssdk.services.iotjobsdataplane.model.DescribeJobExecutionRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.DescribeJobExecutionResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.GetPendingJobExecutionsRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.GetPendingJobExecutionsResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.InvalidRequestException;
import software.amazon.awssdk.services.iotjobsdataplane.model.InvalidStateTransitionException;
import software.amazon.awssdk.services.iotjobsdataplane.model.IotJobsDataPlaneException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ServiceUnavailableException;
import software.amazon.awssdk.services.iotjobsdataplane.model.StartNextPendingJobExecutionRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.StartNextPendingJobExecutionResponse;
import software.amazon.awssdk.services.iotjobsdataplane.model.TerminalStateException;
import software.amazon.awssdk.services.iotjobsdataplane.model.ThrottlingException;
import software.amazon.awssdk.services.iotjobsdataplane.model.UpdateJobExecutionRequest;
import software.amazon.awssdk.services.iotjobsdataplane.model.UpdateJobExecutionResponse;
import software.amazon.awssdk.services.iotjobsdataplane.transform.DescribeJobExecutionRequestMarshaller;
import software.amazon.awssdk.services.iotjobsdataplane.transform.GetPendingJobExecutionsRequestMarshaller;
import software.amazon.awssdk.services.iotjobsdataplane.transform.StartNextPendingJobExecutionRequestMarshaller;
import software.amazon.awssdk.services.iotjobsdataplane.transform.UpdateJobExecutionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IotJobsDataPlaneClient}.
 *
 * @see IotJobsDataPlaneClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotJobsDataPlaneClient implements IotJobsDataPlaneClient {
    private static final Logger log = Logger.loggerFor(DefaultIotJobsDataPlaneClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotJobsDataPlaneClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Gets details of a job execution.
     * </p>
     *
     * @param describeJobExecutionRequest
     * @return Result of the DescribeJobExecution operation returned by the service.
     * @throws InvalidRequestException
     *         The contents of the request were invalid. For example, this code is returned when an UpdateJobExecution
     *         request contains invalid status details. The message contains details about the error.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws CertificateValidationException
     *         The certificate is invalid.
     * @throws TerminalStateException
     *         The job is in a terminal state.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotJobsDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotJobsDataPlaneClient.DescribeJobExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/DescribeJobExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeJobExecutionResponse describeJobExecution(DescribeJobExecutionRequest describeJobExecutionRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, ServiceUnavailableException,
            CertificateValidationException, TerminalStateException, AwsServiceException, SdkClientException,
            IotJobsDataPlaneException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeJobExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeJobExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeJobExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeJobExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeJobExecution");

            return clientHandler.execute(new ClientExecutionParams<DescribeJobExecutionRequest, DescribeJobExecutionResponse>()
                    .withOperationName("DescribeJobExecution").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeJobExecutionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeJobExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the list of all jobs for a thing that are not in a terminal status.
     * </p>
     *
     * @param getPendingJobExecutionsRequest
     * @return Result of the GetPendingJobExecutions operation returned by the service.
     * @throws InvalidRequestException
     *         The contents of the request were invalid. For example, this code is returned when an UpdateJobExecution
     *         request contains invalid status details. The message contains details about the error.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws CertificateValidationException
     *         The certificate is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotJobsDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotJobsDataPlaneClient.GetPendingJobExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/GetPendingJobExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPendingJobExecutionsResponse getPendingJobExecutions(GetPendingJobExecutionsRequest getPendingJobExecutionsRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, ServiceUnavailableException,
            CertificateValidationException, AwsServiceException, SdkClientException, IotJobsDataPlaneException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPendingJobExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPendingJobExecutionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPendingJobExecutionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPendingJobExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPendingJobExecutions");

            return clientHandler
                    .execute(new ClientExecutionParams<GetPendingJobExecutionsRequest, GetPendingJobExecutionsResponse>()
                            .withOperationName("GetPendingJobExecutions").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getPendingJobExecutionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetPendingJobExecutionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets and starts the next pending (status IN_PROGRESS or QUEUED) job execution for a thing.
     * </p>
     *
     * @param startNextPendingJobExecutionRequest
     * @return Result of the StartNextPendingJobExecution operation returned by the service.
     * @throws InvalidRequestException
     *         The contents of the request were invalid. For example, this code is returned when an UpdateJobExecution
     *         request contains invalid status details. The message contains details about the error.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws CertificateValidationException
     *         The certificate is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotJobsDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotJobsDataPlaneClient.StartNextPendingJobExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/StartNextPendingJobExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartNextPendingJobExecutionResponse startNextPendingJobExecution(
            StartNextPendingJobExecutionRequest startNextPendingJobExecutionRequest) throws InvalidRequestException,
            ResourceNotFoundException, ThrottlingException, ServiceUnavailableException, CertificateValidationException,
            AwsServiceException, SdkClientException, IotJobsDataPlaneException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartNextPendingJobExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartNextPendingJobExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startNextPendingJobExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startNextPendingJobExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartNextPendingJobExecution");

            return clientHandler
                    .execute(new ClientExecutionParams<StartNextPendingJobExecutionRequest, StartNextPendingJobExecutionResponse>()
                            .withOperationName("StartNextPendingJobExecution").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startNextPendingJobExecutionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartNextPendingJobExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the status of a job execution.
     * </p>
     *
     * @param updateJobExecutionRequest
     * @return Result of the UpdateJobExecution operation returned by the service.
     * @throws InvalidRequestException
     *         The contents of the request were invalid. For example, this code is returned when an UpdateJobExecution
     *         request contains invalid status details. The message contains details about the error.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws ThrottlingException
     *         The rate exceeds the limit.
     * @throws ServiceUnavailableException
     *         The service is temporarily unavailable.
     * @throws CertificateValidationException
     *         The certificate is invalid.
     * @throws InvalidStateTransitionException
     *         An update attempted to change the job execution to a state that is invalid because of the job execution's
     *         current state (for example, an attempt to change a request in state SUCCESS to state IN_PROGRESS). In
     *         this case, the body of the error message also contains the executionState field.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IotJobsDataPlaneException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IotJobsDataPlaneClient.UpdateJobExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-jobs-data-2017-09-29/UpdateJobExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateJobExecutionResponse updateJobExecution(UpdateJobExecutionRequest updateJobExecutionRequest)
            throws InvalidRequestException, ResourceNotFoundException, ThrottlingException, ServiceUnavailableException,
            CertificateValidationException, InvalidStateTransitionException, AwsServiceException, SdkClientException,
            IotJobsDataPlaneException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateJobExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateJobExecutionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateJobExecutionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT Jobs Data Plane");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJobExecution");

            return clientHandler.execute(new ClientExecutionParams<UpdateJobExecutionRequest, UpdateJobExecutionResponse>()
                    .withOperationName("UpdateJobExecution").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateJobExecutionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateJobExecutionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        IotJobsDataPlaneServiceClientConfigurationBuilder serviceConfigBuilder = new IotJobsDataPlaneServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotJobsDataPlaneException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateTransitionException")
                                .exceptionBuilderSupplier(InvalidStateTransitionException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TerminalStateException")
                                .exceptionBuilderSupplier(TerminalStateException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CertificateValidationException")
                                .exceptionBuilderSupplier(CertificateValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public final IotJobsDataPlaneServiceClientConfiguration serviceClientConfiguration() {
        return new IotJobsDataPlaneServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
