/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotmanagedintegrations.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.document.Document;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Structure representing a capability schema item that defines the functionality and features supported by a managed
 * thing.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CapabilitySchemaItem implements SdkPojo, Serializable,
        ToCopyableBuilder<CapabilitySchemaItem.Builder, CapabilitySchemaItem> {
    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Format")
            .getter(getter(CapabilitySchemaItem::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Format").build()).build();

    private static final SdkField<String> CAPABILITY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CapabilityId").getter(getter(CapabilitySchemaItem::capabilityId)).setter(setter(Builder::capabilityId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapabilityId").build()).build();

    private static final SdkField<String> EXTRINSIC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExtrinsicId").getter(getter(CapabilitySchemaItem::extrinsicId)).setter(setter(Builder::extrinsicId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExtrinsicId").build()).build();

    private static final SdkField<Integer> EXTRINSIC_VERSION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ExtrinsicVersion").getter(getter(CapabilitySchemaItem::extrinsicVersion))
            .setter(setter(Builder::extrinsicVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExtrinsicVersion").build()).build();

    private static final SdkField<Document> SCHEMA_FIELD = SdkField.<Document> builder(MarshallingType.DOCUMENT)
            .memberName("Schema").getter(getter(CapabilitySchemaItem::schema)).setter(setter(Builder::schema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schema").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FORMAT_FIELD,
            CAPABILITY_ID_FIELD, EXTRINSIC_ID_FIELD, EXTRINSIC_VERSION_FIELD, SCHEMA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String format;

    private final String capabilityId;

    private final String extrinsicId;

    private final Integer extrinsicVersion;

    private final Document schema;

    private CapabilitySchemaItem(BuilderImpl builder) {
        this.format = builder.format;
        this.capabilityId = builder.capabilityId;
        this.extrinsicId = builder.extrinsicId;
        this.extrinsicVersion = builder.extrinsicVersion;
        this.schema = builder.schema;
    }

    /**
     * <p>
     * The format of the capability schema, which defines how the schema is structured and interpreted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link SchemaVersionFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the capability schema, which defines how the schema is structured and interpreted.
     * @see SchemaVersionFormat
     */
    public final SchemaVersionFormat format() {
        return SchemaVersionFormat.fromValue(format);
    }

    /**
     * <p>
     * The format of the capability schema, which defines how the schema is structured and interpreted.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link SchemaVersionFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the capability schema, which defines how the schema is structured and interpreted.
     * @see SchemaVersionFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * The unique identifier of the capability defined in the schema.
     * </p>
     * 
     * @return The unique identifier of the capability defined in the schema.
     */
    public final String capabilityId() {
        return capabilityId;
    }

    /**
     * <p>
     * The external identifier for the capability, used when referencing the capability outside of the AWS ecosystem.
     * </p>
     * 
     * @return The external identifier for the capability, used when referencing the capability outside of the AWS
     *         ecosystem.
     */
    public final String extrinsicId() {
        return extrinsicId;
    }

    /**
     * <p>
     * The version of the external capability definition, used to track compatibility with external systems.
     * </p>
     * 
     * @return The version of the external capability definition, used to track compatibility with external systems.
     */
    public final Integer extrinsicVersion() {
        return extrinsicVersion;
    }

    /**
     * <p>
     * The actual schema definition that describes the capability's properties, actions, and events.
     * </p>
     * 
     * @return The actual schema definition that describes the capability's properties, actions, and events.
     */
    public final Document schema() {
        return schema;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(capabilityId());
        hashCode = 31 * hashCode + Objects.hashCode(extrinsicId());
        hashCode = 31 * hashCode + Objects.hashCode(extrinsicVersion());
        hashCode = 31 * hashCode + Objects.hashCode(schema());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CapabilitySchemaItem)) {
            return false;
        }
        CapabilitySchemaItem other = (CapabilitySchemaItem) obj;
        return Objects.equals(formatAsString(), other.formatAsString()) && Objects.equals(capabilityId(), other.capabilityId())
                && Objects.equals(extrinsicId(), other.extrinsicId())
                && Objects.equals(extrinsicVersion(), other.extrinsicVersion()) && Objects.equals(schema(), other.schema());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CapabilitySchemaItem").add("Format", formatAsString()).add("CapabilityId", capabilityId())
                .add("ExtrinsicId", extrinsicId()).add("ExtrinsicVersion", extrinsicVersion()).add("Schema", schema()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "CapabilityId":
            return Optional.ofNullable(clazz.cast(capabilityId()));
        case "ExtrinsicId":
            return Optional.ofNullable(clazz.cast(extrinsicId()));
        case "ExtrinsicVersion":
            return Optional.ofNullable(clazz.cast(extrinsicVersion()));
        case "Schema":
            return Optional.ofNullable(clazz.cast(schema()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Format", FORMAT_FIELD);
        map.put("CapabilityId", CAPABILITY_ID_FIELD);
        map.put("ExtrinsicId", EXTRINSIC_ID_FIELD);
        map.put("ExtrinsicVersion", EXTRINSIC_VERSION_FIELD);
        map.put("Schema", SCHEMA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CapabilitySchemaItem, T> g) {
        return obj -> g.apply((CapabilitySchemaItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CapabilitySchemaItem> {
        /**
         * <p>
         * The format of the capability schema, which defines how the schema is structured and interpreted.
         * </p>
         * 
         * @param format
         *        The format of the capability schema, which defines how the schema is structured and interpreted.
         * @see SchemaVersionFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SchemaVersionFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The format of the capability schema, which defines how the schema is structured and interpreted.
         * </p>
         * 
         * @param format
         *        The format of the capability schema, which defines how the schema is structured and interpreted.
         * @see SchemaVersionFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SchemaVersionFormat
         */
        Builder format(SchemaVersionFormat format);

        /**
         * <p>
         * The unique identifier of the capability defined in the schema.
         * </p>
         * 
         * @param capabilityId
         *        The unique identifier of the capability defined in the schema.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilityId(String capabilityId);

        /**
         * <p>
         * The external identifier for the capability, used when referencing the capability outside of the AWS
         * ecosystem.
         * </p>
         * 
         * @param extrinsicId
         *        The external identifier for the capability, used when referencing the capability outside of the AWS
         *        ecosystem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extrinsicId(String extrinsicId);

        /**
         * <p>
         * The version of the external capability definition, used to track compatibility with external systems.
         * </p>
         * 
         * @param extrinsicVersion
         *        The version of the external capability definition, used to track compatibility with external systems.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extrinsicVersion(Integer extrinsicVersion);

        /**
         * <p>
         * The actual schema definition that describes the capability's properties, actions, and events.
         * </p>
         * 
         * @param schema
         *        The actual schema definition that describes the capability's properties, actions, and events.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schema(Document schema);
    }

    static final class BuilderImpl implements Builder {
        private String format;

        private String capabilityId;

        private String extrinsicId;

        private Integer extrinsicVersion;

        private Document schema;

        private BuilderImpl() {
        }

        private BuilderImpl(CapabilitySchemaItem model) {
            format(model.format);
            capabilityId(model.capabilityId);
            extrinsicId(model.extrinsicId);
            extrinsicVersion(model.extrinsicVersion);
            schema(model.schema);
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(SchemaVersionFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final String getCapabilityId() {
            return capabilityId;
        }

        public final void setCapabilityId(String capabilityId) {
            this.capabilityId = capabilityId;
        }

        @Override
        public final Builder capabilityId(String capabilityId) {
            this.capabilityId = capabilityId;
            return this;
        }

        public final String getExtrinsicId() {
            return extrinsicId;
        }

        public final void setExtrinsicId(String extrinsicId) {
            this.extrinsicId = extrinsicId;
        }

        @Override
        public final Builder extrinsicId(String extrinsicId) {
            this.extrinsicId = extrinsicId;
            return this;
        }

        public final Integer getExtrinsicVersion() {
            return extrinsicVersion;
        }

        public final void setExtrinsicVersion(Integer extrinsicVersion) {
            this.extrinsicVersion = extrinsicVersion;
        }

        @Override
        public final Builder extrinsicVersion(Integer extrinsicVersion) {
            this.extrinsicVersion = extrinsicVersion;
            return this;
        }

        public final Document getSchema() {
            return schema;
        }

        public final void setSchema(Document schema) {
            this.schema = schema;
        }

        @Override
        public final Builder schema(Document schema) {
            this.schema = schema;
            return this;
        }

        @Override
        public CapabilitySchemaItem build() {
            return new CapabilitySchemaItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
