/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotroborunner;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.iotroborunner.model.AccessDeniedException;
import software.amazon.awssdk.services.iotroborunner.model.ConflictException;
import software.amazon.awssdk.services.iotroborunner.model.CreateDestinationRequest;
import software.amazon.awssdk.services.iotroborunner.model.CreateDestinationResponse;
import software.amazon.awssdk.services.iotroborunner.model.CreateSiteRequest;
import software.amazon.awssdk.services.iotroborunner.model.CreateSiteResponse;
import software.amazon.awssdk.services.iotroborunner.model.CreateWorkerFleetRequest;
import software.amazon.awssdk.services.iotroborunner.model.CreateWorkerFleetResponse;
import software.amazon.awssdk.services.iotroborunner.model.CreateWorkerRequest;
import software.amazon.awssdk.services.iotroborunner.model.CreateWorkerResponse;
import software.amazon.awssdk.services.iotroborunner.model.DeleteDestinationRequest;
import software.amazon.awssdk.services.iotroborunner.model.DeleteDestinationResponse;
import software.amazon.awssdk.services.iotroborunner.model.DeleteSiteRequest;
import software.amazon.awssdk.services.iotroborunner.model.DeleteSiteResponse;
import software.amazon.awssdk.services.iotroborunner.model.DeleteWorkerFleetRequest;
import software.amazon.awssdk.services.iotroborunner.model.DeleteWorkerFleetResponse;
import software.amazon.awssdk.services.iotroborunner.model.DeleteWorkerRequest;
import software.amazon.awssdk.services.iotroborunner.model.DeleteWorkerResponse;
import software.amazon.awssdk.services.iotroborunner.model.GetDestinationRequest;
import software.amazon.awssdk.services.iotroborunner.model.GetDestinationResponse;
import software.amazon.awssdk.services.iotroborunner.model.GetSiteRequest;
import software.amazon.awssdk.services.iotroborunner.model.GetSiteResponse;
import software.amazon.awssdk.services.iotroborunner.model.GetWorkerFleetRequest;
import software.amazon.awssdk.services.iotroborunner.model.GetWorkerFleetResponse;
import software.amazon.awssdk.services.iotroborunner.model.GetWorkerRequest;
import software.amazon.awssdk.services.iotroborunner.model.GetWorkerResponse;
import software.amazon.awssdk.services.iotroborunner.model.InternalServerException;
import software.amazon.awssdk.services.iotroborunner.model.IotRoboRunnerException;
import software.amazon.awssdk.services.iotroborunner.model.IotRoboRunnerRequest;
import software.amazon.awssdk.services.iotroborunner.model.ListDestinationsRequest;
import software.amazon.awssdk.services.iotroborunner.model.ListDestinationsResponse;
import software.amazon.awssdk.services.iotroborunner.model.ListSitesRequest;
import software.amazon.awssdk.services.iotroborunner.model.ListSitesResponse;
import software.amazon.awssdk.services.iotroborunner.model.ListWorkerFleetsRequest;
import software.amazon.awssdk.services.iotroborunner.model.ListWorkerFleetsResponse;
import software.amazon.awssdk.services.iotroborunner.model.ListWorkersRequest;
import software.amazon.awssdk.services.iotroborunner.model.ListWorkersResponse;
import software.amazon.awssdk.services.iotroborunner.model.ResourceNotFoundException;
import software.amazon.awssdk.services.iotroborunner.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.iotroborunner.model.ThrottlingException;
import software.amazon.awssdk.services.iotroborunner.model.UpdateDestinationRequest;
import software.amazon.awssdk.services.iotroborunner.model.UpdateDestinationResponse;
import software.amazon.awssdk.services.iotroborunner.model.UpdateSiteRequest;
import software.amazon.awssdk.services.iotroborunner.model.UpdateSiteResponse;
import software.amazon.awssdk.services.iotroborunner.model.UpdateWorkerFleetRequest;
import software.amazon.awssdk.services.iotroborunner.model.UpdateWorkerFleetResponse;
import software.amazon.awssdk.services.iotroborunner.model.UpdateWorkerRequest;
import software.amazon.awssdk.services.iotroborunner.model.UpdateWorkerResponse;
import software.amazon.awssdk.services.iotroborunner.model.ValidationException;
import software.amazon.awssdk.services.iotroborunner.paginators.ListDestinationsPublisher;
import software.amazon.awssdk.services.iotroborunner.paginators.ListSitesPublisher;
import software.amazon.awssdk.services.iotroborunner.paginators.ListWorkerFleetsPublisher;
import software.amazon.awssdk.services.iotroborunner.paginators.ListWorkersPublisher;
import software.amazon.awssdk.services.iotroborunner.transform.CreateDestinationRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.CreateSiteRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.CreateWorkerFleetRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.CreateWorkerRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.DeleteDestinationRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.DeleteSiteRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.DeleteWorkerFleetRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.DeleteWorkerRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.GetDestinationRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.GetSiteRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.GetWorkerFleetRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.GetWorkerRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.ListDestinationsRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.ListSitesRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.ListWorkerFleetsRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.ListWorkersRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.UpdateDestinationRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.UpdateSiteRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.UpdateWorkerFleetRequestMarshaller;
import software.amazon.awssdk.services.iotroborunner.transform.UpdateWorkerRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IotRoboRunnerAsyncClient}.
 *
 * @see IotRoboRunnerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIotRoboRunnerAsyncClient implements IotRoboRunnerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIotRoboRunnerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIotRoboRunnerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Grants permission to create a destination
     * </p>
     *
     * @param createDestinationRequest
     * @return A Java Future containing the result of the CreateDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Exception thrown if a resource in a create request already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>ServiceQuotaExceededException Exception thrown if the user's AWS account has reached a service limit
     *         and the operation cannot proceed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.CreateDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/CreateDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDestinationResponse> createDestination(CreateDestinationRequest createDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDestinationRequest, CreateDestinationResponse>()
                            .withOperationName("CreateDestination")
                            .withMarshaller(new CreateDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDestinationRequest));
            CompletableFuture<CreateDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to create a site
     * </p>
     *
     * @param createSiteRequest
     * @return A Java Future containing the result of the CreateSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Exception thrown if a resource in a create request already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>ServiceQuotaExceededException Exception thrown if the user's AWS account has reached a service limit
     *         and the operation cannot proceed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.CreateSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/CreateSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSiteResponse> createSite(CreateSiteRequest createSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSiteRequest, CreateSiteResponse>().withOperationName("CreateSite")
                            .withMarshaller(new CreateSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSiteRequest));
            CompletableFuture<CreateSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to create a worker
     * </p>
     *
     * @param createWorkerRequest
     * @return A Java Future containing the result of the CreateWorker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Exception thrown if a resource in a create request already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>ServiceQuotaExceededException Exception thrown if the user's AWS account has reached a service limit
     *         and the operation cannot proceed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.CreateWorker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/CreateWorker" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkerResponse> createWorker(CreateWorkerRequest createWorkerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateWorkerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkerRequest, CreateWorkerResponse>()
                            .withOperationName("CreateWorker").withMarshaller(new CreateWorkerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createWorkerRequest));
            CompletableFuture<CreateWorkerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to create a worker fleet
     * </p>
     *
     * @param createWorkerFleetRequest
     * @return A Java Future containing the result of the CreateWorkerFleet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Exception thrown if a resource in a create request already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>ServiceQuotaExceededException Exception thrown if the user's AWS account has reached a service limit
     *         and the operation cannot proceed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.CreateWorkerFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/CreateWorkerFleet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkerFleetResponse> createWorkerFleet(CreateWorkerFleetRequest createWorkerFleetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkerFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkerFleet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkerFleetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkerFleetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkerFleetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkerFleetRequest, CreateWorkerFleetResponse>()
                            .withOperationName("CreateWorkerFleet")
                            .withMarshaller(new CreateWorkerFleetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createWorkerFleetRequest));
            CompletableFuture<CreateWorkerFleetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to delete a destination
     * </p>
     *
     * @param deleteDestinationRequest
     * @return A Java Future containing the result of the DeleteDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Exception thrown if a resource in a create request already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.DeleteDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/DeleteDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDestinationResponse> deleteDestination(DeleteDestinationRequest deleteDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDestinationRequest, DeleteDestinationResponse>()
                            .withOperationName("DeleteDestination")
                            .withMarshaller(new DeleteDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDestinationRequest));
            CompletableFuture<DeleteDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to delete a site
     * </p>
     *
     * @param deleteSiteRequest
     * @return A Java Future containing the result of the DeleteSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Exception thrown if a resource in a create request already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.DeleteSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/DeleteSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSiteResponse> deleteSite(DeleteSiteRequest deleteSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSiteRequest, DeleteSiteResponse>().withOperationName("DeleteSite")
                            .withMarshaller(new DeleteSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteSiteRequest));
            CompletableFuture<DeleteSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to delete a worker
     * </p>
     *
     * @param deleteWorkerRequest
     * @return A Java Future containing the result of the DeleteWorker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Exception thrown if a resource in a create request already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.DeleteWorker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/DeleteWorker" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkerResponse> deleteWorker(DeleteWorkerRequest deleteWorkerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteWorkerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkerRequest, DeleteWorkerResponse>()
                            .withOperationName("DeleteWorker").withMarshaller(new DeleteWorkerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteWorkerRequest));
            CompletableFuture<DeleteWorkerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to delete a worker fleet
     * </p>
     *
     * @param deleteWorkerFleetRequest
     * @return A Java Future containing the result of the DeleteWorkerFleet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Exception thrown if a resource in a create request already exists.</li>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.DeleteWorkerFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/DeleteWorkerFleet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkerFleetResponse> deleteWorkerFleet(DeleteWorkerFleetRequest deleteWorkerFleetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkerFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkerFleet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkerFleetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkerFleetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkerFleetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkerFleetRequest, DeleteWorkerFleetResponse>()
                            .withOperationName("DeleteWorkerFleet")
                            .withMarshaller(new DeleteWorkerFleetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteWorkerFleetRequest));
            CompletableFuture<DeleteWorkerFleetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to get a destination
     * </p>
     *
     * @param getDestinationRequest
     * @return A Java Future containing the result of the GetDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.GetDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/GetDestination" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDestinationResponse> getDestination(GetDestinationRequest getDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDestinationRequest, GetDestinationResponse>()
                            .withOperationName("GetDestination")
                            .withMarshaller(new GetDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDestinationRequest));
            CompletableFuture<GetDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to get a site
     * </p>
     *
     * @param getSiteRequest
     * @return A Java Future containing the result of the GetSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.GetSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/GetSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSiteResponse> getSite(GetSiteRequest getSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSiteRequest, GetSiteResponse>().withOperationName("GetSite")
                            .withMarshaller(new GetSiteRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSiteRequest));
            CompletableFuture<GetSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to get a worker
     * </p>
     *
     * @param getWorkerRequest
     * @return A Java Future containing the result of the GetWorker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.GetWorker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/GetWorker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkerResponse> getWorker(GetWorkerRequest getWorkerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetWorkerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetWorkerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkerRequest, GetWorkerResponse>().withOperationName("GetWorker")
                            .withMarshaller(new GetWorkerRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getWorkerRequest));
            CompletableFuture<GetWorkerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to get a worker fleet
     * </p>
     *
     * @param getWorkerFleetRequest
     * @return A Java Future containing the result of the GetWorkerFleet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.GetWorkerFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/GetWorkerFleet" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetWorkerFleetResponse> getWorkerFleet(GetWorkerFleetRequest getWorkerFleetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getWorkerFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetWorkerFleet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetWorkerFleetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetWorkerFleetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetWorkerFleetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetWorkerFleetRequest, GetWorkerFleetResponse>()
                            .withOperationName("GetWorkerFleet")
                            .withMarshaller(new GetWorkerFleetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getWorkerFleetRequest));
            CompletableFuture<GetWorkerFleetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to list destinations
     * </p>
     *
     * @param listDestinationsRequest
     * @return A Java Future containing the result of the ListDestinations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.ListDestinations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDestinationsResponse> listDestinations(ListDestinationsRequest listDestinationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDestinationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDestinations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDestinationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDestinationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDestinationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDestinationsRequest, ListDestinationsResponse>()
                            .withOperationName("ListDestinations")
                            .withMarshaller(new ListDestinationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDestinationsRequest));
            CompletableFuture<ListDestinationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to list destinations
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDestinations(software.amazon.awssdk.services.iotroborunner.model.ListDestinationsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotroborunner.paginators.ListDestinationsPublisher publisher = client.listDestinationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotroborunner.paginators.ListDestinationsPublisher publisher = client.listDestinationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.iotroborunner.model.ListDestinationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.iotroborunner.model.ListDestinationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDestinations(software.amazon.awssdk.services.iotroborunner.model.ListDestinationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDestinationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.ListDestinations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListDestinations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDestinationsPublisher listDestinationsPaginator(ListDestinationsRequest listDestinationsRequest) {
        return new ListDestinationsPublisher(this, applyPaginatorUserAgent(listDestinationsRequest));
    }

    /**
     * <p>
     * Grants permission to list sites
     * </p>
     *
     * @param listSitesRequest
     * @return A Java Future containing the result of the ListSites operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.ListSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSitesResponse> listSites(ListSitesRequest listSitesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSitesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSites");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSitesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSitesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSitesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSitesRequest, ListSitesResponse>().withOperationName("ListSites")
                            .withMarshaller(new ListSitesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSitesRequest));
            CompletableFuture<ListSitesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to list sites
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listSites(software.amazon.awssdk.services.iotroborunner.model.ListSitesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotroborunner.paginators.ListSitesPublisher publisher = client.listSitesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotroborunner.paginators.ListSitesPublisher publisher = client.listSitesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.iotroborunner.model.ListSitesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.iotroborunner.model.ListSitesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSites(software.amazon.awssdk.services.iotroborunner.model.ListSitesRequest)} operation.</b>
     * </p>
     *
     * @param listSitesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.ListSites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListSites" target="_top">AWS API
     *      Documentation</a>
     */
    public ListSitesPublisher listSitesPaginator(ListSitesRequest listSitesRequest) {
        return new ListSitesPublisher(this, applyPaginatorUserAgent(listSitesRequest));
    }

    /**
     * <p>
     * Grants permission to list worker fleets
     * </p>
     *
     * @param listWorkerFleetsRequest
     * @return A Java Future containing the result of the ListWorkerFleets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.ListWorkerFleets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListWorkerFleets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkerFleetsResponse> listWorkerFleets(ListWorkerFleetsRequest listWorkerFleetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkerFleetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkerFleets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkerFleetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkerFleetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkerFleetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkerFleetsRequest, ListWorkerFleetsResponse>()
                            .withOperationName("ListWorkerFleets")
                            .withMarshaller(new ListWorkerFleetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listWorkerFleetsRequest));
            CompletableFuture<ListWorkerFleetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to list worker fleets
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listWorkerFleets(software.amazon.awssdk.services.iotroborunner.model.ListWorkerFleetsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotroborunner.paginators.ListWorkerFleetsPublisher publisher = client.listWorkerFleetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotroborunner.paginators.ListWorkerFleetsPublisher publisher = client.listWorkerFleetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.iotroborunner.model.ListWorkerFleetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.iotroborunner.model.ListWorkerFleetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkerFleets(software.amazon.awssdk.services.iotroborunner.model.ListWorkerFleetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listWorkerFleetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.ListWorkerFleets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListWorkerFleets"
     *      target="_top">AWS API Documentation</a>
     */
    public ListWorkerFleetsPublisher listWorkerFleetsPaginator(ListWorkerFleetsRequest listWorkerFleetsRequest) {
        return new ListWorkerFleetsPublisher(this, applyPaginatorUserAgent(listWorkerFleetsRequest));
    }

    /**
     * <p>
     * Grants permission to list workers
     * </p>
     *
     * @param listWorkersRequest
     * @return A Java Future containing the result of the ListWorkers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.ListWorkers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListWorkers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkersResponse> listWorkers(ListWorkersRequest listWorkersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListWorkersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkersRequest, ListWorkersResponse>()
                            .withOperationName("ListWorkers").withMarshaller(new ListWorkersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listWorkersRequest));
            CompletableFuture<ListWorkersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to list workers
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listWorkers(software.amazon.awssdk.services.iotroborunner.model.ListWorkersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotroborunner.paginators.ListWorkersPublisher publisher = client.listWorkersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.iotroborunner.paginators.ListWorkersPublisher publisher = client.listWorkersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.iotroborunner.model.ListWorkersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.iotroborunner.model.ListWorkersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listWorkers(software.amazon.awssdk.services.iotroborunner.model.ListWorkersRequest)} operation.</b>
     * </p>
     *
     * @param listWorkersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.ListWorkers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/ListWorkers" target="_top">AWS
     *      API Documentation</a>
     */
    public ListWorkersPublisher listWorkersPaginator(ListWorkersRequest listWorkersRequest) {
        return new ListWorkersPublisher(this, applyPaginatorUserAgent(listWorkersRequest));
    }

    /**
     * <p>
     * Grants permission to update a destination
     * </p>
     *
     * @param updateDestinationRequest
     * @return A Java Future containing the result of the UpdateDestination operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.UpdateDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/UpdateDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDestinationResponse> updateDestination(UpdateDestinationRequest updateDestinationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDestination");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDestinationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDestinationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDestinationRequest, UpdateDestinationResponse>()
                            .withOperationName("UpdateDestination")
                            .withMarshaller(new UpdateDestinationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDestinationRequest));
            CompletableFuture<UpdateDestinationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to update a site
     * </p>
     *
     * @param updateSiteRequest
     * @return A Java Future containing the result of the UpdateSite operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.UpdateSite
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/UpdateSite" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSiteResponse> updateSite(UpdateSiteRequest updateSiteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSiteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSite");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSiteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateSiteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSiteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSiteRequest, UpdateSiteResponse>().withOperationName("UpdateSite")
                            .withMarshaller(new UpdateSiteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSiteRequest));
            CompletableFuture<UpdateSiteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to update a worker
     * </p>
     *
     * @param updateWorkerRequest
     * @return A Java Future containing the result of the UpdateWorker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.UpdateWorker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/UpdateWorker" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWorkerResponse> updateWorker(UpdateWorkerRequest updateWorkerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWorkerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateWorkerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWorkerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkerRequest, UpdateWorkerResponse>()
                            .withOperationName("UpdateWorker").withMarshaller(new UpdateWorkerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateWorkerRequest));
            CompletableFuture<UpdateWorkerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants permission to update a worker fleet
     * </p>
     *
     * @param updateWorkerFleetRequest
     * @return A Java Future containing the result of the UpdateWorkerFleet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException User does not have sufficient access to perform this action.</li>
     *         <li>ValidationException Exception thrown if an invalid parameter is provided to an API.</li>
     *         <li>ResourceNotFoundException Exception thrown if a resource referenced in the request doesn't exist.</li>
     *         <li>ThrottlingException Exception thrown if the api has been called too quickly be the client.</li>
     *         <li>InternalServerException Exception thrown if something goes wrong within the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IotRoboRunnerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IotRoboRunnerAsyncClient.UpdateWorkerFleet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/iot-roborunner-2018-05-10/UpdateWorkerFleet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateWorkerFleetResponse> updateWorkerFleet(UpdateWorkerFleetRequest updateWorkerFleetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateWorkerFleetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IoT RoboRunner");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateWorkerFleet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateWorkerFleetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateWorkerFleetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateWorkerFleetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateWorkerFleetRequest, UpdateWorkerFleetResponse>()
                            .withOperationName("UpdateWorkerFleet")
                            .withMarshaller(new UpdateWorkerFleetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateWorkerFleetRequest));
            CompletableFuture<UpdateWorkerFleetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IotRoboRunnerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends IotRoboRunnerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
