/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotsitewise.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains an asset transform property. A transform is a one-to-one mapping of a property's data points from one form
 * to another. For example, you can use a transform to convert a Celsius data stream to Fahrenheit by applying the
 * transformation expression to each data point of the Celsius stream. A transform can only have a data type of
 * <code>DOUBLE</code> and consume properties with data types of <code>INTEGER</code> or <code>DOUBLE</code>.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/asset-properties.html#transforms">Transforms</a> in
 * the <i>IoT SiteWise User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Transform implements SdkPojo, Serializable, ToCopyableBuilder<Transform.Builder, Transform> {
    private static final SdkField<String> EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("expression").getter(getter(Transform::expression)).setter(setter(Builder::expression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("expression").build()).build();

    private static final SdkField<List<ExpressionVariable>> VARIABLES_FIELD = SdkField
            .<List<ExpressionVariable>> builder(MarshallingType.LIST)
            .memberName("variables")
            .getter(getter(Transform::variables))
            .setter(setter(Builder::variables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("variables").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ExpressionVariable> builder(MarshallingType.SDK_POJO)
                                            .constructor(ExpressionVariable::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TransformProcessingConfig> PROCESSING_CONFIG_FIELD = SdkField
            .<TransformProcessingConfig> builder(MarshallingType.SDK_POJO).memberName("processingConfig")
            .getter(getter(Transform::processingConfig)).setter(setter(Builder::processingConfig))
            .constructor(TransformProcessingConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("processingConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPRESSION_FIELD,
            VARIABLES_FIELD, PROCESSING_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String expression;

    private final List<ExpressionVariable> variables;

    private final TransformProcessingConfig processingConfig;

    private Transform(BuilderImpl builder) {
        this.expression = builder.expression;
        this.variables = builder.variables;
        this.processingConfig = builder.processingConfig;
    }

    /**
     * <p>
     * The mathematical expression that defines the transformation function. You can specify up to 10 variables per
     * expression. You can specify up to 10 functions per expression.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>IoT SiteWise
     * User Guide</i>.
     * </p>
     * 
     * @return The mathematical expression that defines the transformation function. You can specify up to 10 variables
     *         per expression. You can specify up to 10 functions per expression. </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>IoT
     *         SiteWise User Guide</i>.
     */
    public final String expression() {
        return expression;
    }

    /**
     * For responses, this returns true if the service returned a value for the Variables property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasVariables() {
        return variables != null && !(variables instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of variables used in the expression.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVariables} method.
     * </p>
     * 
     * @return The list of variables used in the expression.
     */
    public final List<ExpressionVariable> variables() {
        return variables;
    }

    /**
     * <p>
     * The processing configuration for the given transform property. You can configure transforms to be kept at the
     * edge or forwarded to the Amazon Web Services Cloud. You can also configure transforms to be computed at the edge
     * or in the cloud.
     * </p>
     * 
     * @return The processing configuration for the given transform property. You can configure transforms to be kept at
     *         the edge or forwarded to the Amazon Web Services Cloud. You can also configure transforms to be computed
     *         at the edge or in the cloud.
     */
    public final TransformProcessingConfig processingConfig() {
        return processingConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(expression());
        hashCode = 31 * hashCode + Objects.hashCode(hasVariables() ? variables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(processingConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Transform)) {
            return false;
        }
        Transform other = (Transform) obj;
        return Objects.equals(expression(), other.expression()) && hasVariables() == other.hasVariables()
                && Objects.equals(variables(), other.variables()) && Objects.equals(processingConfig(), other.processingConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Transform").add("Expression", expression())
                .add("Variables", hasVariables() ? variables() : null).add("ProcessingConfig", processingConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "expression":
            return Optional.ofNullable(clazz.cast(expression()));
        case "variables":
            return Optional.ofNullable(clazz.cast(variables()));
        case "processingConfig":
            return Optional.ofNullable(clazz.cast(processingConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Transform, T> g) {
        return obj -> g.apply((Transform) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Transform> {
        /**
         * <p>
         * The mathematical expression that defines the transformation function. You can specify up to 10 variables per
         * expression. You can specify up to 10 functions per expression.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>IoT
         * SiteWise User Guide</i>.
         * </p>
         * 
         * @param expression
         *        The mathematical expression that defines the transformation function. You can specify up to 10
         *        variables per expression. You can specify up to 10 functions per expression. </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>IoT
         *        SiteWise User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expression(String expression);

        /**
         * <p>
         * The list of variables used in the expression.
         * </p>
         * 
         * @param variables
         *        The list of variables used in the expression.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variables(Collection<ExpressionVariable> variables);

        /**
         * <p>
         * The list of variables used in the expression.
         * </p>
         * 
         * @param variables
         *        The list of variables used in the expression.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variables(ExpressionVariable... variables);

        /**
         * <p>
         * The list of variables used in the expression.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<ExpressionVariable>.Builder}
         * avoiding the need to create one manually via {@link List<ExpressionVariable>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ExpressionVariable>.Builder#build()} is called immediately
         * and its result is passed to {@link #variables(List<ExpressionVariable>)}.
         * 
         * @param variables
         *        a consumer that will call methods on {@link List<ExpressionVariable>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #variables(List<ExpressionVariable>)
         */
        Builder variables(Consumer<ExpressionVariable.Builder>... variables);

        /**
         * <p>
         * The processing configuration for the given transform property. You can configure transforms to be kept at the
         * edge or forwarded to the Amazon Web Services Cloud. You can also configure transforms to be computed at the
         * edge or in the cloud.
         * </p>
         * 
         * @param processingConfig
         *        The processing configuration for the given transform property. You can configure transforms to be kept
         *        at the edge or forwarded to the Amazon Web Services Cloud. You can also configure transforms to be
         *        computed at the edge or in the cloud.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingConfig(TransformProcessingConfig processingConfig);

        /**
         * <p>
         * The processing configuration for the given transform property. You can configure transforms to be kept at the
         * edge or forwarded to the Amazon Web Services Cloud. You can also configure transforms to be computed at the
         * edge or in the cloud.
         * </p>
         * This is a convenience method that creates an instance of the {@link TransformProcessingConfig.Builder}
         * avoiding the need to create one manually via {@link TransformProcessingConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link TransformProcessingConfig.Builder#build()} is called immediately
         * and its result is passed to {@link #processingConfig(TransformProcessingConfig)}.
         * 
         * @param processingConfig
         *        a consumer that will call methods on {@link TransformProcessingConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #processingConfig(TransformProcessingConfig)
         */
        default Builder processingConfig(Consumer<TransformProcessingConfig.Builder> processingConfig) {
            return processingConfig(TransformProcessingConfig.builder().applyMutation(processingConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String expression;

        private List<ExpressionVariable> variables = DefaultSdkAutoConstructList.getInstance();

        private TransformProcessingConfig processingConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(Transform model) {
            expression(model.expression);
            variables(model.variables);
            processingConfig(model.processingConfig);
        }

        public final String getExpression() {
            return expression;
        }

        public final void setExpression(String expression) {
            this.expression = expression;
        }

        @Override
        public final Builder expression(String expression) {
            this.expression = expression;
            return this;
        }

        public final List<ExpressionVariable.Builder> getVariables() {
            List<ExpressionVariable.Builder> result = ExpressionVariablesCopier.copyToBuilder(this.variables);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVariables(Collection<ExpressionVariable.BuilderImpl> variables) {
            this.variables = ExpressionVariablesCopier.copyFromBuilder(variables);
        }

        @Override
        public final Builder variables(Collection<ExpressionVariable> variables) {
            this.variables = ExpressionVariablesCopier.copy(variables);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variables(ExpressionVariable... variables) {
            variables(Arrays.asList(variables));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variables(Consumer<ExpressionVariable.Builder>... variables) {
            variables(Stream.of(variables).map(c -> ExpressionVariable.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final TransformProcessingConfig.Builder getProcessingConfig() {
            return processingConfig != null ? processingConfig.toBuilder() : null;
        }

        public final void setProcessingConfig(TransformProcessingConfig.BuilderImpl processingConfig) {
            this.processingConfig = processingConfig != null ? processingConfig.build() : null;
        }

        @Override
        public final Builder processingConfig(TransformProcessingConfig processingConfig) {
            this.processingConfig = processingConfig;
            return this;
        }

        @Override
        public Transform build() {
            return new Transform(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
