/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotsitewise.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.iotsitewise.IoTSiteWiseClient;
import software.amazon.awssdk.services.iotsitewise.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetModelRequest;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetModelResponse;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetRequest;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetResponse;
import software.amazon.awssdk.services.iotsitewise.model.DescribePortalRequest;
import software.amazon.awssdk.services.iotsitewise.model.DescribePortalResponse;
import software.amazon.awssdk.services.iotsitewise.model.IoTSiteWiseRequest;
import software.amazon.awssdk.services.iotsitewise.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultIoTSiteWiseWaiter implements IoTSiteWiseWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final IoTSiteWiseClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeAssetModelResponse> assetModelNotExistsWaiter;

    private final Waiter<DescribeAssetModelResponse> assetModelActiveWaiter;

    private final Waiter<DescribeAssetResponse> assetNotExistsWaiter;

    private final Waiter<DescribeAssetResponse> assetActiveWaiter;

    private final Waiter<DescribePortalResponse> portalNotExistsWaiter;

    private final Waiter<DescribePortalResponse> portalActiveWaiter;

    private DefaultIoTSiteWiseWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = IoTSiteWiseClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.assetModelNotExistsWaiter = Waiter.builder(DescribeAssetModelResponse.class)
                .acceptors(assetModelNotExistsWaiterAcceptors())
                .overrideConfiguration(assetModelNotExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.assetModelActiveWaiter = Waiter.builder(DescribeAssetModelResponse.class)
                .acceptors(assetModelActiveWaiterAcceptors())
                .overrideConfiguration(assetModelActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.assetNotExistsWaiter = Waiter.builder(DescribeAssetResponse.class).acceptors(assetNotExistsWaiterAcceptors())
                .overrideConfiguration(assetNotExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.assetActiveWaiter = Waiter.builder(DescribeAssetResponse.class).acceptors(assetActiveWaiterAcceptors())
                .overrideConfiguration(assetActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.portalNotExistsWaiter = Waiter.builder(DescribePortalResponse.class).acceptors(portalNotExistsWaiterAcceptors())
                .overrideConfiguration(portalNotExistsWaiterConfig(builder.overrideConfiguration)).build();
        this.portalActiveWaiter = Waiter.builder(DescribePortalResponse.class).acceptors(portalActiveWaiterAcceptors())
                .overrideConfiguration(portalActiveWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeAssetResponse> waitUntilAssetActive(DescribeAssetRequest describeAssetRequest) {
        return assetActiveWaiter.run(() -> client.describeAsset(applyWaitersUserAgent(describeAssetRequest)));
    }

    @Override
    public WaiterResponse<DescribeAssetResponse> waitUntilAssetActive(DescribeAssetRequest describeAssetRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return assetActiveWaiter.run(() -> client.describeAsset(applyWaitersUserAgent(describeAssetRequest)),
                assetActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeAssetModelResponse> waitUntilAssetModelActive(
            DescribeAssetModelRequest describeAssetModelRequest) {
        return assetModelActiveWaiter.run(() -> client.describeAssetModel(applyWaitersUserAgent(describeAssetModelRequest)));
    }

    @Override
    public WaiterResponse<DescribeAssetModelResponse> waitUntilAssetModelActive(
            DescribeAssetModelRequest describeAssetModelRequest, WaiterOverrideConfiguration overrideConfig) {
        return assetModelActiveWaiter.run(() -> client.describeAssetModel(applyWaitersUserAgent(describeAssetModelRequest)),
                assetModelActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeAssetModelResponse> waitUntilAssetModelNotExists(
            DescribeAssetModelRequest describeAssetModelRequest) {
        return assetModelNotExistsWaiter.run(() -> client.describeAssetModel(applyWaitersUserAgent(describeAssetModelRequest)));
    }

    @Override
    public WaiterResponse<DescribeAssetModelResponse> waitUntilAssetModelNotExists(
            DescribeAssetModelRequest describeAssetModelRequest, WaiterOverrideConfiguration overrideConfig) {
        return assetModelNotExistsWaiter.run(() -> client.describeAssetModel(applyWaitersUserAgent(describeAssetModelRequest)),
                assetModelNotExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeAssetResponse> waitUntilAssetNotExists(DescribeAssetRequest describeAssetRequest) {
        return assetNotExistsWaiter.run(() -> client.describeAsset(applyWaitersUserAgent(describeAssetRequest)));
    }

    @Override
    public WaiterResponse<DescribeAssetResponse> waitUntilAssetNotExists(DescribeAssetRequest describeAssetRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return assetNotExistsWaiter.run(() -> client.describeAsset(applyWaitersUserAgent(describeAssetRequest)),
                assetNotExistsWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribePortalResponse> waitUntilPortalActive(DescribePortalRequest describePortalRequest) {
        return portalActiveWaiter.run(() -> client.describePortal(applyWaitersUserAgent(describePortalRequest)));
    }

    @Override
    public WaiterResponse<DescribePortalResponse> waitUntilPortalActive(DescribePortalRequest describePortalRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return portalActiveWaiter.run(() -> client.describePortal(applyWaitersUserAgent(describePortalRequest)),
                portalActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribePortalResponse> waitUntilPortalNotExists(DescribePortalRequest describePortalRequest) {
        return portalNotExistsWaiter.run(() -> client.describePortal(applyWaitersUserAgent(describePortalRequest)));
    }

    @Override
    public WaiterResponse<DescribePortalResponse> waitUntilPortalNotExists(DescribePortalRequest describePortalRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return portalNotExistsWaiter.run(() -> client.describePortal(applyWaitersUserAgent(describePortalRequest)),
                portalNotExistsWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeAssetModelResponse>> assetModelNotExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAssetModelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAssetModelResponse>> assetModelActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAssetModelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("assetModelStatus").field("state").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("assetModelStatus").field("state").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (assetModelStatus.state=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAssetResponse>> assetNotExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAssetResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAssetResponse>> assetActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAssetResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("assetStatus").field("state").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("assetStatus").field("state").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (assetStatus.state=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribePortalResponse>> portalNotExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribePortalResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribePortalResponse>> portalActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribePortalResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("portalStatus").field("state").value(), "ACTIVE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration assetModelNotExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration assetModelActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration assetNotExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration assetActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration portalNotExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration portalActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static IoTSiteWiseWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends IoTSiteWiseRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements IoTSiteWiseWaiter.Builder {
        private IoTSiteWiseClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public IoTSiteWiseWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public IoTSiteWiseWaiter.Builder client(IoTSiteWiseClient client) {
            this.client = client;
            return this;
        }

        public IoTSiteWiseWaiter build() {
            return new DefaultIoTSiteWiseWaiter(this);
        }
    }
}
