/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotsitewise.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains an asset metric property. With metrics, you can calculate aggregate functions, such as an average, maximum,
 * or minimum, as specified through an expression. A metric maps several values to a single value (such as a sum).
 * </p>
 * <p>
 * The maximum number of dependent/cascading variables used in any one metric calculation is 10. Therefore, a
 * <i>root</i> metric can have up to 10 cascading metrics in its computational dependency tree. Additionally, a metric
 * can only have a data type of <code>DOUBLE</code> and consume properties with data types of <code>INTEGER</code> or
 * <code>DOUBLE</code>.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/asset-properties.html#metrics">Metrics</a> in the
 * <i>IoT SiteWise User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Metric implements SdkPojo, Serializable, ToCopyableBuilder<Metric.Builder, Metric> {
    private static final SdkField<String> EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("expression").getter(getter(Metric::expression)).setter(setter(Builder::expression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("expression").build()).build();

    private static final SdkField<List<ExpressionVariable>> VARIABLES_FIELD = SdkField
            .<List<ExpressionVariable>> builder(MarshallingType.LIST)
            .memberName("variables")
            .getter(getter(Metric::variables))
            .setter(setter(Builder::variables))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("variables").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ExpressionVariable> builder(MarshallingType.SDK_POJO)
                                            .constructor(ExpressionVariable::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<MetricWindow> WINDOW_FIELD = SdkField.<MetricWindow> builder(MarshallingType.SDK_POJO)
            .memberName("window").getter(getter(Metric::window)).setter(setter(Builder::window))
            .constructor(MetricWindow::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("window").build()).build();

    private static final SdkField<MetricProcessingConfig> PROCESSING_CONFIG_FIELD = SdkField
            .<MetricProcessingConfig> builder(MarshallingType.SDK_POJO).memberName("processingConfig")
            .getter(getter(Metric::processingConfig)).setter(setter(Builder::processingConfig))
            .constructor(MetricProcessingConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("processingConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPRESSION_FIELD,
            VARIABLES_FIELD, WINDOW_FIELD, PROCESSING_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String expression;

    private final List<ExpressionVariable> variables;

    private final MetricWindow window;

    private final MetricProcessingConfig processingConfig;

    private Metric(BuilderImpl builder) {
        this.expression = builder.expression;
        this.variables = builder.variables;
        this.window = builder.window;
        this.processingConfig = builder.processingConfig;
    }

    /**
     * <p>
     * The mathematical expression that defines the metric aggregation function. You can specify up to 10 variables per
     * expression. You can specify up to 10 functions per expression.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>IoT SiteWise
     * User Guide</i>.
     * </p>
     * 
     * @return The mathematical expression that defines the metric aggregation function. You can specify up to 10
     *         variables per expression. You can specify up to 10 functions per expression. </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>IoT
     *         SiteWise User Guide</i>.
     */
    public final String expression() {
        return expression;
    }

    /**
     * For responses, this returns true if the service returned a value for the Variables property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasVariables() {
        return variables != null && !(variables instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of variables used in the expression.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVariables} method.
     * </p>
     * 
     * @return The list of variables used in the expression.
     */
    public final List<ExpressionVariable> variables() {
        return variables;
    }

    /**
     * <p>
     * The window (time interval) over which IoT SiteWise computes the metric's aggregation expression. IoT SiteWise
     * computes one data point per <code>window</code>.
     * </p>
     * 
     * @return The window (time interval) over which IoT SiteWise computes the metric's aggregation expression. IoT
     *         SiteWise computes one data point per <code>window</code>.
     */
    public final MetricWindow window() {
        return window;
    }

    /**
     * <p>
     * The processing configuration for the given metric property. You can configure metrics to be computed at the edge
     * or in the Amazon Web Services Cloud. By default, metrics are forwarded to the cloud.
     * </p>
     * 
     * @return The processing configuration for the given metric property. You can configure metrics to be computed at
     *         the edge or in the Amazon Web Services Cloud. By default, metrics are forwarded to the cloud.
     */
    public final MetricProcessingConfig processingConfig() {
        return processingConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(expression());
        hashCode = 31 * hashCode + Objects.hashCode(hasVariables() ? variables() : null);
        hashCode = 31 * hashCode + Objects.hashCode(window());
        hashCode = 31 * hashCode + Objects.hashCode(processingConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Metric)) {
            return false;
        }
        Metric other = (Metric) obj;
        return Objects.equals(expression(), other.expression()) && hasVariables() == other.hasVariables()
                && Objects.equals(variables(), other.variables()) && Objects.equals(window(), other.window())
                && Objects.equals(processingConfig(), other.processingConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Metric").add("Expression", expression()).add("Variables", hasVariables() ? variables() : null)
                .add("Window", window()).add("ProcessingConfig", processingConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "expression":
            return Optional.ofNullable(clazz.cast(expression()));
        case "variables":
            return Optional.ofNullable(clazz.cast(variables()));
        case "window":
            return Optional.ofNullable(clazz.cast(window()));
        case "processingConfig":
            return Optional.ofNullable(clazz.cast(processingConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("expression", EXPRESSION_FIELD);
        map.put("variables", VARIABLES_FIELD);
        map.put("window", WINDOW_FIELD);
        map.put("processingConfig", PROCESSING_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Metric, T> g) {
        return obj -> g.apply((Metric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Metric> {
        /**
         * <p>
         * The mathematical expression that defines the metric aggregation function. You can specify up to 10 variables
         * per expression. You can specify up to 10 functions per expression.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>IoT
         * SiteWise User Guide</i>.
         * </p>
         * 
         * @param expression
         *        The mathematical expression that defines the metric aggregation function. You can specify up to 10
         *        variables per expression. You can specify up to 10 functions per expression. </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/iot-sitewise/latest/userguide/quotas.html">Quotas</a> in the <i>IoT
         *        SiteWise User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expression(String expression);

        /**
         * <p>
         * The list of variables used in the expression.
         * </p>
         * 
         * @param variables
         *        The list of variables used in the expression.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variables(Collection<ExpressionVariable> variables);

        /**
         * <p>
         * The list of variables used in the expression.
         * </p>
         * 
         * @param variables
         *        The list of variables used in the expression.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder variables(ExpressionVariable... variables);

        /**
         * <p>
         * The list of variables used in the expression.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.iotsitewise.model.ExpressionVariable.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.iotsitewise.model.ExpressionVariable#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.iotsitewise.model.ExpressionVariable.Builder#build()} is called
         * immediately and its result is passed to {@link #variables(List<ExpressionVariable>)}.
         * 
         * @param variables
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.iotsitewise.model.ExpressionVariable.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #variables(java.util.Collection<ExpressionVariable>)
         */
        Builder variables(Consumer<ExpressionVariable.Builder>... variables);

        /**
         * <p>
         * The window (time interval) over which IoT SiteWise computes the metric's aggregation expression. IoT SiteWise
         * computes one data point per <code>window</code>.
         * </p>
         * 
         * @param window
         *        The window (time interval) over which IoT SiteWise computes the metric's aggregation expression. IoT
         *        SiteWise computes one data point per <code>window</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder window(MetricWindow window);

        /**
         * <p>
         * The window (time interval) over which IoT SiteWise computes the metric's aggregation expression. IoT SiteWise
         * computes one data point per <code>window</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link MetricWindow.Builder} avoiding the need
         * to create one manually via {@link MetricWindow#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MetricWindow.Builder#build()} is called immediately and its
         * result is passed to {@link #window(MetricWindow)}.
         * 
         * @param window
         *        a consumer that will call methods on {@link MetricWindow.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #window(MetricWindow)
         */
        default Builder window(Consumer<MetricWindow.Builder> window) {
            return window(MetricWindow.builder().applyMutation(window).build());
        }

        /**
         * <p>
         * The processing configuration for the given metric property. You can configure metrics to be computed at the
         * edge or in the Amazon Web Services Cloud. By default, metrics are forwarded to the cloud.
         * </p>
         * 
         * @param processingConfig
         *        The processing configuration for the given metric property. You can configure metrics to be computed
         *        at the edge or in the Amazon Web Services Cloud. By default, metrics are forwarded to the cloud.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder processingConfig(MetricProcessingConfig processingConfig);

        /**
         * <p>
         * The processing configuration for the given metric property. You can configure metrics to be computed at the
         * edge or in the Amazon Web Services Cloud. By default, metrics are forwarded to the cloud.
         * </p>
         * This is a convenience method that creates an instance of the {@link MetricProcessingConfig.Builder} avoiding
         * the need to create one manually via {@link MetricProcessingConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MetricProcessingConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #processingConfig(MetricProcessingConfig)}.
         * 
         * @param processingConfig
         *        a consumer that will call methods on {@link MetricProcessingConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #processingConfig(MetricProcessingConfig)
         */
        default Builder processingConfig(Consumer<MetricProcessingConfig.Builder> processingConfig) {
            return processingConfig(MetricProcessingConfig.builder().applyMutation(processingConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String expression;

        private List<ExpressionVariable> variables = DefaultSdkAutoConstructList.getInstance();

        private MetricWindow window;

        private MetricProcessingConfig processingConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(Metric model) {
            expression(model.expression);
            variables(model.variables);
            window(model.window);
            processingConfig(model.processingConfig);
        }

        public final String getExpression() {
            return expression;
        }

        public final void setExpression(String expression) {
            this.expression = expression;
        }

        @Override
        public final Builder expression(String expression) {
            this.expression = expression;
            return this;
        }

        public final List<ExpressionVariable.Builder> getVariables() {
            List<ExpressionVariable.Builder> result = ExpressionVariablesCopier.copyToBuilder(this.variables);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVariables(Collection<ExpressionVariable.BuilderImpl> variables) {
            this.variables = ExpressionVariablesCopier.copyFromBuilder(variables);
        }

        @Override
        public final Builder variables(Collection<ExpressionVariable> variables) {
            this.variables = ExpressionVariablesCopier.copy(variables);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variables(ExpressionVariable... variables) {
            variables(Arrays.asList(variables));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder variables(Consumer<ExpressionVariable.Builder>... variables) {
            variables(Stream.of(variables).map(c -> ExpressionVariable.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final MetricWindow.Builder getWindow() {
            return window != null ? window.toBuilder() : null;
        }

        public final void setWindow(MetricWindow.BuilderImpl window) {
            this.window = window != null ? window.build() : null;
        }

        @Override
        public final Builder window(MetricWindow window) {
            this.window = window;
            return this;
        }

        public final MetricProcessingConfig.Builder getProcessingConfig() {
            return processingConfig != null ? processingConfig.toBuilder() : null;
        }

        public final void setProcessingConfig(MetricProcessingConfig.BuilderImpl processingConfig) {
            this.processingConfig = processingConfig != null ? processingConfig.build() : null;
        }

        @Override
        public final Builder processingConfig(MetricProcessingConfig processingConfig) {
            this.processingConfig = processingConfig;
            return this;
        }

        @Override
        public Metric build() {
            return new Metric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
