/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotsitewise.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.iotsitewise.IoTSiteWiseAsyncClient;
import software.amazon.awssdk.services.iotsitewise.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetModelRequest;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetModelResponse;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetRequest;
import software.amazon.awssdk.services.iotsitewise.model.DescribeAssetResponse;
import software.amazon.awssdk.services.iotsitewise.model.DescribePortalRequest;
import software.amazon.awssdk.services.iotsitewise.model.DescribePortalResponse;
import software.amazon.awssdk.services.iotsitewise.model.IoTSiteWiseRequest;
import software.amazon.awssdk.services.iotsitewise.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultIoTSiteWiseAsyncWaiter implements IoTSiteWiseAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final IoTSiteWiseAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeAssetModelResponse> assetModelNotExistsWaiter;

    private final AsyncWaiter<DescribeAssetModelResponse> assetModelActiveWaiter;

    private final AsyncWaiter<DescribeAssetResponse> assetNotExistsWaiter;

    private final AsyncWaiter<DescribeAssetResponse> assetActiveWaiter;

    private final AsyncWaiter<DescribePortalResponse> portalNotExistsWaiter;

    private final AsyncWaiter<DescribePortalResponse> portalActiveWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultIoTSiteWiseAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = IoTSiteWiseAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.assetModelNotExistsWaiter = AsyncWaiter.builder(DescribeAssetModelResponse.class)
                .acceptors(assetModelNotExistsWaiterAcceptors())
                .overrideConfiguration(assetModelNotExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.assetModelActiveWaiter = AsyncWaiter.builder(DescribeAssetModelResponse.class)
                .acceptors(assetModelActiveWaiterAcceptors())
                .overrideConfiguration(assetModelActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.assetNotExistsWaiter = AsyncWaiter.builder(DescribeAssetResponse.class).acceptors(assetNotExistsWaiterAcceptors())
                .overrideConfiguration(assetNotExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.assetActiveWaiter = AsyncWaiter.builder(DescribeAssetResponse.class).acceptors(assetActiveWaiterAcceptors())
                .overrideConfiguration(assetActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.portalNotExistsWaiter = AsyncWaiter.builder(DescribePortalResponse.class)
                .acceptors(portalNotExistsWaiterAcceptors())
                .overrideConfiguration(portalNotExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.portalActiveWaiter = AsyncWaiter.builder(DescribePortalResponse.class).acceptors(portalActiveWaiterAcceptors())
                .overrideConfiguration(portalActiveWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetActive(DescribeAssetRequest describeAssetRequest) {
        return assetActiveWaiter.runAsync(() -> client.describeAsset(applyWaitersUserAgent(describeAssetRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetActive(
            DescribeAssetRequest describeAssetRequest, WaiterOverrideConfiguration overrideConfig) {
        return assetActiveWaiter.runAsync(() -> client.describeAsset(applyWaitersUserAgent(describeAssetRequest)),
                assetActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelActive(
            DescribeAssetModelRequest describeAssetModelRequest) {
        return assetModelActiveWaiter.runAsync(() -> client.describeAssetModel(applyWaitersUserAgent(describeAssetModelRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelActive(
            DescribeAssetModelRequest describeAssetModelRequest, WaiterOverrideConfiguration overrideConfig) {
        return assetModelActiveWaiter.runAsync(() -> client.describeAssetModel(applyWaitersUserAgent(describeAssetModelRequest)),
                assetModelActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelNotExists(
            DescribeAssetModelRequest describeAssetModelRequest) {
        return assetModelNotExistsWaiter.runAsync(() -> client
                .describeAssetModel(applyWaitersUserAgent(describeAssetModelRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAssetModelResponse>> waitUntilAssetModelNotExists(
            DescribeAssetModelRequest describeAssetModelRequest, WaiterOverrideConfiguration overrideConfig) {
        return assetModelNotExistsWaiter.runAsync(
                () -> client.describeAssetModel(applyWaitersUserAgent(describeAssetModelRequest)),
                assetModelNotExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetNotExists(
            DescribeAssetRequest describeAssetRequest) {
        return assetNotExistsWaiter.runAsync(() -> client.describeAsset(applyWaitersUserAgent(describeAssetRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeAssetResponse>> waitUntilAssetNotExists(
            DescribeAssetRequest describeAssetRequest, WaiterOverrideConfiguration overrideConfig) {
        return assetNotExistsWaiter.runAsync(() -> client.describeAsset(applyWaitersUserAgent(describeAssetRequest)),
                assetNotExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalActive(
            DescribePortalRequest describePortalRequest) {
        return portalActiveWaiter.runAsync(() -> client.describePortal(applyWaitersUserAgent(describePortalRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalActive(
            DescribePortalRequest describePortalRequest, WaiterOverrideConfiguration overrideConfig) {
        return portalActiveWaiter.runAsync(() -> client.describePortal(applyWaitersUserAgent(describePortalRequest)),
                portalActiveWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalNotExists(
            DescribePortalRequest describePortalRequest) {
        return portalNotExistsWaiter.runAsync(() -> client.describePortal(applyWaitersUserAgent(describePortalRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribePortalResponse>> waitUntilPortalNotExists(
            DescribePortalRequest describePortalRequest, WaiterOverrideConfiguration overrideConfig) {
        return portalNotExistsWaiter.runAsync(() -> client.describePortal(applyWaitersUserAgent(describePortalRequest)),
                portalNotExistsWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeAssetModelResponse>> assetModelNotExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAssetModelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAssetModelResponse>> assetModelActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAssetModelResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("assetModelStatus").field("state").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("assetModelStatus").field("state").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (assetModelStatus.state=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAssetResponse>> assetNotExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAssetResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAssetResponse>> assetActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAssetResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("assetStatus").field("state").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("assetStatus").field("state").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (assetStatus.state=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribePortalResponse>> portalNotExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribePortalResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribePortalResponse>> portalActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribePortalResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("portalStatus").field("state").value(), "ACTIVE");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration assetModelNotExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration assetModelActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration assetNotExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration assetActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration portalNotExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration portalActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static IoTSiteWiseAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends IoTSiteWiseRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements IoTSiteWiseAsyncWaiter.Builder {
        private IoTSiteWiseAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public IoTSiteWiseAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public IoTSiteWiseAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public IoTSiteWiseAsyncWaiter.Builder client(IoTSiteWiseAsyncClient client) {
            this.client = client;
            return this;
        }

        public IoTSiteWiseAsyncWaiter build() {
            return new DefaultIoTSiteWiseAsyncWaiter(this);
        }
    }
}
