/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotwireless.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The wrapper for a position configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PositionConfigurationItem implements SdkPojo, Serializable,
        ToCopyableBuilder<PositionConfigurationItem.Builder, PositionConfigurationItem> {
    private static final SdkField<String> RESOURCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceIdentifier").getter(getter(PositionConfigurationItem::resourceIdentifier))
            .setter(setter(Builder::resourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIdentifier").build())
            .build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(PositionConfigurationItem::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<PositionSolverDetails> SOLVERS_FIELD = SdkField
            .<PositionSolverDetails> builder(MarshallingType.SDK_POJO).memberName("Solvers")
            .getter(getter(PositionConfigurationItem::solvers)).setter(setter(Builder::solvers))
            .constructor(PositionSolverDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Solvers").build()).build();

    private static final SdkField<String> DESTINATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Destination").getter(getter(PositionConfigurationItem::destination))
            .setter(setter(Builder::destination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Destination").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_IDENTIFIER_FIELD,
            RESOURCE_TYPE_FIELD, SOLVERS_FIELD, DESTINATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resourceIdentifier;

    private final String resourceType;

    private final PositionSolverDetails solvers;

    private final String destination;

    private PositionConfigurationItem(BuilderImpl builder) {
        this.resourceIdentifier = builder.resourceIdentifier;
        this.resourceType = builder.resourceType;
        this.solvers = builder.solvers;
        this.destination = builder.destination;
    }

    /**
     * <p>
     * Resource identifier for the position configuration.
     * </p>
     * 
     * @return Resource identifier for the position configuration.
     */
    public final String resourceIdentifier() {
        return resourceIdentifier;
    }

    /**
     * <p>
     * Resource type of the resource for the position configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link PositionResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return Resource type of the resource for the position configuration.
     * @see PositionResourceType
     */
    public final PositionResourceType resourceType() {
        return PositionResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * Resource type of the resource for the position configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link PositionResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return Resource type of the resource for the position configuration.
     * @see PositionResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The details of the positioning solver object used to compute the location.
     * </p>
     * 
     * @return The details of the positioning solver object used to compute the location.
     */
    public final PositionSolverDetails solvers() {
        return solvers;
    }

    /**
     * <p>
     * The position data destination that describes the AWS IoT rule that processes the device's position data for use
     * by AWS IoT Core for LoRaWAN.
     * </p>
     * 
     * @return The position data destination that describes the AWS IoT rule that processes the device's position data
     *         for use by AWS IoT Core for LoRaWAN.
     */
    public final String destination() {
        return destination;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(solvers());
        hashCode = 31 * hashCode + Objects.hashCode(destination());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PositionConfigurationItem)) {
            return false;
        }
        PositionConfigurationItem other = (PositionConfigurationItem) obj;
        return Objects.equals(resourceIdentifier(), other.resourceIdentifier())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(solvers(), other.solvers()) && Objects.equals(destination(), other.destination());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PositionConfigurationItem").add("ResourceIdentifier", resourceIdentifier())
                .add("ResourceType", resourceTypeAsString()).add("Solvers", solvers()).add("Destination", destination()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceIdentifier":
            return Optional.ofNullable(clazz.cast(resourceIdentifier()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "Solvers":
            return Optional.ofNullable(clazz.cast(solvers()));
        case "Destination":
            return Optional.ofNullable(clazz.cast(destination()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PositionConfigurationItem, T> g) {
        return obj -> g.apply((PositionConfigurationItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PositionConfigurationItem> {
        /**
         * <p>
         * Resource identifier for the position configuration.
         * </p>
         * 
         * @param resourceIdentifier
         *        Resource identifier for the position configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifier(String resourceIdentifier);

        /**
         * <p>
         * Resource type of the resource for the position configuration.
         * </p>
         * 
         * @param resourceType
         *        Resource type of the resource for the position configuration.
         * @see PositionResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PositionResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * Resource type of the resource for the position configuration.
         * </p>
         * 
         * @param resourceType
         *        Resource type of the resource for the position configuration.
         * @see PositionResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PositionResourceType
         */
        Builder resourceType(PositionResourceType resourceType);

        /**
         * <p>
         * The details of the positioning solver object used to compute the location.
         * </p>
         * 
         * @param solvers
         *        The details of the positioning solver object used to compute the location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder solvers(PositionSolverDetails solvers);

        /**
         * <p>
         * The details of the positioning solver object used to compute the location.
         * </p>
         * This is a convenience method that creates an instance of the {@link PositionSolverDetails.Builder} avoiding
         * the need to create one manually via {@link PositionSolverDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link PositionSolverDetails.Builder#build()} is called immediately and
         * its result is passed to {@link #solvers(PositionSolverDetails)}.
         * 
         * @param solvers
         *        a consumer that will call methods on {@link PositionSolverDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #solvers(PositionSolverDetails)
         */
        default Builder solvers(Consumer<PositionSolverDetails.Builder> solvers) {
            return solvers(PositionSolverDetails.builder().applyMutation(solvers).build());
        }

        /**
         * <p>
         * The position data destination that describes the AWS IoT rule that processes the device's position data for
         * use by AWS IoT Core for LoRaWAN.
         * </p>
         * 
         * @param destination
         *        The position data destination that describes the AWS IoT rule that processes the device's position
         *        data for use by AWS IoT Core for LoRaWAN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destination(String destination);
    }

    static final class BuilderImpl implements Builder {
        private String resourceIdentifier;

        private String resourceType;

        private PositionSolverDetails solvers;

        private String destination;

        private BuilderImpl() {
        }

        private BuilderImpl(PositionConfigurationItem model) {
            resourceIdentifier(model.resourceIdentifier);
            resourceType(model.resourceType);
            solvers(model.solvers);
            destination(model.destination);
        }

        public final String getResourceIdentifier() {
            return resourceIdentifier;
        }

        public final void setResourceIdentifier(String resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
        }

        @Override
        public final Builder resourceIdentifier(String resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(PositionResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final PositionSolverDetails.Builder getSolvers() {
            return solvers != null ? solvers.toBuilder() : null;
        }

        public final void setSolvers(PositionSolverDetails.BuilderImpl solvers) {
            this.solvers = solvers != null ? solvers.build() : null;
        }

        @Override
        public final Builder solvers(PositionSolverDetails solvers) {
            this.solvers = solvers;
            return this;
        }

        public final String getDestination() {
            return destination;
        }

        public final void setDestination(String destination) {
            this.destination = destination;
        }

        @Override
        public final Builder destination(String destination) {
            this.destination = destination;
            return this;
        }

        @Override
        public PositionConfigurationItem build() {
            return new PositionConfigurationItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
