/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iotwireless.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Global navigation satellite system (GNSS) object used for positioning.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Gnss implements SdkPojo, Serializable, ToCopyableBuilder<Gnss.Builder, Gnss> {
    private static final SdkField<String> PAYLOAD_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Payload")
            .getter(getter(Gnss::payload)).setter(setter(Builder::payload))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Payload").build()).build();

    private static final SdkField<Float> CAPTURE_TIME_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("CaptureTime").getter(getter(Gnss::captureTime)).setter(setter(Builder::captureTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CaptureTime").build()).build();

    private static final SdkField<Float> CAPTURE_TIME_ACCURACY_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("CaptureTimeAccuracy").getter(getter(Gnss::captureTimeAccuracy))
            .setter(setter(Builder::captureTimeAccuracy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CaptureTimeAccuracy").build())
            .build();

    private static final SdkField<List<Float>> ASSIST_POSITION_FIELD = SdkField
            .<List<Float>> builder(MarshallingType.LIST)
            .memberName("AssistPosition")
            .getter(getter(Gnss::assistPosition))
            .setter(setter(Builder::assistPosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssistPosition").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Float> builder(MarshallingType.FLOAT)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Float> ASSIST_ALTITUDE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("AssistAltitude").getter(getter(Gnss::assistAltitude)).setter(setter(Builder::assistAltitude))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssistAltitude").build()).build();

    private static final SdkField<Boolean> USE2_D_SOLVER_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Use2DSolver").getter(getter(Gnss::use2DSolver)).setter(setter(Builder::use2DSolver))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Use2DSolver").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PAYLOAD_FIELD,
            CAPTURE_TIME_FIELD, CAPTURE_TIME_ACCURACY_FIELD, ASSIST_POSITION_FIELD, ASSIST_ALTITUDE_FIELD, USE2_D_SOLVER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String payload;

    private final Float captureTime;

    private final Float captureTimeAccuracy;

    private final List<Float> assistPosition;

    private final Float assistAltitude;

    private final Boolean use2DSolver;

    private Gnss(BuilderImpl builder) {
        this.payload = builder.payload;
        this.captureTime = builder.captureTime;
        this.captureTimeAccuracy = builder.captureTimeAccuracy;
        this.assistPosition = builder.assistPosition;
        this.assistAltitude = builder.assistAltitude;
        this.use2DSolver = builder.use2DSolver;
    }

    /**
     * <p>
     * Payload that contains the GNSS scan result, or NAV message, in hexadecimal notation.
     * </p>
     * 
     * @return Payload that contains the GNSS scan result, or NAV message, in hexadecimal notation.
     */
    public final String payload() {
        return payload;
    }

    /**
     * <p>
     * Optional parameter that gives an estimate of the time when the GNSS scan information is taken, in seconds GPS
     * time (GPST). If capture time is not specified, the local server time is used.
     * </p>
     * 
     * @return Optional parameter that gives an estimate of the time when the GNSS scan information is taken, in seconds
     *         GPS time (GPST). If capture time is not specified, the local server time is used.
     */
    public final Float captureTime() {
        return captureTime;
    }

    /**
     * <p>
     * Optional value that gives the capture time estimate accuracy, in seconds. If capture time accuracy is not
     * specified, default value of 300 is used.
     * </p>
     * 
     * @return Optional value that gives the capture time estimate accuracy, in seconds. If capture time accuracy is not
     *         specified, default value of 300 is used.
     */
    public final Float captureTimeAccuracy() {
        return captureTimeAccuracy;
    }

    /**
     * For responses, this returns true if the service returned a value for the AssistPosition property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAssistPosition() {
        return assistPosition != null && !(assistPosition instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Optional assistance position information, specified using latitude and longitude values in degrees. The
     * co-ordinates are inside the WGS84 reference frame.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAssistPosition} method.
     * </p>
     * 
     * @return Optional assistance position information, specified using latitude and longitude values in degrees. The
     *         co-ordinates are inside the WGS84 reference frame.
     */
    public final List<Float> assistPosition() {
        return assistPosition;
    }

    /**
     * <p>
     * Optional assistance altitude, which is the altitude of the device at capture time, specified in meters above the
     * WGS84 reference ellipsoid.
     * </p>
     * 
     * @return Optional assistance altitude, which is the altitude of the device at capture time, specified in meters
     *         above the WGS84 reference ellipsoid.
     */
    public final Float assistAltitude() {
        return assistAltitude;
    }

    /**
     * <p>
     * Optional parameter that forces 2D solve, which modifies the positioning algorithm to a 2D solution problem. When
     * this parameter is specified, the assistance altitude should have an accuracy of at least 10 meters.
     * </p>
     * 
     * @return Optional parameter that forces 2D solve, which modifies the positioning algorithm to a 2D solution
     *         problem. When this parameter is specified, the assistance altitude should have an accuracy of at least 10
     *         meters.
     */
    public final Boolean use2DSolver() {
        return use2DSolver;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(payload());
        hashCode = 31 * hashCode + Objects.hashCode(captureTime());
        hashCode = 31 * hashCode + Objects.hashCode(captureTimeAccuracy());
        hashCode = 31 * hashCode + Objects.hashCode(hasAssistPosition() ? assistPosition() : null);
        hashCode = 31 * hashCode + Objects.hashCode(assistAltitude());
        hashCode = 31 * hashCode + Objects.hashCode(use2DSolver());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Gnss)) {
            return false;
        }
        Gnss other = (Gnss) obj;
        return Objects.equals(payload(), other.payload()) && Objects.equals(captureTime(), other.captureTime())
                && Objects.equals(captureTimeAccuracy(), other.captureTimeAccuracy())
                && hasAssistPosition() == other.hasAssistPosition() && Objects.equals(assistPosition(), other.assistPosition())
                && Objects.equals(assistAltitude(), other.assistAltitude()) && Objects.equals(use2DSolver(), other.use2DSolver());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Gnss").add("Payload", payload()).add("CaptureTime", captureTime())
                .add("CaptureTimeAccuracy", captureTimeAccuracy())
                .add("AssistPosition", hasAssistPosition() ? assistPosition() : null).add("AssistAltitude", assistAltitude())
                .add("Use2DSolver", use2DSolver()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Payload":
            return Optional.ofNullable(clazz.cast(payload()));
        case "CaptureTime":
            return Optional.ofNullable(clazz.cast(captureTime()));
        case "CaptureTimeAccuracy":
            return Optional.ofNullable(clazz.cast(captureTimeAccuracy()));
        case "AssistPosition":
            return Optional.ofNullable(clazz.cast(assistPosition()));
        case "AssistAltitude":
            return Optional.ofNullable(clazz.cast(assistAltitude()));
        case "Use2DSolver":
            return Optional.ofNullable(clazz.cast(use2DSolver()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Gnss, T> g) {
        return obj -> g.apply((Gnss) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Gnss> {
        /**
         * <p>
         * Payload that contains the GNSS scan result, or NAV message, in hexadecimal notation.
         * </p>
         * 
         * @param payload
         *        Payload that contains the GNSS scan result, or NAV message, in hexadecimal notation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payload(String payload);

        /**
         * <p>
         * Optional parameter that gives an estimate of the time when the GNSS scan information is taken, in seconds GPS
         * time (GPST). If capture time is not specified, the local server time is used.
         * </p>
         * 
         * @param captureTime
         *        Optional parameter that gives an estimate of the time when the GNSS scan information is taken, in
         *        seconds GPS time (GPST). If capture time is not specified, the local server time is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captureTime(Float captureTime);

        /**
         * <p>
         * Optional value that gives the capture time estimate accuracy, in seconds. If capture time accuracy is not
         * specified, default value of 300 is used.
         * </p>
         * 
         * @param captureTimeAccuracy
         *        Optional value that gives the capture time estimate accuracy, in seconds. If capture time accuracy is
         *        not specified, default value of 300 is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captureTimeAccuracy(Float captureTimeAccuracy);

        /**
         * <p>
         * Optional assistance position information, specified using latitude and longitude values in degrees. The
         * co-ordinates are inside the WGS84 reference frame.
         * </p>
         * 
         * @param assistPosition
         *        Optional assistance position information, specified using latitude and longitude values in degrees.
         *        The co-ordinates are inside the WGS84 reference frame.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assistPosition(Collection<Float> assistPosition);

        /**
         * <p>
         * Optional assistance position information, specified using latitude and longitude values in degrees. The
         * co-ordinates are inside the WGS84 reference frame.
         * </p>
         * 
         * @param assistPosition
         *        Optional assistance position information, specified using latitude and longitude values in degrees.
         *        The co-ordinates are inside the WGS84 reference frame.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assistPosition(Float... assistPosition);

        /**
         * <p>
         * Optional assistance altitude, which is the altitude of the device at capture time, specified in meters above
         * the WGS84 reference ellipsoid.
         * </p>
         * 
         * @param assistAltitude
         *        Optional assistance altitude, which is the altitude of the device at capture time, specified in meters
         *        above the WGS84 reference ellipsoid.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assistAltitude(Float assistAltitude);

        /**
         * <p>
         * Optional parameter that forces 2D solve, which modifies the positioning algorithm to a 2D solution problem.
         * When this parameter is specified, the assistance altitude should have an accuracy of at least 10 meters.
         * </p>
         * 
         * @param use2DSolver
         *        Optional parameter that forces 2D solve, which modifies the positioning algorithm to a 2D solution
         *        problem. When this parameter is specified, the assistance altitude should have an accuracy of at least
         *        10 meters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder use2DSolver(Boolean use2DSolver);
    }

    static final class BuilderImpl implements Builder {
        private String payload;

        private Float captureTime;

        private Float captureTimeAccuracy;

        private List<Float> assistPosition = DefaultSdkAutoConstructList.getInstance();

        private Float assistAltitude;

        private Boolean use2DSolver;

        private BuilderImpl() {
        }

        private BuilderImpl(Gnss model) {
            payload(model.payload);
            captureTime(model.captureTime);
            captureTimeAccuracy(model.captureTimeAccuracy);
            assistPosition(model.assistPosition);
            assistAltitude(model.assistAltitude);
            use2DSolver(model.use2DSolver);
        }

        public final String getPayload() {
            return payload;
        }

        public final void setPayload(String payload) {
            this.payload = payload;
        }

        @Override
        public final Builder payload(String payload) {
            this.payload = payload;
            return this;
        }

        public final Float getCaptureTime() {
            return captureTime;
        }

        public final void setCaptureTime(Float captureTime) {
            this.captureTime = captureTime;
        }

        @Override
        public final Builder captureTime(Float captureTime) {
            this.captureTime = captureTime;
            return this;
        }

        public final Float getCaptureTimeAccuracy() {
            return captureTimeAccuracy;
        }

        public final void setCaptureTimeAccuracy(Float captureTimeAccuracy) {
            this.captureTimeAccuracy = captureTimeAccuracy;
        }

        @Override
        public final Builder captureTimeAccuracy(Float captureTimeAccuracy) {
            this.captureTimeAccuracy = captureTimeAccuracy;
            return this;
        }

        public final Collection<Float> getAssistPosition() {
            if (assistPosition instanceof SdkAutoConstructList) {
                return null;
            }
            return assistPosition;
        }

        public final void setAssistPosition(Collection<Float> assistPosition) {
            this.assistPosition = AssistPositionCopier.copy(assistPosition);
        }

        @Override
        public final Builder assistPosition(Collection<Float> assistPosition) {
            this.assistPosition = AssistPositionCopier.copy(assistPosition);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder assistPosition(Float... assistPosition) {
            assistPosition(Arrays.asList(assistPosition));
            return this;
        }

        public final Float getAssistAltitude() {
            return assistAltitude;
        }

        public final void setAssistAltitude(Float assistAltitude) {
            this.assistAltitude = assistAltitude;
        }

        @Override
        public final Builder assistAltitude(Float assistAltitude) {
            this.assistAltitude = assistAltitude;
            return this;
        }

        public final Boolean getUse2DSolver() {
            return use2DSolver;
        }

        public final void setUse2DSolver(Boolean use2DSolver) {
            this.use2DSolver = use2DSolver;
        }

        @Override
        public final Builder use2DSolver(Boolean use2DSolver) {
            this.use2DSolver = use2DSolver;
            return this;
        }

        @Override
        public Gnss build() {
            return new Gnss(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
