/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ivs;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.ivs.model.AccessDeniedException;
import software.amazon.awssdk.services.ivs.model.BatchGetChannelRequest;
import software.amazon.awssdk.services.ivs.model.BatchGetChannelResponse;
import software.amazon.awssdk.services.ivs.model.BatchGetStreamKeyRequest;
import software.amazon.awssdk.services.ivs.model.BatchGetStreamKeyResponse;
import software.amazon.awssdk.services.ivs.model.ChannelNotBroadcastingException;
import software.amazon.awssdk.services.ivs.model.ConflictException;
import software.amazon.awssdk.services.ivs.model.CreateChannelRequest;
import software.amazon.awssdk.services.ivs.model.CreateChannelResponse;
import software.amazon.awssdk.services.ivs.model.CreateStreamKeyRequest;
import software.amazon.awssdk.services.ivs.model.CreateStreamKeyResponse;
import software.amazon.awssdk.services.ivs.model.DeleteChannelRequest;
import software.amazon.awssdk.services.ivs.model.DeleteChannelResponse;
import software.amazon.awssdk.services.ivs.model.DeletePlaybackKeyPairRequest;
import software.amazon.awssdk.services.ivs.model.DeletePlaybackKeyPairResponse;
import software.amazon.awssdk.services.ivs.model.DeleteStreamKeyRequest;
import software.amazon.awssdk.services.ivs.model.DeleteStreamKeyResponse;
import software.amazon.awssdk.services.ivs.model.GetChannelRequest;
import software.amazon.awssdk.services.ivs.model.GetChannelResponse;
import software.amazon.awssdk.services.ivs.model.GetPlaybackKeyPairRequest;
import software.amazon.awssdk.services.ivs.model.GetPlaybackKeyPairResponse;
import software.amazon.awssdk.services.ivs.model.GetStreamKeyRequest;
import software.amazon.awssdk.services.ivs.model.GetStreamKeyResponse;
import software.amazon.awssdk.services.ivs.model.GetStreamRequest;
import software.amazon.awssdk.services.ivs.model.GetStreamResponse;
import software.amazon.awssdk.services.ivs.model.ImportPlaybackKeyPairRequest;
import software.amazon.awssdk.services.ivs.model.ImportPlaybackKeyPairResponse;
import software.amazon.awssdk.services.ivs.model.InternalServerException;
import software.amazon.awssdk.services.ivs.model.IvsException;
import software.amazon.awssdk.services.ivs.model.ListChannelsRequest;
import software.amazon.awssdk.services.ivs.model.ListChannelsResponse;
import software.amazon.awssdk.services.ivs.model.ListPlaybackKeyPairsRequest;
import software.amazon.awssdk.services.ivs.model.ListPlaybackKeyPairsResponse;
import software.amazon.awssdk.services.ivs.model.ListStreamKeysRequest;
import software.amazon.awssdk.services.ivs.model.ListStreamKeysResponse;
import software.amazon.awssdk.services.ivs.model.ListStreamsRequest;
import software.amazon.awssdk.services.ivs.model.ListStreamsResponse;
import software.amazon.awssdk.services.ivs.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ivs.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ivs.model.PendingVerificationException;
import software.amazon.awssdk.services.ivs.model.PutMetadataRequest;
import software.amazon.awssdk.services.ivs.model.PutMetadataResponse;
import software.amazon.awssdk.services.ivs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ivs.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ivs.model.StopStreamRequest;
import software.amazon.awssdk.services.ivs.model.StopStreamResponse;
import software.amazon.awssdk.services.ivs.model.StreamUnavailableException;
import software.amazon.awssdk.services.ivs.model.TagResourceRequest;
import software.amazon.awssdk.services.ivs.model.TagResourceResponse;
import software.amazon.awssdk.services.ivs.model.ThrottlingException;
import software.amazon.awssdk.services.ivs.model.UntagResourceRequest;
import software.amazon.awssdk.services.ivs.model.UntagResourceResponse;
import software.amazon.awssdk.services.ivs.model.UpdateChannelRequest;
import software.amazon.awssdk.services.ivs.model.UpdateChannelResponse;
import software.amazon.awssdk.services.ivs.model.ValidationException;
import software.amazon.awssdk.services.ivs.paginators.ListChannelsIterable;
import software.amazon.awssdk.services.ivs.paginators.ListPlaybackKeyPairsIterable;
import software.amazon.awssdk.services.ivs.paginators.ListStreamKeysIterable;
import software.amazon.awssdk.services.ivs.paginators.ListStreamsIterable;
import software.amazon.awssdk.services.ivs.paginators.ListTagsForResourceIterable;

/**
 * Service client for accessing Amazon IVS. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * <b>Introduction</b>
 * </p>
 * <p>
 * The Amazon Interactive Video Service (IVS) API is REST compatible, using a standard HTTP API and an <a
 * href="http://aws.amazon.com/sns">AWS SNS</a> event stream for responses. JSON is used for both requests and
 * responses, including errors.
 * </p>
 * <p>
 * The API is an AWS regional service, currently in these regions: us-west-2, us-east-1, and eu-west-1.
 * </p>
 * <p>
 * <i> <b>All API request parameters and URLs are case sensitive. </b> </i>
 * </p>
 * <p>
 * For a summary of notable documentation changes in each release, see <a
 * href="https://docs.aws.amazon.com/ivs/latest/userguide/doc-history.html"> Document History</a>.
 * </p>
 * <p>
 * <b>Service Endpoints</b>
 * </p>
 * <p>
 * The following are the Amazon IVS service endpoints (all HTTPS):
 * </p>
 * <p>
 * Region name: US West (Oregon)
 * </p>
 * <ul>
 * <li>
 * <p>
 * Region: <code>us-west-2</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * Endpoint: <code>ivs.us-west-2.amazonaws.com</code>
 * </p>
 * </li>
 * </ul>
 * <p>
 * Region name: US East (Virginia)
 * </p>
 * <ul>
 * <li>
 * <p>
 * Region: <code>us-east-1</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * Endpoint: <code>ivs.us-east-1.amazonaws.com</code>
 * </p>
 * </li>
 * </ul>
 * <p>
 * Region name: EU West (Dublin)
 * </p>
 * <ul>
 * <li>
 * <p>
 * Region: <code>eu-west-1</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * Endpoint: <code>ivs.eu-west-1.amazonaws.com</code>
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Allowed Header Values</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code> <b>Accept:</b> </code> application/json
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <b>Accept-Encoding:</b> </code> gzip, deflate
 * </p>
 * </li>
 * <li>
 * <p>
 * <code> <b>Content-Type:</b> </code>application/json
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Resources</b>
 * </p>
 * <p>
 * The following resources contain information about your IVS live stream (see <a
 * href="https://docs.aws.amazon.com/ivs/latest/userguide/GSIVS.html"> Getting Started with Amazon IVS</a>):
 * </p>
 * <ul>
 * <li>
 * <p>
 * Channel — Stores configuration data related to your live stream. You first create a channel and then use the
 * channel’s stream key to start your live stream. See the Channel endpoints for more information.
 * </p>
 * </li>
 * <li>
 * <p>
 * Stream key — An identifier assigned by Amazon IVS when you create a channel, which is then used to authorize
 * streaming. See the StreamKey endpoints for more information. <i> <b>Treat the stream key like a secret, since it
 * allows anyone to stream to the channel.</b> </i>
 * </p>
 * </li>
 * <li>
 * <p>
 * Playback key pair — Video playback may be restricted using playback-authorization tokens, which use public-key
 * encryption. A playback key pair is the public-private pair of keys used to sign and validate the
 * playback-authorization token. See the PlaybackKeyPair endpoints for more information.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Tagging</b>
 * </p>
 * <p>
 * A <i>tag</i> is a metadata label that you assign to an AWS resource. A tag comprises a <i>key</i> and a <i>value</i>,
 * both set by you. For example, you might set a tag as <code>topic:nature</code> to label a particular video category.
 * See <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS Resources</a> for more
 * information, including restrictions that apply to tags.
 * </p>
 * <p>
 * Tags can help you identify and organize your AWS resources. For example, you can use the same tag for different
 * resources to indicate that they are related. You can also use tags to manage access (see <a
 * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html"> Access Tags</a>).
 * </p>
 * <p>
 * The Amazon IVS API has these tag-related endpoints: <a>TagResource</a>, <a>UntagResource</a>, and
 * <a>ListTagsForResource</a>. The following resources support tagging: Channels, Stream Keys, and Playback Key Pairs.
 * </p>
 * <p>
 * <b>Channel Endpoints</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreateChannel</a> — Creates a new channel and an associated stream key to start streaming.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GetChannel</a> — Gets the channel configuration for the specified channel ARN (Amazon Resource Name).
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>BatchGetChannel</a> — Performs <a>GetChannel</a> on multiple ARNs simultaneously.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListChannels</a> — Gets summary information about all channels in your account, in the AWS region where the API
 * request is processed. This list can be filtered to match a specified string.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>UpdateChannel</a> — Updates a channel's configuration. This does not affect an ongoing stream of this channel. You
 * must stop and restart the stream for the changes to take effect.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeleteChannel</a> — Deletes the specified channel.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>StreamKey Endpoints</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreateStreamKey</a> — Creates a stream key, used to initiate a stream, for the specified channel ARN.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GetStreamKey</a> — Gets stream key information for the specified ARN.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>BatchGetStreamKey</a> — Performs <a>GetStreamKey</a> on multiple ARNs simultaneously.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListStreamKeys</a> — Gets summary information about stream keys for the specified channel.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeleteStreamKey</a> — Deletes the stream key for the specified ARN, so it can no longer be used to stream.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Stream Endpoints</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>GetStream</a> — Gets information about the active (live) stream on a specified channel.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListStreams</a> — Gets summary information about live streams in your account, in the AWS region where the API
 * request is processed.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>StopStream</a> — Disconnects the incoming RTMPS stream for the specified channel. Can be used in conjunction with
 * <a>DeleteStreamKey</a> to prevent further streaming to a channel.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>PutMetadata</a> — Inserts metadata into an RTMPS stream for the specified channel. A maximum of 5 requests per
 * second per channel is allowed, each with a maximum 1KB payload.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>PlaybackKeyPair Endpoints</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>ImportPlaybackKeyPair</a> — Imports the public portion of a new key pair and returns its <code>arn</code> and
 * <code>fingerprint</code>. The <code>privateKey</code> can then be used to generate viewer authorization tokens, to
 * grant viewers access to authorized channels.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GetPlaybackKeyPair</a> — Gets a specified playback authorization key pair and returns the <code>arn</code> and
 * <code>fingerprint</code>. The <code>privateKey</code> held by the caller can be used to generate viewer authorization
 * tokens, to grant viewers access to authorized channels.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListPlaybackKeyPairs</a> — Gets summary information about playback key pairs.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeletePlaybackKeyPair</a> — Deletes a specified authorization key pair. This invalidates future viewer tokens
 * generated using the key pair’s <code>privateKey</code>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>AWS Tags Endpoints</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>TagResource</a> — Adds or updates tags for the AWS resource with the specified ARN.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>UntagResource</a> — Removes tags from the resource with the specified ARN.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListTagsForResource</a> — Gets information about AWS tags for the specified ARN.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public interface IvsClient extends SdkClient {
    String SERVICE_NAME = "ivs";

    /**
     * Create a {@link IvsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static IvsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link IvsClient}.
     */
    static IvsClientBuilder builder() {
        return new DefaultIvsClientBuilder();
    }

    /**
     * <p>
     * Performs <a>GetChannel</a> on multiple ARNs simultaneously.
     * </p>
     *
     * @param batchGetChannelRequest
     * @return Result of the BatchGetChannel operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.BatchGetChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/BatchGetChannel" target="_top">AWS API
     *      Documentation</a>
     */
    default BatchGetChannelResponse batchGetChannel(BatchGetChannelRequest batchGetChannelRequest) throws AwsServiceException,
            SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Performs <a>GetChannel</a> on multiple ARNs simultaneously.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link BatchGetChannelRequest.Builder} avoiding the need
     * to create one manually via {@link BatchGetChannelRequest#builder()}
     * </p>
     *
     * @param batchGetChannelRequest
     *        A {@link Consumer} that will call methods on {@link BatchGetChannelRequest.Builder} to create a request.
     * @return Result of the BatchGetChannel operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.BatchGetChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/BatchGetChannel" target="_top">AWS API
     *      Documentation</a>
     */
    default BatchGetChannelResponse batchGetChannel(Consumer<BatchGetChannelRequest.Builder> batchGetChannelRequest)
            throws AwsServiceException, SdkClientException, IvsException {
        return batchGetChannel(BatchGetChannelRequest.builder().applyMutation(batchGetChannelRequest).build());
    }

    /**
     * <p>
     * Performs <a>GetStreamKey</a> on multiple ARNs simultaneously.
     * </p>
     *
     * @param batchGetStreamKeyRequest
     * @return Result of the BatchGetStreamKey operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.BatchGetStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/BatchGetStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    default BatchGetStreamKeyResponse batchGetStreamKey(BatchGetStreamKeyRequest batchGetStreamKeyRequest)
            throws AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Performs <a>GetStreamKey</a> on multiple ARNs simultaneously.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link BatchGetStreamKeyRequest.Builder} avoiding the need
     * to create one manually via {@link BatchGetStreamKeyRequest#builder()}
     * </p>
     *
     * @param batchGetStreamKeyRequest
     *        A {@link Consumer} that will call methods on {@link BatchGetStreamKeyRequest.Builder} to create a request.
     * @return Result of the BatchGetStreamKey operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.BatchGetStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/BatchGetStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    default BatchGetStreamKeyResponse batchGetStreamKey(Consumer<BatchGetStreamKeyRequest.Builder> batchGetStreamKeyRequest)
            throws AwsServiceException, SdkClientException, IvsException {
        return batchGetStreamKey(BatchGetStreamKeyRequest.builder().applyMutation(batchGetStreamKeyRequest).build());
    }

    /**
     * <p>
     * Creates a new channel and an associated stream key to start streaming.
     * </p>
     *
     * @param createChannelRequest
     * @return Result of the CreateChannel operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/CreateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateChannelResponse createChannel(CreateChannelRequest createChannelRequest) throws ValidationException,
            AccessDeniedException, ServiceQuotaExceededException, PendingVerificationException, AwsServiceException,
            SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a new channel and an associated stream key to start streaming.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateChannelRequest.Builder} avoiding the need to
     * create one manually via {@link CreateChannelRequest#builder()}
     * </p>
     *
     * @param createChannelRequest
     *        A {@link Consumer} that will call methods on {@link CreateChannelRequest.Builder} to create a request.
     * @return Result of the CreateChannel operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/CreateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateChannelResponse createChannel(Consumer<CreateChannelRequest.Builder> createChannelRequest)
            throws ValidationException, AccessDeniedException, ServiceQuotaExceededException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsException {
        return createChannel(CreateChannelRequest.builder().applyMutation(createChannelRequest).build());
    }

    /**
     * <p>
     * Creates a stream key, used to initiate a stream, for the specified channel ARN.
     * </p>
     * <p>
     * Note that <a>CreateChannel</a> creates a stream key. If you subsequently use CreateStreamKey on the same channel,
     * it will fail because a stream key already exists and there is a limit of 1 stream key per channel. To reset the
     * stream key on a channel, use <a>DeleteStreamKey</a> and then CreateStreamKey.
     * </p>
     *
     * @param createStreamKeyRequest
     * @return Result of the CreateStreamKey operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.CreateStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/CreateStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateStreamKeyResponse createStreamKey(CreateStreamKeyRequest createStreamKeyRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, ServiceQuotaExceededException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a stream key, used to initiate a stream, for the specified channel ARN.
     * </p>
     * <p>
     * Note that <a>CreateChannel</a> creates a stream key. If you subsequently use CreateStreamKey on the same channel,
     * it will fail because a stream key already exists and there is a limit of 1 stream key per channel. To reset the
     * stream key on a channel, use <a>DeleteStreamKey</a> and then CreateStreamKey.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateStreamKeyRequest.Builder} avoiding the need
     * to create one manually via {@link CreateStreamKeyRequest#builder()}
     * </p>
     *
     * @param createStreamKeyRequest
     *        A {@link Consumer} that will call methods on {@link CreateStreamKeyRequest.Builder} to create a request.
     * @return Result of the CreateStreamKey operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.CreateStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/CreateStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateStreamKeyResponse createStreamKey(Consumer<CreateStreamKeyRequest.Builder> createStreamKeyRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, ServiceQuotaExceededException,
            PendingVerificationException, AwsServiceException, SdkClientException, IvsException {
        return createStreamKey(CreateStreamKeyRequest.builder().applyMutation(createStreamKeyRequest).build());
    }

    /**
     * <p>
     * Deletes the specified channel and its associated stream keys.
     * </p>
     *
     * @param deleteChannelRequest
     * @return Result of the DeleteChannel operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ConflictException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/DeleteChannel" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteChannelResponse deleteChannel(DeleteChannelRequest deleteChannelRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, ConflictException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified channel and its associated stream keys.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteChannelRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteChannelRequest#builder()}
     * </p>
     *
     * @param deleteChannelRequest
     *        A {@link Consumer} that will call methods on {@link DeleteChannelRequest.Builder} to create a request.
     * @return Result of the DeleteChannel operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ConflictException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/DeleteChannel" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteChannelResponse deleteChannel(Consumer<DeleteChannelRequest.Builder> deleteChannelRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, ConflictException,
            PendingVerificationException, AwsServiceException, SdkClientException, IvsException {
        return deleteChannel(DeleteChannelRequest.builder().applyMutation(deleteChannelRequest).build());
    }

    /**
     * <p>
     * Deletes a specified authorization key pair. This invalidates future viewer tokens generated using the key pair’s
     * <code>privateKey</code>.
     * </p>
     *
     * @param deletePlaybackKeyPairRequest
     * @return Result of the DeletePlaybackKeyPair operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.DeletePlaybackKeyPair
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/DeletePlaybackKeyPair" target="_top">AWS API
     *      Documentation</a>
     */
    default DeletePlaybackKeyPairResponse deletePlaybackKeyPair(DeletePlaybackKeyPairRequest deletePlaybackKeyPairRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a specified authorization key pair. This invalidates future viewer tokens generated using the key pair’s
     * <code>privateKey</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeletePlaybackKeyPairRequest.Builder} avoiding the
     * need to create one manually via {@link DeletePlaybackKeyPairRequest#builder()}
     * </p>
     *
     * @param deletePlaybackKeyPairRequest
     *        A {@link Consumer} that will call methods on {@link DeletePlaybackKeyPairRequest.Builder} to create a
     *        request.
     * @return Result of the DeletePlaybackKeyPair operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.DeletePlaybackKeyPair
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/DeletePlaybackKeyPair" target="_top">AWS API
     *      Documentation</a>
     */
    default DeletePlaybackKeyPairResponse deletePlaybackKeyPair(
            Consumer<DeletePlaybackKeyPairRequest.Builder> deletePlaybackKeyPairRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, PendingVerificationException, AwsServiceException,
            SdkClientException, IvsException {
        return deletePlaybackKeyPair(DeletePlaybackKeyPairRequest.builder().applyMutation(deletePlaybackKeyPairRequest).build());
    }

    /**
     * <p>
     * Deletes the stream key for the specified ARN, so it can no longer be used to stream.
     * </p>
     *
     * @param deleteStreamKeyRequest
     * @return Result of the DeleteStreamKey operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.DeleteStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/DeleteStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteStreamKeyResponse deleteStreamKey(DeleteStreamKeyRequest deleteStreamKeyRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, PendingVerificationException, AwsServiceException,
            SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the stream key for the specified ARN, so it can no longer be used to stream.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteStreamKeyRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteStreamKeyRequest#builder()}
     * </p>
     *
     * @param deleteStreamKeyRequest
     *        A {@link Consumer} that will call methods on {@link DeleteStreamKeyRequest.Builder} to create a request.
     * @return Result of the DeleteStreamKey operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.DeleteStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/DeleteStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteStreamKeyResponse deleteStreamKey(Consumer<DeleteStreamKeyRequest.Builder> deleteStreamKeyRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsException {
        return deleteStreamKey(DeleteStreamKeyRequest.builder().applyMutation(deleteStreamKeyRequest).build());
    }

    /**
     * <p>
     * Gets the channel configuration for the specified channel ARN. See also <a>BatchGetChannel</a>.
     * </p>
     *
     * @param getChannelRequest
     * @return Result of the GetChannel operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetChannel" target="_top">AWS API
     *      Documentation</a>
     */
    default GetChannelResponse getChannel(GetChannelRequest getChannelRequest) throws ValidationException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the channel configuration for the specified channel ARN. See also <a>BatchGetChannel</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetChannelRequest.Builder} avoiding the need to
     * create one manually via {@link GetChannelRequest#builder()}
     * </p>
     *
     * @param getChannelRequest
     *        A {@link Consumer} that will call methods on {@link GetChannelRequest.Builder} to create a request.
     * @return Result of the GetChannel operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetChannel" target="_top">AWS API
     *      Documentation</a>
     */
    default GetChannelResponse getChannel(Consumer<GetChannelRequest.Builder> getChannelRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, IvsException {
        return getChannel(GetChannelRequest.builder().applyMutation(getChannelRequest).build());
    }

    /**
     * <p>
     * Gets a specified playback authorization key pair and returns the <code>arn</code> and <code>fingerprint</code>.
     * The <code>privateKey</code> held by the caller can be used to generate viewer authorization tokens, to grant
     * viewers access to authorized channels.
     * </p>
     *
     * @param getPlaybackKeyPairRequest
     * @return Result of the GetPlaybackKeyPair operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetPlaybackKeyPair
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetPlaybackKeyPair" target="_top">AWS API
     *      Documentation</a>
     */
    default GetPlaybackKeyPairResponse getPlaybackKeyPair(GetPlaybackKeyPairRequest getPlaybackKeyPairRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets a specified playback authorization key pair and returns the <code>arn</code> and <code>fingerprint</code>.
     * The <code>privateKey</code> held by the caller can be used to generate viewer authorization tokens, to grant
     * viewers access to authorized channels.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetPlaybackKeyPairRequest.Builder} avoiding the
     * need to create one manually via {@link GetPlaybackKeyPairRequest#builder()}
     * </p>
     *
     * @param getPlaybackKeyPairRequest
     *        A {@link Consumer} that will call methods on {@link GetPlaybackKeyPairRequest.Builder} to create a
     *        request.
     * @return Result of the GetPlaybackKeyPair operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetPlaybackKeyPair
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetPlaybackKeyPair" target="_top">AWS API
     *      Documentation</a>
     */
    default GetPlaybackKeyPairResponse getPlaybackKeyPair(Consumer<GetPlaybackKeyPairRequest.Builder> getPlaybackKeyPairRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IvsException {
        return getPlaybackKeyPair(GetPlaybackKeyPairRequest.builder().applyMutation(getPlaybackKeyPairRequest).build());
    }

    /**
     * <p>
     * Gets information about the active (live) stream on a specified channel.
     * </p>
     *
     * @param getStreamRequest
     * @return Result of the GetStream operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ChannelNotBroadcastingException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetStream" target="_top">AWS API
     *      Documentation</a>
     */
    default GetStreamResponse getStream(GetStreamRequest getStreamRequest) throws ResourceNotFoundException, ValidationException,
            AccessDeniedException, ChannelNotBroadcastingException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information about the active (live) stream on a specified channel.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetStreamRequest.Builder} avoiding the need to
     * create one manually via {@link GetStreamRequest#builder()}
     * </p>
     *
     * @param getStreamRequest
     *        A {@link Consumer} that will call methods on {@link GetStreamRequest.Builder} to create a request.
     * @return Result of the GetStream operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ChannelNotBroadcastingException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetStream" target="_top">AWS API
     *      Documentation</a>
     */
    default GetStreamResponse getStream(Consumer<GetStreamRequest.Builder> getStreamRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, ChannelNotBroadcastingException, AwsServiceException, SdkClientException,
            IvsException {
        return getStream(GetStreamRequest.builder().applyMutation(getStreamRequest).build());
    }

    /**
     * <p>
     * Gets stream-key information for a specified ARN.
     * </p>
     *
     * @param getStreamKeyRequest
     * @return Result of the GetStreamKey operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    default GetStreamKeyResponse getStreamKey(GetStreamKeyRequest getStreamKeyRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets stream-key information for a specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetStreamKeyRequest.Builder} avoiding the need to
     * create one manually via {@link GetStreamKeyRequest#builder()}
     * </p>
     *
     * @param getStreamKeyRequest
     *        A {@link Consumer} that will call methods on {@link GetStreamKeyRequest.Builder} to create a request.
     * @return Result of the GetStreamKey operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.GetStreamKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/GetStreamKey" target="_top">AWS API
     *      Documentation</a>
     */
    default GetStreamKeyResponse getStreamKey(Consumer<GetStreamKeyRequest.Builder> getStreamKeyRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IvsException {
        return getStreamKey(GetStreamKeyRequest.builder().applyMutation(getStreamKeyRequest).build());
    }

    /**
     * <p>
     * Imports the public portion of a new key pair and returns its <code>arn</code> and <code>fingerprint</code>. The
     * <code>privateKey</code> can then be used to generate viewer authorization tokens, to grant viewers access to
     * authorized channels.
     * </p>
     *
     * @param importPlaybackKeyPairRequest
     * @return Result of the ImportPlaybackKeyPair operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ImportPlaybackKeyPair
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ImportPlaybackKeyPair" target="_top">AWS API
     *      Documentation</a>
     */
    default ImportPlaybackKeyPairResponse importPlaybackKeyPair(ImportPlaybackKeyPairRequest importPlaybackKeyPairRequest)
            throws ValidationException, ConflictException, AccessDeniedException, ServiceQuotaExceededException,
            PendingVerificationException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Imports the public portion of a new key pair and returns its <code>arn</code> and <code>fingerprint</code>. The
     * <code>privateKey</code> can then be used to generate viewer authorization tokens, to grant viewers access to
     * authorized channels.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ImportPlaybackKeyPairRequest.Builder} avoiding the
     * need to create one manually via {@link ImportPlaybackKeyPairRequest#builder()}
     * </p>
     *
     * @param importPlaybackKeyPairRequest
     *        A {@link Consumer} that will call methods on {@link ImportPlaybackKeyPairRequest.Builder} to create a
     *        request.
     * @return Result of the ImportPlaybackKeyPair operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ImportPlaybackKeyPair
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ImportPlaybackKeyPair" target="_top">AWS API
     *      Documentation</a>
     */
    default ImportPlaybackKeyPairResponse importPlaybackKeyPair(
            Consumer<ImportPlaybackKeyPairRequest.Builder> importPlaybackKeyPairRequest) throws ValidationException,
            ConflictException, AccessDeniedException, ServiceQuotaExceededException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsException {
        return importPlaybackKeyPair(ImportPlaybackKeyPairRequest.builder().applyMutation(importPlaybackKeyPairRequest).build());
    }

    /**
     * <p>
     * Gets summary information about all channels in your account, in the AWS region where the API request is
     * processed. This list can be filtered to match a specified string.
     * </p>
     *
     * @param listChannelsRequest
     * @return Result of the ListChannels operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChannelsResponse listChannels(ListChannelsRequest listChannelsRequest) throws ValidationException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about all channels in your account, in the AWS region where the API request is
     * processed. This list can be filtered to match a specified string.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChannelsRequest.Builder} avoiding the need to
     * create one manually via {@link ListChannelsRequest#builder()}
     * </p>
     *
     * @param listChannelsRequest
     *        A {@link Consumer} that will call methods on {@link ListChannelsRequest.Builder} to create a request.
     * @return Result of the ListChannels operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChannelsResponse listChannels(Consumer<ListChannelsRequest.Builder> listChannelsRequest)
            throws ValidationException, AccessDeniedException, ConflictException, AwsServiceException, SdkClientException,
            IvsException {
        return listChannels(ListChannelsRequest.builder().applyMutation(listChannelsRequest).build());
    }

    /**
     * <p>
     * Gets summary information about all channels in your account, in the AWS region where the API request is
     * processed. This list can be filtered to match a specified string.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listChannels(software.amazon.awssdk.services.ivs.model.ListChannelsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivs.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     *     for (software.amazon.awssdk.services.ivs.model.ListChannelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannels(software.amazon.awssdk.services.ivs.model.ListChannelsRequest)} operation.</b>
     * </p>
     *
     * @param listChannelsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChannelsIterable listChannelsPaginator(ListChannelsRequest listChannelsRequest) throws ValidationException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about all channels in your account, in the AWS region where the API request is
     * processed. This list can be filtered to match a specified string.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listChannels(software.amazon.awssdk.services.ivs.model.ListChannelsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivs.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     *     for (software.amazon.awssdk.services.ivs.model.ListChannelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannels(software.amazon.awssdk.services.ivs.model.ListChannelsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListChannelsRequest.Builder} avoiding the need to
     * create one manually via {@link ListChannelsRequest#builder()}
     * </p>
     *
     * @param listChannelsRequest
     *        A {@link Consumer} that will call methods on {@link ListChannelsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    default ListChannelsIterable listChannelsPaginator(Consumer<ListChannelsRequest.Builder> listChannelsRequest)
            throws ValidationException, AccessDeniedException, ConflictException, AwsServiceException, SdkClientException,
            IvsException {
        return listChannelsPaginator(ListChannelsRequest.builder().applyMutation(listChannelsRequest).build());
    }

    /**
     * <p>
     * Gets summary information about playback key pairs.
     * </p>
     *
     * @param listPlaybackKeyPairsRequest
     * @return Result of the ListPlaybackKeyPairs operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListPlaybackKeyPairs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListPlaybackKeyPairs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListPlaybackKeyPairsResponse listPlaybackKeyPairs(ListPlaybackKeyPairsRequest listPlaybackKeyPairsRequest)
            throws ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about playback key pairs.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPlaybackKeyPairsRequest.Builder} avoiding the
     * need to create one manually via {@link ListPlaybackKeyPairsRequest#builder()}
     * </p>
     *
     * @param listPlaybackKeyPairsRequest
     *        A {@link Consumer} that will call methods on {@link ListPlaybackKeyPairsRequest.Builder} to create a
     *        request.
     * @return Result of the ListPlaybackKeyPairs operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListPlaybackKeyPairs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListPlaybackKeyPairs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListPlaybackKeyPairsResponse listPlaybackKeyPairs(
            Consumer<ListPlaybackKeyPairsRequest.Builder> listPlaybackKeyPairsRequest) throws ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, IvsException {
        return listPlaybackKeyPairs(ListPlaybackKeyPairsRequest.builder().applyMutation(listPlaybackKeyPairsRequest).build());
    }

    /**
     * <p>
     * Gets summary information about playback key pairs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPlaybackKeyPairs(software.amazon.awssdk.services.ivs.model.ListPlaybackKeyPairsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListPlaybackKeyPairsIterable responses = client.listPlaybackKeyPairsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivs.paginators.ListPlaybackKeyPairsIterable responses = client
     *             .listPlaybackKeyPairsPaginator(request);
     *     for (software.amazon.awssdk.services.ivs.model.ListPlaybackKeyPairsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListPlaybackKeyPairsIterable responses = client.listPlaybackKeyPairsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlaybackKeyPairs(software.amazon.awssdk.services.ivs.model.ListPlaybackKeyPairsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listPlaybackKeyPairsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListPlaybackKeyPairs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListPlaybackKeyPairs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListPlaybackKeyPairsIterable listPlaybackKeyPairsPaginator(ListPlaybackKeyPairsRequest listPlaybackKeyPairsRequest)
            throws ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about playback key pairs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPlaybackKeyPairs(software.amazon.awssdk.services.ivs.model.ListPlaybackKeyPairsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListPlaybackKeyPairsIterable responses = client.listPlaybackKeyPairsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivs.paginators.ListPlaybackKeyPairsIterable responses = client
     *             .listPlaybackKeyPairsPaginator(request);
     *     for (software.amazon.awssdk.services.ivs.model.ListPlaybackKeyPairsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListPlaybackKeyPairsIterable responses = client.listPlaybackKeyPairsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlaybackKeyPairs(software.amazon.awssdk.services.ivs.model.ListPlaybackKeyPairsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListPlaybackKeyPairsRequest.Builder} avoiding the
     * need to create one manually via {@link ListPlaybackKeyPairsRequest#builder()}
     * </p>
     *
     * @param listPlaybackKeyPairsRequest
     *        A {@link Consumer} that will call methods on {@link ListPlaybackKeyPairsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListPlaybackKeyPairs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListPlaybackKeyPairs" target="_top">AWS API
     *      Documentation</a>
     */
    default ListPlaybackKeyPairsIterable listPlaybackKeyPairsPaginator(
            Consumer<ListPlaybackKeyPairsRequest.Builder> listPlaybackKeyPairsRequest) throws ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, IvsException {
        return listPlaybackKeyPairsPaginator(ListPlaybackKeyPairsRequest.builder().applyMutation(listPlaybackKeyPairsRequest)
                .build());
    }

    /**
     * <p>
     * Gets summary information about stream keys for the specified channel.
     * </p>
     *
     * @param listStreamKeysRequest
     * @return Result of the ListStreamKeys operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreamKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreamKeys" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStreamKeysResponse listStreamKeys(ListStreamKeysRequest listStreamKeysRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about stream keys for the specified channel.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListStreamKeysRequest.Builder} avoiding the need to
     * create one manually via {@link ListStreamKeysRequest#builder()}
     * </p>
     *
     * @param listStreamKeysRequest
     *        A {@link Consumer} that will call methods on {@link ListStreamKeysRequest.Builder} to create a request.
     * @return Result of the ListStreamKeys operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreamKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreamKeys" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStreamKeysResponse listStreamKeys(Consumer<ListStreamKeysRequest.Builder> listStreamKeysRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IvsException {
        return listStreamKeys(ListStreamKeysRequest.builder().applyMutation(listStreamKeysRequest).build());
    }

    /**
     * <p>
     * Gets summary information about stream keys for the specified channel.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listStreamKeys(software.amazon.awssdk.services.ivs.model.ListStreamKeysRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListStreamKeysIterable responses = client.listStreamKeysPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivs.paginators.ListStreamKeysIterable responses = client.listStreamKeysPaginator(request);
     *     for (software.amazon.awssdk.services.ivs.model.ListStreamKeysResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListStreamKeysIterable responses = client.listStreamKeysPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreamKeys(software.amazon.awssdk.services.ivs.model.ListStreamKeysRequest)} operation.</b>
     * </p>
     *
     * @param listStreamKeysRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreamKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreamKeys" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStreamKeysIterable listStreamKeysPaginator(ListStreamKeysRequest listStreamKeysRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about stream keys for the specified channel.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listStreamKeys(software.amazon.awssdk.services.ivs.model.ListStreamKeysRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListStreamKeysIterable responses = client.listStreamKeysPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivs.paginators.ListStreamKeysIterable responses = client.listStreamKeysPaginator(request);
     *     for (software.amazon.awssdk.services.ivs.model.ListStreamKeysResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListStreamKeysIterable responses = client.listStreamKeysPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreamKeys(software.amazon.awssdk.services.ivs.model.ListStreamKeysRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListStreamKeysRequest.Builder} avoiding the need to
     * create one manually via {@link ListStreamKeysRequest#builder()}
     * </p>
     *
     * @param listStreamKeysRequest
     *        A {@link Consumer} that will call methods on {@link ListStreamKeysRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreamKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreamKeys" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStreamKeysIterable listStreamKeysPaginator(Consumer<ListStreamKeysRequest.Builder> listStreamKeysRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IvsException {
        return listStreamKeysPaginator(ListStreamKeysRequest.builder().applyMutation(listStreamKeysRequest).build());
    }

    /**
     * <p>
     * Gets summary information about live streams in your account, in the AWS region where the API request is
     * processed.
     * </p>
     *
     * @param listStreamsRequest
     * @return Result of the ListStreams operation returned by the service.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStreamsResponse listStreams(ListStreamsRequest listStreamsRequest) throws AccessDeniedException,
            AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about live streams in your account, in the AWS region where the API request is
     * processed.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListStreamsRequest.Builder} avoiding the need to
     * create one manually via {@link ListStreamsRequest#builder()}
     * </p>
     *
     * @param listStreamsRequest
     *        A {@link Consumer} that will call methods on {@link ListStreamsRequest.Builder} to create a request.
     * @return Result of the ListStreams operation returned by the service.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStreamsResponse listStreams(Consumer<ListStreamsRequest.Builder> listStreamsRequest)
            throws AccessDeniedException, AwsServiceException, SdkClientException, IvsException {
        return listStreams(ListStreamsRequest.builder().applyMutation(listStreamsRequest).build());
    }

    /**
     * <p>
     * Gets summary information about live streams in your account, in the AWS region where the API request is
     * processed.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listStreams(software.amazon.awssdk.services.ivs.model.ListStreamsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivs.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     *     for (software.amazon.awssdk.services.ivs.model.ListStreamsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreams(software.amazon.awssdk.services.ivs.model.ListStreamsRequest)} operation.</b>
     * </p>
     *
     * @param listStreamsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStreamsIterable listStreamsPaginator(ListStreamsRequest listStreamsRequest) throws AccessDeniedException,
            AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about live streams in your account, in the AWS region where the API request is
     * processed.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listStreams(software.amazon.awssdk.services.ivs.model.ListStreamsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivs.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     *     for (software.amazon.awssdk.services.ivs.model.ListStreamsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListStreamsIterable responses = client.listStreamsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStreams(software.amazon.awssdk.services.ivs.model.ListStreamsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListStreamsRequest.Builder} avoiding the need to
     * create one manually via {@link ListStreamsRequest#builder()}
     * </p>
     *
     * @param listStreamsRequest
     *        A {@link Consumer} that will call methods on {@link ListStreamsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListStreams" target="_top">AWS API
     *      Documentation</a>
     */
    default ListStreamsIterable listStreamsPaginator(Consumer<ListStreamsRequest.Builder> listStreamsRequest)
            throws AccessDeniedException, AwsServiceException, SdkClientException, IvsException {
        return listStreamsPaginator(ListStreamsRequest.builder().applyMutation(listStreamsRequest).build());
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InternalServerException
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, IvsException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.ivs.model.ListTagsForResourceRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivs.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.ivs.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.ivs.model.ListTagsForResourceRequest)} operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceIterable listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.ivs.model.ListTagsForResourceRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivs.paginators.ListTagsForResourceIterable responses = client
     *             .listTagsForResourcePaginator(request);
     *     for (software.amazon.awssdk.services.ivs.model.ListTagsForResourceResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivs.paginators.ListTagsForResourceIterable responses = client.listTagsForResourcePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.ivs.model.ListTagsForResourceRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    default ListTagsForResourceIterable listTagsForResourcePaginator(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, IvsException {
        return listTagsForResourcePaginator(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest)
                .build());
    }

    /**
     * <p>
     * Inserts metadata into an RTMPS stream for the specified channel. A maximum of 5 requests per second per channel
     * is allowed, each with a maximum 1KB payload.
     * </p>
     *
     * @param putMetadataRequest
     * @return Result of the PutMetadata operation returned by the service.
     * @throws ThrottlingException
     * @throws ResourceNotFoundException
     * @throws ChannelNotBroadcastingException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.PutMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/PutMetadata" target="_top">AWS API
     *      Documentation</a>
     */
    default PutMetadataResponse putMetadata(PutMetadataRequest putMetadataRequest) throws ThrottlingException,
            ResourceNotFoundException, ChannelNotBroadcastingException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Inserts metadata into an RTMPS stream for the specified channel. A maximum of 5 requests per second per channel
     * is allowed, each with a maximum 1KB payload.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutMetadataRequest.Builder} avoiding the need to
     * create one manually via {@link PutMetadataRequest#builder()}
     * </p>
     *
     * @param putMetadataRequest
     *        A {@link Consumer} that will call methods on {@link PutMetadataRequest.Builder} to create a request.
     * @return Result of the PutMetadata operation returned by the service.
     * @throws ThrottlingException
     * @throws ResourceNotFoundException
     * @throws ChannelNotBroadcastingException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.PutMetadata
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/PutMetadata" target="_top">AWS API
     *      Documentation</a>
     */
    default PutMetadataResponse putMetadata(Consumer<PutMetadataRequest.Builder> putMetadataRequest) throws ThrottlingException,
            ResourceNotFoundException, ChannelNotBroadcastingException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, IvsException {
        return putMetadata(PutMetadataRequest.builder().applyMutation(putMetadataRequest).build());
    }

    /**
     * <p>
     * Disconnects the incoming RTMPS stream for the specified channel. Can be used in conjunction with
     * <a>DeleteStreamKey</a> to prevent further streaming to a channel.
     * </p>
     * <note>
     * <p>
     * Many streaming client-software libraries automatically reconnect a dropped RTMPS session, so to stop the stream
     * permanently, you may want to first revoke the <code>streamKey</code> attached to the channel.
     * </p>
     * </note>
     *
     * @param stopStreamRequest
     * @return Result of the StopStream operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ChannelNotBroadcastingException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws StreamUnavailableException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.StopStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/StopStream" target="_top">AWS API
     *      Documentation</a>
     */
    default StopStreamResponse stopStream(StopStreamRequest stopStreamRequest) throws ResourceNotFoundException,
            ChannelNotBroadcastingException, ValidationException, AccessDeniedException, StreamUnavailableException,
            AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Disconnects the incoming RTMPS stream for the specified channel. Can be used in conjunction with
     * <a>DeleteStreamKey</a> to prevent further streaming to a channel.
     * </p>
     * <note>
     * <p>
     * Many streaming client-software libraries automatically reconnect a dropped RTMPS session, so to stop the stream
     * permanently, you may want to first revoke the <code>streamKey</code> attached to the channel.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StopStreamRequest.Builder} avoiding the need to
     * create one manually via {@link StopStreamRequest#builder()}
     * </p>
     *
     * @param stopStreamRequest
     *        A {@link Consumer} that will call methods on {@link StopStreamRequest.Builder} to create a request.
     * @return Result of the StopStream operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ChannelNotBroadcastingException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws StreamUnavailableException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.StopStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/StopStream" target="_top">AWS API
     *      Documentation</a>
     */
    default StopStreamResponse stopStream(Consumer<StopStreamRequest.Builder> stopStreamRequest)
            throws ResourceNotFoundException, ChannelNotBroadcastingException, ValidationException, AccessDeniedException,
            StreamUnavailableException, AwsServiceException, SdkClientException, IvsException {
        return stopStream(StopStreamRequest.builder().applyMutation(stopStreamRequest).build());
    }

    /**
     * <p>
     * Adds or updates tags for the AWS resource with the specified ARN.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds or updates tags for the AWS resource with the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws InternalServerException
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IvsException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes tags from the resource with the specified ARN.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InternalServerException,
            ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes tags from the resource with the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws InternalServerException
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, IvsException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates a channel's configuration. This does not affect an ongoing stream of this channel. You must stop and
     * restart the stream for the changes to take effect.
     * </p>
     *
     * @param updateChannelRequest
     * @return Result of the UpdateChannel operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ConflictException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/UpdateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateChannelResponse updateChannel(UpdateChannelRequest updateChannelRequest) throws ValidationException,
            AccessDeniedException, ResourceNotFoundException, ConflictException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a channel's configuration. This does not affect an ongoing stream of this channel. You must stop and
     * restart the stream for the changes to take effect.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateChannelRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateChannelRequest#builder()}
     * </p>
     *
     * @param updateChannelRequest
     *        A {@link Consumer} that will call methods on {@link UpdateChannelRequest.Builder} to create a request.
     * @return Result of the UpdateChannel operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ConflictException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsClient.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/UpdateChannel" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateChannelResponse updateChannel(Consumer<UpdateChannelRequest.Builder> updateChannelRequest)
            throws ValidationException, AccessDeniedException, ResourceNotFoundException, ConflictException,
            PendingVerificationException, AwsServiceException, SdkClientException, IvsException {
        return updateChannel(UpdateChannelRequest.builder().applyMutation(updateChannelRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("ivs");
    }
}
