/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ivs.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Object that captures the Amazon IVS configuration that the customer provisioned, the ingest configurations that the
 * broadcaster used, and the most recent Amazon IVS stream events it encountered.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StreamSession implements SdkPojo, Serializable, ToCopyableBuilder<StreamSession.Builder, StreamSession> {
    private static final SdkField<String> STREAM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("streamId").getter(getter(StreamSession::streamId)).setter(setter(Builder::streamId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("streamId").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime")
            .getter(getter(StreamSession::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime")
            .getter(getter(StreamSession::endTime))
            .setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Channel> CHANNEL_FIELD = SdkField.<Channel> builder(MarshallingType.SDK_POJO)
            .memberName("channel").getter(getter(StreamSession::channel)).setter(setter(Builder::channel))
            .constructor(Channel::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("channel").build()).build();

    private static final SdkField<IngestConfiguration> INGEST_CONFIGURATION_FIELD = SdkField
            .<IngestConfiguration> builder(MarshallingType.SDK_POJO).memberName("ingestConfiguration")
            .getter(getter(StreamSession::ingestConfiguration)).setter(setter(Builder::ingestConfiguration))
            .constructor(IngestConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ingestConfiguration").build())
            .build();

    private static final SdkField<RecordingConfiguration> RECORDING_CONFIGURATION_FIELD = SdkField
            .<RecordingConfiguration> builder(MarshallingType.SDK_POJO).memberName("recordingConfiguration")
            .getter(getter(StreamSession::recordingConfiguration)).setter(setter(Builder::recordingConfiguration))
            .constructor(RecordingConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recordingConfiguration").build())
            .build();

    private static final SdkField<List<StreamEvent>> TRUNCATED_EVENTS_FIELD = SdkField
            .<List<StreamEvent>> builder(MarshallingType.LIST)
            .memberName("truncatedEvents")
            .getter(getter(StreamSession::truncatedEvents))
            .setter(setter(Builder::truncatedEvents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("truncatedEvents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<StreamEvent> builder(MarshallingType.SDK_POJO)
                                            .constructor(StreamEvent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STREAM_ID_FIELD,
            START_TIME_FIELD, END_TIME_FIELD, CHANNEL_FIELD, INGEST_CONFIGURATION_FIELD, RECORDING_CONFIGURATION_FIELD,
            TRUNCATED_EVENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String streamId;

    private final Instant startTime;

    private final Instant endTime;

    private final Channel channel;

    private final IngestConfiguration ingestConfiguration;

    private final RecordingConfiguration recordingConfiguration;

    private final List<StreamEvent> truncatedEvents;

    private StreamSession(BuilderImpl builder) {
        this.streamId = builder.streamId;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.channel = builder.channel;
        this.ingestConfiguration = builder.ingestConfiguration;
        this.recordingConfiguration = builder.recordingConfiguration;
        this.truncatedEvents = builder.truncatedEvents;
    }

    /**
     * <p>
     * Unique identifier for a live or previously live stream in the specified channel.
     * </p>
     * 
     * @return Unique identifier for a live or previously live stream in the specified channel.
     */
    public final String streamId() {
        return streamId;
    }

    /**
     * <p>
     * Time when the channel went live. This is an ISO 8601 timestamp; <i>note that this is returned as a string</i>.
     * </p>
     * 
     * @return Time when the channel went live. This is an ISO 8601 timestamp; <i>note that this is returned as a
     *         string</i>.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * Time when the channel went offline. This is an ISO 8601 timestamp; <i>note that this is returned as a string</i>.
     * For live streams, this is <code>NULL</code>.
     * </p>
     * 
     * @return Time when the channel went offline. This is an ISO 8601 timestamp; <i>note that this is returned as a
     *         string</i>. For live streams, this is <code>NULL</code>.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The properties of the channel at the time of going live.
     * </p>
     * 
     * @return The properties of the channel at the time of going live.
     */
    public final Channel channel() {
        return channel;
    }

    /**
     * <p>
     * The properties of the incoming RTMP stream for the stream.
     * </p>
     * 
     * @return The properties of the incoming RTMP stream for the stream.
     */
    public final IngestConfiguration ingestConfiguration() {
        return ingestConfiguration;
    }

    /**
     * <p>
     * The properties of recording the live stream.
     * </p>
     * 
     * @return The properties of recording the live stream.
     */
    public final RecordingConfiguration recordingConfiguration() {
        return recordingConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the TruncatedEvents property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTruncatedEvents() {
        return truncatedEvents != null && !(truncatedEvents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of Amazon IVS events that the stream encountered. The list is sorted by most recent events and contains up
     * to 500 events. For Amazon IVS events, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/userguide/eventbridge.html">Using Amazon EventBridge with Amazon
     * IVS</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTruncatedEvents} method.
     * </p>
     * 
     * @return List of Amazon IVS events that the stream encountered. The list is sorted by most recent events and
     *         contains up to 500 events. For Amazon IVS events, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/userguide/eventbridge.html">Using Amazon EventBridge with
     *         Amazon IVS</a>.
     */
    public final List<StreamEvent> truncatedEvents() {
        return truncatedEvents;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(streamId());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(channel());
        hashCode = 31 * hashCode + Objects.hashCode(ingestConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(recordingConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasTruncatedEvents() ? truncatedEvents() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StreamSession)) {
            return false;
        }
        StreamSession other = (StreamSession) obj;
        return Objects.equals(streamId(), other.streamId()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(channel(), other.channel())
                && Objects.equals(ingestConfiguration(), other.ingestConfiguration())
                && Objects.equals(recordingConfiguration(), other.recordingConfiguration())
                && hasTruncatedEvents() == other.hasTruncatedEvents()
                && Objects.equals(truncatedEvents(), other.truncatedEvents());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StreamSession").add("StreamId", streamId()).add("StartTime", startTime())
                .add("EndTime", endTime()).add("Channel", channel()).add("IngestConfiguration", ingestConfiguration())
                .add("RecordingConfiguration", recordingConfiguration())
                .add("TruncatedEvents", hasTruncatedEvents() ? truncatedEvents() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "streamId":
            return Optional.ofNullable(clazz.cast(streamId()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "channel":
            return Optional.ofNullable(clazz.cast(channel()));
        case "ingestConfiguration":
            return Optional.ofNullable(clazz.cast(ingestConfiguration()));
        case "recordingConfiguration":
            return Optional.ofNullable(clazz.cast(recordingConfiguration()));
        case "truncatedEvents":
            return Optional.ofNullable(clazz.cast(truncatedEvents()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StreamSession, T> g) {
        return obj -> g.apply((StreamSession) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StreamSession> {
        /**
         * <p>
         * Unique identifier for a live or previously live stream in the specified channel.
         * </p>
         * 
         * @param streamId
         *        Unique identifier for a live or previously live stream in the specified channel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamId(String streamId);

        /**
         * <p>
         * Time when the channel went live. This is an ISO 8601 timestamp; <i>note that this is returned as a
         * string</i>.
         * </p>
         * 
         * @param startTime
         *        Time when the channel went live. This is an ISO 8601 timestamp; <i>note that this is returned as a
         *        string</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * Time when the channel went offline. This is an ISO 8601 timestamp; <i>note that this is returned as a
         * string</i>. For live streams, this is <code>NULL</code>.
         * </p>
         * 
         * @param endTime
         *        Time when the channel went offline. This is an ISO 8601 timestamp; <i>note that this is returned as a
         *        string</i>. For live streams, this is <code>NULL</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The properties of the channel at the time of going live.
         * </p>
         * 
         * @param channel
         *        The properties of the channel at the time of going live.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder channel(Channel channel);

        /**
         * <p>
         * The properties of the channel at the time of going live.
         * </p>
         * This is a convenience method that creates an instance of the {@link Channel.Builder} avoiding the need to
         * create one manually via {@link Channel#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Channel.Builder#build()} is called immediately and its result is
         * passed to {@link #channel(Channel)}.
         * 
         * @param channel
         *        a consumer that will call methods on {@link Channel.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #channel(Channel)
         */
        default Builder channel(Consumer<Channel.Builder> channel) {
            return channel(Channel.builder().applyMutation(channel).build());
        }

        /**
         * <p>
         * The properties of the incoming RTMP stream for the stream.
         * </p>
         * 
         * @param ingestConfiguration
         *        The properties of the incoming RTMP stream for the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ingestConfiguration(IngestConfiguration ingestConfiguration);

        /**
         * <p>
         * The properties of the incoming RTMP stream for the stream.
         * </p>
         * This is a convenience method that creates an instance of the {@link IngestConfiguration.Builder} avoiding the
         * need to create one manually via {@link IngestConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IngestConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #ingestConfiguration(IngestConfiguration)}.
         * 
         * @param ingestConfiguration
         *        a consumer that will call methods on {@link IngestConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ingestConfiguration(IngestConfiguration)
         */
        default Builder ingestConfiguration(Consumer<IngestConfiguration.Builder> ingestConfiguration) {
            return ingestConfiguration(IngestConfiguration.builder().applyMutation(ingestConfiguration).build());
        }

        /**
         * <p>
         * The properties of recording the live stream.
         * </p>
         * 
         * @param recordingConfiguration
         *        The properties of recording the live stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordingConfiguration(RecordingConfiguration recordingConfiguration);

        /**
         * <p>
         * The properties of recording the live stream.
         * </p>
         * This is a convenience method that creates an instance of the {@link RecordingConfiguration.Builder} avoiding
         * the need to create one manually via {@link RecordingConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RecordingConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #recordingConfiguration(RecordingConfiguration)}.
         * 
         * @param recordingConfiguration
         *        a consumer that will call methods on {@link RecordingConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recordingConfiguration(RecordingConfiguration)
         */
        default Builder recordingConfiguration(Consumer<RecordingConfiguration.Builder> recordingConfiguration) {
            return recordingConfiguration(RecordingConfiguration.builder().applyMutation(recordingConfiguration).build());
        }

        /**
         * <p>
         * List of Amazon IVS events that the stream encountered. The list is sorted by most recent events and contains
         * up to 500 events. For Amazon IVS events, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/userguide/eventbridge.html">Using Amazon EventBridge with Amazon
         * IVS</a>.
         * </p>
         * 
         * @param truncatedEvents
         *        List of Amazon IVS events that the stream encountered. The list is sorted by most recent events and
         *        contains up to 500 events. For Amazon IVS events, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/userguide/eventbridge.html">Using Amazon EventBridge with
         *        Amazon IVS</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder truncatedEvents(Collection<StreamEvent> truncatedEvents);

        /**
         * <p>
         * List of Amazon IVS events that the stream encountered. The list is sorted by most recent events and contains
         * up to 500 events. For Amazon IVS events, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/userguide/eventbridge.html">Using Amazon EventBridge with Amazon
         * IVS</a>.
         * </p>
         * 
         * @param truncatedEvents
         *        List of Amazon IVS events that the stream encountered. The list is sorted by most recent events and
         *        contains up to 500 events. For Amazon IVS events, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/userguide/eventbridge.html">Using Amazon EventBridge with
         *        Amazon IVS</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder truncatedEvents(StreamEvent... truncatedEvents);

        /**
         * <p>
         * List of Amazon IVS events that the stream encountered. The list is sorted by most recent events and contains
         * up to 500 events. For Amazon IVS events, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/userguide/eventbridge.html">Using Amazon EventBridge with Amazon
         * IVS</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ivs.model.StreamEvent.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ivs.model.StreamEvent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ivs.model.StreamEvent.Builder#build()} is called immediately and its
         * result is passed to {@link #truncatedEvents(List<StreamEvent>)}.
         * 
         * @param truncatedEvents
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ivs.model.StreamEvent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #truncatedEvents(java.util.Collection<StreamEvent>)
         */
        Builder truncatedEvents(Consumer<StreamEvent.Builder>... truncatedEvents);
    }

    static final class BuilderImpl implements Builder {
        private String streamId;

        private Instant startTime;

        private Instant endTime;

        private Channel channel;

        private IngestConfiguration ingestConfiguration;

        private RecordingConfiguration recordingConfiguration;

        private List<StreamEvent> truncatedEvents = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StreamSession model) {
            streamId(model.streamId);
            startTime(model.startTime);
            endTime(model.endTime);
            channel(model.channel);
            ingestConfiguration(model.ingestConfiguration);
            recordingConfiguration(model.recordingConfiguration);
            truncatedEvents(model.truncatedEvents);
        }

        public final String getStreamId() {
            return streamId;
        }

        public final void setStreamId(String streamId) {
            this.streamId = streamId;
        }

        @Override
        public final Builder streamId(String streamId) {
            this.streamId = streamId;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final Channel.Builder getChannel() {
            return channel != null ? channel.toBuilder() : null;
        }

        public final void setChannel(Channel.BuilderImpl channel) {
            this.channel = channel != null ? channel.build() : null;
        }

        @Override
        public final Builder channel(Channel channel) {
            this.channel = channel;
            return this;
        }

        public final IngestConfiguration.Builder getIngestConfiguration() {
            return ingestConfiguration != null ? ingestConfiguration.toBuilder() : null;
        }

        public final void setIngestConfiguration(IngestConfiguration.BuilderImpl ingestConfiguration) {
            this.ingestConfiguration = ingestConfiguration != null ? ingestConfiguration.build() : null;
        }

        @Override
        public final Builder ingestConfiguration(IngestConfiguration ingestConfiguration) {
            this.ingestConfiguration = ingestConfiguration;
            return this;
        }

        public final RecordingConfiguration.Builder getRecordingConfiguration() {
            return recordingConfiguration != null ? recordingConfiguration.toBuilder() : null;
        }

        public final void setRecordingConfiguration(RecordingConfiguration.BuilderImpl recordingConfiguration) {
            this.recordingConfiguration = recordingConfiguration != null ? recordingConfiguration.build() : null;
        }

        @Override
        public final Builder recordingConfiguration(RecordingConfiguration recordingConfiguration) {
            this.recordingConfiguration = recordingConfiguration;
            return this;
        }

        public final List<StreamEvent.Builder> getTruncatedEvents() {
            List<StreamEvent.Builder> result = StreamEventsCopier.copyToBuilder(this.truncatedEvents);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTruncatedEvents(Collection<StreamEvent.BuilderImpl> truncatedEvents) {
            this.truncatedEvents = StreamEventsCopier.copyFromBuilder(truncatedEvents);
        }

        @Override
        public final Builder truncatedEvents(Collection<StreamEvent> truncatedEvents) {
            this.truncatedEvents = StreamEventsCopier.copy(truncatedEvents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder truncatedEvents(StreamEvent... truncatedEvents) {
            truncatedEvents(Arrays.asList(truncatedEvents));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder truncatedEvents(Consumer<StreamEvent.Builder>... truncatedEvents) {
            truncatedEvents(Stream.of(truncatedEvents).map(c -> StreamEvent.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public StreamSession build() {
            return new StreamSession(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
