/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ivschat;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.ivschat.model.CreateChatTokenRequest;
import software.amazon.awssdk.services.ivschat.model.CreateChatTokenResponse;
import software.amazon.awssdk.services.ivschat.model.CreateLoggingConfigurationRequest;
import software.amazon.awssdk.services.ivschat.model.CreateLoggingConfigurationResponse;
import software.amazon.awssdk.services.ivschat.model.CreateRoomRequest;
import software.amazon.awssdk.services.ivschat.model.CreateRoomResponse;
import software.amazon.awssdk.services.ivschat.model.DeleteLoggingConfigurationRequest;
import software.amazon.awssdk.services.ivschat.model.DeleteLoggingConfigurationResponse;
import software.amazon.awssdk.services.ivschat.model.DeleteMessageRequest;
import software.amazon.awssdk.services.ivschat.model.DeleteMessageResponse;
import software.amazon.awssdk.services.ivschat.model.DeleteRoomRequest;
import software.amazon.awssdk.services.ivschat.model.DeleteRoomResponse;
import software.amazon.awssdk.services.ivschat.model.DisconnectUserRequest;
import software.amazon.awssdk.services.ivschat.model.DisconnectUserResponse;
import software.amazon.awssdk.services.ivschat.model.GetLoggingConfigurationRequest;
import software.amazon.awssdk.services.ivschat.model.GetLoggingConfigurationResponse;
import software.amazon.awssdk.services.ivschat.model.GetRoomRequest;
import software.amazon.awssdk.services.ivschat.model.GetRoomResponse;
import software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest;
import software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsResponse;
import software.amazon.awssdk.services.ivschat.model.ListRoomsRequest;
import software.amazon.awssdk.services.ivschat.model.ListRoomsResponse;
import software.amazon.awssdk.services.ivschat.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ivschat.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ivschat.model.SendEventRequest;
import software.amazon.awssdk.services.ivschat.model.SendEventResponse;
import software.amazon.awssdk.services.ivschat.model.TagResourceRequest;
import software.amazon.awssdk.services.ivschat.model.TagResourceResponse;
import software.amazon.awssdk.services.ivschat.model.UntagResourceRequest;
import software.amazon.awssdk.services.ivschat.model.UntagResourceResponse;
import software.amazon.awssdk.services.ivschat.model.UpdateLoggingConfigurationRequest;
import software.amazon.awssdk.services.ivschat.model.UpdateLoggingConfigurationResponse;
import software.amazon.awssdk.services.ivschat.model.UpdateRoomRequest;
import software.amazon.awssdk.services.ivschat.model.UpdateRoomResponse;
import software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsPublisher;
import software.amazon.awssdk.services.ivschat.paginators.ListRoomsPublisher;

/**
 * Service client for accessing ivschat asynchronously. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * <b>Introduction</b>
 * </p>
 * <p>
 * The Amazon IVS Chat control-plane API enables you to create and manage Amazon IVS Chat resources. You also need to
 * integrate with the <a href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/chat-messaging-api.html">
 * Amazon IVS Chat Messaging API</a>, to enable users to interact with chat rooms in real time.
 * </p>
 * <p>
 * The API is an AWS regional service. For a list of supported regions and Amazon IVS Chat HTTPS service endpoints, see
 * the Amazon IVS Chat information on the <a href="https://docs.aws.amazon.com/general/latest/gr/ivs.html">Amazon IVS
 * page</a> in the <i>AWS General Reference</i>.
 * </p>
 * <p>
 * <b>Notes on terminology:</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * You create service applications using the Amazon IVS Chat API. We refer to these as <i>applications</i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * You create front-end client applications (browser and Android/iOS apps) using the Amazon IVS Chat Messaging API. We
 * refer to these as <i>clients</i>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Resources</b>
 * </p>
 * <p>
 * The following resources are part of Amazon IVS Chat:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>LoggingConfiguration</b> — A configuration that allows customers to store and record sent messages in a chat room.
 * See the Logging Configuration endpoints for more information.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Room</b> — The central Amazon IVS Chat resource through which clients connect to and exchange chat messages. See
 * the Room endpoints for more information.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Tagging</b>
 * </p>
 * <p>
 * A <i>tag</i> is a metadata label that you assign to an AWS resource. A tag comprises a <i>key</i> and a <i>value</i>,
 * both set by you. For example, you might set a tag as <code>topic:nature</code> to label a particular video category.
 * See <a href="https://docs.aws.amazon.com/general/latest/gr/aws_tagging.html">Tagging AWS Resources</a> for more
 * information, including restrictions that apply to tags and "Tag naming limits and requirements"; Amazon IVS Chat has
 * no service-specific constraints beyond what is documented there.
 * </p>
 * <p>
 * Tags can help you identify and organize your AWS resources. For example, you can use the same tag for different
 * resources to indicate that they are related. You can also use tags to manage access (see <a
 * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html">Access Tags</a>).
 * </p>
 * <p>
 * The Amazon IVS Chat API has these tag-related endpoints: <a>TagResource</a>, <a>UntagResource</a>, and
 * <a>ListTagsForResource</a>. The following resource supports tagging: Room.
 * </p>
 * <p>
 * At most 50 tags can be applied to a resource.
 * </p>
 * <p>
 * <b>API Access Security</b>
 * </p>
 * <p>
 * Your Amazon IVS Chat applications (service applications and clients) must be authenticated and authorized to access
 * Amazon IVS Chat resources. Note the differences between these concepts:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <i>Authentication</i> is about verifying identity. Requests to the Amazon IVS Chat API must be signed to verify your
 * identity.
 * </p>
 * </li>
 * <li>
 * <p>
 * <i>Authorization</i> is about granting permissions. Your IAM roles need to have permissions for Amazon IVS Chat API
 * requests.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Users (viewers) connect to a room using secure access tokens that you create using the <a>CreateChatToken</a>
 * endpoint through the AWS SDK. You call CreateChatToken for every user’s chat session, passing identity and
 * authorization information about the user.
 * </p>
 * <p>
 * <b>Signing API Requests</b>
 * </p>
 * <p>
 * HTTP API requests must be signed with an AWS SigV4 signature using your AWS security credentials. The AWS Command
 * Line Interface (CLI) and the AWS SDKs take care of signing the underlying API calls for you. However, if your
 * application calls the Amazon IVS Chat HTTP API directly, it’s your responsibility to sign the requests.
 * </p>
 * <p>
 * You generate a signature using valid AWS credentials for an IAM role that has permission to perform the requested
 * action. For example, DeleteMessage requests must be made using an IAM role that has the
 * <code>ivschat:DeleteMessage</code> permission.
 * </p>
 * <p>
 * For more information:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Authentication and generating signatures — See <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/API/sig-v4-authenticating-requests.html">Authenticating Requests
 * (Amazon Web Services Signature Version 4)</a> in the <i>Amazon Web Services General Reference</i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Managing Amazon IVS permissions — See <a
 * href="https://docs.aws.amazon.com/ivs/latest/userguide/security-iam.html">Identity and Access Management</a> on the
 * Security page of the <i>Amazon IVS User Guide</i>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Amazon Resource Names (ARNs)</b>
 * </p>
 * <p>
 * ARNs uniquely identify AWS resources. An ARN is required when you need to specify a resource unambiguously across all
 * of AWS, such as in IAM policies and API calls. For more information, see <a
 * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names</a> in the
 * <i>AWS General Reference</i>.
 * </p>
 * <p>
 * <b>Messaging Endpoints</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>DeleteMessage</a> — Sends an event to a specific room which directs clients to delete a specific message; that is,
 * unrender it from view and delete it from the client’s chat history. This event’s <code>EventName</code> is
 * <code>aws:DELETE_MESSAGE</code>. This replicates the <a
 * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-deletemessage-publish.html">
 * DeleteMessage</a> WebSocket operation in the Amazon IVS Chat Messaging API.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DisconnectUser</a> — Disconnects all connections using a specified user ID from a room. This replicates the <a
 * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-disconnectuser-publish.html">
 * DisconnectUser</a> WebSocket operation in the Amazon IVS Chat Messaging API.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>SendEvent</a> — Sends an event to a room. Use this within your application’s business logic to send events to
 * clients of a room; e.g., to notify clients to change the way the chat UI is rendered.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Chat Token Endpoint</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreateChatToken</a> — Creates an encrypted token that is used by a chat participant to establish an individual
 * WebSocket chat connection to a room. When the token is used to connect to chat, the connection is valid for the
 * session duration specified in the request. The token becomes invalid at the token-expiration timestamp included in
 * the response.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Room Endpoints</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreateRoom</a> — Creates a room that allows clients to connect and pass messages.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeleteRoom</a> — Deletes the specified room.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GetRoom</a> — Gets the specified room.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListRooms</a> — Gets summary information about all your rooms in the AWS region where the API request is
 * processed.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>UpdateRoom</a> — Updates a room’s configuration.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Logging Configuration Endpoints</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreateLoggingConfiguration</a> — Creates a logging configuration that allows clients to store and record sent
 * messages.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeleteLoggingConfiguration</a> — Deletes the specified logging configuration.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GetLoggingConfiguration</a> — Gets the specified logging configuration.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListLoggingConfigurations</a> — Gets summary information about all your logging configurations in the AWS region
 * where the API request is processed.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>UpdateLoggingConfiguration</a> — Updates a specified logging configuration.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Tags Endpoints</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>ListTagsForResource</a> — Gets information about AWS tags for the specified ARN.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>TagResource</a> — Adds or updates tags for the AWS resource with the specified ARN.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>UntagResource</a> — Removes tags from the resource with the specified ARN.
 * </p>
 * </li>
 * </ul>
 * <p>
 * All the above are HTTP operations. There is a separate <i>messaging</i> API for managing Chat resources; see the <a
 * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/chat-messaging-api.html"> Amazon IVS Chat Messaging
 * API Reference</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface IvschatAsyncClient extends AwsClient {
    String SERVICE_NAME = "ivschat";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "ivschat";

    /**
     * <p>
     * Creates an encrypted token that is used by a chat participant to establish an individual WebSocket chat
     * connection to a room. When the token is used to connect to chat, the connection is valid for the session duration
     * specified in the request. The token becomes invalid at the token-expiration timestamp included in the response.
     * </p>
     * <p>
     * Use the <code>capabilities</code> field to permit an end user to send messages or moderate a room.
     * </p>
     * <p>
     * The <code>attributes</code> field securely attaches structured data to the chat session; the data is included
     * within each message sent by the end user and received by other participants in the room. Common use cases for
     * attributes include passing end-user profile data like an icon, display name, colors, badges, and other display
     * features.
     * </p>
     * <p>
     * Encryption keys are owned by Amazon IVS Chat and never used directly by your application.
     * </p>
     *
     * @param createChatTokenRequest
     * @return A Java Future containing the result of the CreateChatToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.CreateChatToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateChatToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateChatTokenResponse> createChatToken(CreateChatTokenRequest createChatTokenRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an encrypted token that is used by a chat participant to establish an individual WebSocket chat
     * connection to a room. When the token is used to connect to chat, the connection is valid for the session duration
     * specified in the request. The token becomes invalid at the token-expiration timestamp included in the response.
     * </p>
     * <p>
     * Use the <code>capabilities</code> field to permit an end user to send messages or moderate a room.
     * </p>
     * <p>
     * The <code>attributes</code> field securely attaches structured data to the chat session; the data is included
     * within each message sent by the end user and received by other participants in the room. Common use cases for
     * attributes include passing end-user profile data like an icon, display name, colors, badges, and other display
     * features.
     * </p>
     * <p>
     * Encryption keys are owned by Amazon IVS Chat and never used directly by your application.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateChatTokenRequest.Builder} avoiding the need
     * to create one manually via {@link CreateChatTokenRequest#builder()}
     * </p>
     *
     * @param createChatTokenRequest
     *        A {@link Consumer} that will call methods on {@link CreateChatTokenRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateChatToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.CreateChatToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateChatToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateChatTokenResponse> createChatToken(
            Consumer<CreateChatTokenRequest.Builder> createChatTokenRequest) {
        return createChatToken(CreateChatTokenRequest.builder().applyMutation(createChatTokenRequest).build());
    }

    /**
     * <p>
     * Creates a logging configuration that allows clients to store and record sent messages.
     * </p>
     *
     * @param createLoggingConfigurationRequest
     * @return A Java Future containing the result of the CreateLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.CreateLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateLoggingConfigurationResponse> createLoggingConfiguration(
            CreateLoggingConfigurationRequest createLoggingConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a logging configuration that allows clients to store and record sent messages.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateLoggingConfigurationRequest.Builder} avoiding
     * the need to create one manually via {@link CreateLoggingConfigurationRequest#builder()}
     * </p>
     *
     * @param createLoggingConfigurationRequest
     *        A {@link Consumer} that will call methods on {@link CreateLoggingConfigurationRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the CreateLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.CreateLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateLoggingConfigurationResponse> createLoggingConfiguration(
            Consumer<CreateLoggingConfigurationRequest.Builder> createLoggingConfigurationRequest) {
        return createLoggingConfiguration(CreateLoggingConfigurationRequest.builder()
                .applyMutation(createLoggingConfigurationRequest).build());
    }

    /**
     * <p>
     * Creates a room that allows clients to connect and pass messages.
     * </p>
     *
     * @param createRoomRequest
     * @return A Java Future containing the result of the CreateRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.CreateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateRoomResponse> createRoom(CreateRoomRequest createRoomRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a room that allows clients to connect and pass messages.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateRoomRequest.Builder} avoiding the need to
     * create one manually via {@link CreateRoomRequest#builder()}
     * </p>
     *
     * @param createRoomRequest
     *        A {@link Consumer} that will call methods on {@link CreateRoomRequest.Builder} to create a request.
     * @return A Java Future containing the result of the CreateRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.CreateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<CreateRoomResponse> createRoom(Consumer<CreateRoomRequest.Builder> createRoomRequest) {
        return createRoom(CreateRoomRequest.builder().applyMutation(createRoomRequest).build());
    }

    /**
     * <p>
     * Deletes the specified logging configuration.
     * </p>
     *
     * @param deleteLoggingConfigurationRequest
     * @return A Java Future containing the result of the DeleteLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.DeleteLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteLoggingConfigurationResponse> deleteLoggingConfiguration(
            DeleteLoggingConfigurationRequest deleteLoggingConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified logging configuration.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteLoggingConfigurationRequest.Builder} avoiding
     * the need to create one manually via {@link DeleteLoggingConfigurationRequest#builder()}
     * </p>
     *
     * @param deleteLoggingConfigurationRequest
     *        A {@link Consumer} that will call methods on {@link DeleteLoggingConfigurationRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.DeleteLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteLoggingConfigurationResponse> deleteLoggingConfiguration(
            Consumer<DeleteLoggingConfigurationRequest.Builder> deleteLoggingConfigurationRequest) {
        return deleteLoggingConfiguration(DeleteLoggingConfigurationRequest.builder()
                .applyMutation(deleteLoggingConfigurationRequest).build());
    }

    /**
     * <p>
     * Sends an event to a specific room which directs clients to delete a specific message; that is, unrender it from
     * view and delete it from the client’s chat history. This event’s <code>EventName</code> is
     * <code>aws:DELETE_MESSAGE</code>. This replicates the <a
     * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-deletemessage-publish.html">
     * DeleteMessage</a> WebSocket operation in the Amazon IVS Chat Messaging API.
     * </p>
     *
     * @param deleteMessageRequest
     * @return A Java Future containing the result of the DeleteMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.DeleteMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteMessage" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteMessageResponse> deleteMessage(DeleteMessageRequest deleteMessageRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends an event to a specific room which directs clients to delete a specific message; that is, unrender it from
     * view and delete it from the client’s chat history. This event’s <code>EventName</code> is
     * <code>aws:DELETE_MESSAGE</code>. This replicates the <a
     * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-deletemessage-publish.html">
     * DeleteMessage</a> WebSocket operation in the Amazon IVS Chat Messaging API.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteMessageRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteMessageRequest#builder()}
     * </p>
     *
     * @param deleteMessageRequest
     *        A {@link Consumer} that will call methods on {@link DeleteMessageRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.DeleteMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteMessage" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteMessageResponse> deleteMessage(Consumer<DeleteMessageRequest.Builder> deleteMessageRequest) {
        return deleteMessage(DeleteMessageRequest.builder().applyMutation(deleteMessageRequest).build());
    }

    /**
     * <p>
     * Deletes the specified room.
     * </p>
     *
     * @param deleteRoomRequest
     * @return A Java Future containing the result of the DeleteRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.DeleteRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteRoomResponse> deleteRoom(DeleteRoomRequest deleteRoomRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified room.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteRoomRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteRoomRequest#builder()}
     * </p>
     *
     * @param deleteRoomRequest
     *        A {@link Consumer} that will call methods on {@link DeleteRoomRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.DeleteRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DeleteRoomResponse> deleteRoom(Consumer<DeleteRoomRequest.Builder> deleteRoomRequest) {
        return deleteRoom(DeleteRoomRequest.builder().applyMutation(deleteRoomRequest).build());
    }

    /**
     * <p>
     * Disconnects all connections using a specified user ID from a room. This replicates the <a
     * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-disconnectuser-publish.html">
     * DisconnectUser</a> WebSocket operation in the Amazon IVS Chat Messaging API.
     * </p>
     *
     * @param disconnectUserRequest
     * @return A Java Future containing the result of the DisconnectUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.DisconnectUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DisconnectUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DisconnectUserResponse> disconnectUser(DisconnectUserRequest disconnectUserRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Disconnects all connections using a specified user ID from a room. This replicates the <a
     * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-disconnectuser-publish.html">
     * DisconnectUser</a> WebSocket operation in the Amazon IVS Chat Messaging API.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DisconnectUserRequest.Builder} avoiding the need to
     * create one manually via {@link DisconnectUserRequest#builder()}
     * </p>
     *
     * @param disconnectUserRequest
     *        A {@link Consumer} that will call methods on {@link DisconnectUserRequest.Builder} to create a request.
     * @return A Java Future containing the result of the DisconnectUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.DisconnectUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DisconnectUser" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<DisconnectUserResponse> disconnectUser(Consumer<DisconnectUserRequest.Builder> disconnectUserRequest) {
        return disconnectUser(DisconnectUserRequest.builder().applyMutation(disconnectUserRequest).build());
    }

    /**
     * <p>
     * Gets the specified logging configuration.
     * </p>
     *
     * @param getLoggingConfigurationRequest
     * @return A Java Future containing the result of the GetLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.GetLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/GetLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetLoggingConfigurationResponse> getLoggingConfiguration(
            GetLoggingConfigurationRequest getLoggingConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the specified logging configuration.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetLoggingConfigurationRequest.Builder} avoiding
     * the need to create one manually via {@link GetLoggingConfigurationRequest#builder()}
     * </p>
     *
     * @param getLoggingConfigurationRequest
     *        A {@link Consumer} that will call methods on {@link GetLoggingConfigurationRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the GetLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.GetLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/GetLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetLoggingConfigurationResponse> getLoggingConfiguration(
            Consumer<GetLoggingConfigurationRequest.Builder> getLoggingConfigurationRequest) {
        return getLoggingConfiguration(GetLoggingConfigurationRequest.builder().applyMutation(getLoggingConfigurationRequest)
                .build());
    }

    /**
     * <p>
     * Gets the specified room.
     * </p>
     *
     * @param getRoomRequest
     * @return A Java Future containing the result of the GetRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.GetRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/GetRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetRoomResponse> getRoom(GetRoomRequest getRoomRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the specified room.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRoomRequest.Builder} avoiding the need to create
     * one manually via {@link GetRoomRequest#builder()}
     * </p>
     *
     * @param getRoomRequest
     *        A {@link Consumer} that will call methods on {@link GetRoomRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.GetRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/GetRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetRoomResponse> getRoom(Consumer<GetRoomRequest.Builder> getRoomRequest) {
        return getRoom(GetRoomRequest.builder().applyMutation(getRoomRequest).build());
    }

    /**
     * <p>
     * Gets summary information about all your logging configurations in the AWS region where the API request is
     * processed.
     * </p>
     *
     * @param listLoggingConfigurationsRequest
     * @return A Java Future containing the result of the ListLoggingConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.ListLoggingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListLoggingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListLoggingConfigurationsResponse> listLoggingConfigurations(
            ListLoggingConfigurationsRequest listLoggingConfigurationsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about all your logging configurations in the AWS region where the API request is
     * processed.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListLoggingConfigurationsRequest.Builder} avoiding
     * the need to create one manually via {@link ListLoggingConfigurationsRequest#builder()}
     * </p>
     *
     * @param listLoggingConfigurationsRequest
     *        A {@link Consumer} that will call methods on {@link ListLoggingConfigurationsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListLoggingConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.ListLoggingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListLoggingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListLoggingConfigurationsResponse> listLoggingConfigurations(
            Consumer<ListLoggingConfigurationsRequest.Builder> listLoggingConfigurationsRequest) {
        return listLoggingConfigurations(ListLoggingConfigurationsRequest.builder()
                .applyMutation(listLoggingConfigurationsRequest).build());
    }

    /**
     * <p>
     * Gets summary information about all your logging configurations in the AWS region where the API request is
     * processed.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLoggingConfigurations(software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsPublisher publisher = client.listLoggingConfigurationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsPublisher publisher = client.listLoggingConfigurationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLoggingConfigurations(software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLoggingConfigurationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.ListLoggingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListLoggingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default ListLoggingConfigurationsPublisher listLoggingConfigurationsPaginator(
            ListLoggingConfigurationsRequest listLoggingConfigurationsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about all your logging configurations in the AWS region where the API request is
     * processed.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listLoggingConfigurations(software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsPublisher publisher = client.listLoggingConfigurationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsPublisher publisher = client.listLoggingConfigurationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLoggingConfigurations(software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListLoggingConfigurationsRequest.Builder} avoiding
     * the need to create one manually via {@link ListLoggingConfigurationsRequest#builder()}
     * </p>
     *
     * @param listLoggingConfigurationsRequest
     *        A {@link Consumer} that will call methods on {@link ListLoggingConfigurationsRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.ListLoggingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListLoggingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default ListLoggingConfigurationsPublisher listLoggingConfigurationsPaginator(
            Consumer<ListLoggingConfigurationsRequest.Builder> listLoggingConfigurationsRequest) {
        return listLoggingConfigurationsPaginator(ListLoggingConfigurationsRequest.builder()
                .applyMutation(listLoggingConfigurationsRequest).build());
    }

    /**
     * <p>
     * Gets summary information about all your rooms in the AWS region where the API request is processed. Results are
     * sorted in descending order of <code>updateTime</code>.
     * </p>
     *
     * @param listRoomsRequest
     * @return A Java Future containing the result of the ListRooms operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.ListRooms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListRooms" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListRoomsResponse> listRooms(ListRoomsRequest listRoomsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about all your rooms in the AWS region where the API request is processed. Results are
     * sorted in descending order of <code>updateTime</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRoomsRequest.Builder} avoiding the need to
     * create one manually via {@link ListRoomsRequest#builder()}
     * </p>
     *
     * @param listRoomsRequest
     *        A {@link Consumer} that will call methods on {@link ListRoomsRequest.Builder} to create a request.
     * @return A Java Future containing the result of the ListRooms operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.ListRooms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListRooms" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<ListRoomsResponse> listRooms(Consumer<ListRoomsRequest.Builder> listRoomsRequest) {
        return listRooms(ListRoomsRequest.builder().applyMutation(listRoomsRequest).build());
    }

    /**
     * <p>
     * Gets summary information about all your rooms in the AWS region where the API request is processed. Results are
     * sorted in descending order of <code>updateTime</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listRooms(software.amazon.awssdk.services.ivschat.model.ListRoomsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListRoomsPublisher publisher = client.listRoomsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListRoomsPublisher publisher = client.listRoomsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ivschat.model.ListRoomsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ivschat.model.ListRoomsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRooms(software.amazon.awssdk.services.ivschat.model.ListRoomsRequest)} operation.</b>
     * </p>
     *
     * @param listRoomsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.ListRooms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListRooms" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRoomsPublisher listRoomsPaginator(ListRoomsRequest listRoomsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about all your rooms in the AWS region where the API request is processed. Results are
     * sorted in descending order of <code>updateTime</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listRooms(software.amazon.awssdk.services.ivschat.model.ListRoomsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListRoomsPublisher publisher = client.listRoomsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListRoomsPublisher publisher = client.listRoomsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ivschat.model.ListRoomsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.ivschat.model.ListRoomsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRooms(software.amazon.awssdk.services.ivschat.model.ListRoomsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRoomsRequest.Builder} avoiding the need to
     * create one manually via {@link ListRoomsRequest#builder()}
     * </p>
     *
     * @param listRoomsRequest
     *        A {@link Consumer} that will call methods on {@link ListRoomsRequest.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.ListRooms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListRooms" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRoomsPublisher listRoomsPaginator(Consumer<ListRoomsRequest.Builder> listRoomsRequest) {
        return listRoomsPaginator(ListRoomsRequest.builder().applyMutation(listRoomsRequest).build());
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Sends an event to a room. Use this within your application’s business logic to send events to clients of a room;
     * e.g., to notify clients to change the way the chat UI is rendered.
     * </p>
     *
     * @param sendEventRequest
     * @return A Java Future containing the result of the SendEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.SendEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/SendEvent" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<SendEventResponse> sendEvent(SendEventRequest sendEventRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends an event to a room. Use this within your application’s business logic to send events to clients of a room;
     * e.g., to notify clients to change the way the chat UI is rendered.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SendEventRequest.Builder} avoiding the need to
     * create one manually via {@link SendEventRequest#builder()}
     * </p>
     *
     * @param sendEventRequest
     *        A {@link Consumer} that will call methods on {@link SendEventRequest.Builder} to create a request.
     * @return A Java Future containing the result of the SendEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ThrottlingException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.SendEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/SendEvent" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<SendEventResponse> sendEvent(Consumer<SendEventRequest.Builder> sendEventRequest) {
        return sendEvent(SendEventRequest.builder().applyMutation(sendEventRequest).build());
    }

    /**
     * <p>
     * Adds or updates tags for the AWS resource with the specified ARN.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds or updates tags for the AWS resource with the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes tags from the resource with the specified ARN.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes tags from the resource with the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>InternalServerException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates a specified logging configuration.
     * </p>
     *
     * @param updateLoggingConfigurationRequest
     * @return A Java Future containing the result of the UpdateLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.UpdateLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UpdateLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateLoggingConfigurationResponse> updateLoggingConfiguration(
            UpdateLoggingConfigurationRequest updateLoggingConfigurationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a specified logging configuration.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateLoggingConfigurationRequest.Builder} avoiding
     * the need to create one manually via {@link UpdateLoggingConfigurationRequest#builder()}
     * </p>
     *
     * @param updateLoggingConfigurationRequest
     *        A {@link Consumer} that will call methods on {@link UpdateLoggingConfigurationRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the UpdateLoggingConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.UpdateLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UpdateLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateLoggingConfigurationResponse> updateLoggingConfiguration(
            Consumer<UpdateLoggingConfigurationRequest.Builder> updateLoggingConfigurationRequest) {
        return updateLoggingConfiguration(UpdateLoggingConfigurationRequest.builder()
                .applyMutation(updateLoggingConfigurationRequest).build());
    }

    /**
     * <p>
     * Updates a room’s configuration.
     * </p>
     *
     * @param updateRoomRequest
     * @return A Java Future containing the result of the UpdateRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.UpdateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UpdateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateRoomResponse> updateRoom(UpdateRoomRequest updateRoomRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a room’s configuration.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateRoomRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateRoomRequest#builder()}
     * </p>
     *
     * @param updateRoomRequest
     *        A {@link Consumer} that will call methods on {@link UpdateRoomRequest.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateRoom operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException</li>
     *         <li>ResourceNotFoundException</li>
     *         <li>PendingVerificationException</li>
     *         <li>ValidationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvschatException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvschatAsyncClient.UpdateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UpdateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<UpdateRoomResponse> updateRoom(Consumer<UpdateRoomRequest.Builder> updateRoomRequest) {
        return updateRoom(UpdateRoomRequest.builder().applyMutation(updateRoomRequest).build());
    }

    @Override
    default IvschatServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link IvschatAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static IvschatAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link IvschatAsyncClient}.
     */
    static IvschatAsyncClientBuilder builder() {
        return new DefaultIvschatAsyncClientBuilder();
    }
}
