/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ivschat;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.ivschat.model.AccessDeniedException;
import software.amazon.awssdk.services.ivschat.model.ConflictException;
import software.amazon.awssdk.services.ivschat.model.CreateChatTokenRequest;
import software.amazon.awssdk.services.ivschat.model.CreateChatTokenResponse;
import software.amazon.awssdk.services.ivschat.model.CreateLoggingConfigurationRequest;
import software.amazon.awssdk.services.ivschat.model.CreateLoggingConfigurationResponse;
import software.amazon.awssdk.services.ivschat.model.CreateRoomRequest;
import software.amazon.awssdk.services.ivschat.model.CreateRoomResponse;
import software.amazon.awssdk.services.ivschat.model.DeleteLoggingConfigurationRequest;
import software.amazon.awssdk.services.ivschat.model.DeleteLoggingConfigurationResponse;
import software.amazon.awssdk.services.ivschat.model.DeleteMessageRequest;
import software.amazon.awssdk.services.ivschat.model.DeleteMessageResponse;
import software.amazon.awssdk.services.ivschat.model.DeleteRoomRequest;
import software.amazon.awssdk.services.ivschat.model.DeleteRoomResponse;
import software.amazon.awssdk.services.ivschat.model.DisconnectUserRequest;
import software.amazon.awssdk.services.ivschat.model.DisconnectUserResponse;
import software.amazon.awssdk.services.ivschat.model.GetLoggingConfigurationRequest;
import software.amazon.awssdk.services.ivschat.model.GetLoggingConfigurationResponse;
import software.amazon.awssdk.services.ivschat.model.GetRoomRequest;
import software.amazon.awssdk.services.ivschat.model.GetRoomResponse;
import software.amazon.awssdk.services.ivschat.model.InternalServerException;
import software.amazon.awssdk.services.ivschat.model.IvschatException;
import software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest;
import software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsResponse;
import software.amazon.awssdk.services.ivschat.model.ListRoomsRequest;
import software.amazon.awssdk.services.ivschat.model.ListRoomsResponse;
import software.amazon.awssdk.services.ivschat.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ivschat.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ivschat.model.PendingVerificationException;
import software.amazon.awssdk.services.ivschat.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ivschat.model.SendEventRequest;
import software.amazon.awssdk.services.ivschat.model.SendEventResponse;
import software.amazon.awssdk.services.ivschat.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ivschat.model.TagResourceRequest;
import software.amazon.awssdk.services.ivschat.model.TagResourceResponse;
import software.amazon.awssdk.services.ivschat.model.ThrottlingException;
import software.amazon.awssdk.services.ivschat.model.UntagResourceRequest;
import software.amazon.awssdk.services.ivschat.model.UntagResourceResponse;
import software.amazon.awssdk.services.ivschat.model.UpdateLoggingConfigurationRequest;
import software.amazon.awssdk.services.ivschat.model.UpdateLoggingConfigurationResponse;
import software.amazon.awssdk.services.ivschat.model.UpdateRoomRequest;
import software.amazon.awssdk.services.ivschat.model.UpdateRoomResponse;
import software.amazon.awssdk.services.ivschat.model.ValidationException;
import software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsIterable;
import software.amazon.awssdk.services.ivschat.paginators.ListRoomsIterable;

/**
 * Service client for accessing ivschat. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * <b>Introduction</b>
 * </p>
 * <p>
 * The Amazon IVS Chat control-plane API enables you to create and manage Amazon IVS Chat resources. You also need to
 * integrate with the <a href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/chat-messaging-api.html">
 * Amazon IVS Chat Messaging API</a>, to enable users to interact with chat rooms in real time.
 * </p>
 * <p>
 * The API is an AWS regional service. For a list of supported regions and Amazon IVS Chat HTTPS service endpoints, see
 * the Amazon IVS Chat information on the <a href="https://docs.aws.amazon.com/general/latest/gr/ivs.html">Amazon IVS
 * page</a> in the <i>AWS General Reference</i>.
 * </p>
 * <p>
 * This document describes HTTP operations. There is a separate <i>messaging</i> API for managing Chat resources; see
 * the <a href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/chat-messaging-api.html"> Amazon IVS Chat
 * Messaging API Reference</a>.
 * </p>
 * <p>
 * <b>Notes on terminology:</b>
 * </p>
 * <ul>
 * <li>
 * <p>
 * You create service applications using the Amazon IVS Chat API. We refer to these as <i>applications</i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * You create front-end client applications (browser and Android/iOS apps) using the Amazon IVS Chat Messaging API. We
 * refer to these as <i>clients</i>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Resources</b>
 * </p>
 * <p>
 * The following resources are part of Amazon IVS Chat:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>LoggingConfiguration</b> — A configuration that allows customers to store and record sent messages in a chat room.
 * See the Logging Configuration endpoints for more information.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Room</b> — The central Amazon IVS Chat resource through which clients connect to and exchange chat messages. See
 * the Room endpoints for more information.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Tagging</b>
 * </p>
 * <p>
 * A <i>tag</i> is a metadata label that you assign to an AWS resource. A tag comprises a <i>key</i> and a <i>value</i>,
 * both set by you. For example, you might set a tag as <code>topic:nature</code> to label a particular video category.
 * See <a href="https://docs.aws.amazon.com/tag-editor/latest/userguide/best-practices-and-strats.html">Best practices
 * and strategies</a> in <i>Tagging Amazon Web Services Resources and Tag Editor</i> for details, including restrictions
 * that apply to tags and "Tag naming limits and requirements"; Amazon IVS Chat has no service-specific constraints
 * beyond what is documented there.
 * </p>
 * <p>
 * Tags can help you identify and organize your AWS resources. For example, you can use the same tag for different
 * resources to indicate that they are related. You can also use tags to manage access (see <a
 * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html">Access Tags</a>).
 * </p>
 * <p>
 * The Amazon IVS Chat API has these tag-related operations: <a>TagResource</a>, <a>UntagResource</a>, and
 * <a>ListTagsForResource</a>. The following resource supports tagging: Room.
 * </p>
 * <p>
 * At most 50 tags can be applied to a resource.
 * </p>
 * <p>
 * <b>API Access Security</b>
 * </p>
 * <p>
 * Your Amazon IVS Chat applications (service applications and clients) must be authenticated and authorized to access
 * Amazon IVS Chat resources. Note the differences between these concepts:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <i>Authentication</i> is about verifying identity. Requests to the Amazon IVS Chat API must be signed to verify your
 * identity.
 * </p>
 * </li>
 * <li>
 * <p>
 * <i>Authorization</i> is about granting permissions. Your IAM roles need to have permissions for Amazon IVS Chat API
 * requests.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Users (viewers) connect to a room using secure access tokens that you create using the <a>CreateChatToken</a>
 * operation through the AWS SDK. You call CreateChatToken for every user’s chat session, passing identity and
 * authorization information about the user.
 * </p>
 * <p>
 * <b>Signing API Requests</b>
 * </p>
 * <p>
 * HTTP API requests must be signed with an AWS SigV4 signature using your AWS security credentials. The AWS Command
 * Line Interface (CLI) and the AWS SDKs take care of signing the underlying API calls for you. However, if your
 * application calls the Amazon IVS Chat HTTP API directly, it’s your responsibility to sign the requests.
 * </p>
 * <p>
 * You generate a signature using valid AWS credentials for an IAM role that has permission to perform the requested
 * action. For example, DeleteMessage requests must be made using an IAM role that has the
 * <code>ivschat:DeleteMessage</code> permission.
 * </p>
 * <p>
 * For more information:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Authentication and generating signatures — See <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/API/sig-v4-authenticating-requests.html">Authenticating Requests
 * (Amazon Web Services Signature Version 4)</a> in the <i>Amazon Web Services General Reference</i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Managing Amazon IVS permissions — See <a
 * href="https://docs.aws.amazon.com/ivs/latest/userguide/security-iam.html">Identity and Access Management</a> on the
 * Security page of the <i>Amazon IVS User Guide</i>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>Amazon Resource Names (ARNs)</b>
 * </p>
 * <p>
 * ARNs uniquely identify AWS resources. An ARN is required when you need to specify a resource unambiguously across all
 * of AWS, such as in IAM policies and API calls. For more information, see <a
 * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon Resource Names</a> in the
 * <i>AWS General Reference</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface IvschatClient extends AwsClient {
    String SERVICE_NAME = "ivschat";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "ivschat";

    /**
     * <p>
     * Creates an encrypted token that is used by a chat participant to establish an individual WebSocket chat
     * connection to a room. When the token is used to connect to chat, the connection is valid for the session duration
     * specified in the request. The token becomes invalid at the token-expiration timestamp included in the response.
     * </p>
     * <p>
     * Use the <code>capabilities</code> field to permit an end user to send messages or moderate a room.
     * </p>
     * <p>
     * The <code>attributes</code> field securely attaches structured data to the chat session; the data is included
     * within each message sent by the end user and received by other participants in the room. Common use cases for
     * attributes include passing end-user profile data like an icon, display name, colors, badges, and other display
     * features.
     * </p>
     * <p>
     * Encryption keys are owned by Amazon IVS Chat and never used directly by your application.
     * </p>
     *
     * @param createChatTokenRequest
     * @return Result of the CreateChatToken operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.CreateChatToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateChatToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateChatTokenResponse createChatToken(CreateChatTokenRequest createChatTokenRequest) throws AccessDeniedException,
            ResourceNotFoundException, PendingVerificationException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates an encrypted token that is used by a chat participant to establish an individual WebSocket chat
     * connection to a room. When the token is used to connect to chat, the connection is valid for the session duration
     * specified in the request. The token becomes invalid at the token-expiration timestamp included in the response.
     * </p>
     * <p>
     * Use the <code>capabilities</code> field to permit an end user to send messages or moderate a room.
     * </p>
     * <p>
     * The <code>attributes</code> field securely attaches structured data to the chat session; the data is included
     * within each message sent by the end user and received by other participants in the room. Common use cases for
     * attributes include passing end-user profile data like an icon, display name, colors, badges, and other display
     * features.
     * </p>
     * <p>
     * Encryption keys are owned by Amazon IVS Chat and never used directly by your application.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateChatTokenRequest.Builder} avoiding the need
     * to create one manually via {@link CreateChatTokenRequest#builder()}
     * </p>
     *
     * @param createChatTokenRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.CreateChatTokenRequest.Builder} to create a request.
     * @return Result of the CreateChatToken operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.CreateChatToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateChatToken" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateChatTokenResponse createChatToken(Consumer<CreateChatTokenRequest.Builder> createChatTokenRequest)
            throws AccessDeniedException, ResourceNotFoundException, PendingVerificationException, ValidationException,
            AwsServiceException, SdkClientException, IvschatException {
        return createChatToken(CreateChatTokenRequest.builder().applyMutation(createChatTokenRequest).build());
    }

    /**
     * <p>
     * Creates a logging configuration that allows clients to store and record sent messages.
     * </p>
     *
     * @param createLoggingConfigurationRequest
     * @return Result of the CreateLoggingConfiguration operation returned by the service.
     * @throws ConflictException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.CreateLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateLoggingConfigurationResponse createLoggingConfiguration(
            CreateLoggingConfigurationRequest createLoggingConfigurationRequest) throws ConflictException, AccessDeniedException,
            ResourceNotFoundException, ServiceQuotaExceededException, PendingVerificationException, ValidationException,
            AwsServiceException, SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a logging configuration that allows clients to store and record sent messages.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateLoggingConfigurationRequest.Builder} avoiding
     * the need to create one manually via {@link CreateLoggingConfigurationRequest#builder()}
     * </p>
     *
     * @param createLoggingConfigurationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.CreateLoggingConfigurationRequest.Builder} to create
     *        a request.
     * @return Result of the CreateLoggingConfiguration operation returned by the service.
     * @throws ConflictException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.CreateLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateLoggingConfigurationResponse createLoggingConfiguration(
            Consumer<CreateLoggingConfigurationRequest.Builder> createLoggingConfigurationRequest) throws ConflictException,
            AccessDeniedException, ResourceNotFoundException, ServiceQuotaExceededException, PendingVerificationException,
            ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return createLoggingConfiguration(CreateLoggingConfigurationRequest.builder()
                .applyMutation(createLoggingConfigurationRequest).build());
    }

    /**
     * <p>
     * Creates a room that allows clients to connect and pass messages.
     * </p>
     *
     * @param createRoomRequest
     * @return Result of the CreateRoom operation returned by the service.
     * @throws ConflictException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.CreateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateRoomResponse createRoom(CreateRoomRequest createRoomRequest) throws ConflictException, AccessDeniedException,
            ResourceNotFoundException, ServiceQuotaExceededException, PendingVerificationException, ValidationException,
            AwsServiceException, SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a room that allows clients to connect and pass messages.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateRoomRequest.Builder} avoiding the need to
     * create one manually via {@link CreateRoomRequest#builder()}
     * </p>
     *
     * @param createRoomRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.CreateRoomRequest.Builder} to create a request.
     * @return Result of the CreateRoom operation returned by the service.
     * @throws ConflictException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.CreateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default CreateRoomResponse createRoom(Consumer<CreateRoomRequest.Builder> createRoomRequest) throws ConflictException,
            AccessDeniedException, ResourceNotFoundException, ServiceQuotaExceededException, PendingVerificationException,
            ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return createRoom(CreateRoomRequest.builder().applyMutation(createRoomRequest).build());
    }

    /**
     * <p>
     * Deletes the specified logging configuration.
     * </p>
     *
     * @param deleteLoggingConfigurationRequest
     * @return Result of the DeleteLoggingConfiguration operation returned by the service.
     * @throws ConflictException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.DeleteLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteLoggingConfigurationResponse deleteLoggingConfiguration(
            DeleteLoggingConfigurationRequest deleteLoggingConfigurationRequest) throws ConflictException, AccessDeniedException,
            ResourceNotFoundException, PendingVerificationException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified logging configuration.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteLoggingConfigurationRequest.Builder} avoiding
     * the need to create one manually via {@link DeleteLoggingConfigurationRequest#builder()}
     * </p>
     *
     * @param deleteLoggingConfigurationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.DeleteLoggingConfigurationRequest.Builder} to create
     *        a request.
     * @return Result of the DeleteLoggingConfiguration operation returned by the service.
     * @throws ConflictException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.DeleteLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteLoggingConfigurationResponse deleteLoggingConfiguration(
            Consumer<DeleteLoggingConfigurationRequest.Builder> deleteLoggingConfigurationRequest) throws ConflictException,
            AccessDeniedException, ResourceNotFoundException, PendingVerificationException, ValidationException,
            AwsServiceException, SdkClientException, IvschatException {
        return deleteLoggingConfiguration(DeleteLoggingConfigurationRequest.builder()
                .applyMutation(deleteLoggingConfigurationRequest).build());
    }

    /**
     * <p>
     * Sends an event to a specific room which directs clients to delete a specific message; that is, unrender it from
     * view and delete it from the client’s chat history. This event’s <code>EventName</code> is
     * <code>aws:DELETE_MESSAGE</code>. This replicates the <a
     * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-deletemessage-publish.html">
     * DeleteMessage</a> WebSocket operation in the Amazon IVS Chat Messaging API.
     * </p>
     *
     * @param deleteMessageRequest
     * @return Result of the DeleteMessage operation returned by the service.
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.DeleteMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteMessage" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteMessageResponse deleteMessage(DeleteMessageRequest deleteMessageRequest) throws ThrottlingException,
            AccessDeniedException, ResourceNotFoundException, PendingVerificationException, ValidationException,
            AwsServiceException, SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends an event to a specific room which directs clients to delete a specific message; that is, unrender it from
     * view and delete it from the client’s chat history. This event’s <code>EventName</code> is
     * <code>aws:DELETE_MESSAGE</code>. This replicates the <a
     * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-deletemessage-publish.html">
     * DeleteMessage</a> WebSocket operation in the Amazon IVS Chat Messaging API.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteMessageRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteMessageRequest#builder()}
     * </p>
     *
     * @param deleteMessageRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.DeleteMessageRequest.Builder} to create a request.
     * @return Result of the DeleteMessage operation returned by the service.
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.DeleteMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteMessage" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteMessageResponse deleteMessage(Consumer<DeleteMessageRequest.Builder> deleteMessageRequest)
            throws ThrottlingException, AccessDeniedException, ResourceNotFoundException, PendingVerificationException,
            ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return deleteMessage(DeleteMessageRequest.builder().applyMutation(deleteMessageRequest).build());
    }

    /**
     * <p>
     * Deletes the specified room.
     * </p>
     *
     * @param deleteRoomRequest
     * @return Result of the DeleteRoom operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.DeleteRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteRoomResponse deleteRoom(DeleteRoomRequest deleteRoomRequest) throws AccessDeniedException,
            ResourceNotFoundException, PendingVerificationException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified room.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteRoomRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteRoomRequest#builder()}
     * </p>
     *
     * @param deleteRoomRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.DeleteRoomRequest.Builder} to create a request.
     * @return Result of the DeleteRoom operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.DeleteRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default DeleteRoomResponse deleteRoom(Consumer<DeleteRoomRequest.Builder> deleteRoomRequest) throws AccessDeniedException,
            ResourceNotFoundException, PendingVerificationException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        return deleteRoom(DeleteRoomRequest.builder().applyMutation(deleteRoomRequest).build());
    }

    /**
     * <p>
     * Disconnects all connections using a specified user ID from a room. This replicates the <a
     * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-disconnectuser-publish.html">
     * DisconnectUser</a> WebSocket operation in the Amazon IVS Chat Messaging API.
     * </p>
     *
     * @param disconnectUserRequest
     * @return Result of the DisconnectUser operation returned by the service.
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.DisconnectUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DisconnectUser" target="_top">AWS API
     *      Documentation</a>
     */
    default DisconnectUserResponse disconnectUser(DisconnectUserRequest disconnectUserRequest) throws ThrottlingException,
            AccessDeniedException, ResourceNotFoundException, PendingVerificationException, ValidationException,
            AwsServiceException, SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Disconnects all connections using a specified user ID from a room. This replicates the <a
     * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-disconnectuser-publish.html">
     * DisconnectUser</a> WebSocket operation in the Amazon IVS Chat Messaging API.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DisconnectUserRequest.Builder} avoiding the need to
     * create one manually via {@link DisconnectUserRequest#builder()}
     * </p>
     *
     * @param disconnectUserRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.DisconnectUserRequest.Builder} to create a request.
     * @return Result of the DisconnectUser operation returned by the service.
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.DisconnectUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DisconnectUser" target="_top">AWS API
     *      Documentation</a>
     */
    default DisconnectUserResponse disconnectUser(Consumer<DisconnectUserRequest.Builder> disconnectUserRequest)
            throws ThrottlingException, AccessDeniedException, ResourceNotFoundException, PendingVerificationException,
            ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return disconnectUser(DisconnectUserRequest.builder().applyMutation(disconnectUserRequest).build());
    }

    /**
     * <p>
     * Gets the specified logging configuration.
     * </p>
     *
     * @param getLoggingConfigurationRequest
     * @return Result of the GetLoggingConfiguration operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.GetLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/GetLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default GetLoggingConfigurationResponse getLoggingConfiguration(GetLoggingConfigurationRequest getLoggingConfigurationRequest)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the specified logging configuration.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetLoggingConfigurationRequest.Builder} avoiding
     * the need to create one manually via {@link GetLoggingConfigurationRequest#builder()}
     * </p>
     *
     * @param getLoggingConfigurationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.GetLoggingConfigurationRequest.Builder} to create a
     *        request.
     * @return Result of the GetLoggingConfiguration operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.GetLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/GetLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default GetLoggingConfigurationResponse getLoggingConfiguration(
            Consumer<GetLoggingConfigurationRequest.Builder> getLoggingConfigurationRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return getLoggingConfiguration(GetLoggingConfigurationRequest.builder().applyMutation(getLoggingConfigurationRequest)
                .build());
    }

    /**
     * <p>
     * Gets the specified room.
     * </p>
     *
     * @param getRoomRequest
     * @return Result of the GetRoom operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.GetRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/GetRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default GetRoomResponse getRoom(GetRoomRequest getRoomRequest) throws AccessDeniedException, ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the specified room.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRoomRequest.Builder} avoiding the need to create
     * one manually via {@link GetRoomRequest#builder()}
     * </p>
     *
     * @param getRoomRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.GetRoomRequest.Builder} to create a request.
     * @return Result of the GetRoom operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.GetRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/GetRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default GetRoomResponse getRoom(Consumer<GetRoomRequest.Builder> getRoomRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return getRoom(GetRoomRequest.builder().applyMutation(getRoomRequest).build());
    }

    /**
     * <p>
     * Gets summary information about all your logging configurations in the AWS region where the API request is
     * processed.
     * </p>
     *
     * @param listLoggingConfigurationsRequest
     * @return Result of the ListLoggingConfigurations operation returned by the service.
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.ListLoggingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListLoggingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default ListLoggingConfigurationsResponse listLoggingConfigurations(
            ListLoggingConfigurationsRequest listLoggingConfigurationsRequest) throws AccessDeniedException, ValidationException,
            AwsServiceException, SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about all your logging configurations in the AWS region where the API request is
     * processed.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListLoggingConfigurationsRequest.Builder} avoiding
     * the need to create one manually via {@link ListLoggingConfigurationsRequest#builder()}
     * </p>
     *
     * @param listLoggingConfigurationsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest.Builder} to create a
     *        request.
     * @return Result of the ListLoggingConfigurations operation returned by the service.
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.ListLoggingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListLoggingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default ListLoggingConfigurationsResponse listLoggingConfigurations(
            Consumer<ListLoggingConfigurationsRequest.Builder> listLoggingConfigurationsRequest) throws AccessDeniedException,
            ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return listLoggingConfigurations(ListLoggingConfigurationsRequest.builder()
                .applyMutation(listLoggingConfigurationsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listLoggingConfigurations(software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsIterable responses = client.listLoggingConfigurationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsIterable responses = client
     *             .listLoggingConfigurationsPaginator(request);
     *     for (software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsIterable responses = client.listLoggingConfigurationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLoggingConfigurations(software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listLoggingConfigurationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.ListLoggingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListLoggingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default ListLoggingConfigurationsIterable listLoggingConfigurationsPaginator(
            ListLoggingConfigurationsRequest listLoggingConfigurationsRequest) throws AccessDeniedException, ValidationException,
            AwsServiceException, SdkClientException, IvschatException {
        return new ListLoggingConfigurationsIterable(this, listLoggingConfigurationsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listLoggingConfigurations(software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsIterable responses = client.listLoggingConfigurationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsIterable responses = client
     *             .listLoggingConfigurationsPaginator(request);
     *     for (software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListLoggingConfigurationsIterable responses = client.listLoggingConfigurationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLoggingConfigurations(software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListLoggingConfigurationsRequest.Builder} avoiding
     * the need to create one manually via {@link ListLoggingConfigurationsRequest#builder()}
     * </p>
     *
     * @param listLoggingConfigurationsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.ListLoggingConfigurationsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.ListLoggingConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListLoggingConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    default ListLoggingConfigurationsIterable listLoggingConfigurationsPaginator(
            Consumer<ListLoggingConfigurationsRequest.Builder> listLoggingConfigurationsRequest) throws AccessDeniedException,
            ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return listLoggingConfigurationsPaginator(ListLoggingConfigurationsRequest.builder()
                .applyMutation(listLoggingConfigurationsRequest).build());
    }

    /**
     * <p>
     * Gets summary information about all your rooms in the AWS region where the API request is processed. Results are
     * sorted in descending order of <code>updateTime</code>.
     * </p>
     *
     * @param listRoomsRequest
     * @return Result of the ListRooms operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.ListRooms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListRooms" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRoomsResponse listRooms(ListRoomsRequest listRoomsRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets summary information about all your rooms in the AWS region where the API request is processed. Results are
     * sorted in descending order of <code>updateTime</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRoomsRequest.Builder} avoiding the need to
     * create one manually via {@link ListRoomsRequest#builder()}
     * </p>
     *
     * @param listRoomsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.ListRoomsRequest.Builder} to create a request.
     * @return Result of the ListRooms operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.ListRooms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListRooms" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRoomsResponse listRooms(Consumer<ListRoomsRequest.Builder> listRoomsRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return listRooms(ListRoomsRequest.builder().applyMutation(listRoomsRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listRooms(software.amazon.awssdk.services.ivschat.model.ListRoomsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListRoomsIterable responses = client.listRoomsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivschat.paginators.ListRoomsIterable responses = client.listRoomsPaginator(request);
     *     for (software.amazon.awssdk.services.ivschat.model.ListRoomsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListRoomsIterable responses = client.listRoomsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRooms(software.amazon.awssdk.services.ivschat.model.ListRoomsRequest)} operation.</b>
     * </p>
     *
     * @param listRoomsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.ListRooms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListRooms" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRoomsIterable listRoomsPaginator(ListRoomsRequest listRoomsRequest) throws AccessDeniedException,
            ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return new ListRoomsIterable(this, listRoomsRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listRooms(software.amazon.awssdk.services.ivschat.model.ListRoomsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListRoomsIterable responses = client.listRoomsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivschat.paginators.ListRoomsIterable responses = client.listRoomsPaginator(request);
     *     for (software.amazon.awssdk.services.ivschat.model.ListRoomsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivschat.paginators.ListRoomsIterable responses = client.listRoomsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRooms(software.amazon.awssdk.services.ivschat.model.ListRoomsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRoomsRequest.Builder} avoiding the need to
     * create one manually via {@link ListRoomsRequest#builder()}
     * </p>
     *
     * @param listRoomsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.ListRoomsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.ListRooms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListRooms" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRoomsIterable listRoomsPaginator(Consumer<ListRoomsRequest.Builder> listRoomsRequest)
            throws AccessDeniedException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        return listRoomsPaginator(ListRoomsRequest.builder().applyMutation(listRoomsRequest).build());
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, IvschatException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Sends an event to a room. Use this within your application’s business logic to send events to clients of a room;
     * e.g., to notify clients to change the way the chat UI is rendered.
     * </p>
     *
     * @param sendEventRequest
     * @return Result of the SendEvent operation returned by the service.
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.SendEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/SendEvent" target="_top">AWS API
     *      Documentation</a>
     */
    default SendEventResponse sendEvent(SendEventRequest sendEventRequest) throws ThrottlingException, AccessDeniedException,
            ResourceNotFoundException, PendingVerificationException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Sends an event to a room. Use this within your application’s business logic to send events to clients of a room;
     * e.g., to notify clients to change the way the chat UI is rendered.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link SendEventRequest.Builder} avoiding the need to
     * create one manually via {@link SendEventRequest#builder()}
     * </p>
     *
     * @param sendEventRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.SendEventRequest.Builder} to create a request.
     * @return Result of the SendEvent operation returned by the service.
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.SendEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/SendEvent" target="_top">AWS API
     *      Documentation</a>
     */
    default SendEventResponse sendEvent(Consumer<SendEventRequest.Builder> sendEventRequest) throws ThrottlingException,
            AccessDeniedException, ResourceNotFoundException, PendingVerificationException, ValidationException,
            AwsServiceException, SdkClientException, IvschatException {
        return sendEvent(SendEventRequest.builder().applyMutation(sendEventRequest).build());
    }

    /**
     * <p>
     * Adds or updates tags for the AWS resource with the specified ARN.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds or updates tags for the AWS resource with the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.TagResourceRequest.Builder} to create a request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes tags from the resource with the specified ARN.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            InternalServerException, ValidationException, AwsServiceException, SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes tags from the resource with the specified ARN.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.UntagResourceRequest.Builder} to create a request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws InternalServerException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ResourceNotFoundException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates a specified logging configuration.
     * </p>
     *
     * @param updateLoggingConfigurationRequest
     * @return Result of the UpdateLoggingConfiguration operation returned by the service.
     * @throws ConflictException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.UpdateLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UpdateLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateLoggingConfigurationResponse updateLoggingConfiguration(
            UpdateLoggingConfigurationRequest updateLoggingConfigurationRequest) throws ConflictException, AccessDeniedException,
            ResourceNotFoundException, PendingVerificationException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a specified logging configuration.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateLoggingConfigurationRequest.Builder} avoiding
     * the need to create one manually via {@link UpdateLoggingConfigurationRequest#builder()}
     * </p>
     *
     * @param updateLoggingConfigurationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.UpdateLoggingConfigurationRequest.Builder} to create
     *        a request.
     * @return Result of the UpdateLoggingConfiguration operation returned by the service.
     * @throws ConflictException
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.UpdateLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UpdateLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateLoggingConfigurationResponse updateLoggingConfiguration(
            Consumer<UpdateLoggingConfigurationRequest.Builder> updateLoggingConfigurationRequest) throws ConflictException,
            AccessDeniedException, ResourceNotFoundException, PendingVerificationException, ValidationException,
            AwsServiceException, SdkClientException, IvschatException {
        return updateLoggingConfiguration(UpdateLoggingConfigurationRequest.builder()
                .applyMutation(updateLoggingConfigurationRequest).build());
    }

    /**
     * <p>
     * Updates a room’s configuration.
     * </p>
     *
     * @param updateRoomRequest
     * @return Result of the UpdateRoom operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.UpdateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UpdateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateRoomResponse updateRoom(UpdateRoomRequest updateRoomRequest) throws AccessDeniedException,
            ResourceNotFoundException, PendingVerificationException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a room’s configuration.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateRoomRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateRoomRequest#builder()}
     * </p>
     *
     * @param updateRoomRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.ivschat.model.UpdateRoomRequest.Builder} to create a request.
     * @return Result of the UpdateRoom operation returned by the service.
     * @throws AccessDeniedException
     * @throws ResourceNotFoundException
     * @throws PendingVerificationException
     * @throws ValidationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvschatException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvschatClient.UpdateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UpdateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    default UpdateRoomResponse updateRoom(Consumer<UpdateRoomRequest.Builder> updateRoomRequest) throws AccessDeniedException,
            ResourceNotFoundException, PendingVerificationException, ValidationException, AwsServiceException,
            SdkClientException, IvschatException {
        return updateRoom(UpdateRoomRequest.builder().applyMutation(updateRoomRequest).build());
    }

    /**
     * Create a {@link IvschatClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static IvschatClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link IvschatClient}.
     */
    static IvschatClientBuilder builder() {
        return new DefaultIvschatClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default IvschatServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
