/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ivsrealtime;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ivsrealtime.model.AccessDeniedException;
import software.amazon.awssdk.services.ivsrealtime.model.ConflictException;
import software.amazon.awssdk.services.ivsrealtime.model.CreateParticipantTokenRequest;
import software.amazon.awssdk.services.ivsrealtime.model.CreateParticipantTokenResponse;
import software.amazon.awssdk.services.ivsrealtime.model.CreateStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.CreateStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.DisconnectParticipantRequest;
import software.amazon.awssdk.services.ivsrealtime.model.DisconnectParticipantResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.InternalServerException;
import software.amazon.awssdk.services.ivsrealtime.model.IvsRealTimeException;
import software.amazon.awssdk.services.ivsrealtime.model.IvsRealTimeRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListStagesRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListStagesResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ivsrealtime.model.PendingVerificationException;
import software.amazon.awssdk.services.ivsrealtime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ivsrealtime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ivsrealtime.model.TagResourceRequest;
import software.amazon.awssdk.services.ivsrealtime.model.TagResourceResponse;
import software.amazon.awssdk.services.ivsrealtime.model.UntagResourceRequest;
import software.amazon.awssdk.services.ivsrealtime.model.UntagResourceResponse;
import software.amazon.awssdk.services.ivsrealtime.model.UpdateStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.UpdateStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ValidationException;
import software.amazon.awssdk.services.ivsrealtime.paginators.ListStagesIterable;
import software.amazon.awssdk.services.ivsrealtime.transform.CreateParticipantTokenRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.CreateStageRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.DeleteStageRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.DisconnectParticipantRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetStageRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListStagesRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.UpdateStageRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IvsRealTimeClient}.
 *
 * @see IvsRealTimeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIvsRealTimeClient implements IvsRealTimeClient {
    private static final Logger log = Logger.loggerFor(DefaultIvsRealTimeClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final IvsRealTimeServiceClientConfiguration serviceClientConfiguration;

    protected DefaultIvsRealTimeClient(IvsRealTimeServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an additional token for a specified stage. This can be done after stage creation or when tokens expire.
     * Tokens always are scoped to the stage for which they are created.
     * </p>
     * <p>
     * Encryption keys are owned by Amazon IVS and never used directly by your application.
     * </p>
     *
     * @param createParticipantTokenRequest
     * @return Result of the CreateParticipantToken operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.CreateParticipantToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/CreateParticipantToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateParticipantTokenResponse createParticipantToken(CreateParticipantTokenRequest createParticipantTokenRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, ServiceQuotaExceededException,
            PendingVerificationException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateParticipantTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateParticipantTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createParticipantTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParticipantToken");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateParticipantTokenRequest, CreateParticipantTokenResponse>()
                            .withOperationName("CreateParticipantToken").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createParticipantTokenRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateParticipantTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new stage (and optionally participant tokens).
     * </p>
     *
     * @param createStageRequest
     * @return Result of the CreateStage operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.CreateStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/CreateStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateStageResponse createStage(CreateStageRequest createStageRequest) throws ValidationException,
            AccessDeniedException, ServiceQuotaExceededException, PendingVerificationException, AwsServiceException,
            SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStage");

            return clientHandler.execute(new ClientExecutionParams<CreateStageRequest, CreateStageResponse>()
                    .withOperationName("CreateStage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createStageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Shuts down and deletes the specified stage (disconnecting all participants).
     * </p>
     *
     * @param deleteStageRequest
     * @return Result of the DeleteStage operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.DeleteStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/DeleteStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteStageResponse deleteStage(DeleteStageRequest deleteStageRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, ConflictException, PendingVerificationException, AwsServiceException,
            SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStage");

            return clientHandler.execute(new ClientExecutionParams<DeleteStageRequest, DeleteStageResponse>()
                    .withOperationName("DeleteStage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteStageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disconnects a specified participant and revokes the participant permanently from a specified stage.
     * </p>
     *
     * @param disconnectParticipantRequest
     * @return Result of the DisconnectParticipant operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.DisconnectParticipant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/DisconnectParticipant"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisconnectParticipantResponse disconnectParticipant(DisconnectParticipantRequest disconnectParticipantRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisconnectParticipantResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisconnectParticipantResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectParticipantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectParticipant");

            return clientHandler.execute(new ClientExecutionParams<DisconnectParticipantRequest, DisconnectParticipantResponse>()
                    .withOperationName("DisconnectParticipant").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(disconnectParticipantRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisconnectParticipantRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information for the specified stage.
     * </p>
     *
     * @param getStageRequest
     * @return Result of the GetStage operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.GetStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetStageResponse getStage(GetStageRequest getStageRequest) throws ResourceNotFoundException, ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStage");

            return clientHandler.execute(new ClientExecutionParams<GetStageRequest, GetStageResponse>()
                    .withOperationName("GetStage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getStageRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about all stages in your account, in the AWS region where the API request is processed.
     * </p>
     *
     * @param listStagesRequest
     * @return Result of the ListStages operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListStages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListStages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStagesResponse listStages(ListStagesRequest listStagesRequest) throws ValidationException, AccessDeniedException,
            ConflictException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStagesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStages");

            return clientHandler
                    .execute(new ClientExecutionParams<ListStagesRequest, ListStagesResponse>().withOperationName("ListStages")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listStagesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListStagesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about all stages in your account, in the AWS region where the API request is processed.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listStages(software.amazon.awssdk.services.ivsrealtime.model.ListStagesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivsrealtime.paginators.ListStagesIterable responses = client.listStagesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ivsrealtime.paginators.ListStagesIterable responses = client.listStagesPaginator(request);
     *     for (software.amazon.awssdk.services.ivsrealtime.model.ListStagesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ivsrealtime.paginators.ListStagesIterable responses = client.listStagesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listStages(software.amazon.awssdk.services.ivsrealtime.model.ListStagesRequest)} operation.</b>
     * </p>
     *
     * @param listStagesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListStages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListStages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStagesIterable listStagesPaginator(ListStagesRequest listStagesRequest) throws ValidationException,
            AccessDeniedException, ConflictException, AwsServiceException, SdkClientException, IvsRealTimeException {
        return new ListStagesIterable(this, applyPaginatorUserAgent(listStagesRequest));
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or updates tags for the AWS resource with the specified ARN.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the resource with the specified ARN.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a stage’s configuration.
     * </p>
     *
     * @param updateStageRequest
     * @return Result of the UpdateStage operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.UpdateStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/UpdateStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateStageResponse updateStage(UpdateStageRequest updateStageRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, ServiceQuotaExceededException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStage");

            return clientHandler.execute(new ClientExecutionParams<UpdateStageRequest, UpdateStageResponse>()
                    .withOperationName("UpdateStage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateStageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private <T extends IvsRealTimeRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IvsRealTimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PendingVerification")
                                .exceptionBuilderSupplier(PendingVerificationException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final IvsRealTimeServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
