/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ivsrealtime;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ivsrealtime.internal.IvsRealTimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ivsrealtime.model.AccessDeniedException;
import software.amazon.awssdk.services.ivsrealtime.model.ConflictException;
import software.amazon.awssdk.services.ivsrealtime.model.CreateEncoderConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.CreateEncoderConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.CreateParticipantTokenRequest;
import software.amazon.awssdk.services.ivsrealtime.model.CreateParticipantTokenResponse;
import software.amazon.awssdk.services.ivsrealtime.model.CreateStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.CreateStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.CreateStorageConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.CreateStorageConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteEncoderConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteEncoderConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteStorageConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteStorageConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.DisconnectParticipantRequest;
import software.amazon.awssdk.services.ivsrealtime.model.DisconnectParticipantResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetCompositionRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetCompositionResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetEncoderConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetEncoderConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetParticipantRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetParticipantResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetStageSessionRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetStageSessionResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetStorageConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetStorageConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.InternalServerException;
import software.amazon.awssdk.services.ivsrealtime.model.IvsRealTimeException;
import software.amazon.awssdk.services.ivsrealtime.model.ListCompositionsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListCompositionsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListEncoderConfigurationsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListEncoderConfigurationsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListParticipantEventsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListParticipantEventsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListParticipantsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListParticipantsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListStageSessionsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListStageSessionsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListStagesRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListStagesResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListStorageConfigurationsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListStorageConfigurationsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ivsrealtime.model.PendingVerificationException;
import software.amazon.awssdk.services.ivsrealtime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ivsrealtime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ivsrealtime.model.StartCompositionRequest;
import software.amazon.awssdk.services.ivsrealtime.model.StartCompositionResponse;
import software.amazon.awssdk.services.ivsrealtime.model.StopCompositionRequest;
import software.amazon.awssdk.services.ivsrealtime.model.StopCompositionResponse;
import software.amazon.awssdk.services.ivsrealtime.model.TagResourceRequest;
import software.amazon.awssdk.services.ivsrealtime.model.TagResourceResponse;
import software.amazon.awssdk.services.ivsrealtime.model.UntagResourceRequest;
import software.amazon.awssdk.services.ivsrealtime.model.UntagResourceResponse;
import software.amazon.awssdk.services.ivsrealtime.model.UpdateStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.UpdateStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ValidationException;
import software.amazon.awssdk.services.ivsrealtime.transform.CreateEncoderConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.CreateParticipantTokenRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.CreateStageRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.CreateStorageConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.DeleteEncoderConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.DeleteStageRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.DeleteStorageConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.DisconnectParticipantRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetCompositionRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetEncoderConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetParticipantRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetStageRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetStageSessionRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetStorageConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListCompositionsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListEncoderConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListParticipantEventsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListParticipantsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListStageSessionsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListStagesRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListStorageConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.StartCompositionRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.StopCompositionRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.UpdateStageRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link IvsRealTimeAsyncClient}.
 *
 * @see IvsRealTimeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIvsRealTimeAsyncClient implements IvsRealTimeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultIvsRealTimeAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIvsRealTimeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an EncoderConfiguration object.
     * </p>
     *
     * @param createEncoderConfigurationRequest
     * @return A Java Future containing the result of the CreateEncoderConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>PendingVerificationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.CreateEncoderConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/CreateEncoderConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEncoderConfigurationResponse> createEncoderConfiguration(
            CreateEncoderConfigurationRequest createEncoderConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEncoderConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEncoderConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEncoderConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEncoderConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEncoderConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEncoderConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEncoderConfigurationRequest, CreateEncoderConfigurationResponse>()
                            .withOperationName("CreateEncoderConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEncoderConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEncoderConfigurationRequest));
            CompletableFuture<CreateEncoderConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an additional token for a specified stage. This can be done after stage creation or when tokens expire.
     * Tokens always are scoped to the stage for which they are created.
     * </p>
     * <p>
     * Encryption keys are owned by Amazon IVS and never used directly by your application.
     * </p>
     *
     * @param createParticipantTokenRequest
     * @return A Java Future containing the result of the CreateParticipantToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>PendingVerificationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.CreateParticipantToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/CreateParticipantToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateParticipantTokenResponse> createParticipantToken(
            CreateParticipantTokenRequest createParticipantTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createParticipantTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createParticipantTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParticipantToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateParticipantTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateParticipantTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateParticipantTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateParticipantTokenRequest, CreateParticipantTokenResponse>()
                            .withOperationName("CreateParticipantToken").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateParticipantTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createParticipantTokenRequest));
            CompletableFuture<CreateParticipantTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new stage (and optionally participant tokens).
     * </p>
     *
     * @param createStageRequest
     * @return A Java Future containing the result of the CreateStage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>PendingVerificationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.CreateStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/CreateStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStageResponse> createStage(CreateStageRequest createStageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateStageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateStageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStageRequest, CreateStageResponse>()
                            .withOperationName("CreateStage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateStageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createStageRequest));
            CompletableFuture<CreateStageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new storage configuration, used to enable recording to Amazon S3. When a StorageConfiguration is
     * created, IVS will modify the S3 bucketPolicy of the provided bucket. This will ensure that IVS has sufficient
     * permissions to write content to the provided bucket.
     * </p>
     *
     * @param createStorageConfigurationRequest
     * @return A Java Future containing the result of the CreateStorageConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>PendingVerificationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.CreateStorageConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/CreateStorageConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateStorageConfigurationResponse> createStorageConfiguration(
            CreateStorageConfigurationRequest createStorageConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStorageConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStorageConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStorageConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateStorageConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateStorageConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateStorageConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateStorageConfigurationRequest, CreateStorageConfigurationResponse>()
                            .withOperationName("CreateStorageConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateStorageConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createStorageConfigurationRequest));
            CompletableFuture<CreateStorageConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an EncoderConfiguration resource. Ensures that no Compositions are using this template; otherwise,
     * returns an error.
     * </p>
     *
     * @param deleteEncoderConfigurationRequest
     * @return A Java Future containing the result of the DeleteEncoderConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.DeleteEncoderConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/DeleteEncoderConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEncoderConfigurationResponse> deleteEncoderConfiguration(
            DeleteEncoderConfigurationRequest deleteEncoderConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEncoderConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEncoderConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEncoderConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEncoderConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEncoderConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEncoderConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEncoderConfigurationRequest, DeleteEncoderConfigurationResponse>()
                            .withOperationName("DeleteEncoderConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEncoderConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEncoderConfigurationRequest));
            CompletableFuture<DeleteEncoderConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Shuts down and deletes the specified stage (disconnecting all participants).
     * </p>
     *
     * @param deleteStageRequest
     * @return A Java Future containing the result of the DeleteStage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ConflictException</li>
     *         <li>PendingVerificationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.DeleteStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/DeleteStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStageResponse> deleteStage(DeleteStageRequest deleteStageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteStageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteStageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStageRequest, DeleteStageResponse>()
                            .withOperationName("DeleteStage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteStageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteStageRequest));
            CompletableFuture<DeleteStageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the storage configuration for the specified ARN.
     * </p>
     * <p>
     * If you try to delete a storage configuration that is used by a Composition, you will get an error (409
     * ConflictException). To avoid this, for all Compositions that reference the storage configuration, first use
     * <a>StopComposition</a> and wait for it to complete, then use DeleteStorageConfiguration.
     * </p>
     *
     * @param deleteStorageConfigurationRequest
     * @return A Java Future containing the result of the DeleteStorageConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.DeleteStorageConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/DeleteStorageConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteStorageConfigurationResponse> deleteStorageConfiguration(
            DeleteStorageConfigurationRequest deleteStorageConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStorageConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStorageConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStorageConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteStorageConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteStorageConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteStorageConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteStorageConfigurationRequest, DeleteStorageConfigurationResponse>()
                            .withOperationName("DeleteStorageConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteStorageConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteStorageConfigurationRequest));
            CompletableFuture<DeleteStorageConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disconnects a specified participant and revokes the participant permanently from a specified stage.
     * </p>
     *
     * @param disconnectParticipantRequest
     * @return A Java Future containing the result of the DisconnectParticipant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>PendingVerificationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.DisconnectParticipant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/DisconnectParticipant"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisconnectParticipantResponse> disconnectParticipant(
            DisconnectParticipantRequest disconnectParticipantRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disconnectParticipantRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectParticipantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectParticipant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisconnectParticipantResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisconnectParticipantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisconnectParticipantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisconnectParticipantRequest, DisconnectParticipantResponse>()
                            .withOperationName("DisconnectParticipant").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisconnectParticipantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disconnectParticipantRequest));
            CompletableFuture<DisconnectParticipantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get information about the specified Composition resource.
     * </p>
     *
     * @param getCompositionRequest
     * @return A Java Future containing the result of the GetComposition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.GetComposition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetComposition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCompositionResponse> getComposition(GetCompositionRequest getCompositionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCompositionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCompositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComposition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCompositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCompositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCompositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCompositionRequest, GetCompositionResponse>()
                            .withOperationName("GetComposition").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCompositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCompositionRequest));
            CompletableFuture<GetCompositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the specified EncoderConfiguration resource.
     * </p>
     *
     * @param getEncoderConfigurationRequest
     * @return A Java Future containing the result of the GetEncoderConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.GetEncoderConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetEncoderConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEncoderConfigurationResponse> getEncoderConfiguration(
            GetEncoderConfigurationRequest getEncoderConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEncoderConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEncoderConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEncoderConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEncoderConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEncoderConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEncoderConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEncoderConfigurationRequest, GetEncoderConfigurationResponse>()
                            .withOperationName("GetEncoderConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEncoderConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEncoderConfigurationRequest));
            CompletableFuture<GetEncoderConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the specified participant token.
     * </p>
     *
     * @param getParticipantRequest
     * @return A Java Future containing the result of the GetParticipant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.GetParticipant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetParticipant" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetParticipantResponse> getParticipant(GetParticipantRequest getParticipantRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getParticipantRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParticipantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParticipant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetParticipantResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetParticipantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetParticipantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetParticipantRequest, GetParticipantResponse>()
                            .withOperationName("GetParticipant").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetParticipantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getParticipantRequest));
            CompletableFuture<GetParticipantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information for the specified stage.
     * </p>
     *
     * @param getStageRequest
     * @return A Java Future containing the result of the GetStage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.GetStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetStageResponse> getStage(GetStageRequest getStageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetStageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetStageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStageRequest, GetStageResponse>().withOperationName("GetStage")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetStageRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getStageRequest));
            CompletableFuture<GetStageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information for the specified stage session.
     * </p>
     *
     * @param getStageSessionRequest
     * @return A Java Future containing the result of the GetStageSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.GetStageSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetStageSession" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStageSessionResponse> getStageSession(GetStageSessionRequest getStageSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStageSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStageSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStageSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetStageSessionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetStageSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetStageSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStageSessionRequest, GetStageSessionResponse>()
                            .withOperationName("GetStageSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetStageSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getStageSessionRequest));
            CompletableFuture<GetStageSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the storage configuration for the specified ARN.
     * </p>
     *
     * @param getStorageConfigurationRequest
     * @return A Java Future containing the result of the GetStorageConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.GetStorageConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetStorageConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetStorageConfigurationResponse> getStorageConfiguration(
            GetStorageConfigurationRequest getStorageConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStorageConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStorageConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStorageConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetStorageConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetStorageConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetStorageConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetStorageConfigurationRequest, GetStorageConfigurationResponse>()
                            .withOperationName("GetStorageConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetStorageConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getStorageConfigurationRequest));
            CompletableFuture<GetStorageConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets summary information about all Compositions in your account, in the AWS region where the API request is
     * processed.
     * </p>
     *
     * @param listCompositionsRequest
     * @return A Java Future containing the result of the ListCompositions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.ListCompositions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListCompositions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCompositionsResponse> listCompositions(ListCompositionsRequest listCompositionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCompositionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCompositionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCompositions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCompositionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCompositionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCompositionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCompositionsRequest, ListCompositionsResponse>()
                            .withOperationName("ListCompositions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCompositionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCompositionsRequest));
            CompletableFuture<ListCompositionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets summary information about all EncoderConfigurations in your account, in the AWS region where the API request
     * is processed.
     * </p>
     *
     * @param listEncoderConfigurationsRequest
     * @return A Java Future containing the result of the ListEncoderConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.ListEncoderConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListEncoderConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEncoderConfigurationsResponse> listEncoderConfigurations(
            ListEncoderConfigurationsRequest listEncoderConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEncoderConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEncoderConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEncoderConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEncoderConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEncoderConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEncoderConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEncoderConfigurationsRequest, ListEncoderConfigurationsResponse>()
                            .withOperationName("ListEncoderConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEncoderConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEncoderConfigurationsRequest));
            CompletableFuture<ListEncoderConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists events for a specified participant that occurred during a specified stage session.
     * </p>
     *
     * @param listParticipantEventsRequest
     * @return A Java Future containing the result of the ListParticipantEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.ListParticipantEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListParticipantEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListParticipantEventsResponse> listParticipantEvents(
            ListParticipantEventsRequest listParticipantEventsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listParticipantEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listParticipantEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListParticipantEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListParticipantEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListParticipantEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListParticipantEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListParticipantEventsRequest, ListParticipantEventsResponse>()
                            .withOperationName("ListParticipantEvents").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListParticipantEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listParticipantEventsRequest));
            CompletableFuture<ListParticipantEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all participants in a specified stage session.
     * </p>
     *
     * @param listParticipantsRequest
     * @return A Java Future containing the result of the ListParticipants operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.ListParticipants
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListParticipants" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListParticipantsResponse> listParticipants(ListParticipantsRequest listParticipantsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listParticipantsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listParticipantsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListParticipants");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListParticipantsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListParticipantsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListParticipantsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListParticipantsRequest, ListParticipantsResponse>()
                            .withOperationName("ListParticipants").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListParticipantsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listParticipantsRequest));
            CompletableFuture<ListParticipantsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all sessions for a specified stage.
     * </p>
     *
     * @param listStageSessionsRequest
     * @return A Java Future containing the result of the ListStageSessions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.ListStageSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListStageSessions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStageSessionsResponse> listStageSessions(ListStageSessionsRequest listStageSessionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStageSessionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStageSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStageSessions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStageSessionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStageSessionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStageSessionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStageSessionsRequest, ListStageSessionsResponse>()
                            .withOperationName("ListStageSessions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStageSessionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStageSessionsRequest));
            CompletableFuture<ListStageSessionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets summary information about all stages in your account, in the AWS region where the API request is processed.
     * </p>
     *
     * @param listStagesRequest
     * @return A Java Future containing the result of the ListStages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.ListStages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListStages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListStagesResponse> listStages(ListStagesRequest listStagesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListStagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStagesRequest, ListStagesResponse>().withOperationName("ListStages")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStagesRequest));
            CompletableFuture<ListStagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets summary information about all storage configurations in your account, in the AWS region where the API
     * request is processed.
     * </p>
     *
     * @param listStorageConfigurationsRequest
     * @return A Java Future containing the result of the ListStorageConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.ListStorageConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListStorageConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListStorageConfigurationsResponse> listStorageConfigurations(
            ListStorageConfigurationsRequest listStorageConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStorageConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStorageConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStorageConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListStorageConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListStorageConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListStorageConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListStorageConfigurationsRequest, ListStorageConfigurationsResponse>()
                            .withOperationName("ListStorageConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListStorageConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listStorageConfigurationsRequest));
            CompletableFuture<ListStorageConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>InternalServerException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a Composition from a stage based on the configuration provided in the request.
     * </p>
     * <p>
     * A Composition is an ephemeral resource that exists after this endpoint returns successfully. Composition stops
     * and the resource is deleted:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When <a>StopComposition</a> is called.
     * </p>
     * </li>
     * <li>
     * <p>
     * After a 1-minute timeout, when all participants are disconnected from the stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * After a 1-minute timeout, if there are no participants in the stage when StartComposition is called.
     * </p>
     * </li>
     * <li>
     * <p>
     * When broadcasting to the IVS channel fails and all retries are exhausted.
     * </p>
     * </li>
     * <li>
     * <p>
     * When broadcasting is disconnected and all attempts to reconnect are exhausted.
     * </p>
     * </li>
     * </ul>
     *
     * @param startCompositionRequest
     * @return A Java Future containing the result of the StartComposition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>PendingVerificationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.StartComposition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/StartComposition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartCompositionResponse> startComposition(StartCompositionRequest startCompositionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startCompositionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCompositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartComposition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartCompositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartCompositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartCompositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartCompositionRequest, StartCompositionResponse>()
                            .withOperationName("StartComposition").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StartCompositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startCompositionRequest));
            CompletableFuture<StartCompositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops and deletes a Composition resource. Any broadcast from the Composition resource is stopped.
     * </p>
     *
     * @param stopCompositionRequest
     * @return A Java Future containing the result of the StopComposition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>InternalServerException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>ConflictException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.StopComposition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/StopComposition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StopCompositionResponse> stopComposition(StopCompositionRequest stopCompositionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopCompositionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopCompositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopComposition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopCompositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopCompositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopCompositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopCompositionRequest, StopCompositionResponse>()
                            .withOperationName("StopComposition").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new StopCompositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopCompositionRequest));
            CompletableFuture<StopCompositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates tags for the AWS resource with the specified ARN.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>InternalServerException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the resource with the specified ARN.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>InternalServerException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a stage’s configuration.
     * </p>
     *
     * @param updateStageRequest
     * @return A Java Future containing the result of the UpdateStage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException</li>
     *         <li>ValidationException</li>
     *         <li>AccessDeniedException</li>
     *         <li>ServiceQuotaExceededException</li>
     *         <li>PendingVerificationException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>IvsRealTimeException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample IvsRealTimeAsyncClient.UpdateStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/UpdateStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateStageResponse> updateStage(UpdateStageRequest updateStageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateStageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateStageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateStageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateStageRequest, UpdateStageResponse>()
                            .withOperationName("UpdateStage").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateStageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateStageRequest));
            CompletableFuture<UpdateStageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final IvsRealTimeServiceClientConfiguration serviceClientConfiguration() {
        return new IvsRealTimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IvsRealTimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PendingVerification")
                                .exceptionBuilderSupplier(PendingVerificationException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        IvsRealTimeServiceClientConfigurationBuilder serviceConfigBuilder = new IvsRealTimeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
