/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ivsrealtime;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ivsrealtime.internal.IvsRealTimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.ivsrealtime.model.AccessDeniedException;
import software.amazon.awssdk.services.ivsrealtime.model.ConflictException;
import software.amazon.awssdk.services.ivsrealtime.model.CreateEncoderConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.CreateEncoderConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.CreateParticipantTokenRequest;
import software.amazon.awssdk.services.ivsrealtime.model.CreateParticipantTokenResponse;
import software.amazon.awssdk.services.ivsrealtime.model.CreateStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.CreateStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.CreateStorageConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.CreateStorageConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteEncoderConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteEncoderConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteStorageConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.DeleteStorageConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.DisconnectParticipantRequest;
import software.amazon.awssdk.services.ivsrealtime.model.DisconnectParticipantResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetCompositionRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetCompositionResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetEncoderConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetEncoderConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetParticipantRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetParticipantResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetStageSessionRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetStageSessionResponse;
import software.amazon.awssdk.services.ivsrealtime.model.GetStorageConfigurationRequest;
import software.amazon.awssdk.services.ivsrealtime.model.GetStorageConfigurationResponse;
import software.amazon.awssdk.services.ivsrealtime.model.InternalServerException;
import software.amazon.awssdk.services.ivsrealtime.model.IvsRealTimeException;
import software.amazon.awssdk.services.ivsrealtime.model.ListCompositionsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListCompositionsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListEncoderConfigurationsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListEncoderConfigurationsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListParticipantEventsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListParticipantEventsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListParticipantsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListParticipantsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListStageSessionsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListStageSessionsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListStagesRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListStagesResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListStorageConfigurationsRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListStorageConfigurationsResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.ivsrealtime.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.ivsrealtime.model.PendingVerificationException;
import software.amazon.awssdk.services.ivsrealtime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ivsrealtime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.ivsrealtime.model.StartCompositionRequest;
import software.amazon.awssdk.services.ivsrealtime.model.StartCompositionResponse;
import software.amazon.awssdk.services.ivsrealtime.model.StopCompositionRequest;
import software.amazon.awssdk.services.ivsrealtime.model.StopCompositionResponse;
import software.amazon.awssdk.services.ivsrealtime.model.TagResourceRequest;
import software.amazon.awssdk.services.ivsrealtime.model.TagResourceResponse;
import software.amazon.awssdk.services.ivsrealtime.model.UntagResourceRequest;
import software.amazon.awssdk.services.ivsrealtime.model.UntagResourceResponse;
import software.amazon.awssdk.services.ivsrealtime.model.UpdateStageRequest;
import software.amazon.awssdk.services.ivsrealtime.model.UpdateStageResponse;
import software.amazon.awssdk.services.ivsrealtime.model.ValidationException;
import software.amazon.awssdk.services.ivsrealtime.transform.CreateEncoderConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.CreateParticipantTokenRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.CreateStageRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.CreateStorageConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.DeleteEncoderConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.DeleteStageRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.DeleteStorageConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.DisconnectParticipantRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetCompositionRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetEncoderConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetParticipantRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetStageRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetStageSessionRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.GetStorageConfigurationRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListCompositionsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListEncoderConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListParticipantEventsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListParticipantsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListStageSessionsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListStagesRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListStorageConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.StartCompositionRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.StopCompositionRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.ivsrealtime.transform.UpdateStageRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link IvsRealTimeClient}.
 *
 * @see IvsRealTimeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultIvsRealTimeClient implements IvsRealTimeClient {
    private static final Logger log = Logger.loggerFor(DefaultIvsRealTimeClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultIvsRealTimeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an EncoderConfiguration object.
     * </p>
     *
     * @param createEncoderConfigurationRequest
     * @return Result of the CreateEncoderConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.CreateEncoderConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/CreateEncoderConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateEncoderConfigurationResponse createEncoderConfiguration(
            CreateEncoderConfigurationRequest createEncoderConfigurationRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, InternalServerException, ServiceQuotaExceededException,
            ConflictException, PendingVerificationException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEncoderConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateEncoderConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEncoderConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEncoderConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEncoderConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateEncoderConfigurationRequest, CreateEncoderConfigurationResponse>()
                            .withOperationName("CreateEncoderConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createEncoderConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateEncoderConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an additional token for a specified stage. This can be done after stage creation or when tokens expire.
     * Tokens always are scoped to the stage for which they are created.
     * </p>
     * <p>
     * Encryption keys are owned by Amazon IVS and never used directly by your application.
     * </p>
     *
     * @param createParticipantTokenRequest
     * @return Result of the CreateParticipantToken operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.CreateParticipantToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/CreateParticipantToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateParticipantTokenResponse createParticipantToken(CreateParticipantTokenRequest createParticipantTokenRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, ServiceQuotaExceededException,
            PendingVerificationException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateParticipantTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateParticipantTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createParticipantTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createParticipantTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParticipantToken");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateParticipantTokenRequest, CreateParticipantTokenResponse>()
                            .withOperationName("CreateParticipantToken").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createParticipantTokenRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateParticipantTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new stage (and optionally participant tokens).
     * </p>
     *
     * @param createStageRequest
     * @return Result of the CreateStage operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.CreateStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/CreateStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateStageResponse createStage(CreateStageRequest createStageRequest) throws ValidationException,
            AccessDeniedException, ServiceQuotaExceededException, PendingVerificationException, AwsServiceException,
            SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStage");

            return clientHandler.execute(new ClientExecutionParams<CreateStageRequest, CreateStageResponse>()
                    .withOperationName("CreateStage").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createStageRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new storage configuration, used to enable recording to Amazon S3. When a StorageConfiguration is
     * created, IVS will modify the S3 bucketPolicy of the provided bucket. This will ensure that IVS has sufficient
     * permissions to write content to the provided bucket.
     * </p>
     *
     * @param createStorageConfigurationRequest
     * @return Result of the CreateStorageConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.CreateStorageConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/CreateStorageConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateStorageConfigurationResponse createStorageConfiguration(
            CreateStorageConfigurationRequest createStorageConfigurationRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, InternalServerException, ServiceQuotaExceededException,
            ConflictException, PendingVerificationException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStorageConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateStorageConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStorageConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStorageConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStorageConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateStorageConfigurationRequest, CreateStorageConfigurationResponse>()
                            .withOperationName("CreateStorageConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createStorageConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateStorageConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an EncoderConfiguration resource. Ensures that no Compositions are using this template; otherwise,
     * returns an error.
     * </p>
     *
     * @param deleteEncoderConfigurationRequest
     * @return Result of the DeleteEncoderConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.DeleteEncoderConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/DeleteEncoderConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteEncoderConfigurationResponse deleteEncoderConfiguration(
            DeleteEncoderConfigurationRequest deleteEncoderConfigurationRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, InternalServerException, ServiceQuotaExceededException,
            ConflictException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEncoderConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteEncoderConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEncoderConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEncoderConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEncoderConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteEncoderConfigurationRequest, DeleteEncoderConfigurationResponse>()
                            .withOperationName("DeleteEncoderConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteEncoderConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteEncoderConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Shuts down and deletes the specified stage (disconnecting all participants).
     * </p>
     *
     * @param deleteStageRequest
     * @return Result of the DeleteStage operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.DeleteStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/DeleteStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteStageResponse deleteStage(DeleteStageRequest deleteStageRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, ConflictException, PendingVerificationException, AwsServiceException,
            SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStage");

            return clientHandler.execute(new ClientExecutionParams<DeleteStageRequest, DeleteStageResponse>()
                    .withOperationName("DeleteStage").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteStageRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the storage configuration for the specified ARN.
     * </p>
     * <p>
     * If you try to delete a storage configuration that is used by a Composition, you will get an error (409
     * ConflictException). To avoid this, for all Compositions that reference the storage configuration, first use
     * <a>StopComposition</a> and wait for it to complete, then use DeleteStorageConfiguration.
     * </p>
     *
     * @param deleteStorageConfigurationRequest
     * @return Result of the DeleteStorageConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.DeleteStorageConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/DeleteStorageConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteStorageConfigurationResponse deleteStorageConfiguration(
            DeleteStorageConfigurationRequest deleteStorageConfigurationRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, InternalServerException, ServiceQuotaExceededException,
            ConflictException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStorageConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteStorageConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStorageConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStorageConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStorageConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteStorageConfigurationRequest, DeleteStorageConfigurationResponse>()
                            .withOperationName("DeleteStorageConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteStorageConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteStorageConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disconnects a specified participant and revokes the participant permanently from a specified stage.
     * </p>
     *
     * @param disconnectParticipantRequest
     * @return Result of the DisconnectParticipant operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.DisconnectParticipant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/DisconnectParticipant"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisconnectParticipantResponse disconnectParticipant(DisconnectParticipantRequest disconnectParticipantRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisconnectParticipantResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisconnectParticipantResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disconnectParticipantRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectParticipantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectParticipant");

            return clientHandler.execute(new ClientExecutionParams<DisconnectParticipantRequest, DisconnectParticipantResponse>()
                    .withOperationName("DisconnectParticipant").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disconnectParticipantRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisconnectParticipantRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get information about the specified Composition resource.
     * </p>
     *
     * @param getCompositionRequest
     * @return Result of the GetComposition operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.GetComposition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetComposition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetCompositionResponse getComposition(GetCompositionRequest getCompositionRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, InternalServerException, ServiceQuotaExceededException,
            ConflictException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCompositionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCompositionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCompositionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCompositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetComposition");

            return clientHandler.execute(new ClientExecutionParams<GetCompositionRequest, GetCompositionResponse>()
                    .withOperationName("GetComposition").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getCompositionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCompositionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the specified EncoderConfiguration resource.
     * </p>
     *
     * @param getEncoderConfigurationRequest
     * @return Result of the GetEncoderConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.GetEncoderConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetEncoderConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEncoderConfigurationResponse getEncoderConfiguration(GetEncoderConfigurationRequest getEncoderConfigurationRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEncoderConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEncoderConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEncoderConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEncoderConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEncoderConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetEncoderConfigurationRequest, GetEncoderConfigurationResponse>()
                            .withOperationName("GetEncoderConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getEncoderConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetEncoderConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about the specified participant token.
     * </p>
     *
     * @param getParticipantRequest
     * @return Result of the GetParticipant operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.GetParticipant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetParticipant" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetParticipantResponse getParticipant(GetParticipantRequest getParticipantRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetParticipantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetParticipantResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getParticipantRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getParticipantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetParticipant");

            return clientHandler.execute(new ClientExecutionParams<GetParticipantRequest, GetParticipantResponse>()
                    .withOperationName("GetParticipant").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getParticipantRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetParticipantRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information for the specified stage.
     * </p>
     *
     * @param getStageRequest
     * @return Result of the GetStage operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.GetStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetStageResponse getStage(GetStageRequest getStageRequest) throws ResourceNotFoundException, ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStage");

            return clientHandler.execute(new ClientExecutionParams<GetStageRequest, GetStageResponse>()
                    .withOperationName("GetStage").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getStageRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information for the specified stage session.
     * </p>
     *
     * @param getStageSessionRequest
     * @return Result of the GetStageSession operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.GetStageSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetStageSession" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetStageSessionResponse getStageSession(GetStageSessionRequest getStageSessionRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStageSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetStageSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStageSessionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStageSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStageSession");

            return clientHandler.execute(new ClientExecutionParams<GetStageSessionRequest, GetStageSessionResponse>()
                    .withOperationName("GetStageSession").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getStageSessionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetStageSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the storage configuration for the specified ARN.
     * </p>
     *
     * @param getStorageConfigurationRequest
     * @return Result of the GetStorageConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.GetStorageConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/GetStorageConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetStorageConfigurationResponse getStorageConfiguration(GetStorageConfigurationRequest getStorageConfigurationRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetStorageConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetStorageConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getStorageConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getStorageConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetStorageConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetStorageConfigurationRequest, GetStorageConfigurationResponse>()
                            .withOperationName("GetStorageConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getStorageConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetStorageConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about all Compositions in your account, in the AWS region where the API request is
     * processed.
     * </p>
     *
     * @param listCompositionsRequest
     * @return Result of the ListCompositions operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListCompositions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListCompositions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListCompositionsResponse listCompositions(ListCompositionsRequest listCompositionsRequest) throws ValidationException,
            AccessDeniedException, InternalServerException, ServiceQuotaExceededException, ConflictException,
            AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCompositionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCompositionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCompositionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCompositionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCompositions");

            return clientHandler.execute(new ClientExecutionParams<ListCompositionsRequest, ListCompositionsResponse>()
                    .withOperationName("ListCompositions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listCompositionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCompositionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about all EncoderConfigurations in your account, in the AWS region where the API request
     * is processed.
     * </p>
     *
     * @param listEncoderConfigurationsRequest
     * @return Result of the ListEncoderConfigurations operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListEncoderConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListEncoderConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEncoderConfigurationsResponse listEncoderConfigurations(
            ListEncoderConfigurationsRequest listEncoderConfigurationsRequest) throws ValidationException, AccessDeniedException,
            InternalServerException, ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException,
            IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEncoderConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListEncoderConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEncoderConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEncoderConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEncoderConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListEncoderConfigurationsRequest, ListEncoderConfigurationsResponse>()
                            .withOperationName("ListEncoderConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listEncoderConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListEncoderConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists events for a specified participant that occurred during a specified stage session.
     * </p>
     *
     * @param listParticipantEventsRequest
     * @return Result of the ListParticipantEvents operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListParticipantEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListParticipantEvents"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListParticipantEventsResponse listParticipantEvents(ListParticipantEventsRequest listParticipantEventsRequest)
            throws ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListParticipantEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListParticipantEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listParticipantEventsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listParticipantEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListParticipantEvents");

            return clientHandler.execute(new ClientExecutionParams<ListParticipantEventsRequest, ListParticipantEventsResponse>()
                    .withOperationName("ListParticipantEvents").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listParticipantEventsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListParticipantEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all participants in a specified stage session.
     * </p>
     *
     * @param listParticipantsRequest
     * @return Result of the ListParticipants operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListParticipants
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListParticipants" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListParticipantsResponse listParticipants(ListParticipantsRequest listParticipantsRequest) throws ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListParticipantsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListParticipantsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listParticipantsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listParticipantsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListParticipants");

            return clientHandler.execute(new ClientExecutionParams<ListParticipantsRequest, ListParticipantsResponse>()
                    .withOperationName("ListParticipants").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listParticipantsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListParticipantsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets all sessions for a specified stage.
     * </p>
     *
     * @param listStageSessionsRequest
     * @return Result of the ListStageSessions operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListStageSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListStageSessions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListStageSessionsResponse listStageSessions(ListStageSessionsRequest listStageSessionsRequest)
            throws ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStageSessionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStageSessionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStageSessionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStageSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStageSessions");

            return clientHandler.execute(new ClientExecutionParams<ListStageSessionsRequest, ListStageSessionsResponse>()
                    .withOperationName("ListStageSessions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listStageSessionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStageSessionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about all stages in your account, in the AWS region where the API request is processed.
     * </p>
     *
     * @param listStagesRequest
     * @return Result of the ListStages operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListStages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListStages" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListStagesResponse listStages(ListStagesRequest listStagesRequest) throws ValidationException, AccessDeniedException,
            ConflictException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStagesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStagesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStagesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStages");

            return clientHandler.execute(new ClientExecutionParams<ListStagesRequest, ListStagesResponse>()
                    .withOperationName("ListStages").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listStagesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStagesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets summary information about all storage configurations in your account, in the AWS region where the API
     * request is processed.
     * </p>
     *
     * @param listStorageConfigurationsRequest
     * @return Result of the ListStorageConfigurations operation returned by the service.
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListStorageConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListStorageConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListStorageConfigurationsResponse listStorageConfigurations(
            ListStorageConfigurationsRequest listStorageConfigurationsRequest) throws ValidationException, AccessDeniedException,
            InternalServerException, ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException,
            IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStorageConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListStorageConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStorageConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStorageConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStorageConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListStorageConfigurationsRequest, ListStorageConfigurationsResponse>()
                            .withOperationName("ListStorageConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listStorageConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListStorageConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about AWS tags for the specified ARN.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a Composition from a stage based on the configuration provided in the request.
     * </p>
     * <p>
     * A Composition is an ephemeral resource that exists after this endpoint returns successfully. Composition stops
     * and the resource is deleted:
     * </p>
     * <ul>
     * <li>
     * <p>
     * When <a>StopComposition</a> is called.
     * </p>
     * </li>
     * <li>
     * <p>
     * After a 1-minute timeout, when all participants are disconnected from the stage.
     * </p>
     * </li>
     * <li>
     * <p>
     * After a 1-minute timeout, if there are no participants in the stage when StartComposition is called.
     * </p>
     * </li>
     * <li>
     * <p>
     * When broadcasting to the IVS channel fails and all retries are exhausted.
     * </p>
     * </li>
     * <li>
     * <p>
     * When broadcasting is disconnected and all attempts to reconnect are exhausted.
     * </p>
     * </li>
     * </ul>
     *
     * @param startCompositionRequest
     * @return Result of the StartComposition operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.StartComposition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/StartComposition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartCompositionResponse startComposition(StartCompositionRequest startCompositionRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, ConflictException, PendingVerificationException, AwsServiceException,
            SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartCompositionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartCompositionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startCompositionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startCompositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartComposition");

            return clientHandler.execute(new ClientExecutionParams<StartCompositionRequest, StartCompositionResponse>()
                    .withOperationName("StartComposition").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startCompositionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartCompositionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops and deletes a Composition resource. Any broadcast from the Composition resource is stopped.
     * </p>
     *
     * @param stopCompositionRequest
     * @return Result of the StopComposition operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws ServiceQuotaExceededException
     * @throws ConflictException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.StopComposition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/StopComposition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopCompositionResponse stopComposition(StopCompositionRequest stopCompositionRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, InternalServerException,
            ServiceQuotaExceededException, ConflictException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopCompositionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopCompositionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopCompositionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopCompositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopComposition");

            return clientHandler.execute(new ClientExecutionParams<StopCompositionRequest, StopCompositionResponse>()
                    .withOperationName("StopComposition").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(stopCompositionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopCompositionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or updates tags for the AWS resource with the specified ARN.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the resource with the specified ARN.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a stage’s configuration.
     * </p>
     *
     * @param updateStageRequest
     * @return Result of the UpdateStage operation returned by the service.
     * @throws ResourceNotFoundException
     * @throws ValidationException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws PendingVerificationException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws IvsRealTimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample IvsRealTimeClient.UpdateStage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/ivs-realtime-2020-07-14/UpdateStage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateStageResponse updateStage(UpdateStageRequest updateStageRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, ServiceQuotaExceededException, PendingVerificationException,
            AwsServiceException, SdkClientException, IvsRealTimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateStageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateStageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "IVS RealTime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStage");

            return clientHandler.execute(new ClientExecutionParams<UpdateStageRequest, UpdateStageResponse>()
                    .withOperationName("UpdateStage").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateStageRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateStageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        IvsRealTimeServiceClientConfigurationBuilder serviceConfigBuilder = new IvsRealTimeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(IvsRealTimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PendingVerification")
                                .exceptionBuilderSupplier(PendingVerificationException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    @Override
    public final IvsRealTimeServiceClientConfiguration serviceClientConfiguration() {
        return new IvsRealTimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
