/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ivsrealtime.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A stage session begins when the first participant joins a stage and ends after the last participant leaves the stage.
 * A stage session helps with debugging stages by grouping events and participants into shorter periods of time (i.e., a
 * session), which is helpful when stages are used over long periods of time.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StageSession implements SdkPojo, Serializable, ToCopyableBuilder<StageSession.Builder, StageSession> {
    private static final SdkField<String> SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sessionId").getter(getter(StageSession::sessionId)).setter(setter(Builder::sessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sessionId").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime")
            .getter(getter(StageSession::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime")
            .getter(getter(StageSession::endTime))
            .setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SESSION_ID_FIELD,
            START_TIME_FIELD, END_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String sessionId;

    private final Instant startTime;

    private final Instant endTime;

    private StageSession(BuilderImpl builder) {
        this.sessionId = builder.sessionId;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
    }

    /**
     * <p>
     * ID of the session within the stage.
     * </p>
     * 
     * @return ID of the session within the stage.
     */
    public final String sessionId() {
        return sessionId;
    }

    /**
     * <p>
     * ISO 8601 timestamp (returned as a string) when this stage session began.
     * </p>
     * 
     * @return ISO 8601 timestamp (returned as a string) when this stage session began.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * ISO 8601 timestamp (returned as a string) when the stage session ended. This is null if the stage is active.
     * </p>
     * 
     * @return ISO 8601 timestamp (returned as a string) when the stage session ended. This is null if the stage is
     *         active.
     */
    public final Instant endTime() {
        return endTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sessionId());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StageSession)) {
            return false;
        }
        StageSession other = (StageSession) obj;
        return Objects.equals(sessionId(), other.sessionId()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StageSession").add("SessionId", sessionId()).add("StartTime", startTime())
                .add("EndTime", endTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "sessionId":
            return Optional.ofNullable(clazz.cast(sessionId()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StageSession, T> g) {
        return obj -> g.apply((StageSession) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StageSession> {
        /**
         * <p>
         * ID of the session within the stage.
         * </p>
         * 
         * @param sessionId
         *        ID of the session within the stage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionId(String sessionId);

        /**
         * <p>
         * ISO 8601 timestamp (returned as a string) when this stage session began.
         * </p>
         * 
         * @param startTime
         *        ISO 8601 timestamp (returned as a string) when this stage session began.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * ISO 8601 timestamp (returned as a string) when the stage session ended. This is null if the stage is active.
         * </p>
         * 
         * @param endTime
         *        ISO 8601 timestamp (returned as a string) when the stage session ended. This is null if the stage is
         *        active.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);
    }

    static final class BuilderImpl implements Builder {
        private String sessionId;

        private Instant startTime;

        private Instant endTime;

        private BuilderImpl() {
        }

        private BuilderImpl(StageSession model) {
            sessionId(model.sessionId);
            startTime(model.startTime);
            endTime(model.endTime);
        }

        public final String getSessionId() {
            return sessionId;
        }

        public final void setSessionId(String sessionId) {
            this.sessionId = sessionId;
        }

        @Override
        public final Builder sessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        @Override
        public StageSession build() {
            return new StageSession(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
