/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ivsrealtime.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An occurrence during a stage session.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Event implements SdkPojo, Serializable, ToCopyableBuilder<Event.Builder, Event> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Event::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> PARTICIPANT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("participantId").getter(getter(Event::participantId)).setter(setter(Builder::participantId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("participantId").build()).build();

    private static final SdkField<Instant> EVENT_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("eventTime")
            .getter(getter(Event::eventTime))
            .setter(setter(Builder::eventTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> REMOTE_PARTICIPANT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("remoteParticipantId").getter(getter(Event::remoteParticipantId))
            .setter(setter(Builder::remoteParticipantId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("remoteParticipantId").build())
            .build();

    private static final SdkField<String> ERROR_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("errorCode").getter(getter(Event::errorCodeAsString)).setter(setter(Builder::errorCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("errorCode").build()).build();

    private static final SdkField<String> DESTINATION_STAGE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("destinationStageArn").getter(getter(Event::destinationStageArn))
            .setter(setter(Builder::destinationStageArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("destinationStageArn").build())
            .build();

    private static final SdkField<String> DESTINATION_SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("destinationSessionId").getter(getter(Event::destinationSessionId))
            .setter(setter(Builder::destinationSessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("destinationSessionId").build())
            .build();

    private static final SdkField<Boolean> REPLICA_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("replica").getter(getter(Event::replica)).setter(setter(Builder::replica))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("replica").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            PARTICIPANT_ID_FIELD, EVENT_TIME_FIELD, REMOTE_PARTICIPANT_ID_FIELD, ERROR_CODE_FIELD, DESTINATION_STAGE_ARN_FIELD,
            DESTINATION_SESSION_ID_FIELD, REPLICA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String participantId;

    private final Instant eventTime;

    private final String remoteParticipantId;

    private final String errorCode;

    private final String destinationStageArn;

    private final String destinationSessionId;

    private final Boolean replica;

    private Event(BuilderImpl builder) {
        this.name = builder.name;
        this.participantId = builder.participantId;
        this.eventTime = builder.eventTime;
        this.remoteParticipantId = builder.remoteParticipantId;
        this.errorCode = builder.errorCode;
        this.destinationStageArn = builder.destinationStageArn;
        this.destinationSessionId = builder.destinationSessionId;
        this.replica = builder.replica;
    }

    /**
     * <p>
     * The name of the event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link EventName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the event.
     * @see EventName
     */
    public final EventName name() {
        return EventName.fromValue(name);
    }

    /**
     * <p>
     * The name of the event.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link EventName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the event.
     * @see EventName
     */
    public final String nameAsString() {
        return name;
    }

    /**
     * <p>
     * Unique identifier for the participant who triggered the event. This is assigned by IVS.
     * </p>
     * 
     * @return Unique identifier for the participant who triggered the event. This is assigned by IVS.
     */
    public final String participantId() {
        return participantId;
    }

    /**
     * <p>
     * ISO 8601 timestamp (returned as a string) for when the event occurred.
     * </p>
     * 
     * @return ISO 8601 timestamp (returned as a string) for when the event occurred.
     */
    public final Instant eventTime() {
        return eventTime;
    }

    /**
     * <p>
     * Unique identifier for the remote participant. For a subscribe event, this is the publisher. For a publish or join
     * event, this is null. This is assigned by IVS.
     * </p>
     * 
     * @return Unique identifier for the remote participant. For a subscribe event, this is the publisher. For a publish
     *         or join event, this is null. This is assigned by IVS.
     */
    public final String remoteParticipantId() {
        return remoteParticipantId;
    }

    /**
     * <p>
     * If the event is an error event, the error code is provided to give insight into the specific error that occurred.
     * If the event is not an error event, this field is null.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>B_FRAME_PRESENT</code> — The participant's stream includes B-frames. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP Publishing</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BITRATE_EXCEEDED</code> — The participant exceeded the maximum supported bitrate. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service Quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSUFFICIENT_CAPABILITIES</code> — The participant tried to take an action that the participant’s token is
     * not allowed to do. For details on participant capabilities, see the <code>capabilities</code> field in
     * <a>CreateParticipantToken</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INTERNAL_SERVER_EXCEPTION</code> — The participant failed to publish to the stage due to an internal server
     * error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_AUDIO_CODEC</code> — The participant is using an invalid audio codec. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_INPUT</code> — The participant is using an invalid input stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_PROTOCOL</code> — The participant's IngestConfiguration resource is configured for RTMPS but they
     * tried streaming with RTMP. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP Publishing</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_STREAM_KEY</code> — The participant is using an invalid stream key. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP Publishing</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_VIDEO_CODEC</code> — The participant is using an invalid video codec. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PUBLISHER_NOT_FOUND</code> — The participant tried to subscribe to a publisher that doesn’t exist.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>QUOTA_EXCEEDED</code> — The number of participants who want to publish/subscribe to a stage exceeds the
     * quota. For details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html">
     * Service Quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RESOLUTION_EXCEEDED</code> — The participant exceeded the maximum supported resolution. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service Quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REUSE_OF_STREAM_KEY</code> — The participant tried to use a stream key that is associated with another
     * active stage session.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STREAM_DURATION_EXCEEDED</code> — The participant exceeded the maximum allowed stream duration. For
     * details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
     * Quotas</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorCode} will
     * return {@link EventErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorCodeAsString}.
     * </p>
     * 
     * @return If the event is an error event, the error code is provided to give insight into the specific error that
     *         occurred. If the event is not an error event, this field is null.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>B_FRAME_PRESENT</code> — The participant's stream includes B-frames. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
     *         Publishing</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BITRATE_EXCEEDED</code> — The participant exceeded the maximum supported bitrate. For details, see
     *         <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
     *         Quotas</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INSUFFICIENT_CAPABILITIES</code> — The participant tried to take an action that the participant’s
     *         token is not allowed to do. For details on participant capabilities, see the <code>capabilities</code>
     *         field in <a>CreateParticipantToken</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INTERNAL_SERVER_EXCEPTION</code> — The participant failed to publish to the stage due to an
     *         internal server error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_AUDIO_CODEC</code> — The participant is using an invalid audio codec. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_INPUT</code> — The participant is using an invalid input stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_PROTOCOL</code> — The participant's IngestConfiguration resource is configured for RTMPS
     *         but they tried streaming with RTMP. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
     *         Publishing</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_STREAM_KEY</code> — The participant is using an invalid stream key. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
     *         Publishing</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_VIDEO_CODEC</code> — The participant is using an invalid video codec. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PUBLISHER_NOT_FOUND</code> — The participant tried to subscribe to a publisher that doesn’t exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>QUOTA_EXCEEDED</code> — The number of participants who want to publish/subscribe to a stage exceeds
     *         the quota. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service Quotas</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RESOLUTION_EXCEEDED</code> — The participant exceeded the maximum supported resolution. For
     *         details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html">
     *         Service Quotas</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REUSE_OF_STREAM_KEY</code> — The participant tried to use a stream key that is associated with
     *         another active stage session.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STREAM_DURATION_EXCEEDED</code> — The participant exceeded the maximum allowed stream duration. For
     *         details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html">
     *         Service Quotas</a>.
     *         </p>
     *         </li>
     * @see EventErrorCode
     */
    public final EventErrorCode errorCode() {
        return EventErrorCode.fromValue(errorCode);
    }

    /**
     * <p>
     * If the event is an error event, the error code is provided to give insight into the specific error that occurred.
     * If the event is not an error event, this field is null.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>B_FRAME_PRESENT</code> — The participant's stream includes B-frames. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP Publishing</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BITRATE_EXCEEDED</code> — The participant exceeded the maximum supported bitrate. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service Quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSUFFICIENT_CAPABILITIES</code> — The participant tried to take an action that the participant’s token is
     * not allowed to do. For details on participant capabilities, see the <code>capabilities</code> field in
     * <a>CreateParticipantToken</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INTERNAL_SERVER_EXCEPTION</code> — The participant failed to publish to the stage due to an internal server
     * error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_AUDIO_CODEC</code> — The participant is using an invalid audio codec. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_INPUT</code> — The participant is using an invalid input stream.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_PROTOCOL</code> — The participant's IngestConfiguration resource is configured for RTMPS but they
     * tried streaming with RTMP. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP Publishing</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_STREAM_KEY</code> — The participant is using an invalid stream key. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP Publishing</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INVALID_VIDEO_CODEC</code> — The participant is using an invalid video codec. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PUBLISHER_NOT_FOUND</code> — The participant tried to subscribe to a publisher that doesn’t exist.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>QUOTA_EXCEEDED</code> — The number of participants who want to publish/subscribe to a stage exceeds the
     * quota. For details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html">
     * Service Quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RESOLUTION_EXCEEDED</code> — The participant exceeded the maximum supported resolution. For details, see <a
     * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service Quotas</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REUSE_OF_STREAM_KEY</code> — The participant tried to use a stream key that is associated with another
     * active stage session.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STREAM_DURATION_EXCEEDED</code> — The participant exceeded the maximum allowed stream duration. For
     * details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
     * Quotas</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorCode} will
     * return {@link EventErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorCodeAsString}.
     * </p>
     * 
     * @return If the event is an error event, the error code is provided to give insight into the specific error that
     *         occurred. If the event is not an error event, this field is null.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>B_FRAME_PRESENT</code> — The participant's stream includes B-frames. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
     *         Publishing</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BITRATE_EXCEEDED</code> — The participant exceeded the maximum supported bitrate. For details, see
     *         <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
     *         Quotas</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INSUFFICIENT_CAPABILITIES</code> — The participant tried to take an action that the participant’s
     *         token is not allowed to do. For details on participant capabilities, see the <code>capabilities</code>
     *         field in <a>CreateParticipantToken</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INTERNAL_SERVER_EXCEPTION</code> — The participant failed to publish to the stage due to an
     *         internal server error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_AUDIO_CODEC</code> — The participant is using an invalid audio codec. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_INPUT</code> — The participant is using an invalid input stream.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_PROTOCOL</code> — The participant's IngestConfiguration resource is configured for RTMPS
     *         but they tried streaming with RTMP. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
     *         Publishing</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_STREAM_KEY</code> — The participant is using an invalid stream key. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
     *         Publishing</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INVALID_VIDEO_CODEC</code> — The participant is using an invalid video codec. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PUBLISHER_NOT_FOUND</code> — The participant tried to subscribe to a publisher that doesn’t exist.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>QUOTA_EXCEEDED</code> — The number of participants who want to publish/subscribe to a stage exceeds
     *         the quota. For details, see <a
     *         href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service Quotas</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RESOLUTION_EXCEEDED</code> — The participant exceeded the maximum supported resolution. For
     *         details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html">
     *         Service Quotas</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REUSE_OF_STREAM_KEY</code> — The participant tried to use a stream key that is associated with
     *         another active stage session.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STREAM_DURATION_EXCEEDED</code> — The participant exceeded the maximum allowed stream duration. For
     *         details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html">
     *         Service Quotas</a>.
     *         </p>
     *         </li>
     * @see EventErrorCode
     */
    public final String errorCodeAsString() {
        return errorCode;
    }

    /**
     * <p>
     * ARN of the stage where the participant is replicated. Applicable only if the event name is
     * <code>REPLICATION_STARTED</code> or <code>REPLICATION_STOPPED</code>.
     * </p>
     * 
     * @return ARN of the stage where the participant is replicated. Applicable only if the event name is
     *         <code>REPLICATION_STARTED</code> or <code>REPLICATION_STOPPED</code>.
     */
    public final String destinationStageArn() {
        return destinationStageArn;
    }

    /**
     * <p>
     * ID of the session within the destination stage. Applicable only if the event name is
     * <code>REPLICATION_STARTED</code> or <code>REPLICATION_STOPPED</code>.
     * </p>
     * 
     * @return ID of the session within the destination stage. Applicable only if the event name is
     *         <code>REPLICATION_STARTED</code> or <code>REPLICATION_STOPPED</code>.
     */
    public final String destinationSessionId() {
        return destinationSessionId;
    }

    /**
     * <p>
     * If true, this indicates the <code>participantId</code> is a replicated participant. If this is a subscribe event,
     * then this flag refers to <code>remoteParticipantId</code>. Default: <code>false</code>.
     * </p>
     * 
     * @return If true, this indicates the <code>participantId</code> is a replicated participant. If this is a
     *         subscribe event, then this flag refers to <code>remoteParticipantId</code>. Default: <code>false</code>.
     */
    public final Boolean replica() {
        return replica;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(participantId());
        hashCode = 31 * hashCode + Objects.hashCode(eventTime());
        hashCode = 31 * hashCode + Objects.hashCode(remoteParticipantId());
        hashCode = 31 * hashCode + Objects.hashCode(errorCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(destinationStageArn());
        hashCode = 31 * hashCode + Objects.hashCode(destinationSessionId());
        hashCode = 31 * hashCode + Objects.hashCode(replica());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Event)) {
            return false;
        }
        Event other = (Event) obj;
        return Objects.equals(nameAsString(), other.nameAsString()) && Objects.equals(participantId(), other.participantId())
                && Objects.equals(eventTime(), other.eventTime())
                && Objects.equals(remoteParticipantId(), other.remoteParticipantId())
                && Objects.equals(errorCodeAsString(), other.errorCodeAsString())
                && Objects.equals(destinationStageArn(), other.destinationStageArn())
                && Objects.equals(destinationSessionId(), other.destinationSessionId())
                && Objects.equals(replica(), other.replica());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Event").add("Name", nameAsString()).add("ParticipantId", participantId())
                .add("EventTime", eventTime()).add("RemoteParticipantId", remoteParticipantId())
                .add("ErrorCode", errorCodeAsString()).add("DestinationStageArn", destinationStageArn())
                .add("DestinationSessionId", destinationSessionId()).add("Replica", replica()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        case "participantId":
            return Optional.ofNullable(clazz.cast(participantId()));
        case "eventTime":
            return Optional.ofNullable(clazz.cast(eventTime()));
        case "remoteParticipantId":
            return Optional.ofNullable(clazz.cast(remoteParticipantId()));
        case "errorCode":
            return Optional.ofNullable(clazz.cast(errorCodeAsString()));
        case "destinationStageArn":
            return Optional.ofNullable(clazz.cast(destinationStageArn()));
        case "destinationSessionId":
            return Optional.ofNullable(clazz.cast(destinationSessionId()));
        case "replica":
            return Optional.ofNullable(clazz.cast(replica()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("participantId", PARTICIPANT_ID_FIELD);
        map.put("eventTime", EVENT_TIME_FIELD);
        map.put("remoteParticipantId", REMOTE_PARTICIPANT_ID_FIELD);
        map.put("errorCode", ERROR_CODE_FIELD);
        map.put("destinationStageArn", DESTINATION_STAGE_ARN_FIELD);
        map.put("destinationSessionId", DESTINATION_SESSION_ID_FIELD);
        map.put("replica", REPLICA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Event, T> g) {
        return obj -> g.apply((Event) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Event> {
        /**
         * <p>
         * The name of the event.
         * </p>
         * 
         * @param name
         *        The name of the event.
         * @see EventName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventName
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the event.
         * </p>
         * 
         * @param name
         *        The name of the event.
         * @see EventName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventName
         */
        Builder name(EventName name);

        /**
         * <p>
         * Unique identifier for the participant who triggered the event. This is assigned by IVS.
         * </p>
         * 
         * @param participantId
         *        Unique identifier for the participant who triggered the event. This is assigned by IVS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder participantId(String participantId);

        /**
         * <p>
         * ISO 8601 timestamp (returned as a string) for when the event occurred.
         * </p>
         * 
         * @param eventTime
         *        ISO 8601 timestamp (returned as a string) for when the event occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTime(Instant eventTime);

        /**
         * <p>
         * Unique identifier for the remote participant. For a subscribe event, this is the publisher. For a publish or
         * join event, this is null. This is assigned by IVS.
         * </p>
         * 
         * @param remoteParticipantId
         *        Unique identifier for the remote participant. For a subscribe event, this is the publisher. For a
         *        publish or join event, this is null. This is assigned by IVS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remoteParticipantId(String remoteParticipantId);

        /**
         * <p>
         * If the event is an error event, the error code is provided to give insight into the specific error that
         * occurred. If the event is not an error event, this field is null.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>B_FRAME_PRESENT</code> — The participant's stream includes B-frames. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         * Publishing</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BITRATE_EXCEEDED</code> — The participant exceeded the maximum supported bitrate. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service Quotas</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INSUFFICIENT_CAPABILITIES</code> — The participant tried to take an action that the participant’s token
         * is not allowed to do. For details on participant capabilities, see the <code>capabilities</code> field in
         * <a>CreateParticipantToken</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INTERNAL_SERVER_EXCEPTION</code> — The participant failed to publish to the stage due to an internal
         * server error.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_AUDIO_CODEC</code> — The participant is using an invalid audio codec. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_INPUT</code> — The participant is using an invalid input stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_PROTOCOL</code> — The participant's IngestConfiguration resource is configured for RTMPS but
         * they tried streaming with RTMP. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         * Publishing</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_STREAM_KEY</code> — The participant is using an invalid stream key. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         * Publishing</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_VIDEO_CODEC</code> — The participant is using an invalid video codec. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PUBLISHER_NOT_FOUND</code> — The participant tried to subscribe to a publisher that doesn’t exist.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>QUOTA_EXCEEDED</code> — The number of participants who want to publish/subscribe to a stage exceeds the
         * quota. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service Quotas</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RESOLUTION_EXCEEDED</code> — The participant exceeded the maximum supported resolution. For details,
         * see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
         * Quotas</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REUSE_OF_STREAM_KEY</code> — The participant tried to use a stream key that is associated with another
         * active stage session.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STREAM_DURATION_EXCEEDED</code> — The participant exceeded the maximum allowed stream duration. For
         * details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
         * Quotas</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param errorCode
         *        If the event is an error event, the error code is provided to give insight into the specific error
         *        that occurred. If the event is not an error event, this field is null.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>B_FRAME_PRESENT</code> — The participant's stream includes B-frames. For details, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         *        Publishing</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BITRATE_EXCEEDED</code> — The participant exceeded the maximum supported bitrate. For details,
         *        see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
         *        Quotas</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INSUFFICIENT_CAPABILITIES</code> — The participant tried to take an action that the
         *        participant’s token is not allowed to do. For details on participant capabilities, see the
         *        <code>capabilities</code> field in <a>CreateParticipantToken</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INTERNAL_SERVER_EXCEPTION</code> — The participant failed to publish to the stage due to an
         *        internal server error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_AUDIO_CODEC</code> — The participant is using an invalid audio codec. For details, see
         *        <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream
         *        Ingest</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_INPUT</code> — The participant is using an invalid input stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_PROTOCOL</code> — The participant's IngestConfiguration resource is configured for RTMPS
         *        but they tried streaming with RTMP. For details, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         *        Publishing</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_STREAM_KEY</code> — The participant is using an invalid stream key. For details, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         *        Publishing</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_VIDEO_CODEC</code> — The participant is using an invalid video codec. For details, see
         *        <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream
         *        Ingest</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PUBLISHER_NOT_FOUND</code> — The participant tried to subscribe to a publisher that doesn’t
         *        exist.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>QUOTA_EXCEEDED</code> — The number of participants who want to publish/subscribe to a stage
         *        exceeds the quota. For details, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
         *        Quotas</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RESOLUTION_EXCEEDED</code> — The participant exceeded the maximum supported resolution. For
         *        details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html">
         *        Service Quotas</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REUSE_OF_STREAM_KEY</code> — The participant tried to use a stream key that is associated with
         *        another active stage session.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STREAM_DURATION_EXCEEDED</code> — The participant exceeded the maximum allowed stream duration.
         *        For details, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
         *        Quotas</a>.
         *        </p>
         *        </li>
         * @see EventErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventErrorCode
         */
        Builder errorCode(String errorCode);

        /**
         * <p>
         * If the event is an error event, the error code is provided to give insight into the specific error that
         * occurred. If the event is not an error event, this field is null.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>B_FRAME_PRESENT</code> — The participant's stream includes B-frames. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         * Publishing</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BITRATE_EXCEEDED</code> — The participant exceeded the maximum supported bitrate. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service Quotas</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INSUFFICIENT_CAPABILITIES</code> — The participant tried to take an action that the participant’s token
         * is not allowed to do. For details on participant capabilities, see the <code>capabilities</code> field in
         * <a>CreateParticipantToken</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INTERNAL_SERVER_EXCEPTION</code> — The participant failed to publish to the stage due to an internal
         * server error.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_AUDIO_CODEC</code> — The participant is using an invalid audio codec. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_INPUT</code> — The participant is using an invalid input stream.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_PROTOCOL</code> — The participant's IngestConfiguration resource is configured for RTMPS but
         * they tried streaming with RTMP. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         * Publishing</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_STREAM_KEY</code> — The participant is using an invalid stream key. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         * Publishing</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INVALID_VIDEO_CODEC</code> — The participant is using an invalid video codec. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream Ingest</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PUBLISHER_NOT_FOUND</code> — The participant tried to subscribe to a publisher that doesn’t exist.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>QUOTA_EXCEEDED</code> — The number of participants who want to publish/subscribe to a stage exceeds the
         * quota. For details, see <a
         * href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service Quotas</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RESOLUTION_EXCEEDED</code> — The participant exceeded the maximum supported resolution. For details,
         * see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
         * Quotas</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REUSE_OF_STREAM_KEY</code> — The participant tried to use a stream key that is associated with another
         * active stage session.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STREAM_DURATION_EXCEEDED</code> — The participant exceeded the maximum allowed stream duration. For
         * details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
         * Quotas</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param errorCode
         *        If the event is an error event, the error code is provided to give insight into the specific error
         *        that occurred. If the event is not an error event, this field is null.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>B_FRAME_PRESENT</code> — The participant's stream includes B-frames. For details, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         *        Publishing</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BITRATE_EXCEEDED</code> — The participant exceeded the maximum supported bitrate. For details,
         *        see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
         *        Quotas</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INSUFFICIENT_CAPABILITIES</code> — The participant tried to take an action that the
         *        participant’s token is not allowed to do. For details on participant capabilities, see the
         *        <code>capabilities</code> field in <a>CreateParticipantToken</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INTERNAL_SERVER_EXCEPTION</code> — The participant failed to publish to the stage due to an
         *        internal server error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_AUDIO_CODEC</code> — The participant is using an invalid audio codec. For details, see
         *        <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream
         *        Ingest</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_INPUT</code> — The participant is using an invalid input stream.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_PROTOCOL</code> — The participant's IngestConfiguration resource is configured for RTMPS
         *        but they tried streaming with RTMP. For details, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         *        Publishing</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_STREAM_KEY</code> — The participant is using an invalid stream key. For details, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-rtmp-publishing.html"> IVS RTMP
         *        Publishing</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INVALID_VIDEO_CODEC</code> — The participant is using an invalid video codec. For details, see
         *        <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/rt-stream-ingest.html"> Stream
         *        Ingest</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PUBLISHER_NOT_FOUND</code> — The participant tried to subscribe to a publisher that doesn’t
         *        exist.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>QUOTA_EXCEEDED</code> — The number of participants who want to publish/subscribe to a stage
         *        exceeds the quota. For details, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
         *        Quotas</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RESOLUTION_EXCEEDED</code> — The participant exceeded the maximum supported resolution. For
         *        details, see <a href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html">
         *        Service Quotas</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REUSE_OF_STREAM_KEY</code> — The participant tried to use a stream key that is associated with
         *        another active stage session.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STREAM_DURATION_EXCEEDED</code> — The participant exceeded the maximum allowed stream duration.
         *        For details, see <a
         *        href="https://docs.aws.amazon.com/ivs/latest/RealTimeUserGuide/service-quotas.html"> Service
         *        Quotas</a>.
         *        </p>
         *        </li>
         * @see EventErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventErrorCode
         */
        Builder errorCode(EventErrorCode errorCode);

        /**
         * <p>
         * ARN of the stage where the participant is replicated. Applicable only if the event name is
         * <code>REPLICATION_STARTED</code> or <code>REPLICATION_STOPPED</code>.
         * </p>
         * 
         * @param destinationStageArn
         *        ARN of the stage where the participant is replicated. Applicable only if the event name is
         *        <code>REPLICATION_STARTED</code> or <code>REPLICATION_STOPPED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationStageArn(String destinationStageArn);

        /**
         * <p>
         * ID of the session within the destination stage. Applicable only if the event name is
         * <code>REPLICATION_STARTED</code> or <code>REPLICATION_STOPPED</code>.
         * </p>
         * 
         * @param destinationSessionId
         *        ID of the session within the destination stage. Applicable only if the event name is
         *        <code>REPLICATION_STARTED</code> or <code>REPLICATION_STOPPED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationSessionId(String destinationSessionId);

        /**
         * <p>
         * If true, this indicates the <code>participantId</code> is a replicated participant. If this is a subscribe
         * event, then this flag refers to <code>remoteParticipantId</code>. Default: <code>false</code>.
         * </p>
         * 
         * @param replica
         *        If true, this indicates the <code>participantId</code> is a replicated participant. If this is a
         *        subscribe event, then this flag refers to <code>remoteParticipantId</code>. Default:
         *        <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replica(Boolean replica);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String participantId;

        private Instant eventTime;

        private String remoteParticipantId;

        private String errorCode;

        private String destinationStageArn;

        private String destinationSessionId;

        private Boolean replica;

        private BuilderImpl() {
        }

        private BuilderImpl(Event model) {
            name(model.name);
            participantId(model.participantId);
            eventTime(model.eventTime);
            remoteParticipantId(model.remoteParticipantId);
            errorCode(model.errorCode);
            destinationStageArn(model.destinationStageArn);
            destinationSessionId(model.destinationSessionId);
            replica(model.replica);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(EventName name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        public final String getParticipantId() {
            return participantId;
        }

        public final void setParticipantId(String participantId) {
            this.participantId = participantId;
        }

        @Override
        public final Builder participantId(String participantId) {
            this.participantId = participantId;
            return this;
        }

        public final Instant getEventTime() {
            return eventTime;
        }

        public final void setEventTime(Instant eventTime) {
            this.eventTime = eventTime;
        }

        @Override
        public final Builder eventTime(Instant eventTime) {
            this.eventTime = eventTime;
            return this;
        }

        public final String getRemoteParticipantId() {
            return remoteParticipantId;
        }

        public final void setRemoteParticipantId(String remoteParticipantId) {
            this.remoteParticipantId = remoteParticipantId;
        }

        @Override
        public final Builder remoteParticipantId(String remoteParticipantId) {
            this.remoteParticipantId = remoteParticipantId;
            return this;
        }

        public final String getErrorCode() {
            return errorCode;
        }

        public final void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        @Override
        public final Builder errorCode(String errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        @Override
        public final Builder errorCode(EventErrorCode errorCode) {
            this.errorCode(errorCode == null ? null : errorCode.toString());
            return this;
        }

        public final String getDestinationStageArn() {
            return destinationStageArn;
        }

        public final void setDestinationStageArn(String destinationStageArn) {
            this.destinationStageArn = destinationStageArn;
        }

        @Override
        public final Builder destinationStageArn(String destinationStageArn) {
            this.destinationStageArn = destinationStageArn;
            return this;
        }

        public final String getDestinationSessionId() {
            return destinationSessionId;
        }

        public final void setDestinationSessionId(String destinationSessionId) {
            this.destinationSessionId = destinationSessionId;
        }

        @Override
        public final Builder destinationSessionId(String destinationSessionId) {
            this.destinationSessionId = destinationSessionId;
            return this;
        }

        public final Boolean getReplica() {
            return replica;
        }

        public final void setReplica(Boolean replica) {
            this.replica = replica;
        }

        @Override
        public final Builder replica(Boolean replica) {
            this.replica = replica;
            return this;
        }

        @Override
        public Event build() {
            return new Event(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
