/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kafka.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateConfigurationRequest extends KafkaRequest implements
        ToCopyableBuilder<CreateConfigurationRequest.Builder, CreateConfigurationRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreateConfigurationRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> KAFKA_VERSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("KafkaVersions")
            .getter(getter(CreateConfigurationRequest::kafkaVersions))
            .setter(setter(Builder::kafkaVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("kafkaVersions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CreateConfigurationRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<SdkBytes> SERVER_PROPERTIES_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("ServerProperties").getter(getter(CreateConfigurationRequest::serverProperties))
            .setter(setter(Builder::serverProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serverProperties").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            KAFKA_VERSIONS_FIELD, NAME_FIELD, SERVER_PROPERTIES_FIELD));

    private final String description;

    private final List<String> kafkaVersions;

    private final String name;

    private final SdkBytes serverProperties;

    private CreateConfigurationRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.kafkaVersions = builder.kafkaVersions;
        this.name = builder.name;
        this.serverProperties = builder.serverProperties;
    }

    /**
     *
     * <p>
     * The description of the configuration.
     * </p>
     * 
     * @return <p>
     *         The description of the configuration.
     *         </p>
     */
    public final String description() {
        return description;
    }

    /**
     * Returns true if the KafkaVersions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasKafkaVersions() {
        return kafkaVersions != null && !(kafkaVersions instanceof SdkAutoConstructList);
    }

    /**
     *
     * <p>
     * The versions of Apache Kafka with which you can use this MSK configuration.
     * </p>
     * 
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasKafkaVersions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return <p>
     *         The versions of Apache Kafka with which you can use this MSK configuration.
     *         </p>
     */
    public final List<String> kafkaVersions() {
        return kafkaVersions;
    }

    /**
     *
     * <p>
     * The name of the configuration.
     * </p>
     * 
     * @return <p>
     *         The name of the configuration.
     *         </p>
     */
    public final String name() {
        return name;
    }

    /**
     *
     * <p>
     * Contents of the <filename>server.properties</filename> file. When using the API, you must ensure that the
     * contents of the file are base64 encoded. When using the AWS Management Console, the SDK, or the AWS CLI, the
     * contents of <filename>server.properties</filename> can be in plaintext.
     * </p>
     * 
     * @return <p>
     *         Contents of the <filename>server.properties</filename> file. When using the API, you must ensure that the
     *         contents of the file are base64 encoded. When using the AWS Management Console, the SDK, or the AWS CLI,
     *         the contents of <filename>server.properties</filename> can be in plaintext.
     *         </p>
     */
    public final SdkBytes serverProperties() {
        return serverProperties;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasKafkaVersions() ? kafkaVersions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(serverProperties());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateConfigurationRequest)) {
            return false;
        }
        CreateConfigurationRequest other = (CreateConfigurationRequest) obj;
        return Objects.equals(description(), other.description()) && hasKafkaVersions() == other.hasKafkaVersions()
                && Objects.equals(kafkaVersions(), other.kafkaVersions()) && Objects.equals(name(), other.name())
                && Objects.equals(serverProperties(), other.serverProperties());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateConfigurationRequest").add("Description", description())
                .add("KafkaVersions", hasKafkaVersions() ? kafkaVersions() : null).add("Name", name())
                .add("ServerProperties", serverProperties()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "KafkaVersions":
            return Optional.ofNullable(clazz.cast(kafkaVersions()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ServerProperties":
            return Optional.ofNullable(clazz.cast(serverProperties()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateConfigurationRequest, T> g) {
        return obj -> g.apply((CreateConfigurationRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KafkaRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateConfigurationRequest> {
        /**
         *
         * <p>
         * The description of the configuration.
         * </p>
         * 
         * @param description
         *        <p>
         *        The description of the configuration.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         *
         * <p>
         * The versions of Apache Kafka with which you can use this MSK configuration.
         * </p>
         * 
         * @param kafkaVersions
         *        <p>
         *        The versions of Apache Kafka with which you can use this MSK configuration.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kafkaVersions(Collection<String> kafkaVersions);

        /**
         *
         * <p>
         * The versions of Apache Kafka with which you can use this MSK configuration.
         * </p>
         * 
         * @param kafkaVersions
         *        <p>
         *        The versions of Apache Kafka with which you can use this MSK configuration.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kafkaVersions(String... kafkaVersions);

        /**
         *
         * <p>
         * The name of the configuration.
         * </p>
         * 
         * @param name
         *        <p>
         *        The name of the configuration.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         *
         * <p>
         * Contents of the <filename>server.properties</filename> file. When using the API, you must ensure that the
         * contents of the file are base64 encoded. When using the AWS Management Console, the SDK, or the AWS CLI, the
         * contents of <filename>server.properties</filename> can be in plaintext.
         * </p>
         * 
         * @param serverProperties
         *        <p>
         *        Contents of the <filename>server.properties</filename> file. When using the API, you must ensure that
         *        the contents of the file are base64 encoded. When using the AWS Management Console, the SDK, or the
         *        AWS CLI, the contents of <filename>server.properties</filename> can be in plaintext.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverProperties(SdkBytes serverProperties);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KafkaRequest.BuilderImpl implements Builder {
        private String description;

        private List<String> kafkaVersions = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private SdkBytes serverProperties;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateConfigurationRequest model) {
            super(model);
            description(model.description);
            kafkaVersions(model.kafkaVersions);
            name(model.name);
            serverProperties(model.serverProperties);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<String> getKafkaVersions() {
            if (kafkaVersions instanceof SdkAutoConstructList) {
                return null;
            }
            return kafkaVersions;
        }

        @Override
        public final Builder kafkaVersions(Collection<String> kafkaVersions) {
            this.kafkaVersions = ___listOf__stringCopier.copy(kafkaVersions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder kafkaVersions(String... kafkaVersions) {
            kafkaVersions(Arrays.asList(kafkaVersions));
            return this;
        }

        public final void setKafkaVersions(Collection<String> kafkaVersions) {
            this.kafkaVersions = ___listOf__stringCopier.copy(kafkaVersions);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final ByteBuffer getServerProperties() {
            return serverProperties == null ? null : serverProperties.asByteBuffer();
        }

        @Override
        public final Builder serverProperties(SdkBytes serverProperties) {
            this.serverProperties = serverProperties;
            return this;
        }

        public final void setServerProperties(ByteBuffer serverProperties) {
            serverProperties(serverProperties == null ? null : SdkBytes.fromByteBuffer(serverProperties));
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateConfigurationRequest build() {
            return new CreateConfigurationRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
