/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kafkaconnect;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kafkaconnect.internal.KafkaConnectServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.kafkaconnect.model.BadRequestException;
import software.amazon.awssdk.services.kafkaconnect.model.ConflictException;
import software.amazon.awssdk.services.kafkaconnect.model.CreateConnectorRequest;
import software.amazon.awssdk.services.kafkaconnect.model.CreateConnectorResponse;
import software.amazon.awssdk.services.kafkaconnect.model.CreateCustomPluginRequest;
import software.amazon.awssdk.services.kafkaconnect.model.CreateCustomPluginResponse;
import software.amazon.awssdk.services.kafkaconnect.model.CreateWorkerConfigurationRequest;
import software.amazon.awssdk.services.kafkaconnect.model.CreateWorkerConfigurationResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteConnectorRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteConnectorResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteCustomPluginRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteCustomPluginResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteWorkerConfigurationRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteWorkerConfigurationResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeConnectorRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeConnectorResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeCustomPluginRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeCustomPluginResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeWorkerConfigurationRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeWorkerConfigurationResponse;
import software.amazon.awssdk.services.kafkaconnect.model.ForbiddenException;
import software.amazon.awssdk.services.kafkaconnect.model.InternalServerErrorException;
import software.amazon.awssdk.services.kafkaconnect.model.KafkaConnectException;
import software.amazon.awssdk.services.kafkaconnect.model.ListConnectorsRequest;
import software.amazon.awssdk.services.kafkaconnect.model.ListConnectorsResponse;
import software.amazon.awssdk.services.kafkaconnect.model.ListCustomPluginsRequest;
import software.amazon.awssdk.services.kafkaconnect.model.ListCustomPluginsResponse;
import software.amazon.awssdk.services.kafkaconnect.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.kafkaconnect.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.kafkaconnect.model.ListWorkerConfigurationsRequest;
import software.amazon.awssdk.services.kafkaconnect.model.ListWorkerConfigurationsResponse;
import software.amazon.awssdk.services.kafkaconnect.model.NotFoundException;
import software.amazon.awssdk.services.kafkaconnect.model.ServiceUnavailableException;
import software.amazon.awssdk.services.kafkaconnect.model.TagResourceRequest;
import software.amazon.awssdk.services.kafkaconnect.model.TagResourceResponse;
import software.amazon.awssdk.services.kafkaconnect.model.TooManyRequestsException;
import software.amazon.awssdk.services.kafkaconnect.model.UnauthorizedException;
import software.amazon.awssdk.services.kafkaconnect.model.UntagResourceRequest;
import software.amazon.awssdk.services.kafkaconnect.model.UntagResourceResponse;
import software.amazon.awssdk.services.kafkaconnect.model.UpdateConnectorRequest;
import software.amazon.awssdk.services.kafkaconnect.model.UpdateConnectorResponse;
import software.amazon.awssdk.services.kafkaconnect.transform.CreateConnectorRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.CreateCustomPluginRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.CreateWorkerConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DeleteConnectorRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DeleteCustomPluginRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DeleteWorkerConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DescribeConnectorRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DescribeCustomPluginRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DescribeWorkerConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.ListConnectorsRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.ListCustomPluginsRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.ListWorkerConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.UpdateConnectorRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link KafkaConnectAsyncClient}.
 *
 * @see KafkaConnectAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKafkaConnectAsyncClient implements KafkaConnectAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultKafkaConnectAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKafkaConnectAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a connector using the specified properties.
     * </p>
     *
     * @param createConnectorRequest
     * @return A Java Future containing the result of the CreateConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>ConflictException HTTP Status Code 409: Conflict. A resource with this name already exists. Retry
     *         your request with another name.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.CreateConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/CreateConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConnectorResponse> createConnector(CreateConnectorRequest createConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConnectorRequest, CreateConnectorResponse>()
                            .withOperationName("CreateConnector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConnectorRequest));
            CompletableFuture<CreateConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a custom plugin using the specified properties.
     * </p>
     *
     * @param createCustomPluginRequest
     * @return A Java Future containing the result of the CreateCustomPlugin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>ConflictException HTTP Status Code 409: Conflict. A resource with this name already exists. Retry
     *         your request with another name.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.CreateCustomPlugin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/CreateCustomPlugin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCustomPluginResponse> createCustomPlugin(CreateCustomPluginRequest createCustomPluginRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCustomPluginRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCustomPluginRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCustomPlugin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCustomPluginResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCustomPluginResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCustomPluginResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCustomPluginRequest, CreateCustomPluginResponse>()
                            .withOperationName("CreateCustomPlugin").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateCustomPluginRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createCustomPluginRequest));
            CompletableFuture<CreateCustomPluginResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a worker configuration using the specified properties.
     * </p>
     *
     * @param createWorkerConfigurationRequest
     * @return A Java Future containing the result of the CreateWorkerConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>ConflictException HTTP Status Code 409: Conflict. A resource with this name already exists. Retry
     *         your request with another name.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.CreateWorkerConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/CreateWorkerConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateWorkerConfigurationResponse> createWorkerConfiguration(
            CreateWorkerConfigurationRequest createWorkerConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkerConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkerConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkerConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateWorkerConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateWorkerConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateWorkerConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateWorkerConfigurationRequest, CreateWorkerConfigurationResponse>()
                            .withOperationName("CreateWorkerConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateWorkerConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createWorkerConfigurationRequest));
            CompletableFuture<CreateWorkerConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified connector.
     * </p>
     *
     * @param deleteConnectorRequest
     * @return A Java Future containing the result of the DeleteConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.DeleteConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DeleteConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConnectorResponse> deleteConnector(DeleteConnectorRequest deleteConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConnectorRequest, DeleteConnectorResponse>()
                            .withOperationName("DeleteConnector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteConnectorRequest));
            CompletableFuture<DeleteConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a custom plugin.
     * </p>
     *
     * @param deleteCustomPluginRequest
     * @return A Java Future containing the result of the DeleteCustomPlugin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.DeleteCustomPlugin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DeleteCustomPlugin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCustomPluginResponse> deleteCustomPlugin(DeleteCustomPluginRequest deleteCustomPluginRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCustomPluginRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomPluginRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomPlugin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCustomPluginResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCustomPluginResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCustomPluginResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCustomPluginRequest, DeleteCustomPluginResponse>()
                            .withOperationName("DeleteCustomPlugin").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCustomPluginRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCustomPluginRequest));
            CompletableFuture<DeleteCustomPluginResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified worker configuration.
     * </p>
     *
     * @param deleteWorkerConfigurationRequest
     * @return A Java Future containing the result of the DeleteWorkerConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.DeleteWorkerConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DeleteWorkerConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteWorkerConfigurationResponse> deleteWorkerConfiguration(
            DeleteWorkerConfigurationRequest deleteWorkerConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkerConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkerConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkerConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteWorkerConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteWorkerConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteWorkerConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkerConfigurationRequest, DeleteWorkerConfigurationResponse>()
                            .withOperationName("DeleteWorkerConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteWorkerConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteWorkerConfigurationRequest));
            CompletableFuture<DeleteWorkerConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns summary information about the connector.
     * </p>
     *
     * @param describeConnectorRequest
     * @return A Java Future containing the result of the DescribeConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.DescribeConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DescribeConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeConnectorResponse> describeConnector(DescribeConnectorRequest describeConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeConnectorRequest, DescribeConnectorResponse>()
                            .withOperationName("DescribeConnector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeConnectorRequest));
            CompletableFuture<DescribeConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A summary description of the custom plugin.
     * </p>
     *
     * @param describeCustomPluginRequest
     * @return A Java Future containing the result of the DescribeCustomPlugin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.DescribeCustomPlugin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DescribeCustomPlugin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCustomPluginResponse> describeCustomPlugin(
            DescribeCustomPluginRequest describeCustomPluginRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeCustomPluginRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCustomPluginRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCustomPlugin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCustomPluginResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeCustomPluginResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCustomPluginResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCustomPluginRequest, DescribeCustomPluginResponse>()
                            .withOperationName("DescribeCustomPlugin").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeCustomPluginRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeCustomPluginRequest));
            CompletableFuture<DescribeCustomPluginResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a worker configuration.
     * </p>
     *
     * @param describeWorkerConfigurationRequest
     * @return A Java Future containing the result of the DescribeWorkerConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.DescribeWorkerConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DescribeWorkerConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeWorkerConfigurationResponse> describeWorkerConfiguration(
            DescribeWorkerConfigurationRequest describeWorkerConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWorkerConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWorkerConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkerConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeWorkerConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeWorkerConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeWorkerConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkerConfigurationRequest, DescribeWorkerConfigurationResponse>()
                            .withOperationName("DescribeWorkerConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeWorkerConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeWorkerConfigurationRequest));
            CompletableFuture<DescribeWorkerConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all the connectors in this account and Region. The list is limited to connectors whose name
     * starts with the specified prefix. The response also includes a description of each of the listed connectors.
     * </p>
     *
     * @param listConnectorsRequest
     * @return A Java Future containing the result of the ListConnectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/ListConnectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConnectorsResponse> listConnectors(ListConnectorsRequest listConnectorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConnectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConnectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConnectorsRequest, ListConnectorsResponse>()
                            .withOperationName("ListConnectors").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListConnectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConnectorsRequest));
            CompletableFuture<ListConnectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all of the custom plugins in this account and Region.
     * </p>
     *
     * @param listCustomPluginsRequest
     * @return A Java Future containing the result of the ListCustomPlugins operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.ListCustomPlugins
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/ListCustomPlugins"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCustomPluginsResponse> listCustomPlugins(ListCustomPluginsRequest listCustomPluginsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCustomPluginsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCustomPluginsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCustomPlugins");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCustomPluginsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCustomPluginsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCustomPluginsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCustomPluginsRequest, ListCustomPluginsResponse>()
                            .withOperationName("ListCustomPlugins").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCustomPluginsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCustomPluginsRequest));
            CompletableFuture<ListCustomPluginsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the tags attached to the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all of the worker configurations in this account and Region.
     * </p>
     *
     * @param listWorkerConfigurationsRequest
     * @return A Java Future containing the result of the ListWorkerConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.ListWorkerConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/ListWorkerConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListWorkerConfigurationsResponse> listWorkerConfigurations(
            ListWorkerConfigurationsRequest listWorkerConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkerConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkerConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkerConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListWorkerConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListWorkerConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListWorkerConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListWorkerConfigurationsRequest, ListWorkerConfigurationsResponse>()
                            .withOperationName("ListWorkerConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListWorkerConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listWorkerConfigurationsRequest));
            CompletableFuture<ListWorkerConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>ConflictException HTTP Status Code 409: Conflict. A resource with this name already exists. Retry
     *         your request with another name.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified connector.
     * </p>
     *
     * @param updateConnectorRequest
     * @return A Java Future containing the result of the UpdateConnector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>ServiceUnavailableException HTTP Status Code 503: Service Unavailable. Retrying your request in some
     *         time might resolve the issue.</li>
     *         <li>TooManyRequestsException HTTP Status Code 429: Limit exceeded. Resource limit reached.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>KafkaConnectException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample KafkaConnectAsyncClient.UpdateConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/UpdateConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConnectorResponse> updateConnector(UpdateConnectorRequest updateConnectorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConnector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConnectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConnectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConnectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConnectorRequest, UpdateConnectorResponse>()
                            .withOperationName("UpdateConnector").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateConnectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateConnectorRequest));
            CompletableFuture<UpdateConnectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final KafkaConnectServiceClientConfiguration serviceClientConfiguration() {
        return new KafkaConnectServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KafkaConnectException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        KafkaConnectServiceClientConfigurationBuilder serviceConfigBuilder = new KafkaConnectServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
