/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kafkaconnect;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.kafkaconnect.internal.KafkaConnectServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.kafkaconnect.model.BadRequestException;
import software.amazon.awssdk.services.kafkaconnect.model.ConflictException;
import software.amazon.awssdk.services.kafkaconnect.model.CreateConnectorRequest;
import software.amazon.awssdk.services.kafkaconnect.model.CreateConnectorResponse;
import software.amazon.awssdk.services.kafkaconnect.model.CreateCustomPluginRequest;
import software.amazon.awssdk.services.kafkaconnect.model.CreateCustomPluginResponse;
import software.amazon.awssdk.services.kafkaconnect.model.CreateWorkerConfigurationRequest;
import software.amazon.awssdk.services.kafkaconnect.model.CreateWorkerConfigurationResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteConnectorRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteConnectorResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteCustomPluginRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteCustomPluginResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteWorkerConfigurationRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DeleteWorkerConfigurationResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeConnectorOperationRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeConnectorOperationResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeConnectorRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeConnectorResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeCustomPluginRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeCustomPluginResponse;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeWorkerConfigurationRequest;
import software.amazon.awssdk.services.kafkaconnect.model.DescribeWorkerConfigurationResponse;
import software.amazon.awssdk.services.kafkaconnect.model.ForbiddenException;
import software.amazon.awssdk.services.kafkaconnect.model.InternalServerErrorException;
import software.amazon.awssdk.services.kafkaconnect.model.KafkaConnectException;
import software.amazon.awssdk.services.kafkaconnect.model.ListConnectorOperationsRequest;
import software.amazon.awssdk.services.kafkaconnect.model.ListConnectorOperationsResponse;
import software.amazon.awssdk.services.kafkaconnect.model.ListConnectorsRequest;
import software.amazon.awssdk.services.kafkaconnect.model.ListConnectorsResponse;
import software.amazon.awssdk.services.kafkaconnect.model.ListCustomPluginsRequest;
import software.amazon.awssdk.services.kafkaconnect.model.ListCustomPluginsResponse;
import software.amazon.awssdk.services.kafkaconnect.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.kafkaconnect.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.kafkaconnect.model.ListWorkerConfigurationsRequest;
import software.amazon.awssdk.services.kafkaconnect.model.ListWorkerConfigurationsResponse;
import software.amazon.awssdk.services.kafkaconnect.model.NotFoundException;
import software.amazon.awssdk.services.kafkaconnect.model.ServiceUnavailableException;
import software.amazon.awssdk.services.kafkaconnect.model.TagResourceRequest;
import software.amazon.awssdk.services.kafkaconnect.model.TagResourceResponse;
import software.amazon.awssdk.services.kafkaconnect.model.TooManyRequestsException;
import software.amazon.awssdk.services.kafkaconnect.model.UnauthorizedException;
import software.amazon.awssdk.services.kafkaconnect.model.UntagResourceRequest;
import software.amazon.awssdk.services.kafkaconnect.model.UntagResourceResponse;
import software.amazon.awssdk.services.kafkaconnect.model.UpdateConnectorRequest;
import software.amazon.awssdk.services.kafkaconnect.model.UpdateConnectorResponse;
import software.amazon.awssdk.services.kafkaconnect.transform.CreateConnectorRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.CreateCustomPluginRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.CreateWorkerConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DeleteConnectorRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DeleteCustomPluginRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DeleteWorkerConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DescribeConnectorOperationRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DescribeConnectorRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DescribeCustomPluginRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.DescribeWorkerConfigurationRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.ListConnectorOperationsRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.ListConnectorsRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.ListCustomPluginsRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.ListWorkerConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.kafkaconnect.transform.UpdateConnectorRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link KafkaConnectClient}.
 *
 * @see KafkaConnectClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKafkaConnectClient implements KafkaConnectClient {
    private static final Logger log = Logger.loggerFor(DefaultKafkaConnectClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKafkaConnectClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a connector using the specified properties.
     * </p>
     *
     * @param createConnectorRequest
     * @return Result of the CreateConnector operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws ConflictException
     *         HTTP Status Code 409: Conflict. A resource with this name already exists. Retry your request with another
     *         name.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.CreateConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/CreateConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateConnectorResponse createConnector(CreateConnectorRequest createConnectorRequest) throws NotFoundException,
            ConflictException, BadRequestException, ForbiddenException, ServiceUnavailableException, TooManyRequestsException,
            UnauthorizedException, InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateConnectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConnector");

            return clientHandler.execute(new ClientExecutionParams<CreateConnectorRequest, CreateConnectorResponse>()
                    .withOperationName("CreateConnector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createConnectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateConnectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a custom plugin using the specified properties.
     * </p>
     *
     * @param createCustomPluginRequest
     * @return Result of the CreateCustomPlugin operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws ConflictException
     *         HTTP Status Code 409: Conflict. A resource with this name already exists. Retry your request with another
     *         name.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.CreateCustomPlugin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/CreateCustomPlugin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCustomPluginResponse createCustomPlugin(CreateCustomPluginRequest createCustomPluginRequest)
            throws NotFoundException, ConflictException, BadRequestException, ForbiddenException, ServiceUnavailableException,
            TooManyRequestsException, UnauthorizedException, InternalServerErrorException, AwsServiceException,
            SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCustomPluginResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCustomPluginResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCustomPluginRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCustomPluginRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCustomPlugin");

            return clientHandler.execute(new ClientExecutionParams<CreateCustomPluginRequest, CreateCustomPluginResponse>()
                    .withOperationName("CreateCustomPlugin").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createCustomPluginRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateCustomPluginRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a worker configuration using the specified properties.
     * </p>
     *
     * @param createWorkerConfigurationRequest
     * @return Result of the CreateWorkerConfiguration operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws ConflictException
     *         HTTP Status Code 409: Conflict. A resource with this name already exists. Retry your request with another
     *         name.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.CreateWorkerConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/CreateWorkerConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateWorkerConfigurationResponse createWorkerConfiguration(
            CreateWorkerConfigurationRequest createWorkerConfigurationRequest) throws NotFoundException, ConflictException,
            BadRequestException, ForbiddenException, ServiceUnavailableException, TooManyRequestsException,
            UnauthorizedException, InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateWorkerConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateWorkerConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createWorkerConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createWorkerConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateWorkerConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateWorkerConfigurationRequest, CreateWorkerConfigurationResponse>()
                            .withOperationName("CreateWorkerConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createWorkerConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateWorkerConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified connector.
     * </p>
     *
     * @param deleteConnectorRequest
     * @return Result of the DeleteConnector operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.DeleteConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DeleteConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteConnectorResponse deleteConnector(DeleteConnectorRequest deleteConnectorRequest) throws NotFoundException,
            BadRequestException, ForbiddenException, ServiceUnavailableException, TooManyRequestsException,
            UnauthorizedException, InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteConnectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConnector");

            return clientHandler.execute(new ClientExecutionParams<DeleteConnectorRequest, DeleteConnectorResponse>()
                    .withOperationName("DeleteConnector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteConnectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteConnectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a custom plugin.
     * </p>
     *
     * @param deleteCustomPluginRequest
     * @return Result of the DeleteCustomPlugin operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.DeleteCustomPlugin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DeleteCustomPlugin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteCustomPluginResponse deleteCustomPlugin(DeleteCustomPluginRequest deleteCustomPluginRequest)
            throws NotFoundException, BadRequestException, ForbiddenException, ServiceUnavailableException,
            TooManyRequestsException, UnauthorizedException, InternalServerErrorException, AwsServiceException,
            SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCustomPluginResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteCustomPluginResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCustomPluginRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomPluginRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomPlugin");

            return clientHandler.execute(new ClientExecutionParams<DeleteCustomPluginRequest, DeleteCustomPluginResponse>()
                    .withOperationName("DeleteCustomPlugin").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteCustomPluginRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteCustomPluginRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified worker configuration.
     * </p>
     *
     * @param deleteWorkerConfigurationRequest
     * @return Result of the DeleteWorkerConfiguration operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.DeleteWorkerConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DeleteWorkerConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteWorkerConfigurationResponse deleteWorkerConfiguration(
            DeleteWorkerConfigurationRequest deleteWorkerConfigurationRequest) throws NotFoundException, BadRequestException,
            ForbiddenException, ServiceUnavailableException, TooManyRequestsException, UnauthorizedException,
            InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteWorkerConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteWorkerConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteWorkerConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteWorkerConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteWorkerConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteWorkerConfigurationRequest, DeleteWorkerConfigurationResponse>()
                            .withOperationName("DeleteWorkerConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteWorkerConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteWorkerConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns summary information about the connector.
     * </p>
     *
     * @param describeConnectorRequest
     * @return Result of the DescribeConnector operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.DescribeConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DescribeConnector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeConnectorResponse describeConnector(DescribeConnectorRequest describeConnectorRequest)
            throws NotFoundException, BadRequestException, ForbiddenException, ServiceUnavailableException,
            TooManyRequestsException, UnauthorizedException, InternalServerErrorException, AwsServiceException,
            SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeConnectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnector");

            return clientHandler.execute(new ClientExecutionParams<DescribeConnectorRequest, DescribeConnectorResponse>()
                    .withOperationName("DescribeConnector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeConnectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeConnectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the specified connector's operations.
     * </p>
     *
     * @param describeConnectorOperationRequest
     * @return Result of the DescribeConnectorOperation operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.DescribeConnectorOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DescribeConnectorOperation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeConnectorOperationResponse describeConnectorOperation(
            DescribeConnectorOperationRequest describeConnectorOperationRequest) throws NotFoundException, BadRequestException,
            ForbiddenException, ServiceUnavailableException, TooManyRequestsException, UnauthorizedException,
            InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeConnectorOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeConnectorOperationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeConnectorOperationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConnectorOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConnectorOperation");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeConnectorOperationRequest, DescribeConnectorOperationResponse>()
                            .withOperationName("DescribeConnectorOperation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeConnectorOperationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeConnectorOperationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * A summary description of the custom plugin.
     * </p>
     *
     * @param describeCustomPluginRequest
     * @return Result of the DescribeCustomPlugin operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.DescribeCustomPlugin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DescribeCustomPlugin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeCustomPluginResponse describeCustomPlugin(DescribeCustomPluginRequest describeCustomPluginRequest)
            throws NotFoundException, BadRequestException, ForbiddenException, ServiceUnavailableException,
            TooManyRequestsException, UnauthorizedException, InternalServerErrorException, AwsServiceException,
            SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCustomPluginResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeCustomPluginResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeCustomPluginRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCustomPluginRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCustomPlugin");

            return clientHandler.execute(new ClientExecutionParams<DescribeCustomPluginRequest, DescribeCustomPluginResponse>()
                    .withOperationName("DescribeCustomPlugin").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeCustomPluginRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeCustomPluginRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about a worker configuration.
     * </p>
     *
     * @param describeWorkerConfigurationRequest
     * @return Result of the DescribeWorkerConfiguration operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.DescribeWorkerConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/DescribeWorkerConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeWorkerConfigurationResponse describeWorkerConfiguration(
            DescribeWorkerConfigurationRequest describeWorkerConfigurationRequest) throws NotFoundException, BadRequestException,
            ForbiddenException, ServiceUnavailableException, TooManyRequestsException, UnauthorizedException,
            InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeWorkerConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeWorkerConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeWorkerConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeWorkerConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeWorkerConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeWorkerConfigurationRequest, DescribeWorkerConfigurationResponse>()
                            .withOperationName("DescribeWorkerConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeWorkerConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeWorkerConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists information about a connector's operation(s).
     * </p>
     *
     * @param listConnectorOperationsRequest
     * @return Result of the ListConnectorOperations operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.ListConnectorOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/ListConnectorOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListConnectorOperationsResponse listConnectorOperations(ListConnectorOperationsRequest listConnectorOperationsRequest)
            throws NotFoundException, BadRequestException, ForbiddenException, ServiceUnavailableException,
            TooManyRequestsException, UnauthorizedException, InternalServerErrorException, AwsServiceException,
            SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConnectorOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListConnectorOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectorOperationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectorOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnectorOperations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListConnectorOperationsRequest, ListConnectorOperationsResponse>()
                            .withOperationName("ListConnectorOperations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listConnectorOperationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListConnectorOperationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all the connectors in this account and Region. The list is limited to connectors whose name
     * starts with the specified prefix. The response also includes a description of each of the listed connectors.
     * </p>
     *
     * @param listConnectorsRequest
     * @return Result of the ListConnectors operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.ListConnectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/ListConnectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListConnectorsResponse listConnectors(ListConnectorsRequest listConnectorsRequest) throws NotFoundException,
            BadRequestException, ForbiddenException, ServiceUnavailableException, TooManyRequestsException,
            UnauthorizedException, InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConnectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListConnectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConnectorsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConnectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConnectors");

            return clientHandler.execute(new ClientExecutionParams<ListConnectorsRequest, ListConnectorsResponse>()
                    .withOperationName("ListConnectors").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listConnectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListConnectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all of the custom plugins in this account and Region.
     * </p>
     *
     * @param listCustomPluginsRequest
     * @return Result of the ListCustomPlugins operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.ListCustomPlugins
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/ListCustomPlugins"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCustomPluginsResponse listCustomPlugins(ListCustomPluginsRequest listCustomPluginsRequest)
            throws NotFoundException, BadRequestException, ForbiddenException, ServiceUnavailableException,
            TooManyRequestsException, UnauthorizedException, InternalServerErrorException, AwsServiceException,
            SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCustomPluginsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCustomPluginsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCustomPluginsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCustomPluginsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCustomPlugins");

            return clientHandler.execute(new ClientExecutionParams<ListCustomPluginsRequest, ListCustomPluginsResponse>()
                    .withOperationName("ListCustomPlugins").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listCustomPluginsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCustomPluginsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the tags attached to the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws NotFoundException, BadRequestException, ForbiddenException, ServiceUnavailableException,
            TooManyRequestsException, UnauthorizedException, InternalServerErrorException, AwsServiceException,
            SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all of the worker configurations in this account and Region.
     * </p>
     *
     * @param listWorkerConfigurationsRequest
     * @return Result of the ListWorkerConfigurations operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.ListWorkerConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/ListWorkerConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListWorkerConfigurationsResponse listWorkerConfigurations(
            ListWorkerConfigurationsRequest listWorkerConfigurationsRequest) throws NotFoundException, BadRequestException,
            ForbiddenException, ServiceUnavailableException, TooManyRequestsException, UnauthorizedException,
            InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListWorkerConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListWorkerConfigurationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listWorkerConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listWorkerConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListWorkerConfigurations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListWorkerConfigurationsRequest, ListWorkerConfigurationsResponse>()
                            .withOperationName("ListWorkerConfigurations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listWorkerConfigurationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListWorkerConfigurationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attaches tags to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws ConflictException
     *         HTTP Status Code 409: Conflict. A resource with this name already exists. Retry your request with another
     *         name.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws NotFoundException, ConflictException,
            BadRequestException, ForbiddenException, ServiceUnavailableException, TooManyRequestsException,
            UnauthorizedException, InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws NotFoundException,
            BadRequestException, ForbiddenException, ServiceUnavailableException, TooManyRequestsException,
            UnauthorizedException, InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified connector.
     * </p>
     *
     * @param updateConnectorRequest
     * @return Result of the UpdateConnector operation returned by the service.
     * @throws NotFoundException
     *         HTTP Status Code 404: Resource not found due to incorrect input. Correct your request and then retry it.
     * @throws BadRequestException
     *         HTTP Status Code 400: Bad request due to incorrect input. Correct your request and then retry it.
     * @throws ForbiddenException
     *         HTTP Status Code 403: Access forbidden. Correct your credentials and then retry your request.
     * @throws ServiceUnavailableException
     *         HTTP Status Code 503: Service Unavailable. Retrying your request in some time might resolve the issue.
     * @throws TooManyRequestsException
     *         HTTP Status Code 429: Limit exceeded. Resource limit reached.
     * @throws UnauthorizedException
     *         HTTP Status Code 401: Unauthorized request. The provided credentials couldn't be validated.
     * @throws InternalServerErrorException
     *         HTTP Status Code 500: Unexpected internal server error. Retrying your request might resolve the issue.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KafkaConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KafkaConnectClient.UpdateConnector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/kafkaconnect-2021-09-14/UpdateConnector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateConnectorResponse updateConnector(UpdateConnectorRequest updateConnectorRequest) throws NotFoundException,
            BadRequestException, ForbiddenException, ServiceUnavailableException, TooManyRequestsException,
            UnauthorizedException, InternalServerErrorException, AwsServiceException, SdkClientException, KafkaConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateConnectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateConnectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConnectorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConnectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "KafkaConnect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConnector");

            return clientHandler.execute(new ClientExecutionParams<UpdateConnectorRequest, UpdateConnectorResponse>()
                    .withOperationName("UpdateConnector").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateConnectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateConnectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        KafkaConnectServiceClientConfigurationBuilder serviceConfigBuilder = new KafkaConnectServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KafkaConnectException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public final KafkaConnectServiceClientConfiguration serviceClientConfiguration() {
        return new KafkaConnectServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
