/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides configuration information for a data source to index documents in an Amazon S3 bucket.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3DataSourceConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3DataSourceConfiguration.Builder, S3DataSourceConfiguration> {
    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3DataSourceConfiguration::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketName").build()).build();

    private static final SdkField<List<String>> INCLUSION_PREFIXES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(S3DataSourceConfiguration::inclusionPrefixes))
            .setter(setter(Builder::inclusionPrefixes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InclusionPrefixes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(S3DataSourceConfiguration::exclusionPatterns))
            .setter(setter(Builder::exclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DocumentsMetadataConfiguration> DOCUMENTS_METADATA_CONFIGURATION_FIELD = SdkField
            .<DocumentsMetadataConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(S3DataSourceConfiguration::documentsMetadataConfiguration))
            .setter(setter(Builder::documentsMetadataConfiguration))
            .constructor(DocumentsMetadataConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentsMetadataConfiguration")
                    .build()).build();

    private static final SdkField<AccessControlListConfiguration> ACCESS_CONTROL_LIST_CONFIGURATION_FIELD = SdkField
            .<AccessControlListConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(S3DataSourceConfiguration::accessControlListConfiguration))
            .setter(setter(Builder::accessControlListConfiguration))
            .constructor(AccessControlListConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlListConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_NAME_FIELD,
            INCLUSION_PREFIXES_FIELD, EXCLUSION_PATTERNS_FIELD, DOCUMENTS_METADATA_CONFIGURATION_FIELD,
            ACCESS_CONTROL_LIST_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucketName;

    private final List<String> inclusionPrefixes;

    private final List<String> exclusionPatterns;

    private final DocumentsMetadataConfiguration documentsMetadataConfiguration;

    private final AccessControlListConfiguration accessControlListConfiguration;

    private S3DataSourceConfiguration(BuilderImpl builder) {
        this.bucketName = builder.bucketName;
        this.inclusionPrefixes = builder.inclusionPrefixes;
        this.exclusionPatterns = builder.exclusionPatterns;
        this.documentsMetadataConfiguration = builder.documentsMetadataConfiguration;
        this.accessControlListConfiguration = builder.accessControlListConfiguration;
    }

    /**
     * <p>
     * The name of the bucket that contains the documents.
     * </p>
     * 
     * @return The name of the bucket that contains the documents.
     */
    public String bucketName() {
        return bucketName;
    }

    /**
     * Returns true if the InclusionPrefixes property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasInclusionPrefixes() {
        return inclusionPrefixes != null && !(inclusionPrefixes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of S3 prefixes for the documents that should be included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInclusionPrefixes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of S3 prefixes for the documents that should be included in the index.
     */
    public List<String> inclusionPrefixes() {
        return inclusionPrefixes;
    }

    /**
     * Returns true if the ExclusionPatterns property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasExclusionPatterns() {
        return exclusionPatterns != null && !(exclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of glob patterns for documents that should not be indexed. If a document that matches an inclusion prefix
     * also matches an exclusion pattern, the document is not indexed.
     * </p>
     * <p>
     * For more information about glob patterns, see <a href="http://wikipedia.org/wiki/Glob_%28programming%29">glob
     * (programming)</a> in <i>Wikipedia</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasExclusionPatterns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of glob patterns for documents that should not be indexed. If a document that matches an inclusion
     *         prefix also matches an exclusion pattern, the document is not indexed.</p>
     *         <p>
     *         For more information about glob patterns, see <a
     *         href="http://wikipedia.org/wiki/Glob_%28programming%29">glob (programming)</a> in <i>Wikipedia</i>.
     */
    public List<String> exclusionPatterns() {
        return exclusionPatterns;
    }

    /**
     * Returns the value of the DocumentsMetadataConfiguration property for this object.
     * 
     * @return The value of the DocumentsMetadataConfiguration property for this object.
     */
    public DocumentsMetadataConfiguration documentsMetadataConfiguration() {
        return documentsMetadataConfiguration;
    }

    /**
     * <p>
     * Provides the path to the S3 bucket that contains the user context filtering files for the data source.
     * </p>
     * 
     * @return Provides the path to the S3 bucket that contains the user context filtering files for the data source.
     */
    public AccessControlListConfiguration accessControlListConfiguration() {
        return accessControlListConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(inclusionPrefixes());
        hashCode = 31 * hashCode + Objects.hashCode(exclusionPatterns());
        hashCode = 31 * hashCode + Objects.hashCode(documentsMetadataConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(accessControlListConfiguration());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3DataSourceConfiguration)) {
            return false;
        }
        S3DataSourceConfiguration other = (S3DataSourceConfiguration) obj;
        return Objects.equals(bucketName(), other.bucketName()) && Objects.equals(inclusionPrefixes(), other.inclusionPrefixes())
                && Objects.equals(exclusionPatterns(), other.exclusionPatterns())
                && Objects.equals(documentsMetadataConfiguration(), other.documentsMetadataConfiguration())
                && Objects.equals(accessControlListConfiguration(), other.accessControlListConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("S3DataSourceConfiguration").add("BucketName", bucketName())
                .add("InclusionPrefixes", inclusionPrefixes()).add("ExclusionPatterns", exclusionPatterns())
                .add("DocumentsMetadataConfiguration", documentsMetadataConfiguration())
                .add("AccessControlListConfiguration", accessControlListConfiguration()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "InclusionPrefixes":
            return Optional.ofNullable(clazz.cast(inclusionPrefixes()));
        case "ExclusionPatterns":
            return Optional.ofNullable(clazz.cast(exclusionPatterns()));
        case "DocumentsMetadataConfiguration":
            return Optional.ofNullable(clazz.cast(documentsMetadataConfiguration()));
        case "AccessControlListConfiguration":
            return Optional.ofNullable(clazz.cast(accessControlListConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3DataSourceConfiguration, T> g) {
        return obj -> g.apply((S3DataSourceConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3DataSourceConfiguration> {
        /**
         * <p>
         * The name of the bucket that contains the documents.
         * </p>
         * 
         * @param bucketName
         *        The name of the bucket that contains the documents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * A list of S3 prefixes for the documents that should be included in the index.
         * </p>
         * 
         * @param inclusionPrefixes
         *        A list of S3 prefixes for the documents that should be included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPrefixes(Collection<String> inclusionPrefixes);

        /**
         * <p>
         * A list of S3 prefixes for the documents that should be included in the index.
         * </p>
         * 
         * @param inclusionPrefixes
         *        A list of S3 prefixes for the documents that should be included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPrefixes(String... inclusionPrefixes);

        /**
         * <p>
         * A list of glob patterns for documents that should not be indexed. If a document that matches an inclusion
         * prefix also matches an exclusion pattern, the document is not indexed.
         * </p>
         * <p>
         * For more information about glob patterns, see <a href="http://wikipedia.org/wiki/Glob_%28programming%29">glob
         * (programming)</a> in <i>Wikipedia</i>.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of glob patterns for documents that should not be indexed. If a document that matches an
         *        inclusion prefix also matches an exclusion pattern, the document is not indexed.</p>
         *        <p>
         *        For more information about glob patterns, see <a
         *        href="http://wikipedia.org/wiki/Glob_%28programming%29">glob (programming)</a> in <i>Wikipedia</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(Collection<String> exclusionPatterns);

        /**
         * <p>
         * A list of glob patterns for documents that should not be indexed. If a document that matches an inclusion
         * prefix also matches an exclusion pattern, the document is not indexed.
         * </p>
         * <p>
         * For more information about glob patterns, see <a href="http://wikipedia.org/wiki/Glob_%28programming%29">glob
         * (programming)</a> in <i>Wikipedia</i>.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of glob patterns for documents that should not be indexed. If a document that matches an
         *        inclusion prefix also matches an exclusion pattern, the document is not indexed.</p>
         *        <p>
         *        For more information about glob patterns, see <a
         *        href="http://wikipedia.org/wiki/Glob_%28programming%29">glob (programming)</a> in <i>Wikipedia</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(String... exclusionPatterns);

        /**
         * Sets the value of the DocumentsMetadataConfiguration property for this object.
         *
         * @param documentsMetadataConfiguration
         *        The new value for the DocumentsMetadataConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentsMetadataConfiguration(DocumentsMetadataConfiguration documentsMetadataConfiguration);

        /**
         * Sets the value of the DocumentsMetadataConfiguration property for this object.
         *
         * This is a convenience that creates an instance of the {@link DocumentsMetadataConfiguration.Builder} avoiding
         * the need to create one manually via {@link DocumentsMetadataConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link DocumentsMetadataConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #documentsMetadataConfiguration(DocumentsMetadataConfiguration)}.
         * 
         * @param documentsMetadataConfiguration
         *        a consumer that will call methods on {@link DocumentsMetadataConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #documentsMetadataConfiguration(DocumentsMetadataConfiguration)
         */
        default Builder documentsMetadataConfiguration(
                Consumer<DocumentsMetadataConfiguration.Builder> documentsMetadataConfiguration) {
            return documentsMetadataConfiguration(DocumentsMetadataConfiguration.builder()
                    .applyMutation(documentsMetadataConfiguration).build());
        }

        /**
         * <p>
         * Provides the path to the S3 bucket that contains the user context filtering files for the data source.
         * </p>
         * 
         * @param accessControlListConfiguration
         *        Provides the path to the S3 bucket that contains the user context filtering files for the data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlListConfiguration(AccessControlListConfiguration accessControlListConfiguration);

        /**
         * <p>
         * Provides the path to the S3 bucket that contains the user context filtering files for the data source.
         * </p>
         * This is a convenience that creates an instance of the {@link AccessControlListConfiguration.Builder} avoiding
         * the need to create one manually via {@link AccessControlListConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link AccessControlListConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #accessControlListConfiguration(AccessControlListConfiguration)}.
         * 
         * @param accessControlListConfiguration
         *        a consumer that will call methods on {@link AccessControlListConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlListConfiguration(AccessControlListConfiguration)
         */
        default Builder accessControlListConfiguration(
                Consumer<AccessControlListConfiguration.Builder> accessControlListConfiguration) {
            return accessControlListConfiguration(AccessControlListConfiguration.builder()
                    .applyMutation(accessControlListConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String bucketName;

        private List<String> inclusionPrefixes = DefaultSdkAutoConstructList.getInstance();

        private List<String> exclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private DocumentsMetadataConfiguration documentsMetadataConfiguration;

        private AccessControlListConfiguration accessControlListConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(S3DataSourceConfiguration model) {
            bucketName(model.bucketName);
            inclusionPrefixes(model.inclusionPrefixes);
            exclusionPatterns(model.exclusionPatterns);
            documentsMetadataConfiguration(model.documentsMetadataConfiguration);
            accessControlListConfiguration(model.accessControlListConfiguration);
        }

        public final String getBucketName() {
            return bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        public final Collection<String> getInclusionPrefixes() {
            return inclusionPrefixes;
        }

        @Override
        public final Builder inclusionPrefixes(Collection<String> inclusionPrefixes) {
            this.inclusionPrefixes = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPrefixes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inclusionPrefixes(String... inclusionPrefixes) {
            inclusionPrefixes(Arrays.asList(inclusionPrefixes));
            return this;
        }

        public final void setInclusionPrefixes(Collection<String> inclusionPrefixes) {
            this.inclusionPrefixes = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPrefixes);
        }

        public final Collection<String> getExclusionPatterns() {
            return exclusionPatterns;
        }

        @Override
        public final Builder exclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exclusionPatterns(String... exclusionPatterns) {
            exclusionPatterns(Arrays.asList(exclusionPatterns));
            return this;
        }

        public final void setExclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
        }

        public final DocumentsMetadataConfiguration.Builder getDocumentsMetadataConfiguration() {
            return documentsMetadataConfiguration != null ? documentsMetadataConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder documentsMetadataConfiguration(DocumentsMetadataConfiguration documentsMetadataConfiguration) {
            this.documentsMetadataConfiguration = documentsMetadataConfiguration;
            return this;
        }

        public final void setDocumentsMetadataConfiguration(
                DocumentsMetadataConfiguration.BuilderImpl documentsMetadataConfiguration) {
            this.documentsMetadataConfiguration = documentsMetadataConfiguration != null ? documentsMetadataConfiguration.build()
                    : null;
        }

        public final AccessControlListConfiguration.Builder getAccessControlListConfiguration() {
            return accessControlListConfiguration != null ? accessControlListConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder accessControlListConfiguration(AccessControlListConfiguration accessControlListConfiguration) {
            this.accessControlListConfiguration = accessControlListConfiguration;
            return this;
        }

        public final void setAccessControlListConfiguration(
                AccessControlListConfiguration.BuilderImpl accessControlListConfiguration) {
            this.accessControlListConfiguration = accessControlListConfiguration != null ? accessControlListConfiguration.build()
                    : null;
        }

        @Override
        public S3DataSourceConfiguration build() {
            return new S3DataSourceConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
