/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides configuration information for connecting to a Microsoft SharePoint data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SharePointConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<SharePointConfiguration.Builder, SharePointConfiguration> {
    private static final SdkField<String> SHARE_POINT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SharePointConfiguration::sharePointVersionAsString)).setter(setter(Builder::sharePointVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SharePointVersion").build()).build();

    private static final SdkField<List<String>> URLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(SharePointConfiguration::urls))
            .setter(setter(Builder::urls))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Urls").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SECRET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SharePointConfiguration::secretArn)).setter(setter(Builder::secretArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretArn").build()).build();

    private static final SdkField<Boolean> CRAWL_ATTACHMENTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(SharePointConfiguration::crawlAttachments)).setter(setter(Builder::crawlAttachments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlAttachments").build()).build();

    private static final SdkField<Boolean> USE_CHANGE_LOG_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(SharePointConfiguration::useChangeLog)).setter(setter(Builder::useChangeLog))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UseChangeLog").build()).build();

    private static final SdkField<List<String>> INCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(SharePointConfiguration::inclusionPatterns))
            .setter(setter(Builder::inclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(SharePointConfiguration::exclusionPatterns))
            .setter(setter(Builder::exclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DataSourceVpcConfiguration> VPC_CONFIGURATION_FIELD = SdkField
            .<DataSourceVpcConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(SharePointConfiguration::vpcConfiguration)).setter(setter(Builder::vpcConfiguration))
            .constructor(DataSourceVpcConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfiguration").build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .getter(getter(SharePointConfiguration::fieldMappings))
            .setter(setter(Builder::fieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DOCUMENT_TITLE_FIELD_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SharePointConfiguration::documentTitleFieldName)).setter(setter(Builder::documentTitleFieldName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentTitleFieldName").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SHARE_POINT_VERSION_FIELD,
            URLS_FIELD, SECRET_ARN_FIELD, CRAWL_ATTACHMENTS_FIELD, USE_CHANGE_LOG_FIELD, INCLUSION_PATTERNS_FIELD,
            EXCLUSION_PATTERNS_FIELD, VPC_CONFIGURATION_FIELD, FIELD_MAPPINGS_FIELD, DOCUMENT_TITLE_FIELD_NAME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String sharePointVersion;

    private final List<String> urls;

    private final String secretArn;

    private final Boolean crawlAttachments;

    private final Boolean useChangeLog;

    private final List<String> inclusionPatterns;

    private final List<String> exclusionPatterns;

    private final DataSourceVpcConfiguration vpcConfiguration;

    private final List<DataSourceToIndexFieldMapping> fieldMappings;

    private final String documentTitleFieldName;

    private SharePointConfiguration(BuilderImpl builder) {
        this.sharePointVersion = builder.sharePointVersion;
        this.urls = builder.urls;
        this.secretArn = builder.secretArn;
        this.crawlAttachments = builder.crawlAttachments;
        this.useChangeLog = builder.useChangeLog;
        this.inclusionPatterns = builder.inclusionPatterns;
        this.exclusionPatterns = builder.exclusionPatterns;
        this.vpcConfiguration = builder.vpcConfiguration;
        this.fieldMappings = builder.fieldMappings;
        this.documentTitleFieldName = builder.documentTitleFieldName;
    }

    /**
     * <p>
     * The version of Microsoft SharePoint that you are using as a data source.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sharePointVersion}
     * will return {@link SharePointVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #sharePointVersionAsString}.
     * </p>
     * 
     * @return The version of Microsoft SharePoint that you are using as a data source.
     * @see SharePointVersion
     */
    public SharePointVersion sharePointVersion() {
        return SharePointVersion.fromValue(sharePointVersion);
    }

    /**
     * <p>
     * The version of Microsoft SharePoint that you are using as a data source.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sharePointVersion}
     * will return {@link SharePointVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #sharePointVersionAsString}.
     * </p>
     * 
     * @return The version of Microsoft SharePoint that you are using as a data source.
     * @see SharePointVersion
     */
    public String sharePointVersionAsString() {
        return sharePointVersion;
    }

    /**
     * Returns true if the Urls property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasUrls() {
        return urls != null && !(urls instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The URLs of the Microsoft SharePoint site that contains the documents that should be indexed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUrls()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The URLs of the Microsoft SharePoint site that contains the documents that should be indexed.
     */
    public List<String> urls() {
        return urls;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of credentials stored in AWS Secrets Manager. The credentials should be a
     * user/password pair. For more information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/data-source-sharepoint.html">Using a Microsoft SharePoint Data
     * Source</a>. For more information about AWS Secrets Manager, see <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html"> What Is AWS Secrets Manager </a>
     * in the <i>AWS Secrets Manager</i> user guide.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of credentials stored in AWS Secrets Manager. The credentials should be a
     *         user/password pair. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/data-source-sharepoint.html">Using a Microsoft
     *         SharePoint Data Source</a>. For more information about AWS Secrets Manager, see <a
     *         href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html"> What Is AWS Secrets
     *         Manager </a> in the <i>AWS Secrets Manager</i> user guide.
     */
    public String secretArn() {
        return secretArn;
    }

    /**
     * <p>
     * <code>TRUE</code> to include attachments to documents stored in your Microsoft SharePoint site in the index;
     * otherwise, <code>FALSE</code>.
     * </p>
     * 
     * @return <code>TRUE</code> to include attachments to documents stored in your Microsoft SharePoint site in the
     *         index; otherwise, <code>FALSE</code>.
     */
    public Boolean crawlAttachments() {
        return crawlAttachments;
    }

    /**
     * <p>
     * Set to <code>TRUE</code> to use the Microsoft SharePoint change log to determine the documents that need to be
     * updated in the index. Depending on the size of the SharePoint change log, it may take longer for Amazon Kendra to
     * use the change log than it takes it to determine the changed documents using the Amazon Kendra document crawler.
     * </p>
     * 
     * @return Set to <code>TRUE</code> to use the Microsoft SharePoint change log to determine the documents that need
     *         to be updated in the index. Depending on the size of the SharePoint change log, it may take longer for
     *         Amazon Kendra to use the change log than it takes it to determine the changed documents using the Amazon
     *         Kendra document crawler.
     */
    public Boolean useChangeLog() {
        return useChangeLog;
    }

    /**
     * Returns true if the InclusionPatterns property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasInclusionPatterns() {
        return inclusionPatterns != null && !(inclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns. Documents that match the patterns are included in the index. Documents
     * that don't match the patterns are excluded from the index. If a document matches both an inclusion pattern and an
     * exclusion pattern, the document is not included in the index.
     * </p>
     * <p>
     * The regex is applied to the display URL of the SharePoint document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInclusionPatterns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of regular expression patterns. Documents that match the patterns are included in the index.
     *         Documents that don't match the patterns are excluded from the index. If a document matches both an
     *         inclusion pattern and an exclusion pattern, the document is not included in the index.</p>
     *         <p>
     *         The regex is applied to the display URL of the SharePoint document.
     */
    public List<String> inclusionPatterns() {
        return inclusionPatterns;
    }

    /**
     * Returns true if the ExclusionPatterns property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasExclusionPatterns() {
        return exclusionPatterns != null && !(exclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regulary expression patterns. Documents that match the patterns are excluded from the index. Documents
     * that don't match the patterns are included in the index. If a document matches both an exclusion pattern and an
     * inclusion pattern, the document is not included in the index.
     * </p>
     * <p>
     * The regex is applied to the display URL of the SharePoint document.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasExclusionPatterns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of regulary expression patterns. Documents that match the patterns are excluded from the index.
     *         Documents that don't match the patterns are included in the index. If a document matches both an
     *         exclusion pattern and an inclusion pattern, the document is not included in the index.</p>
     *         <p>
     *         The regex is applied to the display URL of the SharePoint document.
     */
    public List<String> exclusionPatterns() {
        return exclusionPatterns;
    }

    /**
     * Returns the value of the VpcConfiguration property for this object.
     * 
     * @return The value of the VpcConfiguration property for this object.
     */
    public DataSourceVpcConfiguration vpcConfiguration() {
        return vpcConfiguration;
    }

    /**
     * Returns true if the FieldMappings property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasFieldMappings() {
        return fieldMappings != null && !(fieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>DataSourceToIndexFieldMapping</code> objects that map Microsoft SharePoint attributes to custom
     * fields in the Amazon Kendra index. You must first create the index fields using the operation before you map
     * SharePoint attributes. For more information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping Data Source Fields</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFieldMappings()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of <code>DataSourceToIndexFieldMapping</code> objects that map Microsoft SharePoint attributes to
     *         custom fields in the Amazon Kendra index. You must first create the index fields using the operation
     *         before you map SharePoint attributes. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping Data Source Fields</a>.
     */
    public List<DataSourceToIndexFieldMapping> fieldMappings() {
        return fieldMappings;
    }

    /**
     * <p>
     * The Microsoft SharePoint attribute field that contains the title of the document.
     * </p>
     * 
     * @return The Microsoft SharePoint attribute field that contains the title of the document.
     */
    public String documentTitleFieldName() {
        return documentTitleFieldName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sharePointVersionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(urls());
        hashCode = 31 * hashCode + Objects.hashCode(secretArn());
        hashCode = 31 * hashCode + Objects.hashCode(crawlAttachments());
        hashCode = 31 * hashCode + Objects.hashCode(useChangeLog());
        hashCode = 31 * hashCode + Objects.hashCode(inclusionPatterns());
        hashCode = 31 * hashCode + Objects.hashCode(exclusionPatterns());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(fieldMappings());
        hashCode = 31 * hashCode + Objects.hashCode(documentTitleFieldName());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SharePointConfiguration)) {
            return false;
        }
        SharePointConfiguration other = (SharePointConfiguration) obj;
        return Objects.equals(sharePointVersionAsString(), other.sharePointVersionAsString())
                && Objects.equals(urls(), other.urls()) && Objects.equals(secretArn(), other.secretArn())
                && Objects.equals(crawlAttachments(), other.crawlAttachments())
                && Objects.equals(useChangeLog(), other.useChangeLog())
                && Objects.equals(inclusionPatterns(), other.inclusionPatterns())
                && Objects.equals(exclusionPatterns(), other.exclusionPatterns())
                && Objects.equals(vpcConfiguration(), other.vpcConfiguration())
                && Objects.equals(fieldMappings(), other.fieldMappings())
                && Objects.equals(documentTitleFieldName(), other.documentTitleFieldName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SharePointConfiguration").add("SharePointVersion", sharePointVersionAsString())
                .add("Urls", urls()).add("SecretArn", secretArn()).add("CrawlAttachments", crawlAttachments())
                .add("UseChangeLog", useChangeLog()).add("InclusionPatterns", inclusionPatterns())
                .add("ExclusionPatterns", exclusionPatterns()).add("VpcConfiguration", vpcConfiguration())
                .add("FieldMappings", fieldMappings()).add("DocumentTitleFieldName", documentTitleFieldName()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SharePointVersion":
            return Optional.ofNullable(clazz.cast(sharePointVersionAsString()));
        case "Urls":
            return Optional.ofNullable(clazz.cast(urls()));
        case "SecretArn":
            return Optional.ofNullable(clazz.cast(secretArn()));
        case "CrawlAttachments":
            return Optional.ofNullable(clazz.cast(crawlAttachments()));
        case "UseChangeLog":
            return Optional.ofNullable(clazz.cast(useChangeLog()));
        case "InclusionPatterns":
            return Optional.ofNullable(clazz.cast(inclusionPatterns()));
        case "ExclusionPatterns":
            return Optional.ofNullable(clazz.cast(exclusionPatterns()));
        case "VpcConfiguration":
            return Optional.ofNullable(clazz.cast(vpcConfiguration()));
        case "FieldMappings":
            return Optional.ofNullable(clazz.cast(fieldMappings()));
        case "DocumentTitleFieldName":
            return Optional.ofNullable(clazz.cast(documentTitleFieldName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SharePointConfiguration, T> g) {
        return obj -> g.apply((SharePointConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SharePointConfiguration> {
        /**
         * <p>
         * The version of Microsoft SharePoint that you are using as a data source.
         * </p>
         * 
         * @param sharePointVersion
         *        The version of Microsoft SharePoint that you are using as a data source.
         * @see SharePointVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SharePointVersion
         */
        Builder sharePointVersion(String sharePointVersion);

        /**
         * <p>
         * The version of Microsoft SharePoint that you are using as a data source.
         * </p>
         * 
         * @param sharePointVersion
         *        The version of Microsoft SharePoint that you are using as a data source.
         * @see SharePointVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SharePointVersion
         */
        Builder sharePointVersion(SharePointVersion sharePointVersion);

        /**
         * <p>
         * The URLs of the Microsoft SharePoint site that contains the documents that should be indexed.
         * </p>
         * 
         * @param urls
         *        The URLs of the Microsoft SharePoint site that contains the documents that should be indexed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder urls(Collection<String> urls);

        /**
         * <p>
         * The URLs of the Microsoft SharePoint site that contains the documents that should be indexed.
         * </p>
         * 
         * @param urls
         *        The URLs of the Microsoft SharePoint site that contains the documents that should be indexed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder urls(String... urls);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of credentials stored in AWS Secrets Manager. The credentials should be a
         * user/password pair. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/data-source-sharepoint.html">Using a Microsoft SharePoint
         * Data Source</a>. For more information about AWS Secrets Manager, see <a
         * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html"> What Is AWS Secrets Manager
         * </a> in the <i>AWS Secrets Manager</i> user guide.
         * </p>
         * 
         * @param secretArn
         *        The Amazon Resource Name (ARN) of credentials stored in AWS Secrets Manager. The credentials should be
         *        a user/password pair. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/data-source-sharepoint.html">Using a Microsoft
         *        SharePoint Data Source</a>. For more information about AWS Secrets Manager, see <a
         *        href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html"> What Is AWS Secrets
         *        Manager </a> in the <i>AWS Secrets Manager</i> user guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretArn(String secretArn);

        /**
         * <p>
         * <code>TRUE</code> to include attachments to documents stored in your Microsoft SharePoint site in the index;
         * otherwise, <code>FALSE</code>.
         * </p>
         * 
         * @param crawlAttachments
         *        <code>TRUE</code> to include attachments to documents stored in your Microsoft SharePoint site in the
         *        index; otherwise, <code>FALSE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlAttachments(Boolean crawlAttachments);

        /**
         * <p>
         * Set to <code>TRUE</code> to use the Microsoft SharePoint change log to determine the documents that need to
         * be updated in the index. Depending on the size of the SharePoint change log, it may take longer for Amazon
         * Kendra to use the change log than it takes it to determine the changed documents using the Amazon Kendra
         * document crawler.
         * </p>
         * 
         * @param useChangeLog
         *        Set to <code>TRUE</code> to use the Microsoft SharePoint change log to determine the documents that
         *        need to be updated in the index. Depending on the size of the SharePoint change log, it may take
         *        longer for Amazon Kendra to use the change log than it takes it to determine the changed documents
         *        using the Amazon Kendra document crawler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder useChangeLog(Boolean useChangeLog);

        /**
         * <p>
         * A list of regular expression patterns. Documents that match the patterns are included in the index. Documents
         * that don't match the patterns are excluded from the index. If a document matches both an inclusion pattern
         * and an exclusion pattern, the document is not included in the index.
         * </p>
         * <p>
         * The regex is applied to the display URL of the SharePoint document.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns. Documents that match the patterns are included in the index.
         *        Documents that don't match the patterns are excluded from the index. If a document matches both an
         *        inclusion pattern and an exclusion pattern, the document is not included in the index.</p>
         *        <p>
         *        The regex is applied to the display URL of the SharePoint document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(Collection<String> inclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns. Documents that match the patterns are included in the index. Documents
         * that don't match the patterns are excluded from the index. If a document matches both an inclusion pattern
         * and an exclusion pattern, the document is not included in the index.
         * </p>
         * <p>
         * The regex is applied to the display URL of the SharePoint document.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns. Documents that match the patterns are included in the index.
         *        Documents that don't match the patterns are excluded from the index. If a document matches both an
         *        inclusion pattern and an exclusion pattern, the document is not included in the index.</p>
         *        <p>
         *        The regex is applied to the display URL of the SharePoint document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(String... inclusionPatterns);

        /**
         * <p>
         * A list of regulary expression patterns. Documents that match the patterns are excluded from the index.
         * Documents that don't match the patterns are included in the index. If a document matches both an exclusion
         * pattern and an inclusion pattern, the document is not included in the index.
         * </p>
         * <p>
         * The regex is applied to the display URL of the SharePoint document.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of regulary expression patterns. Documents that match the patterns are excluded from the index.
         *        Documents that don't match the patterns are included in the index. If a document matches both an
         *        exclusion pattern and an inclusion pattern, the document is not included in the index.</p>
         *        <p>
         *        The regex is applied to the display URL of the SharePoint document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(Collection<String> exclusionPatterns);

        /**
         * <p>
         * A list of regulary expression patterns. Documents that match the patterns are excluded from the index.
         * Documents that don't match the patterns are included in the index. If a document matches both an exclusion
         * pattern and an inclusion pattern, the document is not included in the index.
         * </p>
         * <p>
         * The regex is applied to the display URL of the SharePoint document.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of regulary expression patterns. Documents that match the patterns are excluded from the index.
         *        Documents that don't match the patterns are included in the index. If a document matches both an
         *        exclusion pattern and an inclusion pattern, the document is not included in the index.</p>
         *        <p>
         *        The regex is applied to the display URL of the SharePoint document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(String... exclusionPatterns);

        /**
         * Sets the value of the VpcConfiguration property for this object.
         *
         * @param vpcConfiguration
         *        The new value for the VpcConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration);

        /**
         * Sets the value of the VpcConfiguration property for this object.
         *
         * This is a convenience that creates an instance of the {@link DataSourceVpcConfiguration.Builder} avoiding the
         * need to create one manually via {@link DataSourceVpcConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataSourceVpcConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #vpcConfiguration(DataSourceVpcConfiguration)}.
         * 
         * @param vpcConfiguration
         *        a consumer that will call methods on {@link DataSourceVpcConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfiguration(DataSourceVpcConfiguration)
         */
        default Builder vpcConfiguration(Consumer<DataSourceVpcConfiguration.Builder> vpcConfiguration) {
            return vpcConfiguration(DataSourceVpcConfiguration.builder().applyMutation(vpcConfiguration).build());
        }

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map Microsoft SharePoint attributes to
         * custom fields in the Amazon Kendra index. You must first create the index fields using the operation before
         * you map SharePoint attributes. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping Data Source Fields</a>.
         * </p>
         * 
         * @param fieldMappings
         *        A list of <code>DataSourceToIndexFieldMapping</code> objects that map Microsoft SharePoint attributes
         *        to custom fields in the Amazon Kendra index. You must first create the index fields using the
         *        operation before you map SharePoint attributes. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping Data Source Fields</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fieldMappings(Collection<DataSourceToIndexFieldMapping> fieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map Microsoft SharePoint attributes to
         * custom fields in the Amazon Kendra index. You must first create the index fields using the operation before
         * you map SharePoint attributes. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping Data Source Fields</a>.
         * </p>
         * 
         * @param fieldMappings
         *        A list of <code>DataSourceToIndexFieldMapping</code> objects that map Microsoft SharePoint attributes
         *        to custom fields in the Amazon Kendra index. You must first create the index fields using the
         *        operation before you map SharePoint attributes. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping Data Source Fields</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fieldMappings(DataSourceToIndexFieldMapping... fieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map Microsoft SharePoint attributes to
         * custom fields in the Amazon Kendra index. You must first create the index fields using the operation before
         * you map SharePoint attributes. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping Data Source Fields</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DataSourceToIndexFieldMapping>.Builder}
         * avoiding the need to create one manually via {@link List<DataSourceToIndexFieldMapping>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DataSourceToIndexFieldMapping>.Builder#build()} is called
         * immediately and its result is passed to {@link #fieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param fieldMappings
         *        a consumer that will call methods on {@link List<DataSourceToIndexFieldMapping>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fieldMappings(List<DataSourceToIndexFieldMapping>)
         */
        Builder fieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... fieldMappings);

        /**
         * <p>
         * The Microsoft SharePoint attribute field that contains the title of the document.
         * </p>
         * 
         * @param documentTitleFieldName
         *        The Microsoft SharePoint attribute field that contains the title of the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentTitleFieldName(String documentTitleFieldName);
    }

    static final class BuilderImpl implements Builder {
        private String sharePointVersion;

        private List<String> urls = DefaultSdkAutoConstructList.getInstance();

        private String secretArn;

        private Boolean crawlAttachments;

        private Boolean useChangeLog;

        private List<String> inclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private List<String> exclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private DataSourceVpcConfiguration vpcConfiguration;

        private List<DataSourceToIndexFieldMapping> fieldMappings = DefaultSdkAutoConstructList.getInstance();

        private String documentTitleFieldName;

        private BuilderImpl() {
        }

        private BuilderImpl(SharePointConfiguration model) {
            sharePointVersion(model.sharePointVersion);
            urls(model.urls);
            secretArn(model.secretArn);
            crawlAttachments(model.crawlAttachments);
            useChangeLog(model.useChangeLog);
            inclusionPatterns(model.inclusionPatterns);
            exclusionPatterns(model.exclusionPatterns);
            vpcConfiguration(model.vpcConfiguration);
            fieldMappings(model.fieldMappings);
            documentTitleFieldName(model.documentTitleFieldName);
        }

        public final String getSharePointVersion() {
            return sharePointVersion;
        }

        @Override
        public final Builder sharePointVersion(String sharePointVersion) {
            this.sharePointVersion = sharePointVersion;
            return this;
        }

        @Override
        public final Builder sharePointVersion(SharePointVersion sharePointVersion) {
            this.sharePointVersion(sharePointVersion == null ? null : sharePointVersion.toString());
            return this;
        }

        public final void setSharePointVersion(String sharePointVersion) {
            this.sharePointVersion = sharePointVersion;
        }

        public final Collection<String> getUrls() {
            return urls;
        }

        @Override
        public final Builder urls(Collection<String> urls) {
            this.urls = SharePointUrlListCopier.copy(urls);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder urls(String... urls) {
            urls(Arrays.asList(urls));
            return this;
        }

        public final void setUrls(Collection<String> urls) {
            this.urls = SharePointUrlListCopier.copy(urls);
        }

        public final String getSecretArn() {
            return secretArn;
        }

        @Override
        public final Builder secretArn(String secretArn) {
            this.secretArn = secretArn;
            return this;
        }

        public final void setSecretArn(String secretArn) {
            this.secretArn = secretArn;
        }

        public final Boolean getCrawlAttachments() {
            return crawlAttachments;
        }

        @Override
        public final Builder crawlAttachments(Boolean crawlAttachments) {
            this.crawlAttachments = crawlAttachments;
            return this;
        }

        public final void setCrawlAttachments(Boolean crawlAttachments) {
            this.crawlAttachments = crawlAttachments;
        }

        public final Boolean getUseChangeLog() {
            return useChangeLog;
        }

        @Override
        public final Builder useChangeLog(Boolean useChangeLog) {
            this.useChangeLog = useChangeLog;
            return this;
        }

        public final void setUseChangeLog(Boolean useChangeLog) {
            this.useChangeLog = useChangeLog;
        }

        public final Collection<String> getInclusionPatterns() {
            return inclusionPatterns;
        }

        @Override
        public final Builder inclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inclusionPatterns(String... inclusionPatterns) {
            inclusionPatterns(Arrays.asList(inclusionPatterns));
            return this;
        }

        public final void setInclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
        }

        public final Collection<String> getExclusionPatterns() {
            return exclusionPatterns;
        }

        @Override
        public final Builder exclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exclusionPatterns(String... exclusionPatterns) {
            exclusionPatterns(Arrays.asList(exclusionPatterns));
            return this;
        }

        public final void setExclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
        }

        public final DataSourceVpcConfiguration.Builder getVpcConfiguration() {
            return vpcConfiguration != null ? vpcConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration;
            return this;
        }

        public final void setVpcConfiguration(DataSourceVpcConfiguration.BuilderImpl vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration != null ? vpcConfiguration.build() : null;
        }

        public final Collection<DataSourceToIndexFieldMapping.Builder> getFieldMappings() {
            return fieldMappings != null ? fieldMappings.stream().map(DataSourceToIndexFieldMapping::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder fieldMappings(Collection<DataSourceToIndexFieldMapping> fieldMappings) {
            this.fieldMappings = DataSourceToIndexFieldMappingListCopier.copy(fieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fieldMappings(DataSourceToIndexFieldMapping... fieldMappings) {
            fieldMappings(Arrays.asList(fieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... fieldMappings) {
            fieldMappings(Stream.of(fieldMappings).map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> fieldMappings) {
            this.fieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(fieldMappings);
        }

        public final String getDocumentTitleFieldName() {
            return documentTitleFieldName;
        }

        @Override
        public final Builder documentTitleFieldName(String documentTitleFieldName) {
            this.documentTitleFieldName = documentTitleFieldName;
            return this;
        }

        public final void setDocumentTitleFieldName(String documentTitleFieldName) {
            this.documentTitleFieldName = documentTitleFieldName;
        }

        @Override
        public SharePointConfiguration build() {
            return new SharePointConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
