/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.kendra.model.AccessDeniedException;
import software.amazon.awssdk.services.kendra.model.BatchDeleteDocumentRequest;
import software.amazon.awssdk.services.kendra.model.BatchDeleteDocumentResponse;
import software.amazon.awssdk.services.kendra.model.BatchPutDocumentRequest;
import software.amazon.awssdk.services.kendra.model.BatchPutDocumentResponse;
import software.amazon.awssdk.services.kendra.model.ConflictException;
import software.amazon.awssdk.services.kendra.model.CreateDataSourceRequest;
import software.amazon.awssdk.services.kendra.model.CreateDataSourceResponse;
import software.amazon.awssdk.services.kendra.model.CreateFaqRequest;
import software.amazon.awssdk.services.kendra.model.CreateFaqResponse;
import software.amazon.awssdk.services.kendra.model.CreateIndexRequest;
import software.amazon.awssdk.services.kendra.model.CreateIndexResponse;
import software.amazon.awssdk.services.kendra.model.DeleteDataSourceRequest;
import software.amazon.awssdk.services.kendra.model.DeleteDataSourceResponse;
import software.amazon.awssdk.services.kendra.model.DeleteFaqRequest;
import software.amazon.awssdk.services.kendra.model.DeleteFaqResponse;
import software.amazon.awssdk.services.kendra.model.DeleteIndexRequest;
import software.amazon.awssdk.services.kendra.model.DeleteIndexResponse;
import software.amazon.awssdk.services.kendra.model.DescribeDataSourceRequest;
import software.amazon.awssdk.services.kendra.model.DescribeDataSourceResponse;
import software.amazon.awssdk.services.kendra.model.DescribeFaqRequest;
import software.amazon.awssdk.services.kendra.model.DescribeFaqResponse;
import software.amazon.awssdk.services.kendra.model.DescribeIndexRequest;
import software.amazon.awssdk.services.kendra.model.DescribeIndexResponse;
import software.amazon.awssdk.services.kendra.model.InternalServerException;
import software.amazon.awssdk.services.kendra.model.KendraException;
import software.amazon.awssdk.services.kendra.model.KendraRequest;
import software.amazon.awssdk.services.kendra.model.ListDataSourceSyncJobsRequest;
import software.amazon.awssdk.services.kendra.model.ListDataSourceSyncJobsResponse;
import software.amazon.awssdk.services.kendra.model.ListDataSourcesRequest;
import software.amazon.awssdk.services.kendra.model.ListDataSourcesResponse;
import software.amazon.awssdk.services.kendra.model.ListFaqsRequest;
import software.amazon.awssdk.services.kendra.model.ListFaqsResponse;
import software.amazon.awssdk.services.kendra.model.ListIndicesRequest;
import software.amazon.awssdk.services.kendra.model.ListIndicesResponse;
import software.amazon.awssdk.services.kendra.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.kendra.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.kendra.model.QueryRequest;
import software.amazon.awssdk.services.kendra.model.QueryResponse;
import software.amazon.awssdk.services.kendra.model.ResourceAlreadyExistException;
import software.amazon.awssdk.services.kendra.model.ResourceInUseException;
import software.amazon.awssdk.services.kendra.model.ResourceNotFoundException;
import software.amazon.awssdk.services.kendra.model.ResourceUnavailableException;
import software.amazon.awssdk.services.kendra.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.kendra.model.StartDataSourceSyncJobRequest;
import software.amazon.awssdk.services.kendra.model.StartDataSourceSyncJobResponse;
import software.amazon.awssdk.services.kendra.model.StopDataSourceSyncJobRequest;
import software.amazon.awssdk.services.kendra.model.StopDataSourceSyncJobResponse;
import software.amazon.awssdk.services.kendra.model.SubmitFeedbackRequest;
import software.amazon.awssdk.services.kendra.model.SubmitFeedbackResponse;
import software.amazon.awssdk.services.kendra.model.TagResourceRequest;
import software.amazon.awssdk.services.kendra.model.TagResourceResponse;
import software.amazon.awssdk.services.kendra.model.ThrottlingException;
import software.amazon.awssdk.services.kendra.model.UntagResourceRequest;
import software.amazon.awssdk.services.kendra.model.UntagResourceResponse;
import software.amazon.awssdk.services.kendra.model.UpdateDataSourceRequest;
import software.amazon.awssdk.services.kendra.model.UpdateDataSourceResponse;
import software.amazon.awssdk.services.kendra.model.UpdateIndexRequest;
import software.amazon.awssdk.services.kendra.model.UpdateIndexResponse;
import software.amazon.awssdk.services.kendra.model.ValidationException;
import software.amazon.awssdk.services.kendra.paginators.ListDataSourceSyncJobsIterable;
import software.amazon.awssdk.services.kendra.paginators.ListDataSourcesIterable;
import software.amazon.awssdk.services.kendra.paginators.ListIndicesIterable;
import software.amazon.awssdk.services.kendra.transform.BatchDeleteDocumentRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.BatchPutDocumentRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.CreateDataSourceRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.CreateFaqRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.CreateIndexRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.DeleteDataSourceRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.DeleteFaqRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.DeleteIndexRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.DescribeDataSourceRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.DescribeFaqRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.DescribeIndexRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.ListDataSourceSyncJobsRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.ListDataSourcesRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.ListFaqsRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.ListIndicesRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.QueryRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.StartDataSourceSyncJobRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.StopDataSourceSyncJobRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.SubmitFeedbackRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.UpdateDataSourceRequestMarshaller;
import software.amazon.awssdk.services.kendra.transform.UpdateIndexRequestMarshaller;

/**
 * Internal implementation of {@link KendraClient}.
 *
 * @see KendraClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultKendraClient implements KendraClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultKendraClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Removes one or more documents from an index. The documents must have been added with the <a>BatchPutDocument</a>
     * operation.
     * </p>
     * <p>
     * The documents are deleted asynchronously. You can see the progress of the deletion by using AWS CloudWatch. Any
     * error messages releated to the processing of the batch are sent to you CloudWatch log.
     * </p>
     *
     * @param batchDeleteDocumentRequest
     * @return Result of the BatchDeleteDocument operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.BatchDeleteDocument
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/BatchDeleteDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchDeleteDocumentResponse batchDeleteDocument(BatchDeleteDocumentRequest batchDeleteDocumentRequest)
            throws ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchDeleteDocumentResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchDeleteDocumentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteDocument");

            return clientHandler.execute(new ClientExecutionParams<BatchDeleteDocumentRequest, BatchDeleteDocumentResponse>()
                    .withOperationName("BatchDeleteDocument").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchDeleteDocumentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchDeleteDocumentRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteDocumentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more documents to an index.
     * </p>
     * <p>
     * The <code>BatchPutDocument</code> operation enables you to ingest inline documents or a set of documents stored
     * in an Amazon S3 bucket. Use this operation to ingest your text and unstructured text into an index, add custom
     * attributes to the documents, and to attach an access control list to the documents added to the index.
     * </p>
     * <p>
     * The documents are indexed asynchronously. You can see the progress of the batch using AWS CloudWatch. Any error
     * messages related to processing the batch are sent to your AWS CloudWatch log.
     * </p>
     *
     * @param batchPutDocumentRequest
     * @return Result of the BatchPutDocument operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.BatchPutDocument
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/BatchPutDocument" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchPutDocumentResponse batchPutDocument(BatchPutDocumentRequest batchPutDocumentRequest) throws ValidationException,
            ConflictException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchPutDocumentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchPutDocumentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutDocument");

            return clientHandler.execute(new ClientExecutionParams<BatchPutDocumentRequest, BatchPutDocumentResponse>()
                    .withOperationName("BatchPutDocument").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchPutDocumentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchPutDocumentRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchPutDocumentRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a data source that you use to with an Amazon Kendra index.
     * </p>
     * <p>
     * You specify a name, connector type and description for your data source. You can choose between an S3 connector,
     * a SharePoint Online connector, and a database connector.
     * </p>
     * <p>
     * You also specify configuration information such as document metadata (author, source URI, and so on) and user
     * context information.
     * </p>
     * <p>
     * <code>CreateDataSource</code> is a synchronous operation. The operation returns 200 if the data source was
     * successfully created. Otherwise, an exception is raised.
     * </p>
     *
     * @param createDataSourceRequest
     * @return Result of the CreateDataSource operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws ResourceNotFoundException
     * @throws ResourceAlreadyExistException
     * @throws ServiceQuotaExceededException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.CreateDataSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/CreateDataSource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateDataSourceResponse createDataSource(CreateDataSourceRequest createDataSourceRequest) throws ValidationException,
            ConflictException, ResourceNotFoundException, ResourceAlreadyExistException, ServiceQuotaExceededException,
            ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDataSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataSource");

            return clientHandler.execute(new ClientExecutionParams<CreateDataSourceRequest, CreateDataSourceResponse>()
                    .withOperationName("CreateDataSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createDataSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDataSourceRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataSourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an new set of frequently asked question (FAQ) questions and answers.
     * </p>
     *
     * @param createFaqRequest
     * @return Result of the CreateFaq operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws ServiceQuotaExceededException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.CreateFaq
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/CreateFaq" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFaqResponse createFaq(CreateFaqRequest createFaqRequest) throws ValidationException, ConflictException,
            ResourceNotFoundException, ThrottlingException, ServiceQuotaExceededException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFaqResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFaqResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFaq");

            return clientHandler.execute(new ClientExecutionParams<CreateFaqRequest, CreateFaqResponse>()
                    .withOperationName("CreateFaq").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createFaqRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CreateFaqRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFaqRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Amazon Kendra index. Index creation is an asynchronous operation. To determine if index creation
     * has completed, check the <code>Status</code> field returned from a call to . The <code>Status</code> field is set
     * to <code>ACTIVE</code> when the index is ready to use.
     * </p>
     * <p>
     * Once the index is active you can index your documents using the operation or using one of the supported data
     * sources.
     * </p>
     *
     * @param createIndexRequest
     * @return Result of the CreateIndex operation returned by the service.
     * @throws ValidationException
     * @throws ResourceAlreadyExistException
     * @throws ServiceQuotaExceededException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.CreateIndex
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/CreateIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateIndexResponse createIndex(CreateIndexRequest createIndexRequest) throws ValidationException,
            ResourceAlreadyExistException, ServiceQuotaExceededException, ThrottlingException, AccessDeniedException,
            ConflictException, InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateIndexResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIndex");

            return clientHandler.execute(new ClientExecutionParams<CreateIndexRequest, CreateIndexResponse>()
                    .withOperationName("CreateIndex").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createIndexRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateIndexRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIndexRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon Kendra data source. An exception is not thrown if the data source is already being deleted.
     * While the data source is being deleted, the <code>Status</code> field returned by a call to the operation is set
     * to <code>DELETING</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/delete-data-source.html">Deleting Data Sources</a>.
     * </p>
     *
     * @param deleteDataSourceRequest
     * @return Result of the DeleteDataSource operation returned by the service.
     * @throws AccessDeniedException
     * @throws ValidationException
     * @throws ConflictException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.DeleteDataSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/DeleteDataSource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteDataSourceResponse deleteDataSource(DeleteDataSourceRequest deleteDataSourceRequest)
            throws AccessDeniedException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDataSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataSource");

            return clientHandler.execute(new ClientExecutionParams<DeleteDataSourceRequest, DeleteDataSourceResponse>()
                    .withOperationName("DeleteDataSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDataSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDataSourceRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataSourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes an FAQ from an index.
     * </p>
     *
     * @param deleteFaqRequest
     * @return Result of the DeleteFaq operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.DeleteFaq
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/DeleteFaq" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFaqResponse deleteFaq(DeleteFaqRequest deleteFaqRequest) throws ValidationException, ConflictException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFaqResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFaqResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFaq");

            return clientHandler.execute(new ClientExecutionParams<DeleteFaqRequest, DeleteFaqResponse>()
                    .withOperationName("DeleteFaq").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteFaqRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new DeleteFaqRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFaqRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing Amazon Kendra index. An exception is not thrown if the index is already being deleted. While
     * the index is being deleted, the <code>Status</code> field returned by a call to the <a>DescribeIndex</a>
     * operation is set to <code>DELETING</code>.
     * </p>
     *
     * @param deleteIndexRequest
     * @return Result of the DeleteIndex operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.DeleteIndex
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/DeleteIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteIndexResponse deleteIndex(DeleteIndexRequest deleteIndexRequest) throws ValidationException, ConflictException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteIndexResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIndex");

            return clientHandler.execute(new ClientExecutionParams<DeleteIndexRequest, DeleteIndexResponse>()
                    .withOperationName("DeleteIndex").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteIndexRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteIndexRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIndexRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about a Amazon Kendra data source.
     * </p>
     *
     * @param describeDataSourceRequest
     * @return Result of the DescribeDataSource operation returned by the service.
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.DescribeDataSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/DescribeDataSource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeDataSourceResponse describeDataSource(DescribeDataSourceRequest describeDataSourceRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDataSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataSource");

            return clientHandler.execute(new ClientExecutionParams<DescribeDataSourceRequest, DescribeDataSourceResponse>()
                    .withOperationName("DescribeDataSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeDataSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDataSourceRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDataSourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about an FAQ list.
     * </p>
     *
     * @param describeFaqRequest
     * @return Result of the DescribeFaq operation returned by the service.
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.DescribeFaq
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/DescribeFaq" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeFaqResponse describeFaq(DescribeFaqRequest describeFaqRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFaqResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeFaqResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFaq");

            return clientHandler.execute(new ClientExecutionParams<DescribeFaqRequest, DescribeFaqResponse>()
                    .withOperationName("DescribeFaq").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeFaqRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFaqRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFaqRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing Amazon Kendra index
     * </p>
     *
     * @param describeIndexRequest
     * @return Result of the DescribeIndex operation returned by the service.
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.DescribeIndex
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/DescribeIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeIndexResponse describeIndex(DescribeIndexRequest describeIndexRequest) throws ValidationException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException,
            SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeIndexResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeIndex");

            return clientHandler.execute(new ClientExecutionParams<DescribeIndexRequest, DescribeIndexResponse>()
                    .withOperationName("DescribeIndex").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeIndexRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeIndexRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeIndexRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets statistics about synchronizing Amazon Kendra with a data source.
     * </p>
     *
     * @param listDataSourceSyncJobsRequest
     * @return Result of the ListDataSourceSyncJobs operation returned by the service.
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.ListDataSourceSyncJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/ListDataSourceSyncJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDataSourceSyncJobsResponse listDataSourceSyncJobs(ListDataSourceSyncJobsRequest listDataSourceSyncJobsRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, AccessDeniedException, ConflictException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataSourceSyncJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDataSourceSyncJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataSourceSyncJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDataSourceSyncJobsRequest, ListDataSourceSyncJobsResponse>()
                            .withOperationName("ListDataSourceSyncJobs").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listDataSourceSyncJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDataSourceSyncJobsRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataSourceSyncJobsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets statistics about synchronizing Amazon Kendra with a data source.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDataSourceSyncJobs(software.amazon.awssdk.services.kendra.model.ListDataSourceSyncJobsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kendra.paginators.ListDataSourceSyncJobsIterable responses = client.listDataSourceSyncJobsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kendra.paginators.ListDataSourceSyncJobsIterable responses = client
     *             .listDataSourceSyncJobsPaginator(request);
     *     for (software.amazon.awssdk.services.kendra.model.ListDataSourceSyncJobsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kendra.paginators.ListDataSourceSyncJobsIterable responses = client.listDataSourceSyncJobsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDataSourceSyncJobs(software.amazon.awssdk.services.kendra.model.ListDataSourceSyncJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDataSourceSyncJobsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.ListDataSourceSyncJobs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/ListDataSourceSyncJobs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDataSourceSyncJobsIterable listDataSourceSyncJobsPaginator(
            ListDataSourceSyncJobsRequest listDataSourceSyncJobsRequest) throws ValidationException, ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, ConflictException, InternalServerException, AwsServiceException,
            SdkClientException, KendraException {
        return new ListDataSourceSyncJobsIterable(this, applyPaginatorUserAgent(listDataSourceSyncJobsRequest));
    }

    /**
     * <p>
     * Lists the data sources that you have created.
     * </p>
     *
     * @param listDataSourcesRequest
     * @return Result of the ListDataSources operation returned by the service.
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.ListDataSources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/ListDataSources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDataSourcesResponse listDataSources(ListDataSourcesRequest listDataSourcesRequest) throws ValidationException,
            ResourceNotFoundException, AccessDeniedException, ThrottlingException, InternalServerException, AwsServiceException,
            SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataSourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDataSourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataSources");

            return clientHandler.execute(new ClientExecutionParams<ListDataSourcesRequest, ListDataSourcesResponse>()
                    .withOperationName("ListDataSources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDataSourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDataSourcesRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataSourcesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the data sources that you have created.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDataSources(software.amazon.awssdk.services.kendra.model.ListDataSourcesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kendra.paginators.ListDataSourcesIterable responses = client.listDataSourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kendra.paginators.ListDataSourcesIterable responses = client
     *             .listDataSourcesPaginator(request);
     *     for (software.amazon.awssdk.services.kendra.model.ListDataSourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kendra.paginators.ListDataSourcesIterable responses = client.listDataSourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDataSources(software.amazon.awssdk.services.kendra.model.ListDataSourcesRequest)} operation.</b>
     * </p>
     *
     * @param listDataSourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws AccessDeniedException
     * @throws ThrottlingException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.ListDataSources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/ListDataSources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDataSourcesIterable listDataSourcesPaginator(ListDataSourcesRequest listDataSourcesRequest)
            throws ValidationException, ResourceNotFoundException, AccessDeniedException, ThrottlingException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        return new ListDataSourcesIterable(this, applyPaginatorUserAgent(listDataSourcesRequest));
    }

    /**
     * <p>
     * Gets a list of FAQ lists associated with an index.
     * </p>
     *
     * @param listFaqsRequest
     * @return Result of the ListFaqs operation returned by the service.
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.ListFaqs
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/ListFaqs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListFaqsResponse listFaqs(ListFaqsRequest listFaqsRequest) throws ValidationException, ResourceNotFoundException,
            ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFaqsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListFaqsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFaqs");

            return clientHandler.execute(new ClientExecutionParams<ListFaqsRequest, ListFaqsResponse>()
                    .withOperationName("ListFaqs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listFaqsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListFaqsRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFaqsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Amazon Kendra indexes that you have created.
     * </p>
     *
     * @param listIndicesRequest
     * @return Result of the ListIndices operation returned by the service.
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.ListIndices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/ListIndices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListIndicesResponse listIndices(ListIndicesRequest listIndicesRequest) throws ValidationException,
            ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListIndicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListIndicesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIndices");

            return clientHandler.execute(new ClientExecutionParams<ListIndicesRequest, ListIndicesResponse>()
                    .withOperationName("ListIndices").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listIndicesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListIndicesRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIndicesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Amazon Kendra indexes that you have created.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listIndices(software.amazon.awssdk.services.kendra.model.ListIndicesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kendra.paginators.ListIndicesIterable responses = client.listIndicesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.kendra.paginators.ListIndicesIterable responses = client.listIndicesPaginator(request);
     *     for (software.amazon.awssdk.services.kendra.model.ListIndicesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.kendra.paginators.ListIndicesIterable responses = client.listIndicesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listIndices(software.amazon.awssdk.services.kendra.model.ListIndicesRequest)} operation.</b>
     * </p>
     *
     * @param listIndicesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.ListIndices
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/ListIndices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListIndicesIterable listIndicesPaginator(ListIndicesRequest listIndicesRequest) throws ValidationException,
            ThrottlingException, AccessDeniedException, InternalServerException, AwsServiceException, SdkClientException,
            KendraException {
        return new ListIndicesIterable(this, applyPaginatorUserAgent(listIndicesRequest));
    }

    /**
     * <p>
     * Gets a list of tags associated with a specified resource. Indexes, FAQs, and data sources can have tags
     * associated with them.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     * @throws ResourceUnavailableException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, ResourceUnavailableException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches an active index. Use this API to search your documents using query. The <code>Query</code> operation
     * enables to do faceted search and to filter results based on document attributes.
     * </p>
     * <p>
     * It also enables you to provide user context that Amazon Kendra uses to enforce document access control in the
     * search results.
     * </p>
     * <p>
     * Amazon Kendra searches your index for text content and question and answer (FAQ) content. By default the response
     * contains three types of results.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Relevant passages
     * </p>
     * </li>
     * <li>
     * <p>
     * Matching FAQs
     * </p>
     * </li>
     * <li>
     * <p>
     * Relevant documents
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can specify that the query return only one type of result using the <code>QueryResultTypeConfig</code>
     * parameter.
     * </p>
     *
     * @param queryRequest
     * @return Result of the Query operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.Query
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/Query" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryResponse query(QueryRequest queryRequest) throws ValidationException, ConflictException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, ServiceQuotaExceededException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<QueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                QueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Query");

            return clientHandler.execute(new ClientExecutionParams<QueryRequest, QueryResponse>().withOperationName("Query")
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(queryRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new QueryRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a synchronization job for a data source. If a synchronization job is already in progress, Amazon Kendra
     * returns a <code>ResourceInUseException</code> exception.
     * </p>
     *
     * @param startDataSourceSyncJobRequest
     * @return Result of the StartDataSourceSyncJob operation returned by the service.
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws ResourceInUseException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ConflictException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.StartDataSourceSyncJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/StartDataSourceSyncJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartDataSourceSyncJobResponse startDataSourceSyncJob(StartDataSourceSyncJobRequest startDataSourceSyncJobRequest)
            throws ValidationException, ResourceNotFoundException, ResourceInUseException, ThrottlingException,
            AccessDeniedException, ConflictException, InternalServerException, AwsServiceException, SdkClientException,
            KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDataSourceSyncJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDataSourceSyncJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDataSourceSyncJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StartDataSourceSyncJobRequest, StartDataSourceSyncJobResponse>()
                            .withOperationName("StartDataSourceSyncJob").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startDataSourceSyncJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartDataSourceSyncJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDataSourceSyncJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a running synchronization job. You can't stop a scheduled synchronization job.
     * </p>
     *
     * @param stopDataSourceSyncJobRequest
     * @return Result of the StopDataSourceSyncJob operation returned by the service.
     * @throws ValidationException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.StopDataSourceSyncJob
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/StopDataSourceSyncJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopDataSourceSyncJobResponse stopDataSourceSyncJob(StopDataSourceSyncJobRequest stopDataSourceSyncJobRequest)
            throws ValidationException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopDataSourceSyncJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopDataSourceSyncJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopDataSourceSyncJob");

            return clientHandler.execute(new ClientExecutionParams<StopDataSourceSyncJobRequest, StopDataSourceSyncJobResponse>()
                    .withOperationName("StopDataSourceSyncJob").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopDataSourceSyncJobRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopDataSourceSyncJobRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopDataSourceSyncJobRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables you to provide feedback to Amazon Kendra to improve the performance of the service.
     * </p>
     *
     * @param submitFeedbackRequest
     * @return Result of the SubmitFeedback operation returned by the service.
     * @throws ValidationException
     * @throws ResourceUnavailableException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.SubmitFeedback
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/SubmitFeedback" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SubmitFeedbackResponse submitFeedback(SubmitFeedbackRequest submitFeedbackRequest) throws ValidationException,
            ResourceUnavailableException, ResourceNotFoundException, ThrottlingException, AccessDeniedException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SubmitFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SubmitFeedbackResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SubmitFeedback");

            return clientHandler.execute(new ClientExecutionParams<SubmitFeedbackRequest, SubmitFeedbackResponse>()
                    .withOperationName("SubmitFeedback").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(submitFeedbackRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SubmitFeedbackRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, submitFeedbackRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified tag to the specified index, FAQ, or data source resource. If the tag already exists, the
     * existing value is replaced with the new value.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     * @throws ResourceUnavailableException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            ResourceUnavailableException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a tag from an index, FAQ, or a data source.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     * @throws ResourceUnavailableException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            ResourceUnavailableException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing Amazon Kendra data source.
     * </p>
     *
     * @param updateDataSourceRequest
     * @return Result of the UpdateDataSource operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.UpdateDataSource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/UpdateDataSource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateDataSourceResponse updateDataSource(UpdateDataSourceRequest updateDataSourceRequest) throws ValidationException,
            ConflictException, ResourceNotFoundException, ThrottlingException, AccessDeniedException, InternalServerException,
            AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDataSourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataSource");

            return clientHandler.execute(new ClientExecutionParams<UpdateDataSourceRequest, UpdateDataSourceResponse>()
                    .withOperationName("UpdateDataSource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateDataSourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDataSourceRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataSourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing Amazon Kendra index.
     * </p>
     *
     * @param updateIndexRequest
     * @return Result of the UpdateIndex operation returned by the service.
     * @throws ValidationException
     * @throws ConflictException
     * @throws ResourceNotFoundException
     * @throws ThrottlingException
     * @throws AccessDeniedException
     * @throws ServiceQuotaExceededException
     * @throws InternalServerException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws KendraException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample KendraClient.UpdateIndex
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/kendra-2019-02-03/UpdateIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateIndexResponse updateIndex(UpdateIndexRequest updateIndexRequest) throws ValidationException, ConflictException,
            ResourceNotFoundException, ThrottlingException, AccessDeniedException, ServiceQuotaExceededException,
            InternalServerException, AwsServiceException, SdkClientException, KendraException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateIndexResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateIndexResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "kendra");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIndex");

            return clientHandler.execute(new ClientExecutionParams<UpdateIndexRequest, UpdateIndexResponse>()
                    .withOperationName("UpdateIndex").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateIndexRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateIndexRequestMarshaller(protocolFactory)));
        } finally {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateIndexRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(KendraException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceUnavailableException")
                                .exceptionBuilderSupplier(ResourceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends KendraRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
