/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.adapter.StandardMemberCopier;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A document in an index.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Document implements SdkPojo, Serializable, ToCopyableBuilder<Document.Builder, Document> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Document::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(Document::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<SdkBytes> BLOB_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("Blob").getter(getter(Document::blob)).setter(setter(Builder::blob))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Blob").build()).build();

    private static final SdkField<S3Path> S3_PATH_FIELD = SdkField.<S3Path> builder(MarshallingType.SDK_POJO)
            .memberName("S3Path").getter(getter(Document::s3Path)).setter(setter(Builder::s3Path)).constructor(S3Path::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Path").build()).build();

    private static final SdkField<List<DocumentAttribute>> ATTRIBUTES_FIELD = SdkField
            .<List<DocumentAttribute>> builder(MarshallingType.LIST)
            .memberName("Attributes")
            .getter(getter(Document::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentAttribute> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentAttribute::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Principal>> ACCESS_CONTROL_LIST_FIELD = SdkField
            .<List<Principal>> builder(MarshallingType.LIST)
            .memberName("AccessControlList")
            .getter(getter(Document::accessControlList))
            .setter(setter(Builder::accessControlList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Principal> builder(MarshallingType.SDK_POJO)
                                            .constructor(Principal::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONTENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentType").getter(getter(Document::contentTypeAsString)).setter(setter(Builder::contentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, TITLE_FIELD,
            BLOB_FIELD, S3_PATH_FIELD, ATTRIBUTES_FIELD, ACCESS_CONTROL_LIST_FIELD, CONTENT_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String title;

    private final SdkBytes blob;

    private final S3Path s3Path;

    private final List<DocumentAttribute> attributes;

    private final List<Principal> accessControlList;

    private final String contentType;

    private Document(BuilderImpl builder) {
        this.id = builder.id;
        this.title = builder.title;
        this.blob = builder.blob;
        this.s3Path = builder.s3Path;
        this.attributes = builder.attributes;
        this.accessControlList = builder.accessControlList;
        this.contentType = builder.contentType;
    }

    /**
     * <p>
     * A unique identifier of the document in the index.
     * </p>
     * 
     * @return A unique identifier of the document in the index.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The title of the document.
     * </p>
     * 
     * @return The title of the document.
     */
    public String title() {
        return title;
    }

    /**
     * <p>
     * The contents of the document.
     * </p>
     * <p>
     * Documents passed to the <code>Blob</code> parameter must be base64 encoded. Your code might not need to encode
     * the document file bytes if you're using an AWS SDK to call Amazon Kendra operations. If you are calling the
     * Amazon Kendra endpoint directly using REST, you must base64 encode the contents before sending.
     * </p>
     * 
     * @return The contents of the document. </p>
     *         <p>
     *         Documents passed to the <code>Blob</code> parameter must be base64 encoded. Your code might not need to
     *         encode the document file bytes if you're using an AWS SDK to call Amazon Kendra operations. If you are
     *         calling the Amazon Kendra endpoint directly using REST, you must base64 encode the contents before
     *         sending.
     */
    public SdkBytes blob() {
        return blob;
    }

    /**
     * Returns the value of the S3Path property for this object.
     * 
     * @return The value of the S3Path property for this object.
     */
    public S3Path s3Path() {
        return s3Path;
    }

    /**
     * Returns true if the Attributes property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Custom attributes to apply to the document. Use the custom attributes to provide additional information for
     * searching, to provide facets for refining searches, and to provide additional information in the query response.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Custom attributes to apply to the document. Use the custom attributes to provide additional information
     *         for searching, to provide facets for refining searches, and to provide additional information in the
     *         query response.
     */
    public List<DocumentAttribute> attributes() {
        return attributes;
    }

    /**
     * Returns true if the AccessControlList property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAccessControlList() {
        return accessControlList != null && !(accessControlList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information to use for user context filtering.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAccessControlList()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Information to use for user context filtering.
     */
    public List<Principal> accessControlList() {
        return accessControlList;
    }

    /**
     * <p>
     * The file type of the document in the <code>Blob</code> field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentType} will
     * return {@link ContentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #contentTypeAsString}.
     * </p>
     * 
     * @return The file type of the document in the <code>Blob</code> field.
     * @see ContentType
     */
    public ContentType contentType() {
        return ContentType.fromValue(contentType);
    }

    /**
     * <p>
     * The file type of the document in the <code>Blob</code> field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentType} will
     * return {@link ContentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #contentTypeAsString}.
     * </p>
     * 
     * @return The file type of the document in the <code>Blob</code> field.
     * @see ContentType
     */
    public String contentTypeAsString() {
        return contentType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(blob());
        hashCode = 31 * hashCode + Objects.hashCode(s3Path());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAccessControlList() ? accessControlList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(contentTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Document)) {
            return false;
        }
        Document other = (Document) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(title(), other.title()) && Objects.equals(blob(), other.blob())
                && Objects.equals(s3Path(), other.s3Path()) && hasAttributes() == other.hasAttributes()
                && Objects.equals(attributes(), other.attributes()) && hasAccessControlList() == other.hasAccessControlList()
                && Objects.equals(accessControlList(), other.accessControlList())
                && Objects.equals(contentTypeAsString(), other.contentTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Document").add("Id", id()).add("Title", title()).add("Blob", blob()).add("S3Path", s3Path())
                .add("Attributes", hasAttributes() ? attributes() : null)
                .add("AccessControlList", hasAccessControlList() ? accessControlList() : null)
                .add("ContentType", contentTypeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Blob":
            return Optional.ofNullable(clazz.cast(blob()));
        case "S3Path":
            return Optional.ofNullable(clazz.cast(s3Path()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "AccessControlList":
            return Optional.ofNullable(clazz.cast(accessControlList()));
        case "ContentType":
            return Optional.ofNullable(clazz.cast(contentTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Document, T> g) {
        return obj -> g.apply((Document) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Document> {
        /**
         * <p>
         * A unique identifier of the document in the index.
         * </p>
         * 
         * @param id
         *        A unique identifier of the document in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The title of the document.
         * </p>
         * 
         * @param title
         *        The title of the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The contents of the document.
         * </p>
         * <p>
         * Documents passed to the <code>Blob</code> parameter must be base64 encoded. Your code might not need to
         * encode the document file bytes if you're using an AWS SDK to call Amazon Kendra operations. If you are
         * calling the Amazon Kendra endpoint directly using REST, you must base64 encode the contents before sending.
         * </p>
         * 
         * @param blob
         *        The contents of the document. </p>
         *        <p>
         *        Documents passed to the <code>Blob</code> parameter must be base64 encoded. Your code might not need
         *        to encode the document file bytes if you're using an AWS SDK to call Amazon Kendra operations. If you
         *        are calling the Amazon Kendra endpoint directly using REST, you must base64 encode the contents before
         *        sending.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blob(SdkBytes blob);

        /**
         * Sets the value of the S3Path property for this object.
         *
         * @param s3Path
         *        The new value for the S3Path property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Path(S3Path s3Path);

        /**
         * Sets the value of the S3Path property for this object.
         *
         * This is a convenience that creates an instance of the {@link S3Path.Builder} avoiding the need to create one
         * manually via {@link S3Path#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Path.Builder#build()} is called immediately and its result is
         * passed to {@link #s3Path(S3Path)}.
         * 
         * @param s3Path
         *        a consumer that will call methods on {@link S3Path.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Path(S3Path)
         */
        default Builder s3Path(Consumer<S3Path.Builder> s3Path) {
            return s3Path(S3Path.builder().applyMutation(s3Path).build());
        }

        /**
         * <p>
         * Custom attributes to apply to the document. Use the custom attributes to provide additional information for
         * searching, to provide facets for refining searches, and to provide additional information in the query
         * response.
         * </p>
         * 
         * @param attributes
         *        Custom attributes to apply to the document. Use the custom attributes to provide additional
         *        information for searching, to provide facets for refining searches, and to provide additional
         *        information in the query response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Collection<DocumentAttribute> attributes);

        /**
         * <p>
         * Custom attributes to apply to the document. Use the custom attributes to provide additional information for
         * searching, to provide facets for refining searches, and to provide additional information in the query
         * response.
         * </p>
         * 
         * @param attributes
         *        Custom attributes to apply to the document. Use the custom attributes to provide additional
         *        information for searching, to provide facets for refining searches, and to provide additional
         *        information in the query response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(DocumentAttribute... attributes);

        /**
         * <p>
         * Custom attributes to apply to the document. Use the custom attributes to provide additional information for
         * searching, to provide facets for refining searches, and to provide additional information in the query
         * response.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DocumentAttribute>.Builder} avoiding the
         * need to create one manually via {@link List<DocumentAttribute>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DocumentAttribute>.Builder#build()} is called immediately
         * and its result is passed to {@link #attributes(List<DocumentAttribute>)}.
         * 
         * @param attributes
         *        a consumer that will call methods on {@link List<DocumentAttribute>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attributes(List<DocumentAttribute>)
         */
        Builder attributes(Consumer<DocumentAttribute.Builder>... attributes);

        /**
         * <p>
         * Information to use for user context filtering.
         * </p>
         * 
         * @param accessControlList
         *        Information to use for user context filtering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlList(Collection<Principal> accessControlList);

        /**
         * <p>
         * Information to use for user context filtering.
         * </p>
         * 
         * @param accessControlList
         *        Information to use for user context filtering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlList(Principal... accessControlList);

        /**
         * <p>
         * Information to use for user context filtering.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Principal>.Builder} avoiding the need to
         * create one manually via {@link List<Principal>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Principal>.Builder#build()} is called immediately and its
         * result is passed to {@link #accessControlList(List<Principal>)}.
         * 
         * @param accessControlList
         *        a consumer that will call methods on {@link List<Principal>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlList(List<Principal>)
         */
        Builder accessControlList(Consumer<Principal.Builder>... accessControlList);

        /**
         * <p>
         * The file type of the document in the <code>Blob</code> field.
         * </p>
         * 
         * @param contentType
         *        The file type of the document in the <code>Blob</code> field.
         * @see ContentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentType
         */
        Builder contentType(String contentType);

        /**
         * <p>
         * The file type of the document in the <code>Blob</code> field.
         * </p>
         * 
         * @param contentType
         *        The file type of the document in the <code>Blob</code> field.
         * @see ContentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentType
         */
        Builder contentType(ContentType contentType);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String title;

        private SdkBytes blob;

        private S3Path s3Path;

        private List<DocumentAttribute> attributes = DefaultSdkAutoConstructList.getInstance();

        private List<Principal> accessControlList = DefaultSdkAutoConstructList.getInstance();

        private String contentType;

        private BuilderImpl() {
        }

        private BuilderImpl(Document model) {
            id(model.id);
            title(model.title);
            blob(model.blob);
            s3Path(model.s3Path);
            attributes(model.attributes);
            accessControlList(model.accessControlList);
            contentType(model.contentType);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getTitle() {
            return title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        public final ByteBuffer getBlob() {
            return blob == null ? null : blob.asByteBuffer();
        }

        @Override
        public final Builder blob(SdkBytes blob) {
            this.blob = StandardMemberCopier.copy(blob);
            return this;
        }

        public final void setBlob(ByteBuffer blob) {
            blob(blob == null ? null : SdkBytes.fromByteBuffer(blob));
        }

        public final S3Path.Builder getS3Path() {
            return s3Path != null ? s3Path.toBuilder() : null;
        }

        @Override
        public final Builder s3Path(S3Path s3Path) {
            this.s3Path = s3Path;
            return this;
        }

        public final void setS3Path(S3Path.BuilderImpl s3Path) {
            this.s3Path = s3Path != null ? s3Path.build() : null;
        }

        public final Collection<DocumentAttribute.Builder> getAttributes() {
            if (attributes instanceof SdkAutoConstructList) {
                return null;
            }
            return attributes != null ? attributes.stream().map(DocumentAttribute::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder attributes(Collection<DocumentAttribute> attributes) {
            this.attributes = DocumentAttributeListCopier.copy(attributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributes(DocumentAttribute... attributes) {
            attributes(Arrays.asList(attributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributes(Consumer<DocumentAttribute.Builder>... attributes) {
            attributes(Stream.of(attributes).map(c -> DocumentAttribute.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAttributes(Collection<DocumentAttribute.BuilderImpl> attributes) {
            this.attributes = DocumentAttributeListCopier.copyFromBuilder(attributes);
        }

        public final Collection<Principal.Builder> getAccessControlList() {
            if (accessControlList instanceof SdkAutoConstructList) {
                return null;
            }
            return accessControlList != null ? accessControlList.stream().map(Principal::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder accessControlList(Collection<Principal> accessControlList) {
            this.accessControlList = PrincipalListCopier.copy(accessControlList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessControlList(Principal... accessControlList) {
            accessControlList(Arrays.asList(accessControlList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessControlList(Consumer<Principal.Builder>... accessControlList) {
            accessControlList(Stream.of(accessControlList).map(c -> Principal.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAccessControlList(Collection<Principal.BuilderImpl> accessControlList) {
            this.accessControlList = PrincipalListCopier.copyFromBuilder(accessControlList);
        }

        public final String getContentType() {
            return contentType;
        }

        @Override
        public final Builder contentType(String contentType) {
            this.contentType = contentType;
            return this;
        }

        @Override
        public final Builder contentType(ContentType contentType) {
            this.contentType(contentType == null ? null : contentType.toString());
            return this;
        }

        public final void setContentType(String contentType) {
            this.contentType = contentType;
        }

        @Override
        public Document build() {
            return new Document(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
