/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about the user context for a Amazon Kendra index.
 * </p>
 * <p>
 * This is used for filtering search results for different users based on their access to documents.
 * </p>
 * <p>
 * You provide one of the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * User token
 * </p>
 * </li>
 * <li>
 * <p>
 * User ID, the groups the user belongs to, and the data sources the groups can access
 * </p>
 * </li>
 * </ul>
 * <p>
 * If you provide both, an exception is thrown.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UserContext implements SdkPojo, Serializable, ToCopyableBuilder<UserContext.Builder, UserContext> {
    private static final SdkField<String> TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Token")
            .getter(getter(UserContext::token)).setter(setter(Builder::token))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Token").build()).build();

    private static final SdkField<String> USER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("UserId")
            .getter(getter(UserContext::userId)).setter(setter(Builder::userId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserId").build()).build();

    private static final SdkField<List<String>> GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Groups")
            .getter(getter(UserContext::groups))
            .setter(setter(Builder::groups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Groups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceGroup>> DATA_SOURCE_GROUPS_FIELD = SdkField
            .<List<DataSourceGroup>> builder(MarshallingType.LIST)
            .memberName("DataSourceGroups")
            .getter(getter(UserContext::dataSourceGroups))
            .setter(setter(Builder::dataSourceGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSourceGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TOKEN_FIELD, USER_ID_FIELD,
            GROUPS_FIELD, DATA_SOURCE_GROUPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String token;

    private final String userId;

    private final List<String> groups;

    private final List<DataSourceGroup> dataSourceGroups;

    private UserContext(BuilderImpl builder) {
        this.token = builder.token;
        this.userId = builder.userId;
        this.groups = builder.groups;
        this.dataSourceGroups = builder.dataSourceGroups;
    }

    /**
     * <p>
     * The user context token for filtering search results for a user. It must be a JWT or a JSON token.
     * </p>
     * 
     * @return The user context token for filtering search results for a user. It must be a JWT or a JSON token.
     */
    public final String token() {
        return token;
    }

    /**
     * <p>
     * The identifier of the user you want to filter search results based on their access to documents.
     * </p>
     * 
     * @return The identifier of the user you want to filter search results based on their access to documents.
     */
    public final String userId() {
        return userId;
    }

    /**
     * Returns true if the Groups property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasGroups() {
        return groups != null && !(groups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of groups you want to filter search results based on the groups' access to documents.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of groups you want to filter search results based on the groups' access to documents.
     */
    public final List<String> groups() {
        return groups;
    }

    /**
     * Returns true if the DataSourceGroups property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasDataSourceGroups() {
        return dataSourceGroups != null && !(dataSourceGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of data source groups you want to filter search results based on groups' access to documents in that
     * data source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDataSourceGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The list of data source groups you want to filter search results based on groups' access to documents in
     *         that data source.
     */
    public final List<DataSourceGroup> dataSourceGroups() {
        return dataSourceGroups;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(token());
        hashCode = 31 * hashCode + Objects.hashCode(userId());
        hashCode = 31 * hashCode + Objects.hashCode(hasGroups() ? groups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDataSourceGroups() ? dataSourceGroups() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UserContext)) {
            return false;
        }
        UserContext other = (UserContext) obj;
        return Objects.equals(token(), other.token()) && Objects.equals(userId(), other.userId())
                && hasGroups() == other.hasGroups() && Objects.equals(groups(), other.groups())
                && hasDataSourceGroups() == other.hasDataSourceGroups()
                && Objects.equals(dataSourceGroups(), other.dataSourceGroups());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UserContext").add("Token", token()).add("UserId", userId())
                .add("Groups", hasGroups() ? groups() : null)
                .add("DataSourceGroups", hasDataSourceGroups() ? dataSourceGroups() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Token":
            return Optional.ofNullable(clazz.cast(token()));
        case "UserId":
            return Optional.ofNullable(clazz.cast(userId()));
        case "Groups":
            return Optional.ofNullable(clazz.cast(groups()));
        case "DataSourceGroups":
            return Optional.ofNullable(clazz.cast(dataSourceGroups()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UserContext, T> g) {
        return obj -> g.apply((UserContext) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UserContext> {
        /**
         * <p>
         * The user context token for filtering search results for a user. It must be a JWT or a JSON token.
         * </p>
         * 
         * @param token
         *        The user context token for filtering search results for a user. It must be a JWT or a JSON token.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder token(String token);

        /**
         * <p>
         * The identifier of the user you want to filter search results based on their access to documents.
         * </p>
         * 
         * @param userId
         *        The identifier of the user you want to filter search results based on their access to documents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userId(String userId);

        /**
         * <p>
         * The list of groups you want to filter search results based on the groups' access to documents.
         * </p>
         * 
         * @param groups
         *        The list of groups you want to filter search results based on the groups' access to documents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(Collection<String> groups);

        /**
         * <p>
         * The list of groups you want to filter search results based on the groups' access to documents.
         * </p>
         * 
         * @param groups
         *        The list of groups you want to filter search results based on the groups' access to documents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(String... groups);

        /**
         * <p>
         * The list of data source groups you want to filter search results based on groups' access to documents in that
         * data source.
         * </p>
         * 
         * @param dataSourceGroups
         *        The list of data source groups you want to filter search results based on groups' access to documents
         *        in that data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSourceGroups(Collection<DataSourceGroup> dataSourceGroups);

        /**
         * <p>
         * The list of data source groups you want to filter search results based on groups' access to documents in that
         * data source.
         * </p>
         * 
         * @param dataSourceGroups
         *        The list of data source groups you want to filter search results based on groups' access to documents
         *        in that data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSourceGroups(DataSourceGroup... dataSourceGroups);

        /**
         * <p>
         * The list of data source groups you want to filter search results based on groups' access to documents in that
         * data source.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DataSourceGroup>.Builder} avoiding the need
         * to create one manually via {@link List<DataSourceGroup>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DataSourceGroup>.Builder#build()} is called immediately and
         * its result is passed to {@link #dataSourceGroups(List<DataSourceGroup>)}.
         * 
         * @param dataSourceGroups
         *        a consumer that will call methods on {@link List<DataSourceGroup>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataSourceGroups(List<DataSourceGroup>)
         */
        Builder dataSourceGroups(Consumer<DataSourceGroup.Builder>... dataSourceGroups);
    }

    static final class BuilderImpl implements Builder {
        private String token;

        private String userId;

        private List<String> groups = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceGroup> dataSourceGroups = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UserContext model) {
            token(model.token);
            userId(model.userId);
            groups(model.groups);
            dataSourceGroups(model.dataSourceGroups);
        }

        public final String getToken() {
            return token;
        }

        public final void setToken(String token) {
            this.token = token;
        }

        @Override
        @Transient
        public final Builder token(String token) {
            this.token = token;
            return this;
        }

        public final String getUserId() {
            return userId;
        }

        public final void setUserId(String userId) {
            this.userId = userId;
        }

        @Override
        @Transient
        public final Builder userId(String userId) {
            this.userId = userId;
            return this;
        }

        public final Collection<String> getGroups() {
            if (groups instanceof SdkAutoConstructList) {
                return null;
            }
            return groups;
        }

        public final void setGroups(Collection<String> groups) {
            this.groups = GroupsCopier.copy(groups);
        }

        @Override
        @Transient
        public final Builder groups(Collection<String> groups) {
            this.groups = GroupsCopier.copy(groups);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder groups(String... groups) {
            groups(Arrays.asList(groups));
            return this;
        }

        public final List<DataSourceGroup.Builder> getDataSourceGroups() {
            List<DataSourceGroup.Builder> result = DataSourceGroupsCopier.copyToBuilder(this.dataSourceGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDataSourceGroups(Collection<DataSourceGroup.BuilderImpl> dataSourceGroups) {
            this.dataSourceGroups = DataSourceGroupsCopier.copyFromBuilder(dataSourceGroups);
        }

        @Override
        @Transient
        public final Builder dataSourceGroups(Collection<DataSourceGroup> dataSourceGroups) {
            this.dataSourceGroups = DataSourceGroupsCopier.copy(dataSourceGroups);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder dataSourceGroups(DataSourceGroup... dataSourceGroups) {
            dataSourceGroups(Arrays.asList(dataSourceGroups));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder dataSourceGroups(Consumer<DataSourceGroup.Builder>... dataSourceGroups) {
            dataSourceGroups(Stream.of(dataSourceGroups).map(c -> DataSourceGroup.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public UserContext build() {
            return new UserContext(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
