/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.beans.Transient;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A document in an index.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Document implements SdkPojo, Serializable, ToCopyableBuilder<Document.Builder, Document> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Document::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(Document::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<SdkBytes> BLOB_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("Blob").getter(getter(Document::blob)).setter(setter(Builder::blob))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Blob").build()).build();

    private static final SdkField<S3Path> S3_PATH_FIELD = SdkField.<S3Path> builder(MarshallingType.SDK_POJO)
            .memberName("S3Path").getter(getter(Document::s3Path)).setter(setter(Builder::s3Path)).constructor(S3Path::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Path").build()).build();

    private static final SdkField<List<DocumentAttribute>> ATTRIBUTES_FIELD = SdkField
            .<List<DocumentAttribute>> builder(MarshallingType.LIST)
            .memberName("Attributes")
            .getter(getter(Document::attributes))
            .setter(setter(Builder::attributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Attributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentAttribute> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentAttribute::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Principal>> ACCESS_CONTROL_LIST_FIELD = SdkField
            .<List<Principal>> builder(MarshallingType.LIST)
            .memberName("AccessControlList")
            .getter(getter(Document::accessControlList))
            .setter(setter(Builder::accessControlList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Principal> builder(MarshallingType.SDK_POJO)
                                            .constructor(Principal::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<HierarchicalPrincipal>> HIERARCHICAL_ACCESS_CONTROL_LIST_FIELD = SdkField
            .<List<HierarchicalPrincipal>> builder(MarshallingType.LIST)
            .memberName("HierarchicalAccessControlList")
            .getter(getter(Document::hierarchicalAccessControlList))
            .setter(setter(Builder::hierarchicalAccessControlList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HierarchicalAccessControlList")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HierarchicalPrincipal> builder(MarshallingType.SDK_POJO)
                                            .constructor(HierarchicalPrincipal::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONTENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentType").getter(getter(Document::contentTypeAsString)).setter(setter(Builder::contentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, TITLE_FIELD,
            BLOB_FIELD, S3_PATH_FIELD, ATTRIBUTES_FIELD, ACCESS_CONTROL_LIST_FIELD, HIERARCHICAL_ACCESS_CONTROL_LIST_FIELD,
            CONTENT_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String title;

    private final SdkBytes blob;

    private final S3Path s3Path;

    private final List<DocumentAttribute> attributes;

    private final List<Principal> accessControlList;

    private final List<HierarchicalPrincipal> hierarchicalAccessControlList;

    private final String contentType;

    private Document(BuilderImpl builder) {
        this.id = builder.id;
        this.title = builder.title;
        this.blob = builder.blob;
        this.s3Path = builder.s3Path;
        this.attributes = builder.attributes;
        this.accessControlList = builder.accessControlList;
        this.hierarchicalAccessControlList = builder.hierarchicalAccessControlList;
        this.contentType = builder.contentType;
    }

    /**
     * <p>
     * A unique identifier of the document in the index.
     * </p>
     * 
     * @return A unique identifier of the document in the index.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The title of the document.
     * </p>
     * 
     * @return The title of the document.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The contents of the document.
     * </p>
     * <p>
     * Documents passed to the <code>Blob</code> parameter must be base64 encoded. Your code might not need to encode
     * the document file bytes if you're using an Amazon Web Services SDK to call Amazon Kendra APIs. If you are calling
     * the Amazon Kendra endpoint directly using REST, you must base64 encode the contents before sending.
     * </p>
     * 
     * @return The contents of the document. </p>
     *         <p>
     *         Documents passed to the <code>Blob</code> parameter must be base64 encoded. Your code might not need to
     *         encode the document file bytes if you're using an Amazon Web Services SDK to call Amazon Kendra APIs. If
     *         you are calling the Amazon Kendra endpoint directly using REST, you must base64 encode the contents
     *         before sending.
     */
    public final SdkBytes blob() {
        return blob;
    }

    /**
     * Returns the value of the S3Path property for this object.
     * 
     * @return The value of the S3Path property for this object.
     */
    public final S3Path s3Path() {
        return s3Path;
    }

    /**
     * For responses, this returns true if the service returned a value for the Attributes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAttributes() {
        return attributes != null && !(attributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Custom attributes to apply to the document. Use the custom attributes to provide additional information for
     * searching, to provide facets for refining searches, and to provide additional information in the query response.
     * </p>
     * <p>
     * For example, 'DataSourceId' and 'DataSourceSyncJobId' are custom attributes that provide information on the
     * synchronization of documents running on a data source. Note, 'DataSourceSyncJobId' could be an optional custom
     * attribute as Amazon Kendra will use the ID of a running sync job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributes} method.
     * </p>
     * 
     * @return Custom attributes to apply to the document. Use the custom attributes to provide additional information
     *         for searching, to provide facets for refining searches, and to provide additional information in the
     *         query response.</p>
     *         <p>
     *         For example, 'DataSourceId' and 'DataSourceSyncJobId' are custom attributes that provide information on
     *         the synchronization of documents running on a data source. Note, 'DataSourceSyncJobId' could be an
     *         optional custom attribute as Amazon Kendra will use the ID of a running sync job.
     */
    public final List<DocumentAttribute> attributes() {
        return attributes;
    }

    /**
     * For responses, this returns true if the service returned a value for the AccessControlList property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAccessControlList() {
        return accessControlList != null && !(accessControlList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information on user and group access rights, which is used for user context filtering.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccessControlList} method.
     * </p>
     * 
     * @return Information on user and group access rights, which is used for user context filtering.
     */
    public final List<Principal> accessControlList() {
        return accessControlList;
    }

    /**
     * For responses, this returns true if the service returned a value for the HierarchicalAccessControlList property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasHierarchicalAccessControlList() {
        return hierarchicalAccessControlList != null && !(hierarchicalAccessControlList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Principal.html">principal</a> lists that
     * define the hierarchy for which documents users should have access to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHierarchicalAccessControlList} method.
     * </p>
     * 
     * @return The list of <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Principal.html">principal</a> lists
     *         that define the hierarchy for which documents users should have access to.
     */
    public final List<HierarchicalPrincipal> hierarchicalAccessControlList() {
        return hierarchicalAccessControlList;
    }

    /**
     * <p>
     * The file type of the document in the <code>Blob</code> field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentType} will
     * return {@link ContentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #contentTypeAsString}.
     * </p>
     * 
     * @return The file type of the document in the <code>Blob</code> field.
     * @see ContentType
     */
    public final ContentType contentType() {
        return ContentType.fromValue(contentType);
    }

    /**
     * <p>
     * The file type of the document in the <code>Blob</code> field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #contentType} will
     * return {@link ContentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #contentTypeAsString}.
     * </p>
     * 
     * @return The file type of the document in the <code>Blob</code> field.
     * @see ContentType
     */
    public final String contentTypeAsString() {
        return contentType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(blob());
        hashCode = 31 * hashCode + Objects.hashCode(s3Path());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributes() ? attributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAccessControlList() ? accessControlList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasHierarchicalAccessControlList() ? hierarchicalAccessControlList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(contentTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Document)) {
            return false;
        }
        Document other = (Document) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(title(), other.title()) && Objects.equals(blob(), other.blob())
                && Objects.equals(s3Path(), other.s3Path()) && hasAttributes() == other.hasAttributes()
                && Objects.equals(attributes(), other.attributes()) && hasAccessControlList() == other.hasAccessControlList()
                && Objects.equals(accessControlList(), other.accessControlList())
                && hasHierarchicalAccessControlList() == other.hasHierarchicalAccessControlList()
                && Objects.equals(hierarchicalAccessControlList(), other.hierarchicalAccessControlList())
                && Objects.equals(contentTypeAsString(), other.contentTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("Document")
                .add("Id", id())
                .add("Title", title())
                .add("Blob", blob())
                .add("S3Path", s3Path())
                .add("Attributes", hasAttributes() ? attributes() : null)
                .add("AccessControlList", hasAccessControlList() ? accessControlList() : null)
                .add("HierarchicalAccessControlList", hasHierarchicalAccessControlList() ? hierarchicalAccessControlList() : null)
                .add("ContentType", contentTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Blob":
            return Optional.ofNullable(clazz.cast(blob()));
        case "S3Path":
            return Optional.ofNullable(clazz.cast(s3Path()));
        case "Attributes":
            return Optional.ofNullable(clazz.cast(attributes()));
        case "AccessControlList":
            return Optional.ofNullable(clazz.cast(accessControlList()));
        case "HierarchicalAccessControlList":
            return Optional.ofNullable(clazz.cast(hierarchicalAccessControlList()));
        case "ContentType":
            return Optional.ofNullable(clazz.cast(contentTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Document, T> g) {
        return obj -> g.apply((Document) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Document> {
        /**
         * <p>
         * A unique identifier of the document in the index.
         * </p>
         * 
         * @param id
         *        A unique identifier of the document in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The title of the document.
         * </p>
         * 
         * @param title
         *        The title of the document.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The contents of the document.
         * </p>
         * <p>
         * Documents passed to the <code>Blob</code> parameter must be base64 encoded. Your code might not need to
         * encode the document file bytes if you're using an Amazon Web Services SDK to call Amazon Kendra APIs. If you
         * are calling the Amazon Kendra endpoint directly using REST, you must base64 encode the contents before
         * sending.
         * </p>
         * 
         * @param blob
         *        The contents of the document. </p>
         *        <p>
         *        Documents passed to the <code>Blob</code> parameter must be base64 encoded. Your code might not need
         *        to encode the document file bytes if you're using an Amazon Web Services SDK to call Amazon Kendra
         *        APIs. If you are calling the Amazon Kendra endpoint directly using REST, you must base64 encode the
         *        contents before sending.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blob(SdkBytes blob);

        /**
         * Sets the value of the S3Path property for this object.
         *
         * @param s3Path
         *        The new value for the S3Path property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Path(S3Path s3Path);

        /**
         * Sets the value of the S3Path property for this object.
         *
         * This is a convenience that creates an instance of the {@link S3Path.Builder} avoiding the need to create one
         * manually via {@link S3Path#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Path.Builder#build()} is called immediately and its result is
         * passed to {@link #s3Path(S3Path)}.
         * 
         * @param s3Path
         *        a consumer that will call methods on {@link S3Path.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Path(S3Path)
         */
        default Builder s3Path(Consumer<S3Path.Builder> s3Path) {
            return s3Path(S3Path.builder().applyMutation(s3Path).build());
        }

        /**
         * <p>
         * Custom attributes to apply to the document. Use the custom attributes to provide additional information for
         * searching, to provide facets for refining searches, and to provide additional information in the query
         * response.
         * </p>
         * <p>
         * For example, 'DataSourceId' and 'DataSourceSyncJobId' are custom attributes that provide information on the
         * synchronization of documents running on a data source. Note, 'DataSourceSyncJobId' could be an optional
         * custom attribute as Amazon Kendra will use the ID of a running sync job.
         * </p>
         * 
         * @param attributes
         *        Custom attributes to apply to the document. Use the custom attributes to provide additional
         *        information for searching, to provide facets for refining searches, and to provide additional
         *        information in the query response.</p>
         *        <p>
         *        For example, 'DataSourceId' and 'DataSourceSyncJobId' are custom attributes that provide information
         *        on the synchronization of documents running on a data source. Note, 'DataSourceSyncJobId' could be an
         *        optional custom attribute as Amazon Kendra will use the ID of a running sync job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(Collection<DocumentAttribute> attributes);

        /**
         * <p>
         * Custom attributes to apply to the document. Use the custom attributes to provide additional information for
         * searching, to provide facets for refining searches, and to provide additional information in the query
         * response.
         * </p>
         * <p>
         * For example, 'DataSourceId' and 'DataSourceSyncJobId' are custom attributes that provide information on the
         * synchronization of documents running on a data source. Note, 'DataSourceSyncJobId' could be an optional
         * custom attribute as Amazon Kendra will use the ID of a running sync job.
         * </p>
         * 
         * @param attributes
         *        Custom attributes to apply to the document. Use the custom attributes to provide additional
         *        information for searching, to provide facets for refining searches, and to provide additional
         *        information in the query response.</p>
         *        <p>
         *        For example, 'DataSourceId' and 'DataSourceSyncJobId' are custom attributes that provide information
         *        on the synchronization of documents running on a data source. Note, 'DataSourceSyncJobId' could be an
         *        optional custom attribute as Amazon Kendra will use the ID of a running sync job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributes(DocumentAttribute... attributes);

        /**
         * <p>
         * Custom attributes to apply to the document. Use the custom attributes to provide additional information for
         * searching, to provide facets for refining searches, and to provide additional information in the query
         * response.
         * </p>
         * <p>
         * For example, 'DataSourceId' and 'DataSourceSyncJobId' are custom attributes that provide information on the
         * synchronization of documents running on a data source. Note, 'DataSourceSyncJobId' could be an optional
         * custom attribute as Amazon Kendra will use the ID of a running sync job.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DocumentAttribute>.Builder} avoiding the
         * need to create one manually via {@link List<DocumentAttribute>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DocumentAttribute>.Builder#build()} is called immediately
         * and its result is passed to {@link #attributes(List<DocumentAttribute>)}.
         * 
         * @param attributes
         *        a consumer that will call methods on {@link List<DocumentAttribute>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attributes(List<DocumentAttribute>)
         */
        Builder attributes(Consumer<DocumentAttribute.Builder>... attributes);

        /**
         * <p>
         * Information on user and group access rights, which is used for user context filtering.
         * </p>
         * 
         * @param accessControlList
         *        Information on user and group access rights, which is used for user context filtering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlList(Collection<Principal> accessControlList);

        /**
         * <p>
         * Information on user and group access rights, which is used for user context filtering.
         * </p>
         * 
         * @param accessControlList
         *        Information on user and group access rights, which is used for user context filtering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlList(Principal... accessControlList);

        /**
         * <p>
         * Information on user and group access rights, which is used for user context filtering.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Principal>.Builder} avoiding the need to
         * create one manually via {@link List<Principal>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Principal>.Builder#build()} is called immediately and its
         * result is passed to {@link #accessControlList(List<Principal>)}.
         * 
         * @param accessControlList
         *        a consumer that will call methods on {@link List<Principal>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlList(List<Principal>)
         */
        Builder accessControlList(Consumer<Principal.Builder>... accessControlList);

        /**
         * <p>
         * The list of <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Principal.html">principal</a> lists
         * that define the hierarchy for which documents users should have access to.
         * </p>
         * 
         * @param hierarchicalAccessControlList
         *        The list of <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Principal.html">principal</a>
         *        lists that define the hierarchy for which documents users should have access to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hierarchicalAccessControlList(Collection<HierarchicalPrincipal> hierarchicalAccessControlList);

        /**
         * <p>
         * The list of <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Principal.html">principal</a> lists
         * that define the hierarchy for which documents users should have access to.
         * </p>
         * 
         * @param hierarchicalAccessControlList
         *        The list of <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Principal.html">principal</a>
         *        lists that define the hierarchy for which documents users should have access to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hierarchicalAccessControlList(HierarchicalPrincipal... hierarchicalAccessControlList);

        /**
         * <p>
         * The list of <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Principal.html">principal</a> lists
         * that define the hierarchy for which documents users should have access to.
         * </p>
         * This is a convenience that creates an instance of the {@link List<HierarchicalPrincipal>.Builder} avoiding
         * the need to create one manually via {@link List<HierarchicalPrincipal>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HierarchicalPrincipal>.Builder#build()} is called
         * immediately and its result is passed to {@link #hierarchicalAccessControlList(List<HierarchicalPrincipal>)}.
         * 
         * @param hierarchicalAccessControlList
         *        a consumer that will call methods on {@link List<HierarchicalPrincipal>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hierarchicalAccessControlList(List<HierarchicalPrincipal>)
         */
        Builder hierarchicalAccessControlList(Consumer<HierarchicalPrincipal.Builder>... hierarchicalAccessControlList);

        /**
         * <p>
         * The file type of the document in the <code>Blob</code> field.
         * </p>
         * 
         * @param contentType
         *        The file type of the document in the <code>Blob</code> field.
         * @see ContentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentType
         */
        Builder contentType(String contentType);

        /**
         * <p>
         * The file type of the document in the <code>Blob</code> field.
         * </p>
         * 
         * @param contentType
         *        The file type of the document in the <code>Blob</code> field.
         * @see ContentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContentType
         */
        Builder contentType(ContentType contentType);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String title;

        private SdkBytes blob;

        private S3Path s3Path;

        private List<DocumentAttribute> attributes = DefaultSdkAutoConstructList.getInstance();

        private List<Principal> accessControlList = DefaultSdkAutoConstructList.getInstance();

        private List<HierarchicalPrincipal> hierarchicalAccessControlList = DefaultSdkAutoConstructList.getInstance();

        private String contentType;

        private BuilderImpl() {
        }

        private BuilderImpl(Document model) {
            id(model.id);
            title(model.title);
            blob(model.blob);
            s3Path(model.s3Path);
            attributes(model.attributes);
            accessControlList(model.accessControlList);
            hierarchicalAccessControlList(model.hierarchicalAccessControlList);
            contentType(model.contentType);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        @Transient
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        @Transient
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final ByteBuffer getBlob() {
            return blob == null ? null : blob.asByteBuffer();
        }

        public final void setBlob(ByteBuffer blob) {
            blob(blob == null ? null : SdkBytes.fromByteBuffer(blob));
        }

        @Override
        @Transient
        public final Builder blob(SdkBytes blob) {
            this.blob = blob;
            return this;
        }

        public final S3Path.Builder getS3Path() {
            return s3Path != null ? s3Path.toBuilder() : null;
        }

        public final void setS3Path(S3Path.BuilderImpl s3Path) {
            this.s3Path = s3Path != null ? s3Path.build() : null;
        }

        @Override
        @Transient
        public final Builder s3Path(S3Path s3Path) {
            this.s3Path = s3Path;
            return this;
        }

        public final List<DocumentAttribute.Builder> getAttributes() {
            List<DocumentAttribute.Builder> result = DocumentAttributeListCopier.copyToBuilder(this.attributes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttributes(Collection<DocumentAttribute.BuilderImpl> attributes) {
            this.attributes = DocumentAttributeListCopier.copyFromBuilder(attributes);
        }

        @Override
        @Transient
        public final Builder attributes(Collection<DocumentAttribute> attributes) {
            this.attributes = DocumentAttributeListCopier.copy(attributes);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder attributes(DocumentAttribute... attributes) {
            attributes(Arrays.asList(attributes));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder attributes(Consumer<DocumentAttribute.Builder>... attributes) {
            attributes(Stream.of(attributes).map(c -> DocumentAttribute.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Principal.Builder> getAccessControlList() {
            List<Principal.Builder> result = PrincipalListCopier.copyToBuilder(this.accessControlList);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAccessControlList(Collection<Principal.BuilderImpl> accessControlList) {
            this.accessControlList = PrincipalListCopier.copyFromBuilder(accessControlList);
        }

        @Override
        @Transient
        public final Builder accessControlList(Collection<Principal> accessControlList) {
            this.accessControlList = PrincipalListCopier.copy(accessControlList);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder accessControlList(Principal... accessControlList) {
            accessControlList(Arrays.asList(accessControlList));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder accessControlList(Consumer<Principal.Builder>... accessControlList) {
            accessControlList(Stream.of(accessControlList).map(c -> Principal.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<HierarchicalPrincipal.Builder> getHierarchicalAccessControlList() {
            List<HierarchicalPrincipal.Builder> result = HierarchicalPrincipalListCopier
                    .copyToBuilder(this.hierarchicalAccessControlList);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setHierarchicalAccessControlList(
                Collection<HierarchicalPrincipal.BuilderImpl> hierarchicalAccessControlList) {
            this.hierarchicalAccessControlList = HierarchicalPrincipalListCopier.copyFromBuilder(hierarchicalAccessControlList);
        }

        @Override
        @Transient
        public final Builder hierarchicalAccessControlList(Collection<HierarchicalPrincipal> hierarchicalAccessControlList) {
            this.hierarchicalAccessControlList = HierarchicalPrincipalListCopier.copy(hierarchicalAccessControlList);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder hierarchicalAccessControlList(HierarchicalPrincipal... hierarchicalAccessControlList) {
            hierarchicalAccessControlList(Arrays.asList(hierarchicalAccessControlList));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder hierarchicalAccessControlList(
                Consumer<HierarchicalPrincipal.Builder>... hierarchicalAccessControlList) {
            hierarchicalAccessControlList(Stream.of(hierarchicalAccessControlList)
                    .map(c -> HierarchicalPrincipal.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getContentType() {
            return contentType;
        }

        public final void setContentType(String contentType) {
            this.contentType = contentType;
        }

        @Override
        @Transient
        public final Builder contentType(String contentType) {
            this.contentType = contentType;
            return this;
        }

        @Override
        @Transient
        public final Builder contentType(ContentType contentType) {
            this.contentType(contentType == null ? null : contentType.toString());
            return this;
        }

        @Override
        public Document build() {
            return new Document(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
