/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the configuration information to connect to Confluence as your data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConfluenceConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ConfluenceConfiguration.Builder, ConfluenceConfiguration> {
    private static final SdkField<String> SERVER_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerUrl").getter(getter(ConfluenceConfiguration::serverUrl)).setter(setter(Builder::serverUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerUrl").build()).build();

    private static final SdkField<String> SECRET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretArn").getter(getter(ConfluenceConfiguration::secretArn)).setter(setter(Builder::secretArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretArn").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(ConfluenceConfiguration::versionAsString)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<ConfluenceSpaceConfiguration> SPACE_CONFIGURATION_FIELD = SdkField
            .<ConfluenceSpaceConfiguration> builder(MarshallingType.SDK_POJO).memberName("SpaceConfiguration")
            .getter(getter(ConfluenceConfiguration::spaceConfiguration)).setter(setter(Builder::spaceConfiguration))
            .constructor(ConfluenceSpaceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpaceConfiguration").build())
            .build();

    private static final SdkField<ConfluencePageConfiguration> PAGE_CONFIGURATION_FIELD = SdkField
            .<ConfluencePageConfiguration> builder(MarshallingType.SDK_POJO).memberName("PageConfiguration")
            .getter(getter(ConfluenceConfiguration::pageConfiguration)).setter(setter(Builder::pageConfiguration))
            .constructor(ConfluencePageConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PageConfiguration").build()).build();

    private static final SdkField<ConfluenceBlogConfiguration> BLOG_CONFIGURATION_FIELD = SdkField
            .<ConfluenceBlogConfiguration> builder(MarshallingType.SDK_POJO).memberName("BlogConfiguration")
            .getter(getter(ConfluenceConfiguration::blogConfiguration)).setter(setter(Builder::blogConfiguration))
            .constructor(ConfluenceBlogConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlogConfiguration").build()).build();

    private static final SdkField<ConfluenceAttachmentConfiguration> ATTACHMENT_CONFIGURATION_FIELD = SdkField
            .<ConfluenceAttachmentConfiguration> builder(MarshallingType.SDK_POJO).memberName("AttachmentConfiguration")
            .getter(getter(ConfluenceConfiguration::attachmentConfiguration)).setter(setter(Builder::attachmentConfiguration))
            .constructor(ConfluenceAttachmentConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentConfiguration").build())
            .build();

    private static final SdkField<DataSourceVpcConfiguration> VPC_CONFIGURATION_FIELD = SdkField
            .<DataSourceVpcConfiguration> builder(MarshallingType.SDK_POJO).memberName("VpcConfiguration")
            .getter(getter(ConfluenceConfiguration::vpcConfiguration)).setter(setter(Builder::vpcConfiguration))
            .constructor(DataSourceVpcConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfiguration").build()).build();

    private static final SdkField<List<String>> INCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InclusionPatterns")
            .getter(getter(ConfluenceConfiguration::inclusionPatterns))
            .setter(setter(Builder::inclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExclusionPatterns")
            .getter(getter(ConfluenceConfiguration::exclusionPatterns))
            .setter(setter(Builder::exclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVER_URL_FIELD,
            SECRET_ARN_FIELD, VERSION_FIELD, SPACE_CONFIGURATION_FIELD, PAGE_CONFIGURATION_FIELD, BLOG_CONFIGURATION_FIELD,
            ATTACHMENT_CONFIGURATION_FIELD, VPC_CONFIGURATION_FIELD, INCLUSION_PATTERNS_FIELD, EXCLUSION_PATTERNS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String serverUrl;

    private final String secretArn;

    private final String version;

    private final ConfluenceSpaceConfiguration spaceConfiguration;

    private final ConfluencePageConfiguration pageConfiguration;

    private final ConfluenceBlogConfiguration blogConfiguration;

    private final ConfluenceAttachmentConfiguration attachmentConfiguration;

    private final DataSourceVpcConfiguration vpcConfiguration;

    private final List<String> inclusionPatterns;

    private final List<String> exclusionPatterns;

    private ConfluenceConfiguration(BuilderImpl builder) {
        this.serverUrl = builder.serverUrl;
        this.secretArn = builder.secretArn;
        this.version = builder.version;
        this.spaceConfiguration = builder.spaceConfiguration;
        this.pageConfiguration = builder.pageConfiguration;
        this.blogConfiguration = builder.blogConfiguration;
        this.attachmentConfiguration = builder.attachmentConfiguration;
        this.vpcConfiguration = builder.vpcConfiguration;
        this.inclusionPatterns = builder.inclusionPatterns;
        this.exclusionPatterns = builder.exclusionPatterns;
    }

    /**
     * <p>
     * The URL of your Confluence instance. Use the full URL of the server. For example,
     * <i>https://server.example.com:port/</i>. You can also use an IP address, for example,
     * <i>https://192.168.1.113/</i>.
     * </p>
     * 
     * @return The URL of your Confluence instance. Use the full URL of the server. For example,
     *         <i>https://server.example.com:port/</i>. You can also use an IP address, for example,
     *         <i>https://192.168.1.113/</i>.
     */
    public final String serverUrl() {
        return serverUrl;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required to connect
     * to your Confluence server. The secret must contain a JSON structure with the following keys:
     * </p>
     * <ul>
     * <li>
     * <p>
     * username—The user name or email address of a user with administrative privileges for the Confluence server.
     * </p>
     * </li>
     * <li>
     * <p>
     * password—The password associated with the user logging in to the Confluence server.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required to
     *         connect to your Confluence server. The secret must contain a JSON structure with the following keys:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         username—The user name or email address of a user with administrative privileges for the Confluence
     *         server.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         password—The password associated with the user logging in to the Confluence server.
     *         </p>
     *         </li>
     */
    public final String secretArn() {
        return secretArn;
    }

    /**
     * <p>
     * Specifies the version of the Confluence installation that you are connecting to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #version} will
     * return {@link ConfluenceVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #versionAsString}.
     * </p>
     * 
     * @return Specifies the version of the Confluence installation that you are connecting to.
     * @see ConfluenceVersion
     */
    public final ConfluenceVersion version() {
        return ConfluenceVersion.fromValue(version);
    }

    /**
     * <p>
     * Specifies the version of the Confluence installation that you are connecting to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #version} will
     * return {@link ConfluenceVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #versionAsString}.
     * </p>
     * 
     * @return Specifies the version of the Confluence installation that you are connecting to.
     * @see ConfluenceVersion
     */
    public final String versionAsString() {
        return version;
    }

    /**
     * <p>
     * Configuration information for indexing Confluence spaces.
     * </p>
     * 
     * @return Configuration information for indexing Confluence spaces.
     */
    public final ConfluenceSpaceConfiguration spaceConfiguration() {
        return spaceConfiguration;
    }

    /**
     * <p>
     * Configuration information for indexing Confluence pages.
     * </p>
     * 
     * @return Configuration information for indexing Confluence pages.
     */
    public final ConfluencePageConfiguration pageConfiguration() {
        return pageConfiguration;
    }

    /**
     * <p>
     * Configuration information for indexing Confluence blogs.
     * </p>
     * 
     * @return Configuration information for indexing Confluence blogs.
     */
    public final ConfluenceBlogConfiguration blogConfiguration() {
        return blogConfiguration;
    }

    /**
     * <p>
     * Configuration information for indexing attachments to Confluence blogs and pages.
     * </p>
     * 
     * @return Configuration information for indexing attachments to Confluence blogs and pages.
     */
    public final ConfluenceAttachmentConfiguration attachmentConfiguration() {
        return attachmentConfiguration;
    }

    /**
     * <p>
     * Configuration information for an Amazon Virtual Private Cloud to connect to your Confluence. For more
     * information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/vpc-configuration.html">Configuring a
     * VPC</a>.
     * </p>
     * 
     * @return Configuration information for an Amazon Virtual Private Cloud to connect to your Confluence. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/vpc-configuration.html">Configuring a VPC</a>.
     */
    public final DataSourceVpcConfiguration vpcConfiguration() {
        return vpcConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the InclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInclusionPatterns() {
        return inclusionPatterns != null && !(inclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to include certain blog posts, pages, spaces, or attachments in your
     * Confluence. Content that matches the patterns are included in the index. Content that doesn't match the patterns
     * is excluded from the index. If content matches both an inclusion and exclusion pattern, the exclusion pattern
     * takes precedence and the content isn't included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to include certain blog posts, pages, spaces, or attachments in
     *         your Confluence. Content that matches the patterns are included in the index. Content that doesn't match
     *         the patterns is excluded from the index. If content matches both an inclusion and exclusion pattern, the
     *         exclusion pattern takes precedence and the content isn't included in the index.
     */
    public final List<String> inclusionPatterns() {
        return inclusionPatterns;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExclusionPatterns() {
        return exclusionPatterns != null && !(exclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * &gt;A list of regular expression patterns to exclude certain blog posts, pages, spaces, or attachments in your
     * Confluence. Content that matches the patterns are excluded from the index. Content that doesn't match the
     * patterns is included in the index. If content matches both an inclusion and exclusion pattern, the exclusion
     * pattern takes precedence and the content isn't included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExclusionPatterns} method.
     * </p>
     * 
     * @return &gt;A list of regular expression patterns to exclude certain blog posts, pages, spaces, or attachments in
     *         your Confluence. Content that matches the patterns are excluded from the index. Content that doesn't
     *         match the patterns is included in the index. If content matches both an inclusion and exclusion pattern,
     *         the exclusion pattern takes precedence and the content isn't included in the index.
     */
    public final List<String> exclusionPatterns() {
        return exclusionPatterns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(serverUrl());
        hashCode = 31 * hashCode + Objects.hashCode(secretArn());
        hashCode = 31 * hashCode + Objects.hashCode(versionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(spaceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(pageConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(blogConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(attachmentConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasInclusionPatterns() ? inclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExclusionPatterns() ? exclusionPatterns() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfluenceConfiguration)) {
            return false;
        }
        ConfluenceConfiguration other = (ConfluenceConfiguration) obj;
        return Objects.equals(serverUrl(), other.serverUrl()) && Objects.equals(secretArn(), other.secretArn())
                && Objects.equals(versionAsString(), other.versionAsString())
                && Objects.equals(spaceConfiguration(), other.spaceConfiguration())
                && Objects.equals(pageConfiguration(), other.pageConfiguration())
                && Objects.equals(blogConfiguration(), other.blogConfiguration())
                && Objects.equals(attachmentConfiguration(), other.attachmentConfiguration())
                && Objects.equals(vpcConfiguration(), other.vpcConfiguration())
                && hasInclusionPatterns() == other.hasInclusionPatterns()
                && Objects.equals(inclusionPatterns(), other.inclusionPatterns())
                && hasExclusionPatterns() == other.hasExclusionPatterns()
                && Objects.equals(exclusionPatterns(), other.exclusionPatterns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConfluenceConfiguration").add("ServerUrl", serverUrl()).add("SecretArn", secretArn())
                .add("Version", versionAsString()).add("SpaceConfiguration", spaceConfiguration())
                .add("PageConfiguration", pageConfiguration()).add("BlogConfiguration", blogConfiguration())
                .add("AttachmentConfiguration", attachmentConfiguration()).add("VpcConfiguration", vpcConfiguration())
                .add("InclusionPatterns", hasInclusionPatterns() ? inclusionPatterns() : null)
                .add("ExclusionPatterns", hasExclusionPatterns() ? exclusionPatterns() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ServerUrl":
            return Optional.ofNullable(clazz.cast(serverUrl()));
        case "SecretArn":
            return Optional.ofNullable(clazz.cast(secretArn()));
        case "Version":
            return Optional.ofNullable(clazz.cast(versionAsString()));
        case "SpaceConfiguration":
            return Optional.ofNullable(clazz.cast(spaceConfiguration()));
        case "PageConfiguration":
            return Optional.ofNullable(clazz.cast(pageConfiguration()));
        case "BlogConfiguration":
            return Optional.ofNullable(clazz.cast(blogConfiguration()));
        case "AttachmentConfiguration":
            return Optional.ofNullable(clazz.cast(attachmentConfiguration()));
        case "VpcConfiguration":
            return Optional.ofNullable(clazz.cast(vpcConfiguration()));
        case "InclusionPatterns":
            return Optional.ofNullable(clazz.cast(inclusionPatterns()));
        case "ExclusionPatterns":
            return Optional.ofNullable(clazz.cast(exclusionPatterns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ConfluenceConfiguration, T> g) {
        return obj -> g.apply((ConfluenceConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConfluenceConfiguration> {
        /**
         * <p>
         * The URL of your Confluence instance. Use the full URL of the server. For example,
         * <i>https://server.example.com:port/</i>. You can also use an IP address, for example,
         * <i>https://192.168.1.113/</i>.
         * </p>
         * 
         * @param serverUrl
         *        The URL of your Confluence instance. Use the full URL of the server. For example,
         *        <i>https://server.example.com:port/</i>. You can also use an IP address, for example,
         *        <i>https://192.168.1.113/</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverUrl(String serverUrl);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required to
         * connect to your Confluence server. The secret must contain a JSON structure with the following keys:
         * </p>
         * <ul>
         * <li>
         * <p>
         * username—The user name or email address of a user with administrative privileges for the Confluence server.
         * </p>
         * </li>
         * <li>
         * <p>
         * password—The password associated with the user logging in to the Confluence server.
         * </p>
         * </li>
         * </ul>
         * 
         * @param secretArn
         *        The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required
         *        to connect to your Confluence server. The secret must contain a JSON structure with the following
         *        keys:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        username—The user name or email address of a user with administrative privileges for the Confluence
         *        server.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        password—The password associated with the user logging in to the Confluence server.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretArn(String secretArn);

        /**
         * <p>
         * Specifies the version of the Confluence installation that you are connecting to.
         * </p>
         * 
         * @param version
         *        Specifies the version of the Confluence installation that you are connecting to.
         * @see ConfluenceVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfluenceVersion
         */
        Builder version(String version);

        /**
         * <p>
         * Specifies the version of the Confluence installation that you are connecting to.
         * </p>
         * 
         * @param version
         *        Specifies the version of the Confluence installation that you are connecting to.
         * @see ConfluenceVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfluenceVersion
         */
        Builder version(ConfluenceVersion version);

        /**
         * <p>
         * Configuration information for indexing Confluence spaces.
         * </p>
         * 
         * @param spaceConfiguration
         *        Configuration information for indexing Confluence spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spaceConfiguration(ConfluenceSpaceConfiguration spaceConfiguration);

        /**
         * <p>
         * Configuration information for indexing Confluence spaces.
         * </p>
         * This is a convenience that creates an instance of the {@link ConfluenceSpaceConfiguration.Builder} avoiding
         * the need to create one manually via {@link ConfluenceSpaceConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConfluenceSpaceConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #spaceConfiguration(ConfluenceSpaceConfiguration)}.
         * 
         * @param spaceConfiguration
         *        a consumer that will call methods on {@link ConfluenceSpaceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #spaceConfiguration(ConfluenceSpaceConfiguration)
         */
        default Builder spaceConfiguration(Consumer<ConfluenceSpaceConfiguration.Builder> spaceConfiguration) {
            return spaceConfiguration(ConfluenceSpaceConfiguration.builder().applyMutation(spaceConfiguration).build());
        }

        /**
         * <p>
         * Configuration information for indexing Confluence pages.
         * </p>
         * 
         * @param pageConfiguration
         *        Configuration information for indexing Confluence pages.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pageConfiguration(ConfluencePageConfiguration pageConfiguration);

        /**
         * <p>
         * Configuration information for indexing Confluence pages.
         * </p>
         * This is a convenience that creates an instance of the {@link ConfluencePageConfiguration.Builder} avoiding
         * the need to create one manually via {@link ConfluencePageConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConfluencePageConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #pageConfiguration(ConfluencePageConfiguration)}.
         * 
         * @param pageConfiguration
         *        a consumer that will call methods on {@link ConfluencePageConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #pageConfiguration(ConfluencePageConfiguration)
         */
        default Builder pageConfiguration(Consumer<ConfluencePageConfiguration.Builder> pageConfiguration) {
            return pageConfiguration(ConfluencePageConfiguration.builder().applyMutation(pageConfiguration).build());
        }

        /**
         * <p>
         * Configuration information for indexing Confluence blogs.
         * </p>
         * 
         * @param blogConfiguration
         *        Configuration information for indexing Confluence blogs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blogConfiguration(ConfluenceBlogConfiguration blogConfiguration);

        /**
         * <p>
         * Configuration information for indexing Confluence blogs.
         * </p>
         * This is a convenience that creates an instance of the {@link ConfluenceBlogConfiguration.Builder} avoiding
         * the need to create one manually via {@link ConfluenceBlogConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConfluenceBlogConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #blogConfiguration(ConfluenceBlogConfiguration)}.
         * 
         * @param blogConfiguration
         *        a consumer that will call methods on {@link ConfluenceBlogConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #blogConfiguration(ConfluenceBlogConfiguration)
         */
        default Builder blogConfiguration(Consumer<ConfluenceBlogConfiguration.Builder> blogConfiguration) {
            return blogConfiguration(ConfluenceBlogConfiguration.builder().applyMutation(blogConfiguration).build());
        }

        /**
         * <p>
         * Configuration information for indexing attachments to Confluence blogs and pages.
         * </p>
         * 
         * @param attachmentConfiguration
         *        Configuration information for indexing attachments to Confluence blogs and pages.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentConfiguration(ConfluenceAttachmentConfiguration attachmentConfiguration);

        /**
         * <p>
         * Configuration information for indexing attachments to Confluence blogs and pages.
         * </p>
         * This is a convenience that creates an instance of the {@link ConfluenceAttachmentConfiguration.Builder}
         * avoiding the need to create one manually via {@link ConfluenceAttachmentConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link ConfluenceAttachmentConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #attachmentConfiguration(ConfluenceAttachmentConfiguration)}.
         * 
         * @param attachmentConfiguration
         *        a consumer that will call methods on {@link ConfluenceAttachmentConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachmentConfiguration(ConfluenceAttachmentConfiguration)
         */
        default Builder attachmentConfiguration(Consumer<ConfluenceAttachmentConfiguration.Builder> attachmentConfiguration) {
            return attachmentConfiguration(ConfluenceAttachmentConfiguration.builder().applyMutation(attachmentConfiguration)
                    .build());
        }

        /**
         * <p>
         * Configuration information for an Amazon Virtual Private Cloud to connect to your Confluence. For more
         * information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/vpc-configuration.html">Configuring a
         * VPC</a>.
         * </p>
         * 
         * @param vpcConfiguration
         *        Configuration information for an Amazon Virtual Private Cloud to connect to your Confluence. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/vpc-configuration.html">Configuring a VPC</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration);

        /**
         * <p>
         * Configuration information for an Amazon Virtual Private Cloud to connect to your Confluence. For more
         * information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/vpc-configuration.html">Configuring a
         * VPC</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link DataSourceVpcConfiguration.Builder} avoiding the
         * need to create one manually via {@link DataSourceVpcConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataSourceVpcConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #vpcConfiguration(DataSourceVpcConfiguration)}.
         * 
         * @param vpcConfiguration
         *        a consumer that will call methods on {@link DataSourceVpcConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfiguration(DataSourceVpcConfiguration)
         */
        default Builder vpcConfiguration(Consumer<DataSourceVpcConfiguration.Builder> vpcConfiguration) {
            return vpcConfiguration(DataSourceVpcConfiguration.builder().applyMutation(vpcConfiguration).build());
        }

        /**
         * <p>
         * A list of regular expression patterns to include certain blog posts, pages, spaces, or attachments in your
         * Confluence. Content that matches the patterns are included in the index. Content that doesn't match the
         * patterns is excluded from the index. If content matches both an inclusion and exclusion pattern, the
         * exclusion pattern takes precedence and the content isn't included in the index.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns to include certain blog posts, pages, spaces, or attachments in
         *        your Confluence. Content that matches the patterns are included in the index. Content that doesn't
         *        match the patterns is excluded from the index. If content matches both an inclusion and exclusion
         *        pattern, the exclusion pattern takes precedence and the content isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(Collection<String> inclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to include certain blog posts, pages, spaces, or attachments in your
         * Confluence. Content that matches the patterns are included in the index. Content that doesn't match the
         * patterns is excluded from the index. If content matches both an inclusion and exclusion pattern, the
         * exclusion pattern takes precedence and the content isn't included in the index.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns to include certain blog posts, pages, spaces, or attachments in
         *        your Confluence. Content that matches the patterns are included in the index. Content that doesn't
         *        match the patterns is excluded from the index. If content matches both an inclusion and exclusion
         *        pattern, the exclusion pattern takes precedence and the content isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(String... inclusionPatterns);

        /**
         * <p>
         * &gt;A list of regular expression patterns to exclude certain blog posts, pages, spaces, or attachments in
         * your Confluence. Content that matches the patterns are excluded from the index. Content that doesn't match
         * the patterns is included in the index. If content matches both an inclusion and exclusion pattern, the
         * exclusion pattern takes precedence and the content isn't included in the index.
         * </p>
         * 
         * @param exclusionPatterns
         *        &gt;A list of regular expression patterns to exclude certain blog posts, pages, spaces, or attachments
         *        in your Confluence. Content that matches the patterns are excluded from the index. Content that
         *        doesn't match the patterns is included in the index. If content matches both an inclusion and
         *        exclusion pattern, the exclusion pattern takes precedence and the content isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(Collection<String> exclusionPatterns);

        /**
         * <p>
         * &gt;A list of regular expression patterns to exclude certain blog posts, pages, spaces, or attachments in
         * your Confluence. Content that matches the patterns are excluded from the index. Content that doesn't match
         * the patterns is included in the index. If content matches both an inclusion and exclusion pattern, the
         * exclusion pattern takes precedence and the content isn't included in the index.
         * </p>
         * 
         * @param exclusionPatterns
         *        &gt;A list of regular expression patterns to exclude certain blog posts, pages, spaces, or attachments
         *        in your Confluence. Content that matches the patterns are excluded from the index. Content that
         *        doesn't match the patterns is included in the index. If content matches both an inclusion and
         *        exclusion pattern, the exclusion pattern takes precedence and the content isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(String... exclusionPatterns);
    }

    static final class BuilderImpl implements Builder {
        private String serverUrl;

        private String secretArn;

        private String version;

        private ConfluenceSpaceConfiguration spaceConfiguration;

        private ConfluencePageConfiguration pageConfiguration;

        private ConfluenceBlogConfiguration blogConfiguration;

        private ConfluenceAttachmentConfiguration attachmentConfiguration;

        private DataSourceVpcConfiguration vpcConfiguration;

        private List<String> inclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private List<String> exclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ConfluenceConfiguration model) {
            serverUrl(model.serverUrl);
            secretArn(model.secretArn);
            version(model.version);
            spaceConfiguration(model.spaceConfiguration);
            pageConfiguration(model.pageConfiguration);
            blogConfiguration(model.blogConfiguration);
            attachmentConfiguration(model.attachmentConfiguration);
            vpcConfiguration(model.vpcConfiguration);
            inclusionPatterns(model.inclusionPatterns);
            exclusionPatterns(model.exclusionPatterns);
        }

        public final String getServerUrl() {
            return serverUrl;
        }

        public final void setServerUrl(String serverUrl) {
            this.serverUrl = serverUrl;
        }

        @Override
        @Transient
        public final Builder serverUrl(String serverUrl) {
            this.serverUrl = serverUrl;
            return this;
        }

        public final String getSecretArn() {
            return secretArn;
        }

        public final void setSecretArn(String secretArn) {
            this.secretArn = secretArn;
        }

        @Override
        @Transient
        public final Builder secretArn(String secretArn) {
            this.secretArn = secretArn;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        @Transient
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        @Override
        @Transient
        public final Builder version(ConfluenceVersion version) {
            this.version(version == null ? null : version.toString());
            return this;
        }

        public final ConfluenceSpaceConfiguration.Builder getSpaceConfiguration() {
            return spaceConfiguration != null ? spaceConfiguration.toBuilder() : null;
        }

        public final void setSpaceConfiguration(ConfluenceSpaceConfiguration.BuilderImpl spaceConfiguration) {
            this.spaceConfiguration = spaceConfiguration != null ? spaceConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder spaceConfiguration(ConfluenceSpaceConfiguration spaceConfiguration) {
            this.spaceConfiguration = spaceConfiguration;
            return this;
        }

        public final ConfluencePageConfiguration.Builder getPageConfiguration() {
            return pageConfiguration != null ? pageConfiguration.toBuilder() : null;
        }

        public final void setPageConfiguration(ConfluencePageConfiguration.BuilderImpl pageConfiguration) {
            this.pageConfiguration = pageConfiguration != null ? pageConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder pageConfiguration(ConfluencePageConfiguration pageConfiguration) {
            this.pageConfiguration = pageConfiguration;
            return this;
        }

        public final ConfluenceBlogConfiguration.Builder getBlogConfiguration() {
            return blogConfiguration != null ? blogConfiguration.toBuilder() : null;
        }

        public final void setBlogConfiguration(ConfluenceBlogConfiguration.BuilderImpl blogConfiguration) {
            this.blogConfiguration = blogConfiguration != null ? blogConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder blogConfiguration(ConfluenceBlogConfiguration blogConfiguration) {
            this.blogConfiguration = blogConfiguration;
            return this;
        }

        public final ConfluenceAttachmentConfiguration.Builder getAttachmentConfiguration() {
            return attachmentConfiguration != null ? attachmentConfiguration.toBuilder() : null;
        }

        public final void setAttachmentConfiguration(ConfluenceAttachmentConfiguration.BuilderImpl attachmentConfiguration) {
            this.attachmentConfiguration = attachmentConfiguration != null ? attachmentConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder attachmentConfiguration(ConfluenceAttachmentConfiguration attachmentConfiguration) {
            this.attachmentConfiguration = attachmentConfiguration;
            return this;
        }

        public final DataSourceVpcConfiguration.Builder getVpcConfiguration() {
            return vpcConfiguration != null ? vpcConfiguration.toBuilder() : null;
        }

        public final void setVpcConfiguration(DataSourceVpcConfiguration.BuilderImpl vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration != null ? vpcConfiguration.build() : null;
        }

        @Override
        @Transient
        public final Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration;
            return this;
        }

        public final Collection<String> getInclusionPatterns() {
            if (inclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return inclusionPatterns;
        }

        public final void setInclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
        }

        @Override
        @Transient
        public final Builder inclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder inclusionPatterns(String... inclusionPatterns) {
            inclusionPatterns(Arrays.asList(inclusionPatterns));
            return this;
        }

        public final Collection<String> getExclusionPatterns() {
            if (exclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return exclusionPatterns;
        }

        public final void setExclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
        }

        @Override
        @Transient
        public final Builder exclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder exclusionPatterns(String... exclusionPatterns) {
            exclusionPatterns(Arrays.asList(exclusionPatterns));
            return this;
        }

        @Override
        public ConfluenceConfiguration build() {
            return new ConfluenceConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
