/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the configuration information to connect to Quip as your data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QuipConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<QuipConfiguration.Builder, QuipConfiguration> {
    private static final SdkField<String> DOMAIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Domain")
            .getter(getter(QuipConfiguration::domain)).setter(setter(Builder::domain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Domain").build()).build();

    private static final SdkField<String> SECRET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretArn").getter(getter(QuipConfiguration::secretArn)).setter(setter(Builder::secretArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretArn").build()).build();

    private static final SdkField<Boolean> CRAWL_FILE_COMMENTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CrawlFileComments").getter(getter(QuipConfiguration::crawlFileComments))
            .setter(setter(Builder::crawlFileComments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlFileComments").build()).build();

    private static final SdkField<Boolean> CRAWL_CHAT_ROOMS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CrawlChatRooms").getter(getter(QuipConfiguration::crawlChatRooms))
            .setter(setter(Builder::crawlChatRooms))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlChatRooms").build()).build();

    private static final SdkField<Boolean> CRAWL_ATTACHMENTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CrawlAttachments").getter(getter(QuipConfiguration::crawlAttachments))
            .setter(setter(Builder::crawlAttachments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlAttachments").build()).build();

    private static final SdkField<List<String>> FOLDER_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FolderIds")
            .getter(getter(QuipConfiguration::folderIds))
            .setter(setter(Builder::folderIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FolderIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> THREAD_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("ThreadFieldMappings")
            .getter(getter(QuipConfiguration::threadFieldMappings))
            .setter(setter(Builder::threadFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ThreadFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> MESSAGE_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("MessageFieldMappings")
            .getter(getter(QuipConfiguration::messageFieldMappings))
            .setter(setter(Builder::messageFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> ATTACHMENT_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("AttachmentFieldMappings")
            .getter(getter(QuipConfiguration::attachmentFieldMappings))
            .setter(setter(Builder::attachmentFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> INCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InclusionPatterns")
            .getter(getter(QuipConfiguration::inclusionPatterns))
            .setter(setter(Builder::inclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExclusionPatterns")
            .getter(getter(QuipConfiguration::exclusionPatterns))
            .setter(setter(Builder::exclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DataSourceVpcConfiguration> VPC_CONFIGURATION_FIELD = SdkField
            .<DataSourceVpcConfiguration> builder(MarshallingType.SDK_POJO).memberName("VpcConfiguration")
            .getter(getter(QuipConfiguration::vpcConfiguration)).setter(setter(Builder::vpcConfiguration))
            .constructor(DataSourceVpcConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_FIELD,
            SECRET_ARN_FIELD, CRAWL_FILE_COMMENTS_FIELD, CRAWL_CHAT_ROOMS_FIELD, CRAWL_ATTACHMENTS_FIELD, FOLDER_IDS_FIELD,
            THREAD_FIELD_MAPPINGS_FIELD, MESSAGE_FIELD_MAPPINGS_FIELD, ATTACHMENT_FIELD_MAPPINGS_FIELD, INCLUSION_PATTERNS_FIELD,
            EXCLUSION_PATTERNS_FIELD, VPC_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String domain;

    private final String secretArn;

    private final Boolean crawlFileComments;

    private final Boolean crawlChatRooms;

    private final Boolean crawlAttachments;

    private final List<String> folderIds;

    private final List<DataSourceToIndexFieldMapping> threadFieldMappings;

    private final List<DataSourceToIndexFieldMapping> messageFieldMappings;

    private final List<DataSourceToIndexFieldMapping> attachmentFieldMappings;

    private final List<String> inclusionPatterns;

    private final List<String> exclusionPatterns;

    private final DataSourceVpcConfiguration vpcConfiguration;

    private QuipConfiguration(BuilderImpl builder) {
        this.domain = builder.domain;
        this.secretArn = builder.secretArn;
        this.crawlFileComments = builder.crawlFileComments;
        this.crawlChatRooms = builder.crawlChatRooms;
        this.crawlAttachments = builder.crawlAttachments;
        this.folderIds = builder.folderIds;
        this.threadFieldMappings = builder.threadFieldMappings;
        this.messageFieldMappings = builder.messageFieldMappings;
        this.attachmentFieldMappings = builder.attachmentFieldMappings;
        this.inclusionPatterns = builder.inclusionPatterns;
        this.exclusionPatterns = builder.exclusionPatterns;
        this.vpcConfiguration = builder.vpcConfiguration;
    }

    /**
     * <p>
     * The configuration information to connect to your Quip data source domain.
     * </p>
     * 
     * @return The configuration information to connect to your Quip data source domain.
     */
    public final String domain() {
        return domain;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs that are required
     * to connect to your Quip file system. Windows is currently the only supported type. The secret must contain a JSON
     * structure with the following keys:
     * </p>
     * <ul>
     * <li>
     * <p>
     * username—The Active Directory user name, along with the Domain Name System (DNS) domain name. For example,
     * <i>user@corp.example.com</i>. The Active Directory user account must have read and mounting access to the Quip
     * file system for Windows.
     * </p>
     * </li>
     * <li>
     * <p>
     * password—The password of the Active Directory user account with read and mounting access to the Quip Windows file
     * system.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs that are
     *         required to connect to your Quip file system. Windows is currently the only supported type. The secret
     *         must contain a JSON structure with the following keys:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         username—The Active Directory user name, along with the Domain Name System (DNS) domain name. For
     *         example, <i>user@corp.example.com</i>. The Active Directory user account must have read and mounting
     *         access to the Quip file system for Windows.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         password—The password of the Active Directory user account with read and mounting access to the Quip
     *         Windows file system.
     *         </p>
     *         </li>
     */
    public final String secretArn() {
        return secretArn;
    }

    /**
     * <p>
     * Specify whether to crawl file comments in your Quip data source. You can specify one or more of these options.
     * </p>
     * 
     * @return Specify whether to crawl file comments in your Quip data source. You can specify one or more of these
     *         options.
     */
    public final Boolean crawlFileComments() {
        return crawlFileComments;
    }

    /**
     * <p>
     * Specify whether to crawl chat rooms in your Quip data source. You can specify one or more of these options.
     * </p>
     * 
     * @return Specify whether to crawl chat rooms in your Quip data source. You can specify one or more of these
     *         options.
     */
    public final Boolean crawlChatRooms() {
        return crawlChatRooms;
    }

    /**
     * <p>
     * Specify whether to crawl attachments in your Quip data source. You can specify one or more of these options.
     * </p>
     * 
     * @return Specify whether to crawl attachments in your Quip data source. You can specify one or more of these
     *         options.
     */
    public final Boolean crawlAttachments() {
        return crawlAttachments;
    }

    /**
     * For responses, this returns true if the service returned a value for the FolderIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFolderIds() {
        return folderIds != null && !(folderIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identifier of the Quip folder IDs to index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFolderIds} method.
     * </p>
     * 
     * @return The identifier of the Quip folder IDs to index.
     */
    public final List<String> folderIds() {
        return folderIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the ThreadFieldMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasThreadFieldMappings() {
        return threadFieldMappings != null && !(threadFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of field mappings to apply when indexing Quip threads.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasThreadFieldMappings} method.
     * </p>
     * 
     * @return A list of field mappings to apply when indexing Quip threads.
     */
    public final List<DataSourceToIndexFieldMapping> threadFieldMappings() {
        return threadFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the MessageFieldMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMessageFieldMappings() {
        return messageFieldMappings != null && !(messageFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of field mappings to apply when indexing Quip messages.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMessageFieldMappings} method.
     * </p>
     * 
     * @return A list of field mappings to apply when indexing Quip messages.
     */
    public final List<DataSourceToIndexFieldMapping> messageFieldMappings() {
        return messageFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the AttachmentFieldMappings property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAttachmentFieldMappings() {
        return attachmentFieldMappings != null && !(attachmentFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of field mappings to apply when indexing Quip attachments.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttachmentFieldMappings} method.
     * </p>
     * 
     * @return A list of field mappings to apply when indexing Quip attachments.
     */
    public final List<DataSourceToIndexFieldMapping> attachmentFieldMappings() {
        return attachmentFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the InclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInclusionPatterns() {
        return inclusionPatterns != null && !(inclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to include certain files in your Quip file system. Files that match the
     * patterns are included in the index. Files that don't match the patterns are excluded from the index. If a file
     * matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence, and the file
     * isn't included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to include certain files in your Quip file system. Files that match
     *         the patterns are included in the index. Files that don't match the patterns are excluded from the index.
     *         If a file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes
     *         precedence, and the file isn't included in the index.
     */
    public final List<String> inclusionPatterns() {
        return inclusionPatterns;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExclusionPatterns() {
        return exclusionPatterns != null && !(exclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to exclude certain files in your Quip file system. Files that match the
     * patterns are excluded from the index. Files that don’t match the patterns are included in the index. If a file
     * matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence, and the file
     * isn't included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to exclude certain files in your Quip file system. Files that match
     *         the patterns are excluded from the index. Files that don’t match the patterns are included in the index.
     *         If a file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes
     *         precedence, and the file isn't included in the index.
     */
    public final List<String> exclusionPatterns() {
        return exclusionPatterns;
    }

    /**
     * <p>
     * Configuration information for connecting to an Amazon Virtual Private Cloud (VPC) for your Quip. Your Quip
     * instance must reside inside your VPC.
     * </p>
     * 
     * @return Configuration information for connecting to an Amazon Virtual Private Cloud (VPC) for your Quip. Your
     *         Quip instance must reside inside your VPC.
     */
    public final DataSourceVpcConfiguration vpcConfiguration() {
        return vpcConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domain());
        hashCode = 31 * hashCode + Objects.hashCode(secretArn());
        hashCode = 31 * hashCode + Objects.hashCode(crawlFileComments());
        hashCode = 31 * hashCode + Objects.hashCode(crawlChatRooms());
        hashCode = 31 * hashCode + Objects.hashCode(crawlAttachments());
        hashCode = 31 * hashCode + Objects.hashCode(hasFolderIds() ? folderIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasThreadFieldMappings() ? threadFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMessageFieldMappings() ? messageFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAttachmentFieldMappings() ? attachmentFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInclusionPatterns() ? inclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExclusionPatterns() ? exclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QuipConfiguration)) {
            return false;
        }
        QuipConfiguration other = (QuipConfiguration) obj;
        return Objects.equals(domain(), other.domain()) && Objects.equals(secretArn(), other.secretArn())
                && Objects.equals(crawlFileComments(), other.crawlFileComments())
                && Objects.equals(crawlChatRooms(), other.crawlChatRooms())
                && Objects.equals(crawlAttachments(), other.crawlAttachments()) && hasFolderIds() == other.hasFolderIds()
                && Objects.equals(folderIds(), other.folderIds()) && hasThreadFieldMappings() == other.hasThreadFieldMappings()
                && Objects.equals(threadFieldMappings(), other.threadFieldMappings())
                && hasMessageFieldMappings() == other.hasMessageFieldMappings()
                && Objects.equals(messageFieldMappings(), other.messageFieldMappings())
                && hasAttachmentFieldMappings() == other.hasAttachmentFieldMappings()
                && Objects.equals(attachmentFieldMappings(), other.attachmentFieldMappings())
                && hasInclusionPatterns() == other.hasInclusionPatterns()
                && Objects.equals(inclusionPatterns(), other.inclusionPatterns())
                && hasExclusionPatterns() == other.hasExclusionPatterns()
                && Objects.equals(exclusionPatterns(), other.exclusionPatterns())
                && Objects.equals(vpcConfiguration(), other.vpcConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("QuipConfiguration").add("Domain", domain()).add("SecretArn", secretArn())
                .add("CrawlFileComments", crawlFileComments()).add("CrawlChatRooms", crawlChatRooms())
                .add("CrawlAttachments", crawlAttachments()).add("FolderIds", hasFolderIds() ? folderIds() : null)
                .add("ThreadFieldMappings", hasThreadFieldMappings() ? threadFieldMappings() : null)
                .add("MessageFieldMappings", hasMessageFieldMappings() ? messageFieldMappings() : null)
                .add("AttachmentFieldMappings", hasAttachmentFieldMappings() ? attachmentFieldMappings() : null)
                .add("InclusionPatterns", hasInclusionPatterns() ? inclusionPatterns() : null)
                .add("ExclusionPatterns", hasExclusionPatterns() ? exclusionPatterns() : null)
                .add("VpcConfiguration", vpcConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Domain":
            return Optional.ofNullable(clazz.cast(domain()));
        case "SecretArn":
            return Optional.ofNullable(clazz.cast(secretArn()));
        case "CrawlFileComments":
            return Optional.ofNullable(clazz.cast(crawlFileComments()));
        case "CrawlChatRooms":
            return Optional.ofNullable(clazz.cast(crawlChatRooms()));
        case "CrawlAttachments":
            return Optional.ofNullable(clazz.cast(crawlAttachments()));
        case "FolderIds":
            return Optional.ofNullable(clazz.cast(folderIds()));
        case "ThreadFieldMappings":
            return Optional.ofNullable(clazz.cast(threadFieldMappings()));
        case "MessageFieldMappings":
            return Optional.ofNullable(clazz.cast(messageFieldMappings()));
        case "AttachmentFieldMappings":
            return Optional.ofNullable(clazz.cast(attachmentFieldMappings()));
        case "InclusionPatterns":
            return Optional.ofNullable(clazz.cast(inclusionPatterns()));
        case "ExclusionPatterns":
            return Optional.ofNullable(clazz.cast(exclusionPatterns()));
        case "VpcConfiguration":
            return Optional.ofNullable(clazz.cast(vpcConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<QuipConfiguration, T> g) {
        return obj -> g.apply((QuipConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, QuipConfiguration> {
        /**
         * <p>
         * The configuration information to connect to your Quip data source domain.
         * </p>
         * 
         * @param domain
         *        The configuration information to connect to your Quip data source domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domain(String domain);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs that are
         * required to connect to your Quip file system. Windows is currently the only supported type. The secret must
         * contain a JSON structure with the following keys:
         * </p>
         * <ul>
         * <li>
         * <p>
         * username—The Active Directory user name, along with the Domain Name System (DNS) domain name. For example,
         * <i>user@corp.example.com</i>. The Active Directory user account must have read and mounting access to the
         * Quip file system for Windows.
         * </p>
         * </li>
         * <li>
         * <p>
         * password—The password of the Active Directory user account with read and mounting access to the Quip Windows
         * file system.
         * </p>
         * </li>
         * </ul>
         * 
         * @param secretArn
         *        The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs that are
         *        required to connect to your Quip file system. Windows is currently the only supported type. The secret
         *        must contain a JSON structure with the following keys:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        username—The Active Directory user name, along with the Domain Name System (DNS) domain name. For
         *        example, <i>user@corp.example.com</i>. The Active Directory user account must have read and mounting
         *        access to the Quip file system for Windows.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        password—The password of the Active Directory user account with read and mounting access to the Quip
         *        Windows file system.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretArn(String secretArn);

        /**
         * <p>
         * Specify whether to crawl file comments in your Quip data source. You can specify one or more of these
         * options.
         * </p>
         * 
         * @param crawlFileComments
         *        Specify whether to crawl file comments in your Quip data source. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlFileComments(Boolean crawlFileComments);

        /**
         * <p>
         * Specify whether to crawl chat rooms in your Quip data source. You can specify one or more of these options.
         * </p>
         * 
         * @param crawlChatRooms
         *        Specify whether to crawl chat rooms in your Quip data source. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlChatRooms(Boolean crawlChatRooms);

        /**
         * <p>
         * Specify whether to crawl attachments in your Quip data source. You can specify one or more of these options.
         * </p>
         * 
         * @param crawlAttachments
         *        Specify whether to crawl attachments in your Quip data source. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlAttachments(Boolean crawlAttachments);

        /**
         * <p>
         * The identifier of the Quip folder IDs to index.
         * </p>
         * 
         * @param folderIds
         *        The identifier of the Quip folder IDs to index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder folderIds(Collection<String> folderIds);

        /**
         * <p>
         * The identifier of the Quip folder IDs to index.
         * </p>
         * 
         * @param folderIds
         *        The identifier of the Quip folder IDs to index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder folderIds(String... folderIds);

        /**
         * <p>
         * A list of field mappings to apply when indexing Quip threads.
         * </p>
         * 
         * @param threadFieldMappings
         *        A list of field mappings to apply when indexing Quip threads.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threadFieldMappings(Collection<DataSourceToIndexFieldMapping> threadFieldMappings);

        /**
         * <p>
         * A list of field mappings to apply when indexing Quip threads.
         * </p>
         * 
         * @param threadFieldMappings
         *        A list of field mappings to apply when indexing Quip threads.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threadFieldMappings(DataSourceToIndexFieldMapping... threadFieldMappings);

        /**
         * <p>
         * A list of field mappings to apply when indexing Quip threads.
         * </p>
         * This is a convenience method that creates an instance of the {@link List
         * <DataSourceToIndexFieldMapping>.Builder} avoiding the need to create one manually via {@link List
         * <DataSourceToIndexFieldMapping>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DataSourceToIndexFieldMapping>.Builder#build()} is called
         * immediately and its result is passed to {@link #threadFieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param threadFieldMappings
         *        a consumer that will call methods on {@link List<DataSourceToIndexFieldMapping>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #threadFieldMappings(List<DataSourceToIndexFieldMapping>)
         */
        Builder threadFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... threadFieldMappings);

        /**
         * <p>
         * A list of field mappings to apply when indexing Quip messages.
         * </p>
         * 
         * @param messageFieldMappings
         *        A list of field mappings to apply when indexing Quip messages.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageFieldMappings(Collection<DataSourceToIndexFieldMapping> messageFieldMappings);

        /**
         * <p>
         * A list of field mappings to apply when indexing Quip messages.
         * </p>
         * 
         * @param messageFieldMappings
         *        A list of field mappings to apply when indexing Quip messages.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageFieldMappings(DataSourceToIndexFieldMapping... messageFieldMappings);

        /**
         * <p>
         * A list of field mappings to apply when indexing Quip messages.
         * </p>
         * This is a convenience method that creates an instance of the {@link List
         * <DataSourceToIndexFieldMapping>.Builder} avoiding the need to create one manually via {@link List
         * <DataSourceToIndexFieldMapping>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DataSourceToIndexFieldMapping>.Builder#build()} is called
         * immediately and its result is passed to {@link #messageFieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param messageFieldMappings
         *        a consumer that will call methods on {@link List<DataSourceToIndexFieldMapping>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #messageFieldMappings(List<DataSourceToIndexFieldMapping>)
         */
        Builder messageFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... messageFieldMappings);

        /**
         * <p>
         * A list of field mappings to apply when indexing Quip attachments.
         * </p>
         * 
         * @param attachmentFieldMappings
         *        A list of field mappings to apply when indexing Quip attachments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentFieldMappings(Collection<DataSourceToIndexFieldMapping> attachmentFieldMappings);

        /**
         * <p>
         * A list of field mappings to apply when indexing Quip attachments.
         * </p>
         * 
         * @param attachmentFieldMappings
         *        A list of field mappings to apply when indexing Quip attachments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentFieldMappings(DataSourceToIndexFieldMapping... attachmentFieldMappings);

        /**
         * <p>
         * A list of field mappings to apply when indexing Quip attachments.
         * </p>
         * This is a convenience method that creates an instance of the {@link List
         * <DataSourceToIndexFieldMapping>.Builder} avoiding the need to create one manually via {@link List
         * <DataSourceToIndexFieldMapping>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DataSourceToIndexFieldMapping>.Builder#build()} is called
         * immediately and its result is passed to {@link #attachmentFieldMappings(List<DataSourceToIndexFieldMapping>)}
         * .
         * 
         * @param attachmentFieldMappings
         *        a consumer that will call methods on {@link List<DataSourceToIndexFieldMapping>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachmentFieldMappings(List<DataSourceToIndexFieldMapping>)
         */
        Builder attachmentFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... attachmentFieldMappings);

        /**
         * <p>
         * A list of regular expression patterns to include certain files in your Quip file system. Files that match the
         * patterns are included in the index. Files that don't match the patterns are excluded from the index. If a
         * file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence, and
         * the file isn't included in the index.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns to include certain files in your Quip file system. Files that
         *        match the patterns are included in the index. Files that don't match the patterns are excluded from
         *        the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern
         *        takes precedence, and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(Collection<String> inclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to include certain files in your Quip file system. Files that match the
         * patterns are included in the index. Files that don't match the patterns are excluded from the index. If a
         * file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence, and
         * the file isn't included in the index.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns to include certain files in your Quip file system. Files that
         *        match the patterns are included in the index. Files that don't match the patterns are excluded from
         *        the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern
         *        takes precedence, and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(String... inclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to exclude certain files in your Quip file system. Files that match the
         * patterns are excluded from the index. Files that don’t match the patterns are included in the index. If a
         * file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence, and
         * the file isn't included in the index.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of regular expression patterns to exclude certain files in your Quip file system. Files that
         *        match the patterns are excluded from the index. Files that don’t match the patterns are included in
         *        the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern
         *        takes precedence, and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(Collection<String> exclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to exclude certain files in your Quip file system. Files that match the
         * patterns are excluded from the index. Files that don’t match the patterns are included in the index. If a
         * file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence, and
         * the file isn't included in the index.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of regular expression patterns to exclude certain files in your Quip file system. Files that
         *        match the patterns are excluded from the index. Files that don’t match the patterns are included in
         *        the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern
         *        takes precedence, and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(String... exclusionPatterns);

        /**
         * <p>
         * Configuration information for connecting to an Amazon Virtual Private Cloud (VPC) for your Quip. Your Quip
         * instance must reside inside your VPC.
         * </p>
         * 
         * @param vpcConfiguration
         *        Configuration information for connecting to an Amazon Virtual Private Cloud (VPC) for your Quip. Your
         *        Quip instance must reside inside your VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration);

        /**
         * <p>
         * Configuration information for connecting to an Amazon Virtual Private Cloud (VPC) for your Quip. Your Quip
         * instance must reside inside your VPC.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataSourceVpcConfiguration.Builder}
         * avoiding the need to create one manually via {@link DataSourceVpcConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataSourceVpcConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #vpcConfiguration(DataSourceVpcConfiguration)}.
         * 
         * @param vpcConfiguration
         *        a consumer that will call methods on {@link DataSourceVpcConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfiguration(DataSourceVpcConfiguration)
         */
        default Builder vpcConfiguration(Consumer<DataSourceVpcConfiguration.Builder> vpcConfiguration) {
            return vpcConfiguration(DataSourceVpcConfiguration.builder().applyMutation(vpcConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String domain;

        private String secretArn;

        private Boolean crawlFileComments;

        private Boolean crawlChatRooms;

        private Boolean crawlAttachments;

        private List<String> folderIds = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> threadFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> messageFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> attachmentFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<String> inclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private List<String> exclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private DataSourceVpcConfiguration vpcConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(QuipConfiguration model) {
            domain(model.domain);
            secretArn(model.secretArn);
            crawlFileComments(model.crawlFileComments);
            crawlChatRooms(model.crawlChatRooms);
            crawlAttachments(model.crawlAttachments);
            folderIds(model.folderIds);
            threadFieldMappings(model.threadFieldMappings);
            messageFieldMappings(model.messageFieldMappings);
            attachmentFieldMappings(model.attachmentFieldMappings);
            inclusionPatterns(model.inclusionPatterns);
            exclusionPatterns(model.exclusionPatterns);
            vpcConfiguration(model.vpcConfiguration);
        }

        public final String getDomain() {
            return domain;
        }

        public final void setDomain(String domain) {
            this.domain = domain;
        }

        @Override
        public final Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        public final String getSecretArn() {
            return secretArn;
        }

        public final void setSecretArn(String secretArn) {
            this.secretArn = secretArn;
        }

        @Override
        public final Builder secretArn(String secretArn) {
            this.secretArn = secretArn;
            return this;
        }

        public final Boolean getCrawlFileComments() {
            return crawlFileComments;
        }

        public final void setCrawlFileComments(Boolean crawlFileComments) {
            this.crawlFileComments = crawlFileComments;
        }

        @Override
        public final Builder crawlFileComments(Boolean crawlFileComments) {
            this.crawlFileComments = crawlFileComments;
            return this;
        }

        public final Boolean getCrawlChatRooms() {
            return crawlChatRooms;
        }

        public final void setCrawlChatRooms(Boolean crawlChatRooms) {
            this.crawlChatRooms = crawlChatRooms;
        }

        @Override
        public final Builder crawlChatRooms(Boolean crawlChatRooms) {
            this.crawlChatRooms = crawlChatRooms;
            return this;
        }

        public final Boolean getCrawlAttachments() {
            return crawlAttachments;
        }

        public final void setCrawlAttachments(Boolean crawlAttachments) {
            this.crawlAttachments = crawlAttachments;
        }

        @Override
        public final Builder crawlAttachments(Boolean crawlAttachments) {
            this.crawlAttachments = crawlAttachments;
            return this;
        }

        public final Collection<String> getFolderIds() {
            if (folderIds instanceof SdkAutoConstructList) {
                return null;
            }
            return folderIds;
        }

        public final void setFolderIds(Collection<String> folderIds) {
            this.folderIds = FolderIdListCopier.copy(folderIds);
        }

        @Override
        public final Builder folderIds(Collection<String> folderIds) {
            this.folderIds = FolderIdListCopier.copy(folderIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder folderIds(String... folderIds) {
            folderIds(Arrays.asList(folderIds));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getThreadFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.threadFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setThreadFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> threadFieldMappings) {
            this.threadFieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(threadFieldMappings);
        }

        @Override
        public final Builder threadFieldMappings(Collection<DataSourceToIndexFieldMapping> threadFieldMappings) {
            this.threadFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(threadFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder threadFieldMappings(DataSourceToIndexFieldMapping... threadFieldMappings) {
            threadFieldMappings(Arrays.asList(threadFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder threadFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... threadFieldMappings) {
            threadFieldMappings(Stream.of(threadFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getMessageFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.messageFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMessageFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> messageFieldMappings) {
            this.messageFieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(messageFieldMappings);
        }

        @Override
        public final Builder messageFieldMappings(Collection<DataSourceToIndexFieldMapping> messageFieldMappings) {
            this.messageFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(messageFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messageFieldMappings(DataSourceToIndexFieldMapping... messageFieldMappings) {
            messageFieldMappings(Arrays.asList(messageFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messageFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... messageFieldMappings) {
            messageFieldMappings(Stream.of(messageFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getAttachmentFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.attachmentFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttachmentFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> attachmentFieldMappings) {
            this.attachmentFieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(attachmentFieldMappings);
        }

        @Override
        public final Builder attachmentFieldMappings(Collection<DataSourceToIndexFieldMapping> attachmentFieldMappings) {
            this.attachmentFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(attachmentFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachmentFieldMappings(DataSourceToIndexFieldMapping... attachmentFieldMappings) {
            attachmentFieldMappings(Arrays.asList(attachmentFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachmentFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... attachmentFieldMappings) {
            attachmentFieldMappings(Stream.of(attachmentFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getInclusionPatterns() {
            if (inclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return inclusionPatterns;
        }

        public final void setInclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
        }

        @Override
        public final Builder inclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inclusionPatterns(String... inclusionPatterns) {
            inclusionPatterns(Arrays.asList(inclusionPatterns));
            return this;
        }

        public final Collection<String> getExclusionPatterns() {
            if (exclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return exclusionPatterns;
        }

        public final void setExclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
        }

        @Override
        public final Builder exclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exclusionPatterns(String... exclusionPatterns) {
            exclusionPatterns(Arrays.asList(exclusionPatterns));
            return this;
        }

        public final DataSourceVpcConfiguration.Builder getVpcConfiguration() {
            return vpcConfiguration != null ? vpcConfiguration.toBuilder() : null;
        }

        public final void setVpcConfiguration(DataSourceVpcConfiguration.BuilderImpl vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration != null ? vpcConfiguration.build() : null;
        }

        @Override
        public final Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration;
            return this;
        }

        @Override
        public QuipConfiguration build() {
            return new QuipConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
