/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the configuration information to connect to Jira as your data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JiraConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<JiraConfiguration.Builder, JiraConfiguration> {
    private static final SdkField<String> JIRA_ACCOUNT_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JiraAccountUrl").getter(getter(JiraConfiguration::jiraAccountUrl))
            .setter(setter(Builder::jiraAccountUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JiraAccountUrl").build()).build();

    private static final SdkField<String> SECRET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretArn").getter(getter(JiraConfiguration::secretArn)).setter(setter(Builder::secretArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretArn").build()).build();

    private static final SdkField<Boolean> USE_CHANGE_LOG_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("UseChangeLog").getter(getter(JiraConfiguration::useChangeLog)).setter(setter(Builder::useChangeLog))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UseChangeLog").build()).build();

    private static final SdkField<List<String>> PROJECT_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Project")
            .getter(getter(JiraConfiguration::project))
            .setter(setter(Builder::project))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Project").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ISSUE_TYPE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IssueType")
            .getter(getter(JiraConfiguration::issueType))
            .setter(setter(Builder::issueType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IssueType").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> STATUS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Status")
            .getter(getter(JiraConfiguration::status))
            .setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ISSUE_SUB_ENTITY_FILTER_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IssueSubEntityFilter")
            .getter(getter(JiraConfiguration::issueSubEntityFilterAsStrings))
            .setter(setter(Builder::issueSubEntityFilterWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IssueSubEntityFilter").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> ATTACHMENT_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("AttachmentFieldMappings")
            .getter(getter(JiraConfiguration::attachmentFieldMappings))
            .setter(setter(Builder::attachmentFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> COMMENT_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("CommentFieldMappings")
            .getter(getter(JiraConfiguration::commentFieldMappings))
            .setter(setter(Builder::commentFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CommentFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> ISSUE_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("IssueFieldMappings")
            .getter(getter(JiraConfiguration::issueFieldMappings))
            .setter(setter(Builder::issueFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IssueFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> PROJECT_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("ProjectFieldMappings")
            .getter(getter(JiraConfiguration::projectFieldMappings))
            .setter(setter(Builder::projectFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProjectFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> WORK_LOG_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("WorkLogFieldMappings")
            .getter(getter(JiraConfiguration::workLogFieldMappings))
            .setter(setter(Builder::workLogFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WorkLogFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> INCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InclusionPatterns")
            .getter(getter(JiraConfiguration::inclusionPatterns))
            .setter(setter(Builder::inclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExclusionPatterns")
            .getter(getter(JiraConfiguration::exclusionPatterns))
            .setter(setter(Builder::exclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DataSourceVpcConfiguration> VPC_CONFIGURATION_FIELD = SdkField
            .<DataSourceVpcConfiguration> builder(MarshallingType.SDK_POJO).memberName("VpcConfiguration")
            .getter(getter(JiraConfiguration::vpcConfiguration)).setter(setter(Builder::vpcConfiguration))
            .constructor(DataSourceVpcConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JIRA_ACCOUNT_URL_FIELD,
            SECRET_ARN_FIELD, USE_CHANGE_LOG_FIELD, PROJECT_FIELD, ISSUE_TYPE_FIELD, STATUS_FIELD, ISSUE_SUB_ENTITY_FILTER_FIELD,
            ATTACHMENT_FIELD_MAPPINGS_FIELD, COMMENT_FIELD_MAPPINGS_FIELD, ISSUE_FIELD_MAPPINGS_FIELD,
            PROJECT_FIELD_MAPPINGS_FIELD, WORK_LOG_FIELD_MAPPINGS_FIELD, INCLUSION_PATTERNS_FIELD, EXCLUSION_PATTERNS_FIELD,
            VPC_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String jiraAccountUrl;

    private final String secretArn;

    private final Boolean useChangeLog;

    private final List<String> project;

    private final List<String> issueType;

    private final List<String> status;

    private final List<String> issueSubEntityFilter;

    private final List<DataSourceToIndexFieldMapping> attachmentFieldMappings;

    private final List<DataSourceToIndexFieldMapping> commentFieldMappings;

    private final List<DataSourceToIndexFieldMapping> issueFieldMappings;

    private final List<DataSourceToIndexFieldMapping> projectFieldMappings;

    private final List<DataSourceToIndexFieldMapping> workLogFieldMappings;

    private final List<String> inclusionPatterns;

    private final List<String> exclusionPatterns;

    private final DataSourceVpcConfiguration vpcConfiguration;

    private JiraConfiguration(BuilderImpl builder) {
        this.jiraAccountUrl = builder.jiraAccountUrl;
        this.secretArn = builder.secretArn;
        this.useChangeLog = builder.useChangeLog;
        this.project = builder.project;
        this.issueType = builder.issueType;
        this.status = builder.status;
        this.issueSubEntityFilter = builder.issueSubEntityFilter;
        this.attachmentFieldMappings = builder.attachmentFieldMappings;
        this.commentFieldMappings = builder.commentFieldMappings;
        this.issueFieldMappings = builder.issueFieldMappings;
        this.projectFieldMappings = builder.projectFieldMappings;
        this.workLogFieldMappings = builder.workLogFieldMappings;
        this.inclusionPatterns = builder.inclusionPatterns;
        this.exclusionPatterns = builder.exclusionPatterns;
        this.vpcConfiguration = builder.vpcConfiguration;
    }

    /**
     * <p>
     * The URL of the Jira account. For example, <i>company.atlassian.net</i> or <i>https://jira.company.com</i>. You
     * can find your Jira account URL in the URL of your profile page for Jira desktop.
     * </p>
     * 
     * @return The URL of the Jira account. For example, <i>company.atlassian.net</i> or
     *         <i>https://jira.company.com</i>. You can find your Jira account URL in the URL of your profile page for
     *         Jira desktop.
     */
    public final String jiraAccountUrl() {
        return jiraAccountUrl;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of a secret in Secrets Manager contains the key-value pairs required to connect to
     * your Jira data source. The secret must contain a JSON structure with the following keys:
     * </p>
     * <ul>
     * <li>
     * <p>
     * jiraId—The Jira username.
     * </p>
     * </li>
     * <li>
     * <p>
     * jiraCredentials—The Jira API token. For more information on creating an API token in Jira, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/data-source-jira.html#jira-authentication"> Authentication for
     * a Jira data source</a>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The Amazon Resource Name (ARN) of a secret in Secrets Manager contains the key-value pairs required to
     *         connect to your Jira data source. The secret must contain a JSON structure with the following keys:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         jiraId—The Jira username.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         jiraCredentials—The Jira API token. For more information on creating an API token in Jira, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/data-source-jira.html#jira-authentication">
     *         Authentication for a Jira data source</a>.
     *         </p>
     *         </li>
     */
    public final String secretArn() {
        return secretArn;
    }

    /**
     * <p>
     * <code>TRUE</code> to use the Jira change log to determine which documents require updating in the index.
     * Depending on the change log's size, it may take longer for Amazon Kendra to use the change log than to scan all
     * of your documents in Jira.
     * </p>
     * 
     * @return <code>TRUE</code> to use the Jira change log to determine which documents require updating in the index.
     *         Depending on the change log's size, it may take longer for Amazon Kendra to use the change log than to
     *         scan all of your documents in Jira.
     */
    public final Boolean useChangeLog() {
        return useChangeLog;
    }

    /**
     * For responses, this returns true if the service returned a value for the Project property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProject() {
        return project != null && !(project instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specify which projects to crawl in your Jira data source. You can specify one or more Jira project IDs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProject} method.
     * </p>
     * 
     * @return Specify which projects to crawl in your Jira data source. You can specify one or more Jira project IDs.
     */
    public final List<String> project() {
        return project;
    }

    /**
     * For responses, this returns true if the service returned a value for the IssueType property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasIssueType() {
        return issueType != null && !(issueType instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specify which issue types to crawl in your Jira data source. You can specify one or more of these options to
     * crawl.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIssueType} method.
     * </p>
     * 
     * @return Specify which issue types to crawl in your Jira data source. You can specify one or more of these options
     *         to crawl.
     */
    public final List<String> issueType() {
        return issueType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Status property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasStatus() {
        return status != null && !(status instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specify which statuses to crawl in your Jira data source. You can specify one or more of these options to crawl.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatus} method.
     * </p>
     * 
     * @return Specify which statuses to crawl in your Jira data source. You can specify one or more of these options to
     *         crawl.
     */
    public final List<String> status() {
        return status;
    }

    /**
     * <p>
     * Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these options.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIssueSubEntityFilter} method.
     * </p>
     * 
     * @return Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these
     *         options.
     */
    public final List<IssueSubEntity> issueSubEntityFilter() {
        return IssueSubEntityFilterCopier.copyStringToEnum(issueSubEntityFilter);
    }

    /**
     * For responses, this returns true if the service returned a value for the IssueSubEntityFilter property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIssueSubEntityFilter() {
        return issueSubEntityFilter != null && !(issueSubEntityFilter instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these options.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIssueSubEntityFilter} method.
     * </p>
     * 
     * @return Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these
     *         options.
     */
    public final List<String> issueSubEntityFilterAsStrings() {
        return issueSubEntityFilter;
    }

    /**
     * For responses, this returns true if the service returned a value for the AttachmentFieldMappings property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAttachmentFieldMappings() {
        return attachmentFieldMappings != null && !(attachmentFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira attachments to Amazon
     * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
     * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
     * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttachmentFieldMappings} method.
     * </p>
     * 
     * @return A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira attachments to
     *         Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
     *         fields. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>.
     *         The Jira data source field names must exist in your Jira custom metadata.
     */
    public final List<DataSourceToIndexFieldMapping> attachmentFieldMappings() {
        return attachmentFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the CommentFieldMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCommentFieldMappings() {
        return commentFieldMappings != null && !(commentFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira comments to Amazon
     * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
     * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
     * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommentFieldMappings} method.
     * </p>
     * 
     * @return A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira comments to
     *         Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
     *         fields. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>.
     *         The Jira data source field names must exist in your Jira custom metadata.
     */
    public final List<DataSourceToIndexFieldMapping> commentFieldMappings() {
        return commentFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the IssueFieldMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIssueFieldMappings() {
        return issueFieldMappings != null && !(issueFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira issues to Amazon
     * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
     * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
     * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIssueFieldMappings} method.
     * </p>
     * 
     * @return A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira issues to
     *         Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
     *         fields. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>.
     *         The Jira data source field names must exist in your Jira custom metadata.
     */
    public final List<DataSourceToIndexFieldMapping> issueFieldMappings() {
        return issueFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProjectFieldMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasProjectFieldMappings() {
        return projectFieldMappings != null && !(projectFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira projects to Amazon
     * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
     * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
     * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProjectFieldMappings} method.
     * </p>
     * 
     * @return A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira projects to
     *         Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
     *         fields. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>.
     *         The Jira data source field names must exist in your Jira custom metadata.
     */
    public final List<DataSourceToIndexFieldMapping> projectFieldMappings() {
        return projectFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the WorkLogFieldMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasWorkLogFieldMappings() {
        return workLogFieldMappings != null && !(workLogFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira work logs to Amazon
     * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
     * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
     * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasWorkLogFieldMappings} method.
     * </p>
     * 
     * @return A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira work logs to
     *         Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
     *         fields. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>.
     *         The Jira data source field names must exist in your Jira custom metadata.
     */
    public final List<DataSourceToIndexFieldMapping> workLogFieldMappings() {
        return workLogFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the InclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInclusionPatterns() {
        return inclusionPatterns != null && !(inclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to include certain file paths, file names, and file types in your Jira data
     * source. Files that match the patterns are included in the index. Files that don't match the patterns are excluded
     * from the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes
     * precedence and the file isn't included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to include certain file paths, file names, and file types in your
     *         Jira data source. Files that match the patterns are included in the index. Files that don't match the
     *         patterns are excluded from the index. If a file matches both an inclusion pattern and an exclusion
     *         pattern, the exclusion pattern takes precedence and the file isn't included in the index.
     */
    public final List<String> inclusionPatterns() {
        return inclusionPatterns;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExclusionPatterns() {
        return exclusionPatterns != null && !(exclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to exclude certain file paths, file names, and file types in your Jira data
     * source. Files that match the patterns are excluded from the index. Files that don’t match the patterns are
     * included in the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion
     * pattern takes precedence and the file isn't included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to exclude certain file paths, file names, and file types in your
     *         Jira data source. Files that match the patterns are excluded from the index. Files that don’t match the
     *         patterns are included in the index. If a file matches both an inclusion pattern and an exclusion pattern,
     *         the exclusion pattern takes precedence and the file isn't included in the index.
     */
    public final List<String> exclusionPatterns() {
        return exclusionPatterns;
    }

    /**
     * <p>
     * Configuration information for an Amazon Virtual Private Cloud to connect to your Jira. Your Jira account must
     * reside inside your VPC.
     * </p>
     * 
     * @return Configuration information for an Amazon Virtual Private Cloud to connect to your Jira. Your Jira account
     *         must reside inside your VPC.
     */
    public final DataSourceVpcConfiguration vpcConfiguration() {
        return vpcConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jiraAccountUrl());
        hashCode = 31 * hashCode + Objects.hashCode(secretArn());
        hashCode = 31 * hashCode + Objects.hashCode(useChangeLog());
        hashCode = 31 * hashCode + Objects.hashCode(hasProject() ? project() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasIssueType() ? issueType() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStatus() ? status() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasIssueSubEntityFilter() ? issueSubEntityFilterAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAttachmentFieldMappings() ? attachmentFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCommentFieldMappings() ? commentFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasIssueFieldMappings() ? issueFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasProjectFieldMappings() ? projectFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasWorkLogFieldMappings() ? workLogFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInclusionPatterns() ? inclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExclusionPatterns() ? exclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JiraConfiguration)) {
            return false;
        }
        JiraConfiguration other = (JiraConfiguration) obj;
        return Objects.equals(jiraAccountUrl(), other.jiraAccountUrl()) && Objects.equals(secretArn(), other.secretArn())
                && Objects.equals(useChangeLog(), other.useChangeLog()) && hasProject() == other.hasProject()
                && Objects.equals(project(), other.project()) && hasIssueType() == other.hasIssueType()
                && Objects.equals(issueType(), other.issueType()) && hasStatus() == other.hasStatus()
                && Objects.equals(status(), other.status()) && hasIssueSubEntityFilter() == other.hasIssueSubEntityFilter()
                && Objects.equals(issueSubEntityFilterAsStrings(), other.issueSubEntityFilterAsStrings())
                && hasAttachmentFieldMappings() == other.hasAttachmentFieldMappings()
                && Objects.equals(attachmentFieldMappings(), other.attachmentFieldMappings())
                && hasCommentFieldMappings() == other.hasCommentFieldMappings()
                && Objects.equals(commentFieldMappings(), other.commentFieldMappings())
                && hasIssueFieldMappings() == other.hasIssueFieldMappings()
                && Objects.equals(issueFieldMappings(), other.issueFieldMappings())
                && hasProjectFieldMappings() == other.hasProjectFieldMappings()
                && Objects.equals(projectFieldMappings(), other.projectFieldMappings())
                && hasWorkLogFieldMappings() == other.hasWorkLogFieldMappings()
                && Objects.equals(workLogFieldMappings(), other.workLogFieldMappings())
                && hasInclusionPatterns() == other.hasInclusionPatterns()
                && Objects.equals(inclusionPatterns(), other.inclusionPatterns())
                && hasExclusionPatterns() == other.hasExclusionPatterns()
                && Objects.equals(exclusionPatterns(), other.exclusionPatterns())
                && Objects.equals(vpcConfiguration(), other.vpcConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JiraConfiguration").add("JiraAccountUrl", jiraAccountUrl()).add("SecretArn", secretArn())
                .add("UseChangeLog", useChangeLog()).add("Project", hasProject() ? project() : null)
                .add("IssueType", hasIssueType() ? issueType() : null).add("Status", hasStatus() ? status() : null)
                .add("IssueSubEntityFilter", hasIssueSubEntityFilter() ? issueSubEntityFilterAsStrings() : null)
                .add("AttachmentFieldMappings", hasAttachmentFieldMappings() ? attachmentFieldMappings() : null)
                .add("CommentFieldMappings", hasCommentFieldMappings() ? commentFieldMappings() : null)
                .add("IssueFieldMappings", hasIssueFieldMappings() ? issueFieldMappings() : null)
                .add("ProjectFieldMappings", hasProjectFieldMappings() ? projectFieldMappings() : null)
                .add("WorkLogFieldMappings", hasWorkLogFieldMappings() ? workLogFieldMappings() : null)
                .add("InclusionPatterns", hasInclusionPatterns() ? inclusionPatterns() : null)
                .add("ExclusionPatterns", hasExclusionPatterns() ? exclusionPatterns() : null)
                .add("VpcConfiguration", vpcConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "JiraAccountUrl":
            return Optional.ofNullable(clazz.cast(jiraAccountUrl()));
        case "SecretArn":
            return Optional.ofNullable(clazz.cast(secretArn()));
        case "UseChangeLog":
            return Optional.ofNullable(clazz.cast(useChangeLog()));
        case "Project":
            return Optional.ofNullable(clazz.cast(project()));
        case "IssueType":
            return Optional.ofNullable(clazz.cast(issueType()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "IssueSubEntityFilter":
            return Optional.ofNullable(clazz.cast(issueSubEntityFilterAsStrings()));
        case "AttachmentFieldMappings":
            return Optional.ofNullable(clazz.cast(attachmentFieldMappings()));
        case "CommentFieldMappings":
            return Optional.ofNullable(clazz.cast(commentFieldMappings()));
        case "IssueFieldMappings":
            return Optional.ofNullable(clazz.cast(issueFieldMappings()));
        case "ProjectFieldMappings":
            return Optional.ofNullable(clazz.cast(projectFieldMappings()));
        case "WorkLogFieldMappings":
            return Optional.ofNullable(clazz.cast(workLogFieldMappings()));
        case "InclusionPatterns":
            return Optional.ofNullable(clazz.cast(inclusionPatterns()));
        case "ExclusionPatterns":
            return Optional.ofNullable(clazz.cast(exclusionPatterns()));
        case "VpcConfiguration":
            return Optional.ofNullable(clazz.cast(vpcConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JiraConfiguration, T> g) {
        return obj -> g.apply((JiraConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JiraConfiguration> {
        /**
         * <p>
         * The URL of the Jira account. For example, <i>company.atlassian.net</i> or <i>https://jira.company.com</i>.
         * You can find your Jira account URL in the URL of your profile page for Jira desktop.
         * </p>
         * 
         * @param jiraAccountUrl
         *        The URL of the Jira account. For example, <i>company.atlassian.net</i> or
         *        <i>https://jira.company.com</i>. You can find your Jira account URL in the URL of your profile page
         *        for Jira desktop.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jiraAccountUrl(String jiraAccountUrl);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of a secret in Secrets Manager contains the key-value pairs required to
         * connect to your Jira data source. The secret must contain a JSON structure with the following keys:
         * </p>
         * <ul>
         * <li>
         * <p>
         * jiraId—The Jira username.
         * </p>
         * </li>
         * <li>
         * <p>
         * jiraCredentials—The Jira API token. For more information on creating an API token in Jira, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/data-source-jira.html#jira-authentication"> Authentication
         * for a Jira data source</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param secretArn
         *        The Amazon Resource Name (ARN) of a secret in Secrets Manager contains the key-value pairs required to
         *        connect to your Jira data source. The secret must contain a JSON structure with the following
         *        keys:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        jiraId—The Jira username.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        jiraCredentials—The Jira API token. For more information on creating an API token in Jira, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/data-source-jira.html#jira-authentication">
         *        Authentication for a Jira data source</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretArn(String secretArn);

        /**
         * <p>
         * <code>TRUE</code> to use the Jira change log to determine which documents require updating in the index.
         * Depending on the change log's size, it may take longer for Amazon Kendra to use the change log than to scan
         * all of your documents in Jira.
         * </p>
         * 
         * @param useChangeLog
         *        <code>TRUE</code> to use the Jira change log to determine which documents require updating in the
         *        index. Depending on the change log's size, it may take longer for Amazon Kendra to use the change log
         *        than to scan all of your documents in Jira.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder useChangeLog(Boolean useChangeLog);

        /**
         * <p>
         * Specify which projects to crawl in your Jira data source. You can specify one or more Jira project IDs.
         * </p>
         * 
         * @param project
         *        Specify which projects to crawl in your Jira data source. You can specify one or more Jira project
         *        IDs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder project(Collection<String> project);

        /**
         * <p>
         * Specify which projects to crawl in your Jira data source. You can specify one or more Jira project IDs.
         * </p>
         * 
         * @param project
         *        Specify which projects to crawl in your Jira data source. You can specify one or more Jira project
         *        IDs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder project(String... project);

        /**
         * <p>
         * Specify which issue types to crawl in your Jira data source. You can specify one or more of these options to
         * crawl.
         * </p>
         * 
         * @param issueType
         *        Specify which issue types to crawl in your Jira data source. You can specify one or more of these
         *        options to crawl.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issueType(Collection<String> issueType);

        /**
         * <p>
         * Specify which issue types to crawl in your Jira data source. You can specify one or more of these options to
         * crawl.
         * </p>
         * 
         * @param issueType
         *        Specify which issue types to crawl in your Jira data source. You can specify one or more of these
         *        options to crawl.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issueType(String... issueType);

        /**
         * <p>
         * Specify which statuses to crawl in your Jira data source. You can specify one or more of these options to
         * crawl.
         * </p>
         * 
         * @param status
         *        Specify which statuses to crawl in your Jira data source. You can specify one or more of these options
         *        to crawl.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(Collection<String> status);

        /**
         * <p>
         * Specify which statuses to crawl in your Jira data source. You can specify one or more of these options to
         * crawl.
         * </p>
         * 
         * @param status
         *        Specify which statuses to crawl in your Jira data source. You can specify one or more of these options
         *        to crawl.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String... status);

        /**
         * <p>
         * Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these options.
         * </p>
         * 
         * @param issueSubEntityFilter
         *        Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issueSubEntityFilterWithStrings(Collection<String> issueSubEntityFilter);

        /**
         * <p>
         * Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these options.
         * </p>
         * 
         * @param issueSubEntityFilter
         *        Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issueSubEntityFilterWithStrings(String... issueSubEntityFilter);

        /**
         * <p>
         * Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these options.
         * </p>
         * 
         * @param issueSubEntityFilter
         *        Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issueSubEntityFilter(Collection<IssueSubEntity> issueSubEntityFilter);

        /**
         * <p>
         * Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these options.
         * </p>
         * 
         * @param issueSubEntityFilter
         *        Specify whether to crawl comments, attachments, and work logs. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issueSubEntityFilter(IssueSubEntity... issueSubEntityFilter);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira attachments to
         * Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
         * fields. For more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">
         * Mapping data source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * 
         * @param attachmentFieldMappings
         *        A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira attachments
         *        to Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to
         *        Jira fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentFieldMappings(Collection<DataSourceToIndexFieldMapping> attachmentFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira attachments to
         * Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
         * fields. For more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">
         * Mapping data source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * 
         * @param attachmentFieldMappings
         *        A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira attachments
         *        to Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to
         *        Jira fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachmentFieldMappings(DataSourceToIndexFieldMapping... attachmentFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira attachments to
         * Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
         * fields. For more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">
         * Mapping data source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder#build()} is called
         * immediately and its result is passed to {@link #attachmentFieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param attachmentFieldMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attachmentFieldMappings(java.util.Collection<DataSourceToIndexFieldMapping>)
         */
        Builder attachmentFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... attachmentFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira comments to Amazon
         * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
         * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
         * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * 
         * @param commentFieldMappings
         *        A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira comments to
         *        Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to
         *        Jira fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commentFieldMappings(Collection<DataSourceToIndexFieldMapping> commentFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira comments to Amazon
         * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
         * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
         * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * 
         * @param commentFieldMappings
         *        A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira comments to
         *        Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to
         *        Jira fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commentFieldMappings(DataSourceToIndexFieldMapping... commentFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira comments to Amazon
         * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
         * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
         * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder#build()} is called
         * immediately and its result is passed to {@link #commentFieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param commentFieldMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #commentFieldMappings(java.util.Collection<DataSourceToIndexFieldMapping>)
         */
        Builder commentFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... commentFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira issues to Amazon
         * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
         * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
         * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * 
         * @param issueFieldMappings
         *        A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira issues to
         *        Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to
         *        Jira fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issueFieldMappings(Collection<DataSourceToIndexFieldMapping> issueFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira issues to Amazon
         * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
         * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
         * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * 
         * @param issueFieldMappings
         *        A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira issues to
         *        Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to
         *        Jira fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issueFieldMappings(DataSourceToIndexFieldMapping... issueFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira issues to Amazon
         * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
         * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
         * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder#build()} is called
         * immediately and its result is passed to {@link #issueFieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param issueFieldMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #issueFieldMappings(java.util.Collection<DataSourceToIndexFieldMapping>)
         */
        Builder issueFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... issueFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira projects to Amazon
         * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
         * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
         * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * 
         * @param projectFieldMappings
         *        A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira projects to
         *        Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to
         *        Jira fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectFieldMappings(Collection<DataSourceToIndexFieldMapping> projectFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira projects to Amazon
         * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
         * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
         * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * 
         * @param projectFieldMappings
         *        A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira projects to
         *        Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to
         *        Jira fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectFieldMappings(DataSourceToIndexFieldMapping... projectFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira projects to Amazon
         * Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira fields. For
         * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data
         * source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder#build()} is called
         * immediately and its result is passed to {@link #projectFieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param projectFieldMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #projectFieldMappings(java.util.Collection<DataSourceToIndexFieldMapping>)
         */
        Builder projectFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... projectFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira work logs to
         * Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
         * fields. For more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">
         * Mapping data source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * 
         * @param workLogFieldMappings
         *        A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira work logs
         *        to Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to
         *        Jira fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workLogFieldMappings(Collection<DataSourceToIndexFieldMapping> workLogFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira work logs to
         * Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
         * fields. For more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">
         * Mapping data source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * 
         * @param workLogFieldMappings
         *        A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira work logs
         *        to Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to
         *        Jira fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder workLogFieldMappings(DataSourceToIndexFieldMapping... workLogFieldMappings);

        /**
         * <p>
         * A list of DataSourceToIndexFieldMapping objects that map attributes or field names of Jira work logs to
         * Amazon Kendra index field names. To create custom fields, use the UpdateIndex API before you map to Jira
         * fields. For more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">
         * Mapping data source fields</a>. The Jira data source field names must exist in your Jira custom metadata.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder#build()} is called
         * immediately and its result is passed to {@link #workLogFieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param workLogFieldMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #workLogFieldMappings(java.util.Collection<DataSourceToIndexFieldMapping>)
         */
        Builder workLogFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... workLogFieldMappings);

        /**
         * <p>
         * A list of regular expression patterns to include certain file paths, file names, and file types in your Jira
         * data source. Files that match the patterns are included in the index. Files that don't match the patterns are
         * excluded from the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion
         * pattern takes precedence and the file isn't included in the index.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns to include certain file paths, file names, and file types in
         *        your Jira data source. Files that match the patterns are included in the index. Files that don't match
         *        the patterns are excluded from the index. If a file matches both an inclusion pattern and an exclusion
         *        pattern, the exclusion pattern takes precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(Collection<String> inclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to include certain file paths, file names, and file types in your Jira
         * data source. Files that match the patterns are included in the index. Files that don't match the patterns are
         * excluded from the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion
         * pattern takes precedence and the file isn't included in the index.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns to include certain file paths, file names, and file types in
         *        your Jira data source. Files that match the patterns are included in the index. Files that don't match
         *        the patterns are excluded from the index. If a file matches both an inclusion pattern and an exclusion
         *        pattern, the exclusion pattern takes precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(String... inclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to exclude certain file paths, file names, and file types in your Jira
         * data source. Files that match the patterns are excluded from the index. Files that don’t match the patterns
         * are included in the index. If a file matches both an inclusion pattern and an exclusion pattern, the
         * exclusion pattern takes precedence and the file isn't included in the index.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of regular expression patterns to exclude certain file paths, file names, and file types in
         *        your Jira data source. Files that match the patterns are excluded from the index. Files that don’t
         *        match the patterns are included in the index. If a file matches both an inclusion pattern and an
         *        exclusion pattern, the exclusion pattern takes precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(Collection<String> exclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to exclude certain file paths, file names, and file types in your Jira
         * data source. Files that match the patterns are excluded from the index. Files that don’t match the patterns
         * are included in the index. If a file matches both an inclusion pattern and an exclusion pattern, the
         * exclusion pattern takes precedence and the file isn't included in the index.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of regular expression patterns to exclude certain file paths, file names, and file types in
         *        your Jira data source. Files that match the patterns are excluded from the index. Files that don’t
         *        match the patterns are included in the index. If a file matches both an inclusion pattern and an
         *        exclusion pattern, the exclusion pattern takes precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(String... exclusionPatterns);

        /**
         * <p>
         * Configuration information for an Amazon Virtual Private Cloud to connect to your Jira. Your Jira account must
         * reside inside your VPC.
         * </p>
         * 
         * @param vpcConfiguration
         *        Configuration information for an Amazon Virtual Private Cloud to connect to your Jira. Your Jira
         *        account must reside inside your VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration);

        /**
         * <p>
         * Configuration information for an Amazon Virtual Private Cloud to connect to your Jira. Your Jira account must
         * reside inside your VPC.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataSourceVpcConfiguration.Builder}
         * avoiding the need to create one manually via {@link DataSourceVpcConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataSourceVpcConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #vpcConfiguration(DataSourceVpcConfiguration)}.
         * 
         * @param vpcConfiguration
         *        a consumer that will call methods on {@link DataSourceVpcConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfiguration(DataSourceVpcConfiguration)
         */
        default Builder vpcConfiguration(Consumer<DataSourceVpcConfiguration.Builder> vpcConfiguration) {
            return vpcConfiguration(DataSourceVpcConfiguration.builder().applyMutation(vpcConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String jiraAccountUrl;

        private String secretArn;

        private Boolean useChangeLog;

        private List<String> project = DefaultSdkAutoConstructList.getInstance();

        private List<String> issueType = DefaultSdkAutoConstructList.getInstance();

        private List<String> status = DefaultSdkAutoConstructList.getInstance();

        private List<String> issueSubEntityFilter = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> attachmentFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> commentFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> issueFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> projectFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> workLogFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<String> inclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private List<String> exclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private DataSourceVpcConfiguration vpcConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(JiraConfiguration model) {
            jiraAccountUrl(model.jiraAccountUrl);
            secretArn(model.secretArn);
            useChangeLog(model.useChangeLog);
            project(model.project);
            issueType(model.issueType);
            status(model.status);
            issueSubEntityFilterWithStrings(model.issueSubEntityFilter);
            attachmentFieldMappings(model.attachmentFieldMappings);
            commentFieldMappings(model.commentFieldMappings);
            issueFieldMappings(model.issueFieldMappings);
            projectFieldMappings(model.projectFieldMappings);
            workLogFieldMappings(model.workLogFieldMappings);
            inclusionPatterns(model.inclusionPatterns);
            exclusionPatterns(model.exclusionPatterns);
            vpcConfiguration(model.vpcConfiguration);
        }

        public final String getJiraAccountUrl() {
            return jiraAccountUrl;
        }

        public final void setJiraAccountUrl(String jiraAccountUrl) {
            this.jiraAccountUrl = jiraAccountUrl;
        }

        @Override
        public final Builder jiraAccountUrl(String jiraAccountUrl) {
            this.jiraAccountUrl = jiraAccountUrl;
            return this;
        }

        public final String getSecretArn() {
            return secretArn;
        }

        public final void setSecretArn(String secretArn) {
            this.secretArn = secretArn;
        }

        @Override
        public final Builder secretArn(String secretArn) {
            this.secretArn = secretArn;
            return this;
        }

        public final Boolean getUseChangeLog() {
            return useChangeLog;
        }

        public final void setUseChangeLog(Boolean useChangeLog) {
            this.useChangeLog = useChangeLog;
        }

        @Override
        public final Builder useChangeLog(Boolean useChangeLog) {
            this.useChangeLog = useChangeLog;
            return this;
        }

        public final Collection<String> getProject() {
            if (project instanceof SdkAutoConstructList) {
                return null;
            }
            return project;
        }

        public final void setProject(Collection<String> project) {
            this.project = ProjectCopier.copy(project);
        }

        @Override
        public final Builder project(Collection<String> project) {
            this.project = ProjectCopier.copy(project);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder project(String... project) {
            project(Arrays.asList(project));
            return this;
        }

        public final Collection<String> getIssueType() {
            if (issueType instanceof SdkAutoConstructList) {
                return null;
            }
            return issueType;
        }

        public final void setIssueType(Collection<String> issueType) {
            this.issueType = IssueTypeCopier.copy(issueType);
        }

        @Override
        public final Builder issueType(Collection<String> issueType) {
            this.issueType = IssueTypeCopier.copy(issueType);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder issueType(String... issueType) {
            issueType(Arrays.asList(issueType));
            return this;
        }

        public final Collection<String> getStatus() {
            if (status instanceof SdkAutoConstructList) {
                return null;
            }
            return status;
        }

        public final void setStatus(Collection<String> status) {
            this.status = JiraStatusCopier.copy(status);
        }

        @Override
        public final Builder status(Collection<String> status) {
            this.status = JiraStatusCopier.copy(status);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder status(String... status) {
            status(Arrays.asList(status));
            return this;
        }

        public final Collection<String> getIssueSubEntityFilter() {
            if (issueSubEntityFilter instanceof SdkAutoConstructList) {
                return null;
            }
            return issueSubEntityFilter;
        }

        public final void setIssueSubEntityFilter(Collection<String> issueSubEntityFilter) {
            this.issueSubEntityFilter = IssueSubEntityFilterCopier.copy(issueSubEntityFilter);
        }

        @Override
        public final Builder issueSubEntityFilterWithStrings(Collection<String> issueSubEntityFilter) {
            this.issueSubEntityFilter = IssueSubEntityFilterCopier.copy(issueSubEntityFilter);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder issueSubEntityFilterWithStrings(String... issueSubEntityFilter) {
            issueSubEntityFilterWithStrings(Arrays.asList(issueSubEntityFilter));
            return this;
        }

        @Override
        public final Builder issueSubEntityFilter(Collection<IssueSubEntity> issueSubEntityFilter) {
            this.issueSubEntityFilter = IssueSubEntityFilterCopier.copyEnumToString(issueSubEntityFilter);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder issueSubEntityFilter(IssueSubEntity... issueSubEntityFilter) {
            issueSubEntityFilter(Arrays.asList(issueSubEntityFilter));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getAttachmentFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.attachmentFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttachmentFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> attachmentFieldMappings) {
            this.attachmentFieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(attachmentFieldMappings);
        }

        @Override
        public final Builder attachmentFieldMappings(Collection<DataSourceToIndexFieldMapping> attachmentFieldMappings) {
            this.attachmentFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(attachmentFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachmentFieldMappings(DataSourceToIndexFieldMapping... attachmentFieldMappings) {
            attachmentFieldMappings(Arrays.asList(attachmentFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attachmentFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... attachmentFieldMappings) {
            attachmentFieldMappings(Stream.of(attachmentFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getCommentFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.commentFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCommentFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> commentFieldMappings) {
            this.commentFieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(commentFieldMappings);
        }

        @Override
        public final Builder commentFieldMappings(Collection<DataSourceToIndexFieldMapping> commentFieldMappings) {
            this.commentFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(commentFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder commentFieldMappings(DataSourceToIndexFieldMapping... commentFieldMappings) {
            commentFieldMappings(Arrays.asList(commentFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder commentFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... commentFieldMappings) {
            commentFieldMappings(Stream.of(commentFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getIssueFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.issueFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setIssueFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> issueFieldMappings) {
            this.issueFieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(issueFieldMappings);
        }

        @Override
        public final Builder issueFieldMappings(Collection<DataSourceToIndexFieldMapping> issueFieldMappings) {
            this.issueFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(issueFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder issueFieldMappings(DataSourceToIndexFieldMapping... issueFieldMappings) {
            issueFieldMappings(Arrays.asList(issueFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder issueFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... issueFieldMappings) {
            issueFieldMappings(Stream.of(issueFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getProjectFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.projectFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setProjectFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> projectFieldMappings) {
            this.projectFieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(projectFieldMappings);
        }

        @Override
        public final Builder projectFieldMappings(Collection<DataSourceToIndexFieldMapping> projectFieldMappings) {
            this.projectFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(projectFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder projectFieldMappings(DataSourceToIndexFieldMapping... projectFieldMappings) {
            projectFieldMappings(Arrays.asList(projectFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder projectFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... projectFieldMappings) {
            projectFieldMappings(Stream.of(projectFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getWorkLogFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.workLogFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setWorkLogFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> workLogFieldMappings) {
            this.workLogFieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(workLogFieldMappings);
        }

        @Override
        public final Builder workLogFieldMappings(Collection<DataSourceToIndexFieldMapping> workLogFieldMappings) {
            this.workLogFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(workLogFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder workLogFieldMappings(DataSourceToIndexFieldMapping... workLogFieldMappings) {
            workLogFieldMappings(Arrays.asList(workLogFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder workLogFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... workLogFieldMappings) {
            workLogFieldMappings(Stream.of(workLogFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getInclusionPatterns() {
            if (inclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return inclusionPatterns;
        }

        public final void setInclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
        }

        @Override
        public final Builder inclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inclusionPatterns(String... inclusionPatterns) {
            inclusionPatterns(Arrays.asList(inclusionPatterns));
            return this;
        }

        public final Collection<String> getExclusionPatterns() {
            if (exclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return exclusionPatterns;
        }

        public final void setExclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
        }

        @Override
        public final Builder exclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exclusionPatterns(String... exclusionPatterns) {
            exclusionPatterns(Arrays.asList(exclusionPatterns));
            return this;
        }

        public final DataSourceVpcConfiguration.Builder getVpcConfiguration() {
            return vpcConfiguration != null ? vpcConfiguration.toBuilder() : null;
        }

        public final void setVpcConfiguration(DataSourceVpcConfiguration.BuilderImpl vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration != null ? vpcConfiguration.build() : null;
        }

        @Override
        public final Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration;
            return this;
        }

        @Override
        public JiraConfiguration build() {
            return new JiraConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
