/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the configuration information required for Amazon Kendra Web Crawler.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WebCrawlerConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<WebCrawlerConfiguration.Builder, WebCrawlerConfiguration> {
    private static final SdkField<Urls> URLS_FIELD = SdkField.<Urls> builder(MarshallingType.SDK_POJO).memberName("Urls")
            .getter(getter(WebCrawlerConfiguration::urls)).setter(setter(Builder::urls)).constructor(Urls::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Urls").build()).build();

    private static final SdkField<Integer> CRAWL_DEPTH_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("CrawlDepth").getter(getter(WebCrawlerConfiguration::crawlDepth)).setter(setter(Builder::crawlDepth))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlDepth").build()).build();

    private static final SdkField<Integer> MAX_LINKS_PER_PAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxLinksPerPage").getter(getter(WebCrawlerConfiguration::maxLinksPerPage))
            .setter(setter(Builder::maxLinksPerPage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxLinksPerPage").build()).build();

    private static final SdkField<Float> MAX_CONTENT_SIZE_PER_PAGE_IN_MEGA_BYTES_FIELD = SdkField
            .<Float> builder(MarshallingType.FLOAT)
            .memberName("MaxContentSizePerPageInMegaBytes")
            .getter(getter(WebCrawlerConfiguration::maxContentSizePerPageInMegaBytes))
            .setter(setter(Builder::maxContentSizePerPageInMegaBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxContentSizePerPageInMegaBytes")
                    .build()).build();

    private static final SdkField<Integer> MAX_URLS_PER_MINUTE_CRAWL_RATE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("MaxUrlsPerMinuteCrawlRate")
            .getter(getter(WebCrawlerConfiguration::maxUrlsPerMinuteCrawlRate))
            .setter(setter(Builder::maxUrlsPerMinuteCrawlRate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxUrlsPerMinuteCrawlRate").build())
            .build();

    private static final SdkField<List<String>> URL_INCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("UrlInclusionPatterns")
            .getter(getter(WebCrawlerConfiguration::urlInclusionPatterns))
            .setter(setter(Builder::urlInclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UrlInclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> URL_EXCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("UrlExclusionPatterns")
            .getter(getter(WebCrawlerConfiguration::urlExclusionPatterns))
            .setter(setter(Builder::urlExclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UrlExclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ProxyConfiguration> PROXY_CONFIGURATION_FIELD = SdkField
            .<ProxyConfiguration> builder(MarshallingType.SDK_POJO).memberName("ProxyConfiguration")
            .getter(getter(WebCrawlerConfiguration::proxyConfiguration)).setter(setter(Builder::proxyConfiguration))
            .constructor(ProxyConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProxyConfiguration").build())
            .build();

    private static final SdkField<AuthenticationConfiguration> AUTHENTICATION_CONFIGURATION_FIELD = SdkField
            .<AuthenticationConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("AuthenticationConfiguration")
            .getter(getter(WebCrawlerConfiguration::authenticationConfiguration))
            .setter(setter(Builder::authenticationConfiguration))
            .constructor(AuthenticationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(URLS_FIELD, CRAWL_DEPTH_FIELD,
            MAX_LINKS_PER_PAGE_FIELD, MAX_CONTENT_SIZE_PER_PAGE_IN_MEGA_BYTES_FIELD, MAX_URLS_PER_MINUTE_CRAWL_RATE_FIELD,
            URL_INCLUSION_PATTERNS_FIELD, URL_EXCLUSION_PATTERNS_FIELD, PROXY_CONFIGURATION_FIELD,
            AUTHENTICATION_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final Urls urls;

    private final Integer crawlDepth;

    private final Integer maxLinksPerPage;

    private final Float maxContentSizePerPageInMegaBytes;

    private final Integer maxUrlsPerMinuteCrawlRate;

    private final List<String> urlInclusionPatterns;

    private final List<String> urlExclusionPatterns;

    private final ProxyConfiguration proxyConfiguration;

    private final AuthenticationConfiguration authenticationConfiguration;

    private WebCrawlerConfiguration(BuilderImpl builder) {
        this.urls = builder.urls;
        this.crawlDepth = builder.crawlDepth;
        this.maxLinksPerPage = builder.maxLinksPerPage;
        this.maxContentSizePerPageInMegaBytes = builder.maxContentSizePerPageInMegaBytes;
        this.maxUrlsPerMinuteCrawlRate = builder.maxUrlsPerMinuteCrawlRate;
        this.urlInclusionPatterns = builder.urlInclusionPatterns;
        this.urlExclusionPatterns = builder.urlExclusionPatterns;
        this.proxyConfiguration = builder.proxyConfiguration;
        this.authenticationConfiguration = builder.authenticationConfiguration;
    }

    /**
     * <p>
     * Specifies the seed or starting point URLs of the websites or the sitemap URLs of the websites you want to crawl.
     * </p>
     * <p>
     * You can include website subdomains. You can list up to 100 seed URLs and up to three sitemap URLs.
     * </p>
     * <p>
     * You can only crawl websites that use the secure communication protocol, Hypertext Transfer Protocol Secure
     * (HTTPS). If you receive an error when crawling a website, it could be that the website is blocked from crawling.
     * </p>
     * <p>
     * <i>When selecting websites to index, you must adhere to the <a href="https://aws.amazon.com/aup/">Amazon
     * Acceptable Use Policy</a> and all other Amazon terms. Remember that you must only use Amazon Kendra Web Crawler
     * to index your own web pages, or web pages that you have authorization to index.</i>
     * </p>
     * 
     * @return Specifies the seed or starting point URLs of the websites or the sitemap URLs of the websites you want to
     *         crawl.</p>
     *         <p>
     *         You can include website subdomains. You can list up to 100 seed URLs and up to three sitemap URLs.
     *         </p>
     *         <p>
     *         You can only crawl websites that use the secure communication protocol, Hypertext Transfer Protocol
     *         Secure (HTTPS). If you receive an error when crawling a website, it could be that the website is blocked
     *         from crawling.
     *         </p>
     *         <p>
     *         <i>When selecting websites to index, you must adhere to the <a href="https://aws.amazon.com/aup/">Amazon
     *         Acceptable Use Policy</a> and all other Amazon terms. Remember that you must only use Amazon Kendra Web
     *         Crawler to index your own web pages, or web pages that you have authorization to index.</i>
     */
    public final Urls urls() {
        return urls;
    }

    /**
     * <p>
     * The 'depth' or number of levels from the seed level to crawl. For example, the seed URL page is depth 1 and any
     * hyperlinks on this page that are also crawled are depth 2.
     * </p>
     * 
     * @return The 'depth' or number of levels from the seed level to crawl. For example, the seed URL page is depth 1
     *         and any hyperlinks on this page that are also crawled are depth 2.
     */
    public final Integer crawlDepth() {
        return crawlDepth;
    }

    /**
     * <p>
     * The maximum number of URLs on a web page to include when crawling a website. This number is per web page.
     * </p>
     * <p>
     * As a website’s web pages are crawled, any URLs the web pages link to are also crawled. URLs on a web page are
     * crawled in order of appearance.
     * </p>
     * <p>
     * The default maximum links per page is 100.
     * </p>
     * 
     * @return The maximum number of URLs on a web page to include when crawling a website. This number is per web
     *         page.</p>
     *         <p>
     *         As a website’s web pages are crawled, any URLs the web pages link to are also crawled. URLs on a web page
     *         are crawled in order of appearance.
     *         </p>
     *         <p>
     *         The default maximum links per page is 100.
     */
    public final Integer maxLinksPerPage() {
        return maxLinksPerPage;
    }

    /**
     * <p>
     * The maximum size (in MB) of a web page or attachment to crawl.
     * </p>
     * <p>
     * Files larger than this size (in MB) are skipped/not crawled.
     * </p>
     * <p>
     * The default maximum size of a web page or attachment is set to 50 MB.
     * </p>
     * 
     * @return The maximum size (in MB) of a web page or attachment to crawl.</p>
     *         <p>
     *         Files larger than this size (in MB) are skipped/not crawled.
     *         </p>
     *         <p>
     *         The default maximum size of a web page or attachment is set to 50 MB.
     */
    public final Float maxContentSizePerPageInMegaBytes() {
        return maxContentSizePerPageInMegaBytes;
    }

    /**
     * <p>
     * The maximum number of URLs crawled per website host per minute.
     * </p>
     * <p>
     * A minimum of one URL is required.
     * </p>
     * <p>
     * The default maximum number of URLs crawled per website host per minute is 300.
     * </p>
     * 
     * @return The maximum number of URLs crawled per website host per minute.</p>
     *         <p>
     *         A minimum of one URL is required.
     *         </p>
     *         <p>
     *         The default maximum number of URLs crawled per website host per minute is 300.
     */
    public final Integer maxUrlsPerMinuteCrawlRate() {
        return maxUrlsPerMinuteCrawlRate;
    }

    /**
     * For responses, this returns true if the service returned a value for the UrlInclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUrlInclusionPatterns() {
        return urlInclusionPatterns != null && !(urlInclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to include certain URLs to crawl. URLs that match the patterns are included
     * in the index. URLs that don't match the patterns are excluded from the index. If a URL matches both an inclusion
     * and exclusion pattern, the exclusion pattern takes precedence and the URL file isn't included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUrlInclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to include certain URLs to crawl. URLs that match the patterns are
     *         included in the index. URLs that don't match the patterns are excluded from the index. If a URL matches
     *         both an inclusion and exclusion pattern, the exclusion pattern takes precedence and the URL file isn't
     *         included in the index.
     */
    public final List<String> urlInclusionPatterns() {
        return urlInclusionPatterns;
    }

    /**
     * For responses, this returns true if the service returned a value for the UrlExclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUrlExclusionPatterns() {
        return urlExclusionPatterns != null && !(urlExclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to exclude certain URLs to crawl. URLs that match the patterns are excluded
     * from the index. URLs that don't match the patterns are included in the index. If a URL matches both an inclusion
     * and exclusion pattern, the exclusion pattern takes precedence and the URL file isn't included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUrlExclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to exclude certain URLs to crawl. URLs that match the patterns are
     *         excluded from the index. URLs that don't match the patterns are included in the index. If a URL matches
     *         both an inclusion and exclusion pattern, the exclusion pattern takes precedence and the URL file isn't
     *         included in the index.
     */
    public final List<String> urlExclusionPatterns() {
        return urlExclusionPatterns;
    }

    /**
     * <p>
     * Configuration information required to connect to your internal websites via a web proxy.
     * </p>
     * <p>
     * You must provide the website host name and port number. For example, the host name of
     * https://a.example.com/page1.html is "a.example.com" and the port is 443, the standard port for HTTPS.
     * </p>
     * <p>
     * Web proxy credentials are optional and you can use them to connect to a web proxy server that requires basic
     * authentication. To store web proxy credentials, you use a secret in <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">Secrets Manager</a>.
     * </p>
     * 
     * @return Configuration information required to connect to your internal websites via a web proxy.</p>
     *         <p>
     *         You must provide the website host name and port number. For example, the host name of
     *         https://a.example.com/page1.html is "a.example.com" and the port is 443, the standard port for HTTPS.
     *         </p>
     *         <p>
     *         Web proxy credentials are optional and you can use them to connect to a web proxy server that requires
     *         basic authentication. To store web proxy credentials, you use a secret in <a
     *         href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">Secrets Manager</a>.
     */
    public final ProxyConfiguration proxyConfiguration() {
        return proxyConfiguration;
    }

    /**
     * <p>
     * Configuration information required to connect to websites using authentication.
     * </p>
     * <p>
     * You can connect to websites using basic authentication of user name and password. You use a secret in <a
     * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">Secrets Manager</a> to store your
     * authentication credentials.
     * </p>
     * <p>
     * You must provide the website host name and port number. For example, the host name of
     * https://a.example.com/page1.html is "a.example.com" and the port is 443, the standard port for HTTPS.
     * </p>
     * 
     * @return Configuration information required to connect to websites using authentication.</p>
     *         <p>
     *         You can connect to websites using basic authentication of user name and password. You use a secret in <a
     *         href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">Secrets Manager</a> to
     *         store your authentication credentials.
     *         </p>
     *         <p>
     *         You must provide the website host name and port number. For example, the host name of
     *         https://a.example.com/page1.html is "a.example.com" and the port is 443, the standard port for HTTPS.
     */
    public final AuthenticationConfiguration authenticationConfiguration() {
        return authenticationConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(urls());
        hashCode = 31 * hashCode + Objects.hashCode(crawlDepth());
        hashCode = 31 * hashCode + Objects.hashCode(maxLinksPerPage());
        hashCode = 31 * hashCode + Objects.hashCode(maxContentSizePerPageInMegaBytes());
        hashCode = 31 * hashCode + Objects.hashCode(maxUrlsPerMinuteCrawlRate());
        hashCode = 31 * hashCode + Objects.hashCode(hasUrlInclusionPatterns() ? urlInclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUrlExclusionPatterns() ? urlExclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(proxyConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(authenticationConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WebCrawlerConfiguration)) {
            return false;
        }
        WebCrawlerConfiguration other = (WebCrawlerConfiguration) obj;
        return Objects.equals(urls(), other.urls()) && Objects.equals(crawlDepth(), other.crawlDepth())
                && Objects.equals(maxLinksPerPage(), other.maxLinksPerPage())
                && Objects.equals(maxContentSizePerPageInMegaBytes(), other.maxContentSizePerPageInMegaBytes())
                && Objects.equals(maxUrlsPerMinuteCrawlRate(), other.maxUrlsPerMinuteCrawlRate())
                && hasUrlInclusionPatterns() == other.hasUrlInclusionPatterns()
                && Objects.equals(urlInclusionPatterns(), other.urlInclusionPatterns())
                && hasUrlExclusionPatterns() == other.hasUrlExclusionPatterns()
                && Objects.equals(urlExclusionPatterns(), other.urlExclusionPatterns())
                && Objects.equals(proxyConfiguration(), other.proxyConfiguration())
                && Objects.equals(authenticationConfiguration(), other.authenticationConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WebCrawlerConfiguration").add("Urls", urls()).add("CrawlDepth", crawlDepth())
                .add("MaxLinksPerPage", maxLinksPerPage())
                .add("MaxContentSizePerPageInMegaBytes", maxContentSizePerPageInMegaBytes())
                .add("MaxUrlsPerMinuteCrawlRate", maxUrlsPerMinuteCrawlRate())
                .add("UrlInclusionPatterns", hasUrlInclusionPatterns() ? urlInclusionPatterns() : null)
                .add("UrlExclusionPatterns", hasUrlExclusionPatterns() ? urlExclusionPatterns() : null)
                .add("ProxyConfiguration", proxyConfiguration())
                .add("AuthenticationConfiguration", authenticationConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Urls":
            return Optional.ofNullable(clazz.cast(urls()));
        case "CrawlDepth":
            return Optional.ofNullable(clazz.cast(crawlDepth()));
        case "MaxLinksPerPage":
            return Optional.ofNullable(clazz.cast(maxLinksPerPage()));
        case "MaxContentSizePerPageInMegaBytes":
            return Optional.ofNullable(clazz.cast(maxContentSizePerPageInMegaBytes()));
        case "MaxUrlsPerMinuteCrawlRate":
            return Optional.ofNullable(clazz.cast(maxUrlsPerMinuteCrawlRate()));
        case "UrlInclusionPatterns":
            return Optional.ofNullable(clazz.cast(urlInclusionPatterns()));
        case "UrlExclusionPatterns":
            return Optional.ofNullable(clazz.cast(urlExclusionPatterns()));
        case "ProxyConfiguration":
            return Optional.ofNullable(clazz.cast(proxyConfiguration()));
        case "AuthenticationConfiguration":
            return Optional.ofNullable(clazz.cast(authenticationConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<WebCrawlerConfiguration, T> g) {
        return obj -> g.apply((WebCrawlerConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WebCrawlerConfiguration> {
        /**
         * <p>
         * Specifies the seed or starting point URLs of the websites or the sitemap URLs of the websites you want to
         * crawl.
         * </p>
         * <p>
         * You can include website subdomains. You can list up to 100 seed URLs and up to three sitemap URLs.
         * </p>
         * <p>
         * You can only crawl websites that use the secure communication protocol, Hypertext Transfer Protocol Secure
         * (HTTPS). If you receive an error when crawling a website, it could be that the website is blocked from
         * crawling.
         * </p>
         * <p>
         * <i>When selecting websites to index, you must adhere to the <a href="https://aws.amazon.com/aup/">Amazon
         * Acceptable Use Policy</a> and all other Amazon terms. Remember that you must only use Amazon Kendra Web
         * Crawler to index your own web pages, or web pages that you have authorization to index.</i>
         * </p>
         * 
         * @param urls
         *        Specifies the seed or starting point URLs of the websites or the sitemap URLs of the websites you want
         *        to crawl.</p>
         *        <p>
         *        You can include website subdomains. You can list up to 100 seed URLs and up to three sitemap URLs.
         *        </p>
         *        <p>
         *        You can only crawl websites that use the secure communication protocol, Hypertext Transfer Protocol
         *        Secure (HTTPS). If you receive an error when crawling a website, it could be that the website is
         *        blocked from crawling.
         *        </p>
         *        <p>
         *        <i>When selecting websites to index, you must adhere to the <a
         *        href="https://aws.amazon.com/aup/">Amazon Acceptable Use Policy</a> and all other Amazon terms.
         *        Remember that you must only use Amazon Kendra Web Crawler to index your own web pages, or web pages
         *        that you have authorization to index.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder urls(Urls urls);

        /**
         * <p>
         * Specifies the seed or starting point URLs of the websites or the sitemap URLs of the websites you want to
         * crawl.
         * </p>
         * <p>
         * You can include website subdomains. You can list up to 100 seed URLs and up to three sitemap URLs.
         * </p>
         * <p>
         * You can only crawl websites that use the secure communication protocol, Hypertext Transfer Protocol Secure
         * (HTTPS). If you receive an error when crawling a website, it could be that the website is blocked from
         * crawling.
         * </p>
         * <p>
         * <i>When selecting websites to index, you must adhere to the <a href="https://aws.amazon.com/aup/">Amazon
         * Acceptable Use Policy</a> and all other Amazon terms. Remember that you must only use Amazon Kendra Web
         * Crawler to index your own web pages, or web pages that you have authorization to index.</i>
         * </p>
         * This is a convenience method that creates an instance of the {@link Urls.Builder} avoiding the need to create
         * one manually via {@link Urls#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Urls.Builder#build()} is called immediately and its result is
         * passed to {@link #urls(Urls)}.
         * 
         * @param urls
         *        a consumer that will call methods on {@link Urls.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #urls(Urls)
         */
        default Builder urls(Consumer<Urls.Builder> urls) {
            return urls(Urls.builder().applyMutation(urls).build());
        }

        /**
         * <p>
         * The 'depth' or number of levels from the seed level to crawl. For example, the seed URL page is depth 1 and
         * any hyperlinks on this page that are also crawled are depth 2.
         * </p>
         * 
         * @param crawlDepth
         *        The 'depth' or number of levels from the seed level to crawl. For example, the seed URL page is depth
         *        1 and any hyperlinks on this page that are also crawled are depth 2.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlDepth(Integer crawlDepth);

        /**
         * <p>
         * The maximum number of URLs on a web page to include when crawling a website. This number is per web page.
         * </p>
         * <p>
         * As a website’s web pages are crawled, any URLs the web pages link to are also crawled. URLs on a web page are
         * crawled in order of appearance.
         * </p>
         * <p>
         * The default maximum links per page is 100.
         * </p>
         * 
         * @param maxLinksPerPage
         *        The maximum number of URLs on a web page to include when crawling a website. This number is per web
         *        page.</p>
         *        <p>
         *        As a website’s web pages are crawled, any URLs the web pages link to are also crawled. URLs on a web
         *        page are crawled in order of appearance.
         *        </p>
         *        <p>
         *        The default maximum links per page is 100.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxLinksPerPage(Integer maxLinksPerPage);

        /**
         * <p>
         * The maximum size (in MB) of a web page or attachment to crawl.
         * </p>
         * <p>
         * Files larger than this size (in MB) are skipped/not crawled.
         * </p>
         * <p>
         * The default maximum size of a web page or attachment is set to 50 MB.
         * </p>
         * 
         * @param maxContentSizePerPageInMegaBytes
         *        The maximum size (in MB) of a web page or attachment to crawl.</p>
         *        <p>
         *        Files larger than this size (in MB) are skipped/not crawled.
         *        </p>
         *        <p>
         *        The default maximum size of a web page or attachment is set to 50 MB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxContentSizePerPageInMegaBytes(Float maxContentSizePerPageInMegaBytes);

        /**
         * <p>
         * The maximum number of URLs crawled per website host per minute.
         * </p>
         * <p>
         * A minimum of one URL is required.
         * </p>
         * <p>
         * The default maximum number of URLs crawled per website host per minute is 300.
         * </p>
         * 
         * @param maxUrlsPerMinuteCrawlRate
         *        The maximum number of URLs crawled per website host per minute.</p>
         *        <p>
         *        A minimum of one URL is required.
         *        </p>
         *        <p>
         *        The default maximum number of URLs crawled per website host per minute is 300.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxUrlsPerMinuteCrawlRate(Integer maxUrlsPerMinuteCrawlRate);

        /**
         * <p>
         * A list of regular expression patterns to include certain URLs to crawl. URLs that match the patterns are
         * included in the index. URLs that don't match the patterns are excluded from the index. If a URL matches both
         * an inclusion and exclusion pattern, the exclusion pattern takes precedence and the URL file isn't included in
         * the index.
         * </p>
         * 
         * @param urlInclusionPatterns
         *        A list of regular expression patterns to include certain URLs to crawl. URLs that match the patterns
         *        are included in the index. URLs that don't match the patterns are excluded from the index. If a URL
         *        matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence and the URL
         *        file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder urlInclusionPatterns(Collection<String> urlInclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to include certain URLs to crawl. URLs that match the patterns are
         * included in the index. URLs that don't match the patterns are excluded from the index. If a URL matches both
         * an inclusion and exclusion pattern, the exclusion pattern takes precedence and the URL file isn't included in
         * the index.
         * </p>
         * 
         * @param urlInclusionPatterns
         *        A list of regular expression patterns to include certain URLs to crawl. URLs that match the patterns
         *        are included in the index. URLs that don't match the patterns are excluded from the index. If a URL
         *        matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence and the URL
         *        file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder urlInclusionPatterns(String... urlInclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to exclude certain URLs to crawl. URLs that match the patterns are
         * excluded from the index. URLs that don't match the patterns are included in the index. If a URL matches both
         * an inclusion and exclusion pattern, the exclusion pattern takes precedence and the URL file isn't included in
         * the index.
         * </p>
         * 
         * @param urlExclusionPatterns
         *        A list of regular expression patterns to exclude certain URLs to crawl. URLs that match the patterns
         *        are excluded from the index. URLs that don't match the patterns are included in the index. If a URL
         *        matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence and the URL
         *        file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder urlExclusionPatterns(Collection<String> urlExclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to exclude certain URLs to crawl. URLs that match the patterns are
         * excluded from the index. URLs that don't match the patterns are included in the index. If a URL matches both
         * an inclusion and exclusion pattern, the exclusion pattern takes precedence and the URL file isn't included in
         * the index.
         * </p>
         * 
         * @param urlExclusionPatterns
         *        A list of regular expression patterns to exclude certain URLs to crawl. URLs that match the patterns
         *        are excluded from the index. URLs that don't match the patterns are included in the index. If a URL
         *        matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence and the URL
         *        file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder urlExclusionPatterns(String... urlExclusionPatterns);

        /**
         * <p>
         * Configuration information required to connect to your internal websites via a web proxy.
         * </p>
         * <p>
         * You must provide the website host name and port number. For example, the host name of
         * https://a.example.com/page1.html is "a.example.com" and the port is 443, the standard port for HTTPS.
         * </p>
         * <p>
         * Web proxy credentials are optional and you can use them to connect to a web proxy server that requires basic
         * authentication. To store web proxy credentials, you use a secret in <a
         * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">Secrets Manager</a>.
         * </p>
         * 
         * @param proxyConfiguration
         *        Configuration information required to connect to your internal websites via a web proxy.</p>
         *        <p>
         *        You must provide the website host name and port number. For example, the host name of
         *        https://a.example.com/page1.html is "a.example.com" and the port is 443, the standard port for HTTPS.
         *        </p>
         *        <p>
         *        Web proxy credentials are optional and you can use them to connect to a web proxy server that requires
         *        basic authentication. To store web proxy credentials, you use a secret in <a
         *        href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">Secrets Manager</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder proxyConfiguration(ProxyConfiguration proxyConfiguration);

        /**
         * <p>
         * Configuration information required to connect to your internal websites via a web proxy.
         * </p>
         * <p>
         * You must provide the website host name and port number. For example, the host name of
         * https://a.example.com/page1.html is "a.example.com" and the port is 443, the standard port for HTTPS.
         * </p>
         * <p>
         * Web proxy credentials are optional and you can use them to connect to a web proxy server that requires basic
         * authentication. To store web proxy credentials, you use a secret in <a
         * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">Secrets Manager</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProxyConfiguration.Builder} avoiding the
         * need to create one manually via {@link ProxyConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProxyConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #proxyConfiguration(ProxyConfiguration)}.
         * 
         * @param proxyConfiguration
         *        a consumer that will call methods on {@link ProxyConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #proxyConfiguration(ProxyConfiguration)
         */
        default Builder proxyConfiguration(Consumer<ProxyConfiguration.Builder> proxyConfiguration) {
            return proxyConfiguration(ProxyConfiguration.builder().applyMutation(proxyConfiguration).build());
        }

        /**
         * <p>
         * Configuration information required to connect to websites using authentication.
         * </p>
         * <p>
         * You can connect to websites using basic authentication of user name and password. You use a secret in <a
         * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">Secrets Manager</a> to store
         * your authentication credentials.
         * </p>
         * <p>
         * You must provide the website host name and port number. For example, the host name of
         * https://a.example.com/page1.html is "a.example.com" and the port is 443, the standard port for HTTPS.
         * </p>
         * 
         * @param authenticationConfiguration
         *        Configuration information required to connect to websites using authentication.</p>
         *        <p>
         *        You can connect to websites using basic authentication of user name and password. You use a secret in
         *        <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">Secrets Manager</a>
         *        to store your authentication credentials.
         *        </p>
         *        <p>
         *        You must provide the website host name and port number. For example, the host name of
         *        https://a.example.com/page1.html is "a.example.com" and the port is 443, the standard port for HTTPS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationConfiguration(AuthenticationConfiguration authenticationConfiguration);

        /**
         * <p>
         * Configuration information required to connect to websites using authentication.
         * </p>
         * <p>
         * You can connect to websites using basic authentication of user name and password. You use a secret in <a
         * href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/intro.html">Secrets Manager</a> to store
         * your authentication credentials.
         * </p>
         * <p>
         * You must provide the website host name and port number. For example, the host name of
         * https://a.example.com/page1.html is "a.example.com" and the port is 443, the standard port for HTTPS.
         * </p>
         * This is a convenience method that creates an instance of the {@link AuthenticationConfiguration.Builder}
         * avoiding the need to create one manually via {@link AuthenticationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AuthenticationConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #authenticationConfiguration(AuthenticationConfiguration)}.
         * 
         * @param authenticationConfiguration
         *        a consumer that will call methods on {@link AuthenticationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authenticationConfiguration(AuthenticationConfiguration)
         */
        default Builder authenticationConfiguration(Consumer<AuthenticationConfiguration.Builder> authenticationConfiguration) {
            return authenticationConfiguration(AuthenticationConfiguration.builder().applyMutation(authenticationConfiguration)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Urls urls;

        private Integer crawlDepth;

        private Integer maxLinksPerPage;

        private Float maxContentSizePerPageInMegaBytes;

        private Integer maxUrlsPerMinuteCrawlRate;

        private List<String> urlInclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private List<String> urlExclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private ProxyConfiguration proxyConfiguration;

        private AuthenticationConfiguration authenticationConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(WebCrawlerConfiguration model) {
            urls(model.urls);
            crawlDepth(model.crawlDepth);
            maxLinksPerPage(model.maxLinksPerPage);
            maxContentSizePerPageInMegaBytes(model.maxContentSizePerPageInMegaBytes);
            maxUrlsPerMinuteCrawlRate(model.maxUrlsPerMinuteCrawlRate);
            urlInclusionPatterns(model.urlInclusionPatterns);
            urlExclusionPatterns(model.urlExclusionPatterns);
            proxyConfiguration(model.proxyConfiguration);
            authenticationConfiguration(model.authenticationConfiguration);
        }

        public final Urls.Builder getUrls() {
            return urls != null ? urls.toBuilder() : null;
        }

        public final void setUrls(Urls.BuilderImpl urls) {
            this.urls = urls != null ? urls.build() : null;
        }

        @Override
        public final Builder urls(Urls urls) {
            this.urls = urls;
            return this;
        }

        public final Integer getCrawlDepth() {
            return crawlDepth;
        }

        public final void setCrawlDepth(Integer crawlDepth) {
            this.crawlDepth = crawlDepth;
        }

        @Override
        public final Builder crawlDepth(Integer crawlDepth) {
            this.crawlDepth = crawlDepth;
            return this;
        }

        public final Integer getMaxLinksPerPage() {
            return maxLinksPerPage;
        }

        public final void setMaxLinksPerPage(Integer maxLinksPerPage) {
            this.maxLinksPerPage = maxLinksPerPage;
        }

        @Override
        public final Builder maxLinksPerPage(Integer maxLinksPerPage) {
            this.maxLinksPerPage = maxLinksPerPage;
            return this;
        }

        public final Float getMaxContentSizePerPageInMegaBytes() {
            return maxContentSizePerPageInMegaBytes;
        }

        public final void setMaxContentSizePerPageInMegaBytes(Float maxContentSizePerPageInMegaBytes) {
            this.maxContentSizePerPageInMegaBytes = maxContentSizePerPageInMegaBytes;
        }

        @Override
        public final Builder maxContentSizePerPageInMegaBytes(Float maxContentSizePerPageInMegaBytes) {
            this.maxContentSizePerPageInMegaBytes = maxContentSizePerPageInMegaBytes;
            return this;
        }

        public final Integer getMaxUrlsPerMinuteCrawlRate() {
            return maxUrlsPerMinuteCrawlRate;
        }

        public final void setMaxUrlsPerMinuteCrawlRate(Integer maxUrlsPerMinuteCrawlRate) {
            this.maxUrlsPerMinuteCrawlRate = maxUrlsPerMinuteCrawlRate;
        }

        @Override
        public final Builder maxUrlsPerMinuteCrawlRate(Integer maxUrlsPerMinuteCrawlRate) {
            this.maxUrlsPerMinuteCrawlRate = maxUrlsPerMinuteCrawlRate;
            return this;
        }

        public final Collection<String> getUrlInclusionPatterns() {
            if (urlInclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return urlInclusionPatterns;
        }

        public final void setUrlInclusionPatterns(Collection<String> urlInclusionPatterns) {
            this.urlInclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(urlInclusionPatterns);
        }

        @Override
        public final Builder urlInclusionPatterns(Collection<String> urlInclusionPatterns) {
            this.urlInclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(urlInclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder urlInclusionPatterns(String... urlInclusionPatterns) {
            urlInclusionPatterns(Arrays.asList(urlInclusionPatterns));
            return this;
        }

        public final Collection<String> getUrlExclusionPatterns() {
            if (urlExclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return urlExclusionPatterns;
        }

        public final void setUrlExclusionPatterns(Collection<String> urlExclusionPatterns) {
            this.urlExclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(urlExclusionPatterns);
        }

        @Override
        public final Builder urlExclusionPatterns(Collection<String> urlExclusionPatterns) {
            this.urlExclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(urlExclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder urlExclusionPatterns(String... urlExclusionPatterns) {
            urlExclusionPatterns(Arrays.asList(urlExclusionPatterns));
            return this;
        }

        public final ProxyConfiguration.Builder getProxyConfiguration() {
            return proxyConfiguration != null ? proxyConfiguration.toBuilder() : null;
        }

        public final void setProxyConfiguration(ProxyConfiguration.BuilderImpl proxyConfiguration) {
            this.proxyConfiguration = proxyConfiguration != null ? proxyConfiguration.build() : null;
        }

        @Override
        public final Builder proxyConfiguration(ProxyConfiguration proxyConfiguration) {
            this.proxyConfiguration = proxyConfiguration;
            return this;
        }

        public final AuthenticationConfiguration.Builder getAuthenticationConfiguration() {
            return authenticationConfiguration != null ? authenticationConfiguration.toBuilder() : null;
        }

        public final void setAuthenticationConfiguration(AuthenticationConfiguration.BuilderImpl authenticationConfiguration) {
            this.authenticationConfiguration = authenticationConfiguration != null ? authenticationConfiguration.build() : null;
        }

        @Override
        public final Builder authenticationConfiguration(AuthenticationConfiguration authenticationConfiguration) {
            this.authenticationConfiguration = authenticationConfiguration;
            return this;
        }

        @Override
        public WebCrawlerConfiguration build() {
            return new WebCrawlerConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
