/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A set of featured results that are displayed at the top of your search results. Featured results are placed above all
 * other results for certain queries. If there's an exact match of a query, then one or more specific documents are
 * featured in the search results.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FeaturedResultsSet implements SdkPojo, Serializable,
        ToCopyableBuilder<FeaturedResultsSet.Builder, FeaturedResultsSet> {
    private static final SdkField<String> FEATURED_RESULTS_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FeaturedResultsSetId").getter(getter(FeaturedResultsSet::featuredResultsSetId))
            .setter(setter(Builder::featuredResultsSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeaturedResultsSetId").build())
            .build();

    private static final SdkField<String> FEATURED_RESULTS_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FeaturedResultsSetName").getter(getter(FeaturedResultsSet::featuredResultsSetName))
            .setter(setter(Builder::featuredResultsSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeaturedResultsSetName").build())
            .build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(FeaturedResultsSet::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(FeaturedResultsSet::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<String>> QUERY_TEXTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("QueryTexts")
            .getter(getter(FeaturedResultsSet::queryTexts))
            .setter(setter(Builder::queryTexts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryTexts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<FeaturedDocument>> FEATURED_DOCUMENTS_FIELD = SdkField
            .<List<FeaturedDocument>> builder(MarshallingType.LIST)
            .memberName("FeaturedDocuments")
            .getter(getter(FeaturedResultsSet::featuredDocuments))
            .setter(setter(Builder::featuredDocuments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FeaturedDocuments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FeaturedDocument> builder(MarshallingType.SDK_POJO)
                                            .constructor(FeaturedDocument::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> LAST_UPDATED_TIMESTAMP_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("LastUpdatedTimestamp").getter(getter(FeaturedResultsSet::lastUpdatedTimestamp))
            .setter(setter(Builder::lastUpdatedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTimestamp").build())
            .build();

    private static final SdkField<Long> CREATION_TIMESTAMP_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CreationTimestamp").getter(getter(FeaturedResultsSet::creationTimestamp))
            .setter(setter(Builder::creationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTimestamp").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FEATURED_RESULTS_SET_ID_FIELD,
            FEATURED_RESULTS_SET_NAME_FIELD, DESCRIPTION_FIELD, STATUS_FIELD, QUERY_TEXTS_FIELD, FEATURED_DOCUMENTS_FIELD,
            LAST_UPDATED_TIMESTAMP_FIELD, CREATION_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String featuredResultsSetId;

    private final String featuredResultsSetName;

    private final String description;

    private final String status;

    private final List<String> queryTexts;

    private final List<FeaturedDocument> featuredDocuments;

    private final Long lastUpdatedTimestamp;

    private final Long creationTimestamp;

    private FeaturedResultsSet(BuilderImpl builder) {
        this.featuredResultsSetId = builder.featuredResultsSetId;
        this.featuredResultsSetName = builder.featuredResultsSetName;
        this.description = builder.description;
        this.status = builder.status;
        this.queryTexts = builder.queryTexts;
        this.featuredDocuments = builder.featuredDocuments;
        this.lastUpdatedTimestamp = builder.lastUpdatedTimestamp;
        this.creationTimestamp = builder.creationTimestamp;
    }

    /**
     * <p>
     * The identifier of the set of featured results.
     * </p>
     * 
     * @return The identifier of the set of featured results.
     */
    public final String featuredResultsSetId() {
        return featuredResultsSetId;
    }

    /**
     * <p>
     * The name for the set of featured results.
     * </p>
     * 
     * @return The name for the set of featured results.
     */
    public final String featuredResultsSetName() {
        return featuredResultsSetName;
    }

    /**
     * <p>
     * The description for the set of featured results.
     * </p>
     * 
     * @return The description for the set of featured results.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The current status of the set of featured results. When the value is <code>ACTIVE</code>, featured results are
     * ready for use. You can still configure your settings before setting the status to <code>ACTIVE</code>. You can
     * set the status to <code>ACTIVE</code> or <code>INACTIVE</code> using the <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateFeaturedResultsSet.html"
     * >UpdateFeaturedResultsSet</a> API. The queries you specify for featured results must be unique per featured
     * results set for each index, whether the status is <code>ACTIVE</code> or <code>INACTIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FeaturedResultsSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the set of featured results. When the value is <code>ACTIVE</code>, featured
     *         results are ready for use. You can still configure your settings before setting the status to
     *         <code>ACTIVE</code>. You can set the status to <code>ACTIVE</code> or <code>INACTIVE</code> using the <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateFeaturedResultsSet.html">
     *         UpdateFeaturedResultsSet</a> API. The queries you specify for featured results must be unique per
     *         featured results set for each index, whether the status is <code>ACTIVE</code> or <code>INACTIVE</code>.
     * @see FeaturedResultsSetStatus
     */
    public final FeaturedResultsSetStatus status() {
        return FeaturedResultsSetStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the set of featured results. When the value is <code>ACTIVE</code>, featured results are
     * ready for use. You can still configure your settings before setting the status to <code>ACTIVE</code>. You can
     * set the status to <code>ACTIVE</code> or <code>INACTIVE</code> using the <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateFeaturedResultsSet.html"
     * >UpdateFeaturedResultsSet</a> API. The queries you specify for featured results must be unique per featured
     * results set for each index, whether the status is <code>ACTIVE</code> or <code>INACTIVE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FeaturedResultsSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the set of featured results. When the value is <code>ACTIVE</code>, featured
     *         results are ready for use. You can still configure your settings before setting the status to
     *         <code>ACTIVE</code>. You can set the status to <code>ACTIVE</code> or <code>INACTIVE</code> using the <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateFeaturedResultsSet.html">
     *         UpdateFeaturedResultsSet</a> API. The queries you specify for featured results must be unique per
     *         featured results set for each index, whether the status is <code>ACTIVE</code> or <code>INACTIVE</code>.
     * @see FeaturedResultsSetStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the QueryTexts property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasQueryTexts() {
        return queryTexts != null && !(queryTexts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of queries for featuring results.
     * </p>
     * <p>
     * Specific queries are mapped to specific documents for featuring in the results. If a query contains an exact
     * match, then one or more specific documents are featured in the results. The exact match applies to the full
     * query. For example, if you only specify 'Kendra', queries such as 'How does kendra semantically rank results?'
     * will not render the featured results. Featured results are designed for specific queries, rather than queries
     * that are too broad in scope.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasQueryTexts} method.
     * </p>
     * 
     * @return The list of queries for featuring results.</p>
     *         <p>
     *         Specific queries are mapped to specific documents for featuring in the results. If a query contains an
     *         exact match, then one or more specific documents are featured in the results. The exact match applies to
     *         the full query. For example, if you only specify 'Kendra', queries such as 'How does kendra semantically
     *         rank results?' will not render the featured results. Featured results are designed for specific queries,
     *         rather than queries that are too broad in scope.
     */
    public final List<String> queryTexts() {
        return queryTexts;
    }

    /**
     * For responses, this returns true if the service returned a value for the FeaturedDocuments property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFeaturedDocuments() {
        return featuredDocuments != null && !(featuredDocuments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of document IDs for the documents you want to feature at the top of the search results page. You can use
     * the <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Query.html">Query</a> API to search for specific
     * documents with their document IDs included in the result items, or you can use the console.
     * </p>
     * <p>
     * You can add up to four featured documents. You can request to increase this limit by contacting <a
     * href="http://aws.amazon.com/contact-us/">Support</a>.
     * </p>
     * <p>
     * Specific queries are mapped to specific documents for featuring in the results. If a query contains an exact
     * match, then one or more specific documents are featured in the results. The exact match applies to the full
     * query. For example, if you only specify 'Kendra', queries such as 'How does kendra semantically rank results?'
     * will not render the featured results. Featured results are designed for specific queries, rather than queries
     * that are too broad in scope.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeaturedDocuments} method.
     * </p>
     * 
     * @return The list of document IDs for the documents you want to feature at the top of the search results page. You
     *         can use the <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Query.html">Query</a> API to search
     *         for specific documents with their document IDs included in the result items, or you can use the
     *         console.</p>
     *         <p>
     *         You can add up to four featured documents. You can request to increase this limit by contacting <a
     *         href="http://aws.amazon.com/contact-us/">Support</a>.
     *         </p>
     *         <p>
     *         Specific queries are mapped to specific documents for featuring in the results. If a query contains an
     *         exact match, then one or more specific documents are featured in the results. The exact match applies to
     *         the full query. For example, if you only specify 'Kendra', queries such as 'How does kendra semantically
     *         rank results?' will not render the featured results. Featured results are designed for specific queries,
     *         rather than queries that are too broad in scope.
     */
    public final List<FeaturedDocument> featuredDocuments() {
        return featuredDocuments;
    }

    /**
     * <p>
     * The Unix timestamp when the set of featured results was last updated.
     * </p>
     * 
     * @return The Unix timestamp when the set of featured results was last updated.
     */
    public final Long lastUpdatedTimestamp() {
        return lastUpdatedTimestamp;
    }

    /**
     * <p>
     * The Unix timestamp when the set of featured results was created.
     * </p>
     * 
     * @return The Unix timestamp when the set of featured results was created.
     */
    public final Long creationTimestamp() {
        return creationTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(featuredResultsSetId());
        hashCode = 31 * hashCode + Objects.hashCode(featuredResultsSetName());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasQueryTexts() ? queryTexts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFeaturedDocuments() ? featuredDocuments() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(creationTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FeaturedResultsSet)) {
            return false;
        }
        FeaturedResultsSet other = (FeaturedResultsSet) obj;
        return Objects.equals(featuredResultsSetId(), other.featuredResultsSetId())
                && Objects.equals(featuredResultsSetName(), other.featuredResultsSetName())
                && Objects.equals(description(), other.description()) && Objects.equals(statusAsString(), other.statusAsString())
                && hasQueryTexts() == other.hasQueryTexts() && Objects.equals(queryTexts(), other.queryTexts())
                && hasFeaturedDocuments() == other.hasFeaturedDocuments()
                && Objects.equals(featuredDocuments(), other.featuredDocuments())
                && Objects.equals(lastUpdatedTimestamp(), other.lastUpdatedTimestamp())
                && Objects.equals(creationTimestamp(), other.creationTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FeaturedResultsSet").add("FeaturedResultsSetId", featuredResultsSetId())
                .add("FeaturedResultsSetName", featuredResultsSetName()).add("Description", description())
                .add("Status", statusAsString()).add("QueryTexts", hasQueryTexts() ? queryTexts() : null)
                .add("FeaturedDocuments", hasFeaturedDocuments() ? featuredDocuments() : null)
                .add("LastUpdatedTimestamp", lastUpdatedTimestamp()).add("CreationTimestamp", creationTimestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FeaturedResultsSetId":
            return Optional.ofNullable(clazz.cast(featuredResultsSetId()));
        case "FeaturedResultsSetName":
            return Optional.ofNullable(clazz.cast(featuredResultsSetName()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "QueryTexts":
            return Optional.ofNullable(clazz.cast(queryTexts()));
        case "FeaturedDocuments":
            return Optional.ofNullable(clazz.cast(featuredDocuments()));
        case "LastUpdatedTimestamp":
            return Optional.ofNullable(clazz.cast(lastUpdatedTimestamp()));
        case "CreationTimestamp":
            return Optional.ofNullable(clazz.cast(creationTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FeaturedResultsSet, T> g) {
        return obj -> g.apply((FeaturedResultsSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FeaturedResultsSet> {
        /**
         * <p>
         * The identifier of the set of featured results.
         * </p>
         * 
         * @param featuredResultsSetId
         *        The identifier of the set of featured results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featuredResultsSetId(String featuredResultsSetId);

        /**
         * <p>
         * The name for the set of featured results.
         * </p>
         * 
         * @param featuredResultsSetName
         *        The name for the set of featured results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featuredResultsSetName(String featuredResultsSetName);

        /**
         * <p>
         * The description for the set of featured results.
         * </p>
         * 
         * @param description
         *        The description for the set of featured results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The current status of the set of featured results. When the value is <code>ACTIVE</code>, featured results
         * are ready for use. You can still configure your settings before setting the status to <code>ACTIVE</code>.
         * You can set the status to <code>ACTIVE</code> or <code>INACTIVE</code> using the <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateFeaturedResultsSet.html"
         * >UpdateFeaturedResultsSet</a> API. The queries you specify for featured results must be unique per featured
         * results set for each index, whether the status is <code>ACTIVE</code> or <code>INACTIVE</code>.
         * </p>
         * 
         * @param status
         *        The current status of the set of featured results. When the value is <code>ACTIVE</code>, featured
         *        results are ready for use. You can still configure your settings before setting the status to
         *        <code>ACTIVE</code>. You can set the status to <code>ACTIVE</code> or <code>INACTIVE</code> using the
         *        <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateFeaturedResultsSet.html">
         *        UpdateFeaturedResultsSet</a> API. The queries you specify for featured results must be unique per
         *        featured results set for each index, whether the status is <code>ACTIVE</code> or
         *        <code>INACTIVE</code>.
         * @see FeaturedResultsSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FeaturedResultsSetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the set of featured results. When the value is <code>ACTIVE</code>, featured results
         * are ready for use. You can still configure your settings before setting the status to <code>ACTIVE</code>.
         * You can set the status to <code>ACTIVE</code> or <code>INACTIVE</code> using the <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateFeaturedResultsSet.html"
         * >UpdateFeaturedResultsSet</a> API. The queries you specify for featured results must be unique per featured
         * results set for each index, whether the status is <code>ACTIVE</code> or <code>INACTIVE</code>.
         * </p>
         * 
         * @param status
         *        The current status of the set of featured results. When the value is <code>ACTIVE</code>, featured
         *        results are ready for use. You can still configure your settings before setting the status to
         *        <code>ACTIVE</code>. You can set the status to <code>ACTIVE</code> or <code>INACTIVE</code> using the
         *        <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateFeaturedResultsSet.html">
         *        UpdateFeaturedResultsSet</a> API. The queries you specify for featured results must be unique per
         *        featured results set for each index, whether the status is <code>ACTIVE</code> or
         *        <code>INACTIVE</code>.
         * @see FeaturedResultsSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FeaturedResultsSetStatus
         */
        Builder status(FeaturedResultsSetStatus status);

        /**
         * <p>
         * The list of queries for featuring results.
         * </p>
         * <p>
         * Specific queries are mapped to specific documents for featuring in the results. If a query contains an exact
         * match, then one or more specific documents are featured in the results. The exact match applies to the full
         * query. For example, if you only specify 'Kendra', queries such as 'How does kendra semantically rank
         * results?' will not render the featured results. Featured results are designed for specific queries, rather
         * than queries that are too broad in scope.
         * </p>
         * 
         * @param queryTexts
         *        The list of queries for featuring results.</p>
         *        <p>
         *        Specific queries are mapped to specific documents for featuring in the results. If a query contains an
         *        exact match, then one or more specific documents are featured in the results. The exact match applies
         *        to the full query. For example, if you only specify 'Kendra', queries such as 'How does kendra
         *        semantically rank results?' will not render the featured results. Featured results are designed for
         *        specific queries, rather than queries that are too broad in scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryTexts(Collection<String> queryTexts);

        /**
         * <p>
         * The list of queries for featuring results.
         * </p>
         * <p>
         * Specific queries are mapped to specific documents for featuring in the results. If a query contains an exact
         * match, then one or more specific documents are featured in the results. The exact match applies to the full
         * query. For example, if you only specify 'Kendra', queries such as 'How does kendra semantically rank
         * results?' will not render the featured results. Featured results are designed for specific queries, rather
         * than queries that are too broad in scope.
         * </p>
         * 
         * @param queryTexts
         *        The list of queries for featuring results.</p>
         *        <p>
         *        Specific queries are mapped to specific documents for featuring in the results. If a query contains an
         *        exact match, then one or more specific documents are featured in the results. The exact match applies
         *        to the full query. For example, if you only specify 'Kendra', queries such as 'How does kendra
         *        semantically rank results?' will not render the featured results. Featured results are designed for
         *        specific queries, rather than queries that are too broad in scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryTexts(String... queryTexts);

        /**
         * <p>
         * The list of document IDs for the documents you want to feature at the top of the search results page. You can
         * use the <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Query.html">Query</a> API to search for
         * specific documents with their document IDs included in the result items, or you can use the console.
         * </p>
         * <p>
         * You can add up to four featured documents. You can request to increase this limit by contacting <a
         * href="http://aws.amazon.com/contact-us/">Support</a>.
         * </p>
         * <p>
         * Specific queries are mapped to specific documents for featuring in the results. If a query contains an exact
         * match, then one or more specific documents are featured in the results. The exact match applies to the full
         * query. For example, if you only specify 'Kendra', queries such as 'How does kendra semantically rank
         * results?' will not render the featured results. Featured results are designed for specific queries, rather
         * than queries that are too broad in scope.
         * </p>
         * 
         * @param featuredDocuments
         *        The list of document IDs for the documents you want to feature at the top of the search results page.
         *        You can use the <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Query.html">Query</a> API to
         *        search for specific documents with their document IDs included in the result items, or you can use the
         *        console.</p>
         *        <p>
         *        You can add up to four featured documents. You can request to increase this limit by contacting <a
         *        href="http://aws.amazon.com/contact-us/">Support</a>.
         *        </p>
         *        <p>
         *        Specific queries are mapped to specific documents for featuring in the results. If a query contains an
         *        exact match, then one or more specific documents are featured in the results. The exact match applies
         *        to the full query. For example, if you only specify 'Kendra', queries such as 'How does kendra
         *        semantically rank results?' will not render the featured results. Featured results are designed for
         *        specific queries, rather than queries that are too broad in scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featuredDocuments(Collection<FeaturedDocument> featuredDocuments);

        /**
         * <p>
         * The list of document IDs for the documents you want to feature at the top of the search results page. You can
         * use the <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Query.html">Query</a> API to search for
         * specific documents with their document IDs included in the result items, or you can use the console.
         * </p>
         * <p>
         * You can add up to four featured documents. You can request to increase this limit by contacting <a
         * href="http://aws.amazon.com/contact-us/">Support</a>.
         * </p>
         * <p>
         * Specific queries are mapped to specific documents for featuring in the results. If a query contains an exact
         * match, then one or more specific documents are featured in the results. The exact match applies to the full
         * query. For example, if you only specify 'Kendra', queries such as 'How does kendra semantically rank
         * results?' will not render the featured results. Featured results are designed for specific queries, rather
         * than queries that are too broad in scope.
         * </p>
         * 
         * @param featuredDocuments
         *        The list of document IDs for the documents you want to feature at the top of the search results page.
         *        You can use the <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Query.html">Query</a> API to
         *        search for specific documents with their document IDs included in the result items, or you can use the
         *        console.</p>
         *        <p>
         *        You can add up to four featured documents. You can request to increase this limit by contacting <a
         *        href="http://aws.amazon.com/contact-us/">Support</a>.
         *        </p>
         *        <p>
         *        Specific queries are mapped to specific documents for featuring in the results. If a query contains an
         *        exact match, then one or more specific documents are featured in the results. The exact match applies
         *        to the full query. For example, if you only specify 'Kendra', queries such as 'How does kendra
         *        semantically rank results?' will not render the featured results. Featured results are designed for
         *        specific queries, rather than queries that are too broad in scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder featuredDocuments(FeaturedDocument... featuredDocuments);

        /**
         * <p>
         * The list of document IDs for the documents you want to feature at the top of the search results page. You can
         * use the <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_Query.html">Query</a> API to search for
         * specific documents with their document IDs included in the result items, or you can use the console.
         * </p>
         * <p>
         * You can add up to four featured documents. You can request to increase this limit by contacting <a
         * href="http://aws.amazon.com/contact-us/">Support</a>.
         * </p>
         * <p>
         * Specific queries are mapped to specific documents for featuring in the results. If a query contains an exact
         * match, then one or more specific documents are featured in the results. The exact match applies to the full
         * query. For example, if you only specify 'Kendra', queries such as 'How does kendra semantically rank
         * results?' will not render the featured results. Featured results are designed for specific queries, rather
         * than queries that are too broad in scope.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.FeaturedDocument.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.kendra.model.FeaturedDocument#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.FeaturedDocument.Builder#build()} is called immediately
         * and its result is passed to {@link #featuredDocuments(List<FeaturedDocument>)}.
         * 
         * @param featuredDocuments
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.FeaturedDocument.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #featuredDocuments(java.util.Collection<FeaturedDocument>)
         */
        Builder featuredDocuments(Consumer<FeaturedDocument.Builder>... featuredDocuments);

        /**
         * <p>
         * The Unix timestamp when the set of featured results was last updated.
         * </p>
         * 
         * @param lastUpdatedTimestamp
         *        The Unix timestamp when the set of featured results was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTimestamp(Long lastUpdatedTimestamp);

        /**
         * <p>
         * The Unix timestamp when the set of featured results was created.
         * </p>
         * 
         * @param creationTimestamp
         *        The Unix timestamp when the set of featured results was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTimestamp(Long creationTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String featuredResultsSetId;

        private String featuredResultsSetName;

        private String description;

        private String status;

        private List<String> queryTexts = DefaultSdkAutoConstructList.getInstance();

        private List<FeaturedDocument> featuredDocuments = DefaultSdkAutoConstructList.getInstance();

        private Long lastUpdatedTimestamp;

        private Long creationTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(FeaturedResultsSet model) {
            featuredResultsSetId(model.featuredResultsSetId);
            featuredResultsSetName(model.featuredResultsSetName);
            description(model.description);
            status(model.status);
            queryTexts(model.queryTexts);
            featuredDocuments(model.featuredDocuments);
            lastUpdatedTimestamp(model.lastUpdatedTimestamp);
            creationTimestamp(model.creationTimestamp);
        }

        public final String getFeaturedResultsSetId() {
            return featuredResultsSetId;
        }

        public final void setFeaturedResultsSetId(String featuredResultsSetId) {
            this.featuredResultsSetId = featuredResultsSetId;
        }

        @Override
        public final Builder featuredResultsSetId(String featuredResultsSetId) {
            this.featuredResultsSetId = featuredResultsSetId;
            return this;
        }

        public final String getFeaturedResultsSetName() {
            return featuredResultsSetName;
        }

        public final void setFeaturedResultsSetName(String featuredResultsSetName) {
            this.featuredResultsSetName = featuredResultsSetName;
        }

        @Override
        public final Builder featuredResultsSetName(String featuredResultsSetName) {
            this.featuredResultsSetName = featuredResultsSetName;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(FeaturedResultsSetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Collection<String> getQueryTexts() {
            if (queryTexts instanceof SdkAutoConstructList) {
                return null;
            }
            return queryTexts;
        }

        public final void setQueryTexts(Collection<String> queryTexts) {
            this.queryTexts = QueryTextListCopier.copy(queryTexts);
        }

        @Override
        public final Builder queryTexts(Collection<String> queryTexts) {
            this.queryTexts = QueryTextListCopier.copy(queryTexts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder queryTexts(String... queryTexts) {
            queryTexts(Arrays.asList(queryTexts));
            return this;
        }

        public final List<FeaturedDocument.Builder> getFeaturedDocuments() {
            List<FeaturedDocument.Builder> result = FeaturedDocumentListCopier.copyToBuilder(this.featuredDocuments);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFeaturedDocuments(Collection<FeaturedDocument.BuilderImpl> featuredDocuments) {
            this.featuredDocuments = FeaturedDocumentListCopier.copyFromBuilder(featuredDocuments);
        }

        @Override
        public final Builder featuredDocuments(Collection<FeaturedDocument> featuredDocuments) {
            this.featuredDocuments = FeaturedDocumentListCopier.copy(featuredDocuments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featuredDocuments(FeaturedDocument... featuredDocuments) {
            featuredDocuments(Arrays.asList(featuredDocuments));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder featuredDocuments(Consumer<FeaturedDocument.Builder>... featuredDocuments) {
            featuredDocuments(Stream.of(featuredDocuments).map(c -> FeaturedDocument.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Long getLastUpdatedTimestamp() {
            return lastUpdatedTimestamp;
        }

        public final void setLastUpdatedTimestamp(Long lastUpdatedTimestamp) {
            this.lastUpdatedTimestamp = lastUpdatedTimestamp;
        }

        @Override
        public final Builder lastUpdatedTimestamp(Long lastUpdatedTimestamp) {
            this.lastUpdatedTimestamp = lastUpdatedTimestamp;
            return this;
        }

        public final Long getCreationTimestamp() {
            return creationTimestamp;
        }

        public final void setCreationTimestamp(Long creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
        }

        @Override
        public final Builder creationTimestamp(Long creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
            return this;
        }

        @Override
        public FeaturedResultsSet build() {
            return new FeaturedResultsSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
