/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class RetrieveRequest extends KendraRequest implements ToCopyableBuilder<RetrieveRequest.Builder, RetrieveRequest> {
    private static final SdkField<String> INDEX_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IndexId").getter(getter(RetrieveRequest::indexId)).setter(setter(Builder::indexId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IndexId").build()).build();

    private static final SdkField<String> QUERY_TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("QueryText").getter(getter(RetrieveRequest::queryText)).setter(setter(Builder::queryText))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryText").build()).build();

    private static final SdkField<AttributeFilter> ATTRIBUTE_FILTER_FIELD = SdkField
            .<AttributeFilter> builder(MarshallingType.SDK_POJO).memberName("AttributeFilter")
            .getter(getter(RetrieveRequest::attributeFilter)).setter(setter(Builder::attributeFilter))
            .constructor(AttributeFilter::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeFilter").build()).build();

    private static final SdkField<List<String>> REQUESTED_DOCUMENT_ATTRIBUTES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RequestedDocumentAttributes")
            .getter(getter(RetrieveRequest::requestedDocumentAttributes))
            .setter(setter(Builder::requestedDocumentAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedDocumentAttributes")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DocumentRelevanceConfiguration>> DOCUMENT_RELEVANCE_OVERRIDE_CONFIGURATIONS_FIELD = SdkField
            .<List<DocumentRelevanceConfiguration>> builder(MarshallingType.LIST)
            .memberName("DocumentRelevanceOverrideConfigurations")
            .getter(getter(RetrieveRequest::documentRelevanceOverrideConfigurations))
            .setter(setter(Builder::documentRelevanceOverrideConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("DocumentRelevanceOverrideConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentRelevanceConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentRelevanceConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> PAGE_NUMBER_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("PageNumber").getter(getter(RetrieveRequest::pageNumber)).setter(setter(Builder::pageNumber))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PageNumber").build()).build();

    private static final SdkField<Integer> PAGE_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("PageSize").getter(getter(RetrieveRequest::pageSize)).setter(setter(Builder::pageSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PageSize").build()).build();

    private static final SdkField<UserContext> USER_CONTEXT_FIELD = SdkField.<UserContext> builder(MarshallingType.SDK_POJO)
            .memberName("UserContext").getter(getter(RetrieveRequest::userContext)).setter(setter(Builder::userContext))
            .constructor(UserContext::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserContext").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INDEX_ID_FIELD,
            QUERY_TEXT_FIELD, ATTRIBUTE_FILTER_FIELD, REQUESTED_DOCUMENT_ATTRIBUTES_FIELD,
            DOCUMENT_RELEVANCE_OVERRIDE_CONFIGURATIONS_FIELD, PAGE_NUMBER_FIELD, PAGE_SIZE_FIELD, USER_CONTEXT_FIELD));

    private final String indexId;

    private final String queryText;

    private final AttributeFilter attributeFilter;

    private final List<String> requestedDocumentAttributes;

    private final List<DocumentRelevanceConfiguration> documentRelevanceOverrideConfigurations;

    private final Integer pageNumber;

    private final Integer pageSize;

    private final UserContext userContext;

    private RetrieveRequest(BuilderImpl builder) {
        super(builder);
        this.indexId = builder.indexId;
        this.queryText = builder.queryText;
        this.attributeFilter = builder.attributeFilter;
        this.requestedDocumentAttributes = builder.requestedDocumentAttributes;
        this.documentRelevanceOverrideConfigurations = builder.documentRelevanceOverrideConfigurations;
        this.pageNumber = builder.pageNumber;
        this.pageSize = builder.pageSize;
        this.userContext = builder.userContext;
    }

    /**
     * <p>
     * The identifier of the index to retrieve relevant passages for the search.
     * </p>
     * 
     * @return The identifier of the index to retrieve relevant passages for the search.
     */
    public final String indexId() {
        return indexId;
    }

    /**
     * <p>
     * The input query text to retrieve relevant passages for the search. Amazon Kendra truncates queries at 30 token
     * words, which excludes punctuation and stop words. Truncation still applies if you use Boolean or more advanced,
     * complex queries. For example, <code>Timeoff AND October AND Category:HR</code> is counted as 3 tokens:
     * <code>timeoff</code>, <code>october</code>, <code>hr</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/searching-example.html#searching-index-query-syntax">Searching
     * with advanced query syntax</a> in the Amazon Kendra Developer Guide.
     * </p>
     * 
     * @return The input query text to retrieve relevant passages for the search. Amazon Kendra truncates queries at 30
     *         token words, which excludes punctuation and stop words. Truncation still applies if you use Boolean or
     *         more advanced, complex queries. For example, <code>Timeoff AND October AND Category:HR</code> is counted
     *         as 3 tokens: <code>timeoff</code>, <code>october</code>, <code>hr</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/searching-example.html#searching-index-query-syntax">
     *         Searching with advanced query syntax</a> in the Amazon Kendra Developer Guide.
     */
    public final String queryText() {
        return queryText;
    }

    /**
     * <p>
     * Filters search results by document fields/attributes. You can only provide one attribute filter; however, the
     * <code>AndAllFilters</code>, <code>NotFilter</code>, and <code>OrAllFilters</code> parameters contain a list of
     * other filters.
     * </p>
     * <p>
     * The <code>AttributeFilter</code> parameter means you can create a set of filtering rules that a document must
     * satisfy to be included in the query results.
     * </p>
     * 
     * @return Filters search results by document fields/attributes. You can only provide one attribute filter; however,
     *         the <code>AndAllFilters</code>, <code>NotFilter</code>, and <code>OrAllFilters</code> parameters contain
     *         a list of other filters.</p>
     *         <p>
     *         The <code>AttributeFilter</code> parameter means you can create a set of filtering rules that a document
     *         must satisfy to be included in the query results.
     */
    public final AttributeFilter attributeFilter() {
        return attributeFilter;
    }

    /**
     * For responses, this returns true if the service returned a value for the RequestedDocumentAttributes property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRequestedDocumentAttributes() {
        return requestedDocumentAttributes != null && !(requestedDocumentAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of document fields/attributes to include in the response. You can limit the response to include certain
     * document fields. By default, all document fields are included in the response.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequestedDocumentAttributes} method.
     * </p>
     * 
     * @return A list of document fields/attributes to include in the response. You can limit the response to include
     *         certain document fields. By default, all document fields are included in the response.
     */
    public final List<String> requestedDocumentAttributes() {
        return requestedDocumentAttributes;
    }

    /**
     * For responses, this returns true if the service returned a value for the DocumentRelevanceOverrideConfigurations
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasDocumentRelevanceOverrideConfigurations() {
        return documentRelevanceOverrideConfigurations != null
                && !(documentRelevanceOverrideConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Overrides relevance tuning configurations of fields/attributes set at the index level.
     * </p>
     * <p>
     * If you use this API to override the relevance tuning configured at the index level, but there is no relevance
     * tuning configured at the index level, then Amazon Kendra does not apply any relevance tuning.
     * </p>
     * <p>
     * If there is relevance tuning configured for fields at the index level, and you use this API to override only some
     * of these fields, then for the fields you did not override, the importance is set to 1.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the
     * {@link #hasDocumentRelevanceOverrideConfigurations} method.
     * </p>
     * 
     * @return Overrides relevance tuning configurations of fields/attributes set at the index level.</p>
     *         <p>
     *         If you use this API to override the relevance tuning configured at the index level, but there is no
     *         relevance tuning configured at the index level, then Amazon Kendra does not apply any relevance tuning.
     *         </p>
     *         <p>
     *         If there is relevance tuning configured for fields at the index level, and you use this API to override
     *         only some of these fields, then for the fields you did not override, the importance is set to 1.
     */
    public final List<DocumentRelevanceConfiguration> documentRelevanceOverrideConfigurations() {
        return documentRelevanceOverrideConfigurations;
    }

    /**
     * <p>
     * Retrieved relevant passages are returned in pages the size of the <code>PageSize</code> parameter. By default,
     * Amazon Kendra returns the first page of results. Use this parameter to get result pages after the first one.
     * </p>
     * 
     * @return Retrieved relevant passages are returned in pages the size of the <code>PageSize</code> parameter. By
     *         default, Amazon Kendra returns the first page of results. Use this parameter to get result pages after
     *         the first one.
     */
    public final Integer pageNumber() {
        return pageNumber;
    }

    /**
     * <p>
     * Sets the number of retrieved relevant passages that are returned in each page of results. The default page size
     * is 10. The maximum number of results returned is 100. If you ask for more than 100 results, only 100 are
     * returned.
     * </p>
     * 
     * @return Sets the number of retrieved relevant passages that are returned in each page of results. The default
     *         page size is 10. The maximum number of results returned is 100. If you ask for more than 100 results,
     *         only 100 are returned.
     */
    public final Integer pageSize() {
        return pageSize;
    }

    /**
     * <p>
     * The user context token or user and group information.
     * </p>
     * 
     * @return The user context token or user and group information.
     */
    public final UserContext userContext() {
        return userContext;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(indexId());
        hashCode = 31 * hashCode + Objects.hashCode(queryText());
        hashCode = 31 * hashCode + Objects.hashCode(attributeFilter());
        hashCode = 31 * hashCode + Objects.hashCode(hasRequestedDocumentAttributes() ? requestedDocumentAttributes() : null);
        hashCode = 31
                * hashCode
                + Objects.hashCode(hasDocumentRelevanceOverrideConfigurations() ? documentRelevanceOverrideConfigurations()
                        : null);
        hashCode = 31 * hashCode + Objects.hashCode(pageNumber());
        hashCode = 31 * hashCode + Objects.hashCode(pageSize());
        hashCode = 31 * hashCode + Objects.hashCode(userContext());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RetrieveRequest)) {
            return false;
        }
        RetrieveRequest other = (RetrieveRequest) obj;
        return Objects.equals(indexId(), other.indexId()) && Objects.equals(queryText(), other.queryText())
                && Objects.equals(attributeFilter(), other.attributeFilter())
                && hasRequestedDocumentAttributes() == other.hasRequestedDocumentAttributes()
                && Objects.equals(requestedDocumentAttributes(), other.requestedDocumentAttributes())
                && hasDocumentRelevanceOverrideConfigurations() == other.hasDocumentRelevanceOverrideConfigurations()
                && Objects.equals(documentRelevanceOverrideConfigurations(), other.documentRelevanceOverrideConfigurations())
                && Objects.equals(pageNumber(), other.pageNumber()) && Objects.equals(pageSize(), other.pageSize())
                && Objects.equals(userContext(), other.userContext());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("RetrieveRequest")
                .add("IndexId", indexId())
                .add("QueryText", queryText())
                .add("AttributeFilter", attributeFilter())
                .add("RequestedDocumentAttributes", hasRequestedDocumentAttributes() ? requestedDocumentAttributes() : null)
                .add("DocumentRelevanceOverrideConfigurations",
                        hasDocumentRelevanceOverrideConfigurations() ? documentRelevanceOverrideConfigurations() : null)
                .add("PageNumber", pageNumber()).add("PageSize", pageSize()).add("UserContext", userContext()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IndexId":
            return Optional.ofNullable(clazz.cast(indexId()));
        case "QueryText":
            return Optional.ofNullable(clazz.cast(queryText()));
        case "AttributeFilter":
            return Optional.ofNullable(clazz.cast(attributeFilter()));
        case "RequestedDocumentAttributes":
            return Optional.ofNullable(clazz.cast(requestedDocumentAttributes()));
        case "DocumentRelevanceOverrideConfigurations":
            return Optional.ofNullable(clazz.cast(documentRelevanceOverrideConfigurations()));
        case "PageNumber":
            return Optional.ofNullable(clazz.cast(pageNumber()));
        case "PageSize":
            return Optional.ofNullable(clazz.cast(pageSize()));
        case "UserContext":
            return Optional.ofNullable(clazz.cast(userContext()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RetrieveRequest, T> g) {
        return obj -> g.apply((RetrieveRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KendraRequest.Builder, SdkPojo, CopyableBuilder<Builder, RetrieveRequest> {
        /**
         * <p>
         * The identifier of the index to retrieve relevant passages for the search.
         * </p>
         * 
         * @param indexId
         *        The identifier of the index to retrieve relevant passages for the search.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder indexId(String indexId);

        /**
         * <p>
         * The input query text to retrieve relevant passages for the search. Amazon Kendra truncates queries at 30
         * token words, which excludes punctuation and stop words. Truncation still applies if you use Boolean or more
         * advanced, complex queries. For example, <code>Timeoff AND October AND Category:HR</code> is counted as 3
         * tokens: <code>timeoff</code>, <code>october</code>, <code>hr</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/searching-example.html#searching-index-query-syntax"
         * >Searching with advanced query syntax</a> in the Amazon Kendra Developer Guide.
         * </p>
         * 
         * @param queryText
         *        The input query text to retrieve relevant passages for the search. Amazon Kendra truncates queries at
         *        30 token words, which excludes punctuation and stop words. Truncation still applies if you use Boolean
         *        or more advanced, complex queries. For example, <code>Timeoff AND October AND Category:HR</code> is
         *        counted as 3 tokens: <code>timeoff</code>, <code>october</code>, <code>hr</code>. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/kendra/latest/dg/searching-example.html#searching-index-query-syntax"
         *        >Searching with advanced query syntax</a> in the Amazon Kendra Developer Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryText(String queryText);

        /**
         * <p>
         * Filters search results by document fields/attributes. You can only provide one attribute filter; however, the
         * <code>AndAllFilters</code>, <code>NotFilter</code>, and <code>OrAllFilters</code> parameters contain a list
         * of other filters.
         * </p>
         * <p>
         * The <code>AttributeFilter</code> parameter means you can create a set of filtering rules that a document must
         * satisfy to be included in the query results.
         * </p>
         * 
         * @param attributeFilter
         *        Filters search results by document fields/attributes. You can only provide one attribute filter;
         *        however, the <code>AndAllFilters</code>, <code>NotFilter</code>, and <code>OrAllFilters</code>
         *        parameters contain a list of other filters.</p>
         *        <p>
         *        The <code>AttributeFilter</code> parameter means you can create a set of filtering rules that a
         *        document must satisfy to be included in the query results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeFilter(AttributeFilter attributeFilter);

        /**
         * <p>
         * Filters search results by document fields/attributes. You can only provide one attribute filter; however, the
         * <code>AndAllFilters</code>, <code>NotFilter</code>, and <code>OrAllFilters</code> parameters contain a list
         * of other filters.
         * </p>
         * <p>
         * The <code>AttributeFilter</code> parameter means you can create a set of filtering rules that a document must
         * satisfy to be included in the query results.
         * </p>
         * This is a convenience method that creates an instance of the {@link AttributeFilter.Builder} avoiding the
         * need to create one manually via {@link AttributeFilter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AttributeFilter.Builder#build()} is called immediately and its
         * result is passed to {@link #attributeFilter(AttributeFilter)}.
         * 
         * @param attributeFilter
         *        a consumer that will call methods on {@link AttributeFilter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attributeFilter(AttributeFilter)
         */
        default Builder attributeFilter(Consumer<AttributeFilter.Builder> attributeFilter) {
            return attributeFilter(AttributeFilter.builder().applyMutation(attributeFilter).build());
        }

        /**
         * <p>
         * A list of document fields/attributes to include in the response. You can limit the response to include
         * certain document fields. By default, all document fields are included in the response.
         * </p>
         * 
         * @param requestedDocumentAttributes
         *        A list of document fields/attributes to include in the response. You can limit the response to include
         *        certain document fields. By default, all document fields are included in the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedDocumentAttributes(Collection<String> requestedDocumentAttributes);

        /**
         * <p>
         * A list of document fields/attributes to include in the response. You can limit the response to include
         * certain document fields. By default, all document fields are included in the response.
         * </p>
         * 
         * @param requestedDocumentAttributes
         *        A list of document fields/attributes to include in the response. You can limit the response to include
         *        certain document fields. By default, all document fields are included in the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedDocumentAttributes(String... requestedDocumentAttributes);

        /**
         * <p>
         * Overrides relevance tuning configurations of fields/attributes set at the index level.
         * </p>
         * <p>
         * If you use this API to override the relevance tuning configured at the index level, but there is no relevance
         * tuning configured at the index level, then Amazon Kendra does not apply any relevance tuning.
         * </p>
         * <p>
         * If there is relevance tuning configured for fields at the index level, and you use this API to override only
         * some of these fields, then for the fields you did not override, the importance is set to 1.
         * </p>
         * 
         * @param documentRelevanceOverrideConfigurations
         *        Overrides relevance tuning configurations of fields/attributes set at the index level.</p>
         *        <p>
         *        If you use this API to override the relevance tuning configured at the index level, but there is no
         *        relevance tuning configured at the index level, then Amazon Kendra does not apply any relevance
         *        tuning.
         *        </p>
         *        <p>
         *        If there is relevance tuning configured for fields at the index level, and you use this API to
         *        override only some of these fields, then for the fields you did not override, the importance is set to
         *        1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentRelevanceOverrideConfigurations(
                Collection<DocumentRelevanceConfiguration> documentRelevanceOverrideConfigurations);

        /**
         * <p>
         * Overrides relevance tuning configurations of fields/attributes set at the index level.
         * </p>
         * <p>
         * If you use this API to override the relevance tuning configured at the index level, but there is no relevance
         * tuning configured at the index level, then Amazon Kendra does not apply any relevance tuning.
         * </p>
         * <p>
         * If there is relevance tuning configured for fields at the index level, and you use this API to override only
         * some of these fields, then for the fields you did not override, the importance is set to 1.
         * </p>
         * 
         * @param documentRelevanceOverrideConfigurations
         *        Overrides relevance tuning configurations of fields/attributes set at the index level.</p>
         *        <p>
         *        If you use this API to override the relevance tuning configured at the index level, but there is no
         *        relevance tuning configured at the index level, then Amazon Kendra does not apply any relevance
         *        tuning.
         *        </p>
         *        <p>
         *        If there is relevance tuning configured for fields at the index level, and you use this API to
         *        override only some of these fields, then for the fields you did not override, the importance is set to
         *        1.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentRelevanceOverrideConfigurations(DocumentRelevanceConfiguration... documentRelevanceOverrideConfigurations);

        /**
         * <p>
         * Overrides relevance tuning configurations of fields/attributes set at the index level.
         * </p>
         * <p>
         * If you use this API to override the relevance tuning configured at the index level, but there is no relevance
         * tuning configured at the index level, then Amazon Kendra does not apply any relevance tuning.
         * </p>
         * <p>
         * If there is relevance tuning configured for fields at the index level, and you use this API to override only
         * some of these fields, then for the fields you did not override, the importance is set to 1.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.DocumentRelevanceConfiguration.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.DocumentRelevanceConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.DocumentRelevanceConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #documentRelevanceOverrideConfigurations(List<DocumentRelevanceConfiguration>)}.
         * 
         * @param documentRelevanceOverrideConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.DocumentRelevanceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #documentRelevanceOverrideConfigurations(java.util.Collection<DocumentRelevanceConfiguration>)
         */
        Builder documentRelevanceOverrideConfigurations(
                Consumer<DocumentRelevanceConfiguration.Builder>... documentRelevanceOverrideConfigurations);

        /**
         * <p>
         * Retrieved relevant passages are returned in pages the size of the <code>PageSize</code> parameter. By
         * default, Amazon Kendra returns the first page of results. Use this parameter to get result pages after the
         * first one.
         * </p>
         * 
         * @param pageNumber
         *        Retrieved relevant passages are returned in pages the size of the <code>PageSize</code> parameter. By
         *        default, Amazon Kendra returns the first page of results. Use this parameter to get result pages after
         *        the first one.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pageNumber(Integer pageNumber);

        /**
         * <p>
         * Sets the number of retrieved relevant passages that are returned in each page of results. The default page
         * size is 10. The maximum number of results returned is 100. If you ask for more than 100 results, only 100 are
         * returned.
         * </p>
         * 
         * @param pageSize
         *        Sets the number of retrieved relevant passages that are returned in each page of results. The default
         *        page size is 10. The maximum number of results returned is 100. If you ask for more than 100 results,
         *        only 100 are returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pageSize(Integer pageSize);

        /**
         * <p>
         * The user context token or user and group information.
         * </p>
         * 
         * @param userContext
         *        The user context token or user and group information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userContext(UserContext userContext);

        /**
         * <p>
         * The user context token or user and group information.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserContext.Builder} avoiding the need to
         * create one manually via {@link UserContext#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UserContext.Builder#build()} is called immediately and its result
         * is passed to {@link #userContext(UserContext)}.
         * 
         * @param userContext
         *        a consumer that will call methods on {@link UserContext.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userContext(UserContext)
         */
        default Builder userContext(Consumer<UserContext.Builder> userContext) {
            return userContext(UserContext.builder().applyMutation(userContext).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KendraRequest.BuilderImpl implements Builder {
        private String indexId;

        private String queryText;

        private AttributeFilter attributeFilter;

        private List<String> requestedDocumentAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<DocumentRelevanceConfiguration> documentRelevanceOverrideConfigurations = DefaultSdkAutoConstructList
                .getInstance();

        private Integer pageNumber;

        private Integer pageSize;

        private UserContext userContext;

        private BuilderImpl() {
        }

        private BuilderImpl(RetrieveRequest model) {
            super(model);
            indexId(model.indexId);
            queryText(model.queryText);
            attributeFilter(model.attributeFilter);
            requestedDocumentAttributes(model.requestedDocumentAttributes);
            documentRelevanceOverrideConfigurations(model.documentRelevanceOverrideConfigurations);
            pageNumber(model.pageNumber);
            pageSize(model.pageSize);
            userContext(model.userContext);
        }

        public final String getIndexId() {
            return indexId;
        }

        public final void setIndexId(String indexId) {
            this.indexId = indexId;
        }

        @Override
        public final Builder indexId(String indexId) {
            this.indexId = indexId;
            return this;
        }

        public final String getQueryText() {
            return queryText;
        }

        public final void setQueryText(String queryText) {
            this.queryText = queryText;
        }

        @Override
        public final Builder queryText(String queryText) {
            this.queryText = queryText;
            return this;
        }

        public final AttributeFilter.Builder getAttributeFilter() {
            return attributeFilter != null ? attributeFilter.toBuilder() : null;
        }

        public final void setAttributeFilter(AttributeFilter.BuilderImpl attributeFilter) {
            this.attributeFilter = attributeFilter != null ? attributeFilter.build() : null;
        }

        @Override
        public final Builder attributeFilter(AttributeFilter attributeFilter) {
            this.attributeFilter = attributeFilter;
            return this;
        }

        public final Collection<String> getRequestedDocumentAttributes() {
            if (requestedDocumentAttributes instanceof SdkAutoConstructList) {
                return null;
            }
            return requestedDocumentAttributes;
        }

        public final void setRequestedDocumentAttributes(Collection<String> requestedDocumentAttributes) {
            this.requestedDocumentAttributes = DocumentAttributeKeyListCopier.copy(requestedDocumentAttributes);
        }

        @Override
        public final Builder requestedDocumentAttributes(Collection<String> requestedDocumentAttributes) {
            this.requestedDocumentAttributes = DocumentAttributeKeyListCopier.copy(requestedDocumentAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder requestedDocumentAttributes(String... requestedDocumentAttributes) {
            requestedDocumentAttributes(Arrays.asList(requestedDocumentAttributes));
            return this;
        }

        public final List<DocumentRelevanceConfiguration.Builder> getDocumentRelevanceOverrideConfigurations() {
            List<DocumentRelevanceConfiguration.Builder> result = DocumentRelevanceOverrideConfigurationListCopier
                    .copyToBuilder(this.documentRelevanceOverrideConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDocumentRelevanceOverrideConfigurations(
                Collection<DocumentRelevanceConfiguration.BuilderImpl> documentRelevanceOverrideConfigurations) {
            this.documentRelevanceOverrideConfigurations = DocumentRelevanceOverrideConfigurationListCopier
                    .copyFromBuilder(documentRelevanceOverrideConfigurations);
        }

        @Override
        public final Builder documentRelevanceOverrideConfigurations(
                Collection<DocumentRelevanceConfiguration> documentRelevanceOverrideConfigurations) {
            this.documentRelevanceOverrideConfigurations = DocumentRelevanceOverrideConfigurationListCopier
                    .copy(documentRelevanceOverrideConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder documentRelevanceOverrideConfigurations(
                DocumentRelevanceConfiguration... documentRelevanceOverrideConfigurations) {
            documentRelevanceOverrideConfigurations(Arrays.asList(documentRelevanceOverrideConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder documentRelevanceOverrideConfigurations(
                Consumer<DocumentRelevanceConfiguration.Builder>... documentRelevanceOverrideConfigurations) {
            documentRelevanceOverrideConfigurations(Stream.of(documentRelevanceOverrideConfigurations)
                    .map(c -> DocumentRelevanceConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getPageNumber() {
            return pageNumber;
        }

        public final void setPageNumber(Integer pageNumber) {
            this.pageNumber = pageNumber;
        }

        @Override
        public final Builder pageNumber(Integer pageNumber) {
            this.pageNumber = pageNumber;
            return this;
        }

        public final Integer getPageSize() {
            return pageSize;
        }

        public final void setPageSize(Integer pageSize) {
            this.pageSize = pageSize;
        }

        @Override
        public final Builder pageSize(Integer pageSize) {
            this.pageSize = pageSize;
            return this;
        }

        public final UserContext.Builder getUserContext() {
            return userContext != null ? userContext.toBuilder() : null;
        }

        public final void setUserContext(UserContext.BuilderImpl userContext) {
            this.userContext = userContext != null ? userContext.build() : null;
        }

        @Override
        public final Builder userContext(UserContext userContext) {
            this.userContext = userContext;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public RetrieveRequest build() {
            return new RetrieveRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
