/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the configuration information to connect to an Amazon S3 bucket.
 * </p>
 * <note>
 * <p>
 * Amazon Kendra now supports an upgraded Amazon S3 connector.
 * </p>
 * <p>
 * You must now use the <a
 * href="https://docs.aws.amazon.com/kendra/latest/APIReference/API_TemplateConfiguration.html">TemplateConfiguration
 * </a> object instead of the <code>S3DataSourceConfiguration</code> object to configure your connector.
 * </p>
 * <p>
 * Connectors configured using the older console and API architecture will continue to function as configured. However,
 * you won't be able to edit or update them. If you want to edit or update your connector configuration, you must create
 * a new connector.
 * </p>
 * <p>
 * We recommended migrating your connector workflow to the upgraded version. Support for connectors configured using the
 * older architecture is scheduled to end by June 2024.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3DataSourceConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3DataSourceConfiguration.Builder, S3DataSourceConfiguration> {
    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BucketName").getter(getter(S3DataSourceConfiguration::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketName").build()).build();

    private static final SdkField<List<String>> INCLUSION_PREFIXES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InclusionPrefixes")
            .getter(getter(S3DataSourceConfiguration::inclusionPrefixes))
            .setter(setter(Builder::inclusionPrefixes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InclusionPrefixes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> INCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InclusionPatterns")
            .getter(getter(S3DataSourceConfiguration::inclusionPatterns))
            .setter(setter(Builder::inclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExclusionPatterns")
            .getter(getter(S3DataSourceConfiguration::exclusionPatterns))
            .setter(setter(Builder::exclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DocumentsMetadataConfiguration> DOCUMENTS_METADATA_CONFIGURATION_FIELD = SdkField
            .<DocumentsMetadataConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("DocumentsMetadataConfiguration")
            .getter(getter(S3DataSourceConfiguration::documentsMetadataConfiguration))
            .setter(setter(Builder::documentsMetadataConfiguration))
            .constructor(DocumentsMetadataConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentsMetadataConfiguration")
                    .build()).build();

    private static final SdkField<AccessControlListConfiguration> ACCESS_CONTROL_LIST_CONFIGURATION_FIELD = SdkField
            .<AccessControlListConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("AccessControlListConfiguration")
            .getter(getter(S3DataSourceConfiguration::accessControlListConfiguration))
            .setter(setter(Builder::accessControlListConfiguration))
            .constructor(AccessControlListConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlListConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_NAME_FIELD,
            INCLUSION_PREFIXES_FIELD, INCLUSION_PATTERNS_FIELD, EXCLUSION_PATTERNS_FIELD, DOCUMENTS_METADATA_CONFIGURATION_FIELD,
            ACCESS_CONTROL_LIST_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucketName;

    private final List<String> inclusionPrefixes;

    private final List<String> inclusionPatterns;

    private final List<String> exclusionPatterns;

    private final DocumentsMetadataConfiguration documentsMetadataConfiguration;

    private final AccessControlListConfiguration accessControlListConfiguration;

    private S3DataSourceConfiguration(BuilderImpl builder) {
        this.bucketName = builder.bucketName;
        this.inclusionPrefixes = builder.inclusionPrefixes;
        this.inclusionPatterns = builder.inclusionPatterns;
        this.exclusionPatterns = builder.exclusionPatterns;
        this.documentsMetadataConfiguration = builder.documentsMetadataConfiguration;
        this.accessControlListConfiguration = builder.accessControlListConfiguration;
    }

    /**
     * <p>
     * The name of the bucket that contains the documents.
     * </p>
     * 
     * @return The name of the bucket that contains the documents.
     */
    public final String bucketName() {
        return bucketName;
    }

    /**
     * For responses, this returns true if the service returned a value for the InclusionPrefixes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInclusionPrefixes() {
        return inclusionPrefixes != null && !(inclusionPrefixes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of S3 prefixes for the documents that should be included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInclusionPrefixes} method.
     * </p>
     * 
     * @return A list of S3 prefixes for the documents that should be included in the index.
     */
    public final List<String> inclusionPrefixes() {
        return inclusionPrefixes;
    }

    /**
     * For responses, this returns true if the service returned a value for the InclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInclusionPatterns() {
        return inclusionPatterns != null && !(inclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that match the
     * given pattern) for certain file names and file types to include in your index. If a document matches both an
     * inclusion and exclusion prefix or pattern, the exclusion prefix takes precendence and the document is not
     * indexed. Examples of glob patterns include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>/myapp/config/*</i>—All files inside config directory.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>**&#47;*.png</i>—All .png files in all directories.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and .jpg).
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as 'internal',
     * 'internal_only', 'company_internal'.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more examples, see <a
     * href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of Exclude and
     * Include Filters</a> in the Amazon Web Services CLI Command Reference.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInclusionPatterns} method.
     * </p>
     * 
     * @return A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that match
     *         the given pattern) for certain file names and file types to include in your index. If a document matches
     *         both an inclusion and exclusion prefix or pattern, the exclusion prefix takes precendence and the
     *         document is not indexed. Examples of glob patterns include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>/myapp/config/*</i>—All files inside config directory.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>**&#47;*.png</i>—All .png files in all directories.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and
     *         .jpg).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as 'internal',
     *         'internal_only', 'company_internal'.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more examples, see <a
     *         href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of
     *         Exclude and Include Filters</a> in the Amazon Web Services CLI Command Reference.
     */
    public final List<String> inclusionPatterns() {
        return inclusionPatterns;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExclusionPatterns() {
        return exclusionPatterns != null && !(exclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that match the
     * given pattern) for certain file names and file types to exclude from your index. If a document matches both an
     * inclusion and exclusion prefix or pattern, the exclusion prefix takes precendence and the document is not
     * indexed. Examples of glob patterns include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>/myapp/config/*</i>—All files inside config directory.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>**&#47;*.png</i>—All .png files in all directories.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and .jpg).
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as 'internal',
     * 'internal_only', 'company_internal'.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more examples, see <a
     * href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of Exclude and
     * Include Filters</a> in the Amazon Web Services CLI Command Reference.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExclusionPatterns} method.
     * </p>
     * 
     * @return A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that match
     *         the given pattern) for certain file names and file types to exclude from your index. If a document
     *         matches both an inclusion and exclusion prefix or pattern, the exclusion prefix takes precendence and the
     *         document is not indexed. Examples of glob patterns include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>/myapp/config/*</i>—All files inside config directory.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>**&#47;*.png</i>—All .png files in all directories.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and
     *         .jpg).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as 'internal',
     *         'internal_only', 'company_internal'.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more examples, see <a
     *         href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of
     *         Exclude and Include Filters</a> in the Amazon Web Services CLI Command Reference.
     */
    public final List<String> exclusionPatterns() {
        return exclusionPatterns;
    }

    /**
     * Returns the value of the DocumentsMetadataConfiguration property for this object.
     * 
     * @return The value of the DocumentsMetadataConfiguration property for this object.
     */
    public final DocumentsMetadataConfiguration documentsMetadataConfiguration() {
        return documentsMetadataConfiguration;
    }

    /**
     * <p>
     * Provides the path to the S3 bucket that contains the user context filtering files for the data source. For the
     * format of the file, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/s3-acl.html">Access control for S3
     * data sources</a>.
     * </p>
     * 
     * @return Provides the path to the S3 bucket that contains the user context filtering files for the data source.
     *         For the format of the file, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/s3-acl.html">Access
     *         control for S3 data sources</a>.
     */
    public final AccessControlListConfiguration accessControlListConfiguration() {
        return accessControlListConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(hasInclusionPrefixes() ? inclusionPrefixes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInclusionPatterns() ? inclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExclusionPatterns() ? exclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(documentsMetadataConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(accessControlListConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3DataSourceConfiguration)) {
            return false;
        }
        S3DataSourceConfiguration other = (S3DataSourceConfiguration) obj;
        return Objects.equals(bucketName(), other.bucketName()) && hasInclusionPrefixes() == other.hasInclusionPrefixes()
                && Objects.equals(inclusionPrefixes(), other.inclusionPrefixes())
                && hasInclusionPatterns() == other.hasInclusionPatterns()
                && Objects.equals(inclusionPatterns(), other.inclusionPatterns())
                && hasExclusionPatterns() == other.hasExclusionPatterns()
                && Objects.equals(exclusionPatterns(), other.exclusionPatterns())
                && Objects.equals(documentsMetadataConfiguration(), other.documentsMetadataConfiguration())
                && Objects.equals(accessControlListConfiguration(), other.accessControlListConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3DataSourceConfiguration").add("BucketName", bucketName())
                .add("InclusionPrefixes", hasInclusionPrefixes() ? inclusionPrefixes() : null)
                .add("InclusionPatterns", hasInclusionPatterns() ? inclusionPatterns() : null)
                .add("ExclusionPatterns", hasExclusionPatterns() ? exclusionPatterns() : null)
                .add("DocumentsMetadataConfiguration", documentsMetadataConfiguration())
                .add("AccessControlListConfiguration", accessControlListConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "InclusionPrefixes":
            return Optional.ofNullable(clazz.cast(inclusionPrefixes()));
        case "InclusionPatterns":
            return Optional.ofNullable(clazz.cast(inclusionPatterns()));
        case "ExclusionPatterns":
            return Optional.ofNullable(clazz.cast(exclusionPatterns()));
        case "DocumentsMetadataConfiguration":
            return Optional.ofNullable(clazz.cast(documentsMetadataConfiguration()));
        case "AccessControlListConfiguration":
            return Optional.ofNullable(clazz.cast(accessControlListConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3DataSourceConfiguration, T> g) {
        return obj -> g.apply((S3DataSourceConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3DataSourceConfiguration> {
        /**
         * <p>
         * The name of the bucket that contains the documents.
         * </p>
         * 
         * @param bucketName
         *        The name of the bucket that contains the documents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * A list of S3 prefixes for the documents that should be included in the index.
         * </p>
         * 
         * @param inclusionPrefixes
         *        A list of S3 prefixes for the documents that should be included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPrefixes(Collection<String> inclusionPrefixes);

        /**
         * <p>
         * A list of S3 prefixes for the documents that should be included in the index.
         * </p>
         * 
         * @param inclusionPrefixes
         *        A list of S3 prefixes for the documents that should be included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPrefixes(String... inclusionPrefixes);

        /**
         * <p>
         * A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that match the
         * given pattern) for certain file names and file types to include in your index. If a document matches both an
         * inclusion and exclusion prefix or pattern, the exclusion prefix takes precendence and the document is not
         * indexed. Examples of glob patterns include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>/myapp/config/*</i>—All files inside config directory.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*.png</i>—All .png files in all directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and .jpg).
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as 'internal',
         * 'internal_only', 'company_internal'.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more examples, see <a
         * href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of Exclude
         * and Include Filters</a> in the Amazon Web Services CLI Command Reference.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that
         *        match the given pattern) for certain file names and file types to include in your index. If a document
         *        matches both an inclusion and exclusion prefix or pattern, the exclusion prefix takes precendence and
         *        the document is not indexed. Examples of glob patterns include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>/myapp/config/*</i>—All files inside config directory.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*.png</i>—All .png files in all directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and
         *        .jpg).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as
         *        'internal', 'internal_only', 'company_internal'.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more examples, see <a
         *        href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of
         *        Exclude and Include Filters</a> in the Amazon Web Services CLI Command Reference.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(Collection<String> inclusionPatterns);

        /**
         * <p>
         * A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that match the
         * given pattern) for certain file names and file types to include in your index. If a document matches both an
         * inclusion and exclusion prefix or pattern, the exclusion prefix takes precendence and the document is not
         * indexed. Examples of glob patterns include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>/myapp/config/*</i>—All files inside config directory.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*.png</i>—All .png files in all directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and .jpg).
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as 'internal',
         * 'internal_only', 'company_internal'.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more examples, see <a
         * href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of Exclude
         * and Include Filters</a> in the Amazon Web Services CLI Command Reference.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that
         *        match the given pattern) for certain file names and file types to include in your index. If a document
         *        matches both an inclusion and exclusion prefix or pattern, the exclusion prefix takes precendence and
         *        the document is not indexed. Examples of glob patterns include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>/myapp/config/*</i>—All files inside config directory.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*.png</i>—All .png files in all directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and
         *        .jpg).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as
         *        'internal', 'internal_only', 'company_internal'.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more examples, see <a
         *        href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of
         *        Exclude and Include Filters</a> in the Amazon Web Services CLI Command Reference.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(String... inclusionPatterns);

        /**
         * <p>
         * A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that match the
         * given pattern) for certain file names and file types to exclude from your index. If a document matches both
         * an inclusion and exclusion prefix or pattern, the exclusion prefix takes precendence and the document is not
         * indexed. Examples of glob patterns include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>/myapp/config/*</i>—All files inside config directory.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*.png</i>—All .png files in all directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and .jpg).
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as 'internal',
         * 'internal_only', 'company_internal'.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more examples, see <a
         * href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of Exclude
         * and Include Filters</a> in the Amazon Web Services CLI Command Reference.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that
         *        match the given pattern) for certain file names and file types to exclude from your index. If a
         *        document matches both an inclusion and exclusion prefix or pattern, the exclusion prefix takes
         *        precendence and the document is not indexed. Examples of glob patterns include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>/myapp/config/*</i>—All files inside config directory.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*.png</i>—All .png files in all directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and
         *        .jpg).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as
         *        'internal', 'internal_only', 'company_internal'.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more examples, see <a
         *        href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of
         *        Exclude and Include Filters</a> in the Amazon Web Services CLI Command Reference.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(Collection<String> exclusionPatterns);

        /**
         * <p>
         * A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that match the
         * given pattern) for certain file names and file types to exclude from your index. If a document matches both
         * an inclusion and exclusion prefix or pattern, the exclusion prefix takes precendence and the document is not
         * indexed. Examples of glob patterns include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>/myapp/config/*</i>—All files inside config directory.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*.png</i>—All .png files in all directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and .jpg).
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as 'internal',
         * 'internal_only', 'company_internal'.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more examples, see <a
         * href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of Exclude
         * and Include Filters</a> in the Amazon Web Services CLI Command Reference.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of glob patterns (patterns that can expand a wildcard pattern into a list of path names that
         *        match the given pattern) for certain file names and file types to exclude from your index. If a
         *        document matches both an inclusion and exclusion prefix or pattern, the exclusion prefix takes
         *        precendence and the document is not indexed. Examples of glob patterns include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>/myapp/config/*</i>—All files inside config directory.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*.png</i>—All .png files in all directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*.{png, ico, md}</i>—All .png, .ico or .md files in all directories.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>/myapp/src/**&#47;*.ts</i>—All .ts files inside src directory (and all its subdirectories).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;!(*.module).ts</i>—All .ts files but not .module.ts
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>*.png , *.jpg</i>—All PNG and JPEG image files in a directory (files with the extensions .png and
         *        .jpg).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>*internal*</i>—All files in a directory that contain 'internal' in the file name, such as
         *        'internal', 'internal_only', 'company_internal'.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>**&#47;*internal*</i>—All internal-related files in a directory and its subdirectories.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more examples, see <a
         *        href="https://docs.aws.amazon.com/cli/latest/reference/s3/#use-of-exclude-and-include-filters">Use of
         *        Exclude and Include Filters</a> in the Amazon Web Services CLI Command Reference.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(String... exclusionPatterns);

        /**
         * Sets the value of the DocumentsMetadataConfiguration property for this object.
         *
         * @param documentsMetadataConfiguration
         *        The new value for the DocumentsMetadataConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentsMetadataConfiguration(DocumentsMetadataConfiguration documentsMetadataConfiguration);

        /**
         * Sets the value of the DocumentsMetadataConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link DocumentsMetadataConfiguration.Builder}
         * avoiding the need to create one manually via {@link DocumentsMetadataConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DocumentsMetadataConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #documentsMetadataConfiguration(DocumentsMetadataConfiguration)}.
         * 
         * @param documentsMetadataConfiguration
         *        a consumer that will call methods on {@link DocumentsMetadataConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #documentsMetadataConfiguration(DocumentsMetadataConfiguration)
         */
        default Builder documentsMetadataConfiguration(
                Consumer<DocumentsMetadataConfiguration.Builder> documentsMetadataConfiguration) {
            return documentsMetadataConfiguration(DocumentsMetadataConfiguration.builder()
                    .applyMutation(documentsMetadataConfiguration).build());
        }

        /**
         * <p>
         * Provides the path to the S3 bucket that contains the user context filtering files for the data source. For
         * the format of the file, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/s3-acl.html">Access control
         * for S3 data sources</a>.
         * </p>
         * 
         * @param accessControlListConfiguration
         *        Provides the path to the S3 bucket that contains the user context filtering files for the data source.
         *        For the format of the file, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/s3-acl.html">Access control for S3 data
         *        sources</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlListConfiguration(AccessControlListConfiguration accessControlListConfiguration);

        /**
         * <p>
         * Provides the path to the S3 bucket that contains the user context filtering files for the data source. For
         * the format of the file, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/s3-acl.html">Access control
         * for S3 data sources</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link AccessControlListConfiguration.Builder}
         * avoiding the need to create one manually via {@link AccessControlListConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AccessControlListConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #accessControlListConfiguration(AccessControlListConfiguration)}.
         * 
         * @param accessControlListConfiguration
         *        a consumer that will call methods on {@link AccessControlListConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlListConfiguration(AccessControlListConfiguration)
         */
        default Builder accessControlListConfiguration(
                Consumer<AccessControlListConfiguration.Builder> accessControlListConfiguration) {
            return accessControlListConfiguration(AccessControlListConfiguration.builder()
                    .applyMutation(accessControlListConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String bucketName;

        private List<String> inclusionPrefixes = DefaultSdkAutoConstructList.getInstance();

        private List<String> inclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private List<String> exclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private DocumentsMetadataConfiguration documentsMetadataConfiguration;

        private AccessControlListConfiguration accessControlListConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(S3DataSourceConfiguration model) {
            bucketName(model.bucketName);
            inclusionPrefixes(model.inclusionPrefixes);
            inclusionPatterns(model.inclusionPatterns);
            exclusionPatterns(model.exclusionPatterns);
            documentsMetadataConfiguration(model.documentsMetadataConfiguration);
            accessControlListConfiguration(model.accessControlListConfiguration);
        }

        public final String getBucketName() {
            return bucketName;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final Collection<String> getInclusionPrefixes() {
            if (inclusionPrefixes instanceof SdkAutoConstructList) {
                return null;
            }
            return inclusionPrefixes;
        }

        public final void setInclusionPrefixes(Collection<String> inclusionPrefixes) {
            this.inclusionPrefixes = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPrefixes);
        }

        @Override
        public final Builder inclusionPrefixes(Collection<String> inclusionPrefixes) {
            this.inclusionPrefixes = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPrefixes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inclusionPrefixes(String... inclusionPrefixes) {
            inclusionPrefixes(Arrays.asList(inclusionPrefixes));
            return this;
        }

        public final Collection<String> getInclusionPatterns() {
            if (inclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return inclusionPatterns;
        }

        public final void setInclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
        }

        @Override
        public final Builder inclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inclusionPatterns(String... inclusionPatterns) {
            inclusionPatterns(Arrays.asList(inclusionPatterns));
            return this;
        }

        public final Collection<String> getExclusionPatterns() {
            if (exclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return exclusionPatterns;
        }

        public final void setExclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
        }

        @Override
        public final Builder exclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exclusionPatterns(String... exclusionPatterns) {
            exclusionPatterns(Arrays.asList(exclusionPatterns));
            return this;
        }

        public final DocumentsMetadataConfiguration.Builder getDocumentsMetadataConfiguration() {
            return documentsMetadataConfiguration != null ? documentsMetadataConfiguration.toBuilder() : null;
        }

        public final void setDocumentsMetadataConfiguration(
                DocumentsMetadataConfiguration.BuilderImpl documentsMetadataConfiguration) {
            this.documentsMetadataConfiguration = documentsMetadataConfiguration != null ? documentsMetadataConfiguration.build()
                    : null;
        }

        @Override
        public final Builder documentsMetadataConfiguration(DocumentsMetadataConfiguration documentsMetadataConfiguration) {
            this.documentsMetadataConfiguration = documentsMetadataConfiguration;
            return this;
        }

        public final AccessControlListConfiguration.Builder getAccessControlListConfiguration() {
            return accessControlListConfiguration != null ? accessControlListConfiguration.toBuilder() : null;
        }

        public final void setAccessControlListConfiguration(
                AccessControlListConfiguration.BuilderImpl accessControlListConfiguration) {
            this.accessControlListConfiguration = accessControlListConfiguration != null ? accessControlListConfiguration.build()
                    : null;
        }

        @Override
        public final Builder accessControlListConfiguration(AccessControlListConfiguration accessControlListConfiguration) {
            this.accessControlListConfiguration = accessControlListConfiguration;
            return this;
        }

        @Override
        public S3DataSourceConfiguration build() {
            return new S3DataSourceConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
