/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the configuration information to connect to Alfresco as your data source.
 * </p>
 * <note>
 * <p>
 * Support for <code>AlfrescoConfiguration</code> ended May 2023. We recommend migrating to or using the Alfresco data
 * source template schema / <a
 * href="https://docs.aws.amazon.com/kendra/latest/APIReference/API_TemplateConfiguration.html"
 * >TemplateConfiguration</a> API.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AlfrescoConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<AlfrescoConfiguration.Builder, AlfrescoConfiguration> {
    private static final SdkField<String> SITE_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SiteUrl").getter(getter(AlfrescoConfiguration::siteUrl)).setter(setter(Builder::siteUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SiteUrl").build()).build();

    private static final SdkField<String> SITE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("SiteId")
            .getter(getter(AlfrescoConfiguration::siteId)).setter(setter(Builder::siteId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SiteId").build()).build();

    private static final SdkField<String> SECRET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretArn").getter(getter(AlfrescoConfiguration::secretArn)).setter(setter(Builder::secretArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretArn").build()).build();

    private static final SdkField<S3Path> SSL_CERTIFICATE_S3_PATH_FIELD = SdkField.<S3Path> builder(MarshallingType.SDK_POJO)
            .memberName("SslCertificateS3Path").getter(getter(AlfrescoConfiguration::sslCertificateS3Path))
            .setter(setter(Builder::sslCertificateS3Path)).constructor(S3Path::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SslCertificateS3Path").build())
            .build();

    private static final SdkField<Boolean> CRAWL_SYSTEM_FOLDERS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CrawlSystemFolders").getter(getter(AlfrescoConfiguration::crawlSystemFolders))
            .setter(setter(Builder::crawlSystemFolders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlSystemFolders").build())
            .build();

    private static final SdkField<Boolean> CRAWL_COMMENTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CrawlComments").getter(getter(AlfrescoConfiguration::crawlComments))
            .setter(setter(Builder::crawlComments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlComments").build()).build();

    private static final SdkField<List<String>> ENTITY_FILTER_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("EntityFilter")
            .getter(getter(AlfrescoConfiguration::entityFilterAsStrings))
            .setter(setter(Builder::entityFilterWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntityFilter").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> DOCUMENT_LIBRARY_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("DocumentLibraryFieldMappings")
            .getter(getter(AlfrescoConfiguration::documentLibraryFieldMappings))
            .setter(setter(Builder::documentLibraryFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentLibraryFieldMappings")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> BLOG_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("BlogFieldMappings")
            .getter(getter(AlfrescoConfiguration::blogFieldMappings))
            .setter(setter(Builder::blogFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlogFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> WIKI_FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("WikiFieldMappings")
            .getter(getter(AlfrescoConfiguration::wikiFieldMappings))
            .setter(setter(Builder::wikiFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WikiFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> INCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InclusionPatterns")
            .getter(getter(AlfrescoConfiguration::inclusionPatterns))
            .setter(setter(Builder::inclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExclusionPatterns")
            .getter(getter(AlfrescoConfiguration::exclusionPatterns))
            .setter(setter(Builder::exclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DataSourceVpcConfiguration> VPC_CONFIGURATION_FIELD = SdkField
            .<DataSourceVpcConfiguration> builder(MarshallingType.SDK_POJO).memberName("VpcConfiguration")
            .getter(getter(AlfrescoConfiguration::vpcConfiguration)).setter(setter(Builder::vpcConfiguration))
            .constructor(DataSourceVpcConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfiguration").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SITE_URL_FIELD, SITE_ID_FIELD,
            SECRET_ARN_FIELD, SSL_CERTIFICATE_S3_PATH_FIELD, CRAWL_SYSTEM_FOLDERS_FIELD, CRAWL_COMMENTS_FIELD,
            ENTITY_FILTER_FIELD, DOCUMENT_LIBRARY_FIELD_MAPPINGS_FIELD, BLOG_FIELD_MAPPINGS_FIELD, WIKI_FIELD_MAPPINGS_FIELD,
            INCLUSION_PATTERNS_FIELD, EXCLUSION_PATTERNS_FIELD, VPC_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String siteUrl;

    private final String siteId;

    private final String secretArn;

    private final S3Path sslCertificateS3Path;

    private final Boolean crawlSystemFolders;

    private final Boolean crawlComments;

    private final List<String> entityFilter;

    private final List<DataSourceToIndexFieldMapping> documentLibraryFieldMappings;

    private final List<DataSourceToIndexFieldMapping> blogFieldMappings;

    private final List<DataSourceToIndexFieldMapping> wikiFieldMappings;

    private final List<String> inclusionPatterns;

    private final List<String> exclusionPatterns;

    private final DataSourceVpcConfiguration vpcConfiguration;

    private AlfrescoConfiguration(BuilderImpl builder) {
        this.siteUrl = builder.siteUrl;
        this.siteId = builder.siteId;
        this.secretArn = builder.secretArn;
        this.sslCertificateS3Path = builder.sslCertificateS3Path;
        this.crawlSystemFolders = builder.crawlSystemFolders;
        this.crawlComments = builder.crawlComments;
        this.entityFilter = builder.entityFilter;
        this.documentLibraryFieldMappings = builder.documentLibraryFieldMappings;
        this.blogFieldMappings = builder.blogFieldMappings;
        this.wikiFieldMappings = builder.wikiFieldMappings;
        this.inclusionPatterns = builder.inclusionPatterns;
        this.exclusionPatterns = builder.exclusionPatterns;
        this.vpcConfiguration = builder.vpcConfiguration;
    }

    /**
     * <p>
     * The URL of the Alfresco site. For example, <i>https://hostname:8080</i>.
     * </p>
     * 
     * @return The URL of the Alfresco site. For example, <i>https://hostname:8080</i>.
     */
    public final String siteUrl() {
        return siteUrl;
    }

    /**
     * <p>
     * The identifier of the Alfresco site. For example, <i>my-site</i>.
     * </p>
     * 
     * @return The identifier of the Alfresco site. For example, <i>my-site</i>.
     */
    public final String siteId() {
        return siteId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required to connect
     * to your Alfresco data source. The secret must contain a JSON structure with the following keys:
     * </p>
     * <ul>
     * <li>
     * <p>
     * username—The user name of the Alfresco account.
     * </p>
     * </li>
     * <li>
     * <p>
     * password—The password of the Alfresco account.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required to
     *         connect to your Alfresco data source. The secret must contain a JSON structure with the following
     *         keys:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         username—The user name of the Alfresco account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         password—The password of the Alfresco account.
     *         </p>
     *         </li>
     */
    public final String secretArn() {
        return secretArn;
    }

    /**
     * <p>
     * The path to the SSL certificate stored in an Amazon S3 bucket. You use this to connect to Alfresco if you require
     * a secure SSL connection.
     * </p>
     * <p>
     * You can simply generate a self-signed X509 certificate on any computer using OpenSSL. For an example of using
     * OpenSSL to create an X509 certificate, see <a
     * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/configuring-https-ssl.html">Create and sign an X509
     * certificate</a>.
     * </p>
     * 
     * @return The path to the SSL certificate stored in an Amazon S3 bucket. You use this to connect to Alfresco if you
     *         require a secure SSL connection.</p>
     *         <p>
     *         You can simply generate a self-signed X509 certificate on any computer using OpenSSL. For an example of
     *         using OpenSSL to create an X509 certificate, see <a
     *         href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/configuring-https-ssl.html">Create and sign
     *         an X509 certificate</a>.
     */
    public final S3Path sslCertificateS3Path() {
        return sslCertificateS3Path;
    }

    /**
     * <p>
     * <code>TRUE</code> to index shared files.
     * </p>
     * 
     * @return <code>TRUE</code> to index shared files.
     */
    public final Boolean crawlSystemFolders() {
        return crawlSystemFolders;
    }

    /**
     * <p>
     * <code>TRUE</code> to index comments of blogs and other content.
     * </p>
     * 
     * @return <code>TRUE</code> to index comments of blogs and other content.
     */
    public final Boolean crawlComments() {
        return crawlComments;
    }

    /**
     * <p>
     * Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these options.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntityFilter} method.
     * </p>
     * 
     * @return Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these
     *         options.
     */
    public final List<AlfrescoEntity> entityFilter() {
        return EntityFilterCopier.copyStringToEnum(entityFilter);
    }

    /**
     * For responses, this returns true if the service returned a value for the EntityFilter property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEntityFilter() {
        return entityFilter != null && !(entityFilter instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these options.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntityFilter} method.
     * </p>
     * 
     * @return Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these
     *         options.
     */
    public final List<String> entityFilterAsStrings() {
        return entityFilter;
    }

    /**
     * For responses, this returns true if the service returned a value for the DocumentLibraryFieldMappings property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDocumentLibraryFieldMappings() {
        return documentLibraryFieldMappings != null && !(documentLibraryFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco
     * document libraries to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code>
     * API before you map to Alfresco fields. For more information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
     * Alfresco data source field names must exist in your Alfresco custom metadata.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDocumentLibraryFieldMappings} method.
     * </p>
     * 
     * @return A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of
     *         Alfresco document libraries to Amazon Kendra index field names. To create custom fields, use the
     *         <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>.
     *         The Alfresco data source field names must exist in your Alfresco custom metadata.
     */
    public final List<DataSourceToIndexFieldMapping> documentLibraryFieldMappings() {
        return documentLibraryFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the BlogFieldMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBlogFieldMappings() {
        return blogFieldMappings != null && !(blogFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco blogs
     * to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API before you map
     * to Alfresco fields. For more information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
     * Alfresco data source field names must exist in your Alfresco custom metadata.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBlogFieldMappings} method.
     * </p>
     * 
     * @return A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of
     *         Alfresco blogs to Amazon Kendra index field names. To create custom fields, use the
     *         <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>.
     *         The Alfresco data source field names must exist in your Alfresco custom metadata.
     */
    public final List<DataSourceToIndexFieldMapping> blogFieldMappings() {
        return blogFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the WikiFieldMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasWikiFieldMappings() {
        return wikiFieldMappings != null && !(wikiFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco wikis
     * to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API before you map
     * to Alfresco fields. For more information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
     * Alfresco data source field names must exist in your Alfresco custom metadata.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasWikiFieldMappings} method.
     * </p>
     * 
     * @return A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of
     *         Alfresco wikis to Amazon Kendra index field names. To create custom fields, use the
     *         <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>.
     *         The Alfresco data source field names must exist in your Alfresco custom metadata.
     */
    public final List<DataSourceToIndexFieldMapping> wikiFieldMappings() {
        return wikiFieldMappings;
    }

    /**
     * For responses, this returns true if the service returned a value for the InclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInclusionPatterns() {
        return inclusionPatterns != null && !(inclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to include certain files in your Alfresco data source. Files that match the
     * patterns are included in the index. Files that don't match the patterns are excluded from the index. If a file
     * matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence and the file
     * isn't included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to include certain files in your Alfresco data source. Files that
     *         match the patterns are included in the index. Files that don't match the patterns are excluded from the
     *         index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes
     *         precedence and the file isn't included in the index.
     */
    public final List<String> inclusionPatterns() {
        return inclusionPatterns;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExclusionPatterns() {
        return exclusionPatterns != null && !(exclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to exclude certain files in your Alfresco data source. Files that match the
     * patterns are excluded from the index. Files that don't match the patterns are included in the index. If a file
     * matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence and the file
     * isn't included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to exclude certain files in your Alfresco data source. Files that
     *         match the patterns are excluded from the index. Files that don't match the patterns are included in the
     *         index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes
     *         precedence and the file isn't included in the index.
     */
    public final List<String> exclusionPatterns() {
        return exclusionPatterns;
    }

    /**
     * <p>
     * Configuration information for an Amazon Virtual Private Cloud to connect to your Alfresco. For more information,
     * see <a href="https://docs.aws.amazon.com/kendra/latest/dg/vpc-configuration.html">Configuring a VPC</a>.
     * </p>
     * 
     * @return Configuration information for an Amazon Virtual Private Cloud to connect to your Alfresco. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/vpc-configuration.html">Configuring a VPC</a>.
     */
    public final DataSourceVpcConfiguration vpcConfiguration() {
        return vpcConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(siteUrl());
        hashCode = 31 * hashCode + Objects.hashCode(siteId());
        hashCode = 31 * hashCode + Objects.hashCode(secretArn());
        hashCode = 31 * hashCode + Objects.hashCode(sslCertificateS3Path());
        hashCode = 31 * hashCode + Objects.hashCode(crawlSystemFolders());
        hashCode = 31 * hashCode + Objects.hashCode(crawlComments());
        hashCode = 31 * hashCode + Objects.hashCode(hasEntityFilter() ? entityFilterAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDocumentLibraryFieldMappings() ? documentLibraryFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasBlogFieldMappings() ? blogFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasWikiFieldMappings() ? wikiFieldMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInclusionPatterns() ? inclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExclusionPatterns() ? exclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AlfrescoConfiguration)) {
            return false;
        }
        AlfrescoConfiguration other = (AlfrescoConfiguration) obj;
        return Objects.equals(siteUrl(), other.siteUrl()) && Objects.equals(siteId(), other.siteId())
                && Objects.equals(secretArn(), other.secretArn())
                && Objects.equals(sslCertificateS3Path(), other.sslCertificateS3Path())
                && Objects.equals(crawlSystemFolders(), other.crawlSystemFolders())
                && Objects.equals(crawlComments(), other.crawlComments()) && hasEntityFilter() == other.hasEntityFilter()
                && Objects.equals(entityFilterAsStrings(), other.entityFilterAsStrings())
                && hasDocumentLibraryFieldMappings() == other.hasDocumentLibraryFieldMappings()
                && Objects.equals(documentLibraryFieldMappings(), other.documentLibraryFieldMappings())
                && hasBlogFieldMappings() == other.hasBlogFieldMappings()
                && Objects.equals(blogFieldMappings(), other.blogFieldMappings())
                && hasWikiFieldMappings() == other.hasWikiFieldMappings()
                && Objects.equals(wikiFieldMappings(), other.wikiFieldMappings())
                && hasInclusionPatterns() == other.hasInclusionPatterns()
                && Objects.equals(inclusionPatterns(), other.inclusionPatterns())
                && hasExclusionPatterns() == other.hasExclusionPatterns()
                && Objects.equals(exclusionPatterns(), other.exclusionPatterns())
                && Objects.equals(vpcConfiguration(), other.vpcConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AlfrescoConfiguration").add("SiteUrl", siteUrl()).add("SiteId", siteId())
                .add("SecretArn", secretArn()).add("SslCertificateS3Path", sslCertificateS3Path())
                .add("CrawlSystemFolders", crawlSystemFolders()).add("CrawlComments", crawlComments())
                .add("EntityFilter", hasEntityFilter() ? entityFilterAsStrings() : null)
                .add("DocumentLibraryFieldMappings", hasDocumentLibraryFieldMappings() ? documentLibraryFieldMappings() : null)
                .add("BlogFieldMappings", hasBlogFieldMappings() ? blogFieldMappings() : null)
                .add("WikiFieldMappings", hasWikiFieldMappings() ? wikiFieldMappings() : null)
                .add("InclusionPatterns", hasInclusionPatterns() ? inclusionPatterns() : null)
                .add("ExclusionPatterns", hasExclusionPatterns() ? exclusionPatterns() : null)
                .add("VpcConfiguration", vpcConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SiteUrl":
            return Optional.ofNullable(clazz.cast(siteUrl()));
        case "SiteId":
            return Optional.ofNullable(clazz.cast(siteId()));
        case "SecretArn":
            return Optional.ofNullable(clazz.cast(secretArn()));
        case "SslCertificateS3Path":
            return Optional.ofNullable(clazz.cast(sslCertificateS3Path()));
        case "CrawlSystemFolders":
            return Optional.ofNullable(clazz.cast(crawlSystemFolders()));
        case "CrawlComments":
            return Optional.ofNullable(clazz.cast(crawlComments()));
        case "EntityFilter":
            return Optional.ofNullable(clazz.cast(entityFilterAsStrings()));
        case "DocumentLibraryFieldMappings":
            return Optional.ofNullable(clazz.cast(documentLibraryFieldMappings()));
        case "BlogFieldMappings":
            return Optional.ofNullable(clazz.cast(blogFieldMappings()));
        case "WikiFieldMappings":
            return Optional.ofNullable(clazz.cast(wikiFieldMappings()));
        case "InclusionPatterns":
            return Optional.ofNullable(clazz.cast(inclusionPatterns()));
        case "ExclusionPatterns":
            return Optional.ofNullable(clazz.cast(exclusionPatterns()));
        case "VpcConfiguration":
            return Optional.ofNullable(clazz.cast(vpcConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SiteUrl", SITE_URL_FIELD);
        map.put("SiteId", SITE_ID_FIELD);
        map.put("SecretArn", SECRET_ARN_FIELD);
        map.put("SslCertificateS3Path", SSL_CERTIFICATE_S3_PATH_FIELD);
        map.put("CrawlSystemFolders", CRAWL_SYSTEM_FOLDERS_FIELD);
        map.put("CrawlComments", CRAWL_COMMENTS_FIELD);
        map.put("EntityFilter", ENTITY_FILTER_FIELD);
        map.put("DocumentLibraryFieldMappings", DOCUMENT_LIBRARY_FIELD_MAPPINGS_FIELD);
        map.put("BlogFieldMappings", BLOG_FIELD_MAPPINGS_FIELD);
        map.put("WikiFieldMappings", WIKI_FIELD_MAPPINGS_FIELD);
        map.put("InclusionPatterns", INCLUSION_PATTERNS_FIELD);
        map.put("ExclusionPatterns", EXCLUSION_PATTERNS_FIELD);
        map.put("VpcConfiguration", VPC_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AlfrescoConfiguration, T> g) {
        return obj -> g.apply((AlfrescoConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AlfrescoConfiguration> {
        /**
         * <p>
         * The URL of the Alfresco site. For example, <i>https://hostname:8080</i>.
         * </p>
         * 
         * @param siteUrl
         *        The URL of the Alfresco site. For example, <i>https://hostname:8080</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder siteUrl(String siteUrl);

        /**
         * <p>
         * The identifier of the Alfresco site. For example, <i>my-site</i>.
         * </p>
         * 
         * @param siteId
         *        The identifier of the Alfresco site. For example, <i>my-site</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder siteId(String siteId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required to
         * connect to your Alfresco data source. The secret must contain a JSON structure with the following keys:
         * </p>
         * <ul>
         * <li>
         * <p>
         * username—The user name of the Alfresco account.
         * </p>
         * </li>
         * <li>
         * <p>
         * password—The password of the Alfresco account.
         * </p>
         * </li>
         * </ul>
         * 
         * @param secretArn
         *        The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required
         *        to connect to your Alfresco data source. The secret must contain a JSON structure with the following
         *        keys:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        username—The user name of the Alfresco account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        password—The password of the Alfresco account.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretArn(String secretArn);

        /**
         * <p>
         * The path to the SSL certificate stored in an Amazon S3 bucket. You use this to connect to Alfresco if you
         * require a secure SSL connection.
         * </p>
         * <p>
         * You can simply generate a self-signed X509 certificate on any computer using OpenSSL. For an example of using
         * OpenSSL to create an X509 certificate, see <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/configuring-https-ssl.html">Create and sign an
         * X509 certificate</a>.
         * </p>
         * 
         * @param sslCertificateS3Path
         *        The path to the SSL certificate stored in an Amazon S3 bucket. You use this to connect to Alfresco if
         *        you require a secure SSL connection.</p>
         *        <p>
         *        You can simply generate a self-signed X509 certificate on any computer using OpenSSL. For an example
         *        of using OpenSSL to create an X509 certificate, see <a
         *        href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/configuring-https-ssl.html">Create and
         *        sign an X509 certificate</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sslCertificateS3Path(S3Path sslCertificateS3Path);

        /**
         * <p>
         * The path to the SSL certificate stored in an Amazon S3 bucket. You use this to connect to Alfresco if you
         * require a secure SSL connection.
         * </p>
         * <p>
         * You can simply generate a self-signed X509 certificate on any computer using OpenSSL. For an example of using
         * OpenSSL to create an X509 certificate, see <a
         * href="https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/configuring-https-ssl.html">Create and sign an
         * X509 certificate</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3Path.Builder} avoiding the need to
         * create one manually via {@link S3Path#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3Path.Builder#build()} is called immediately and its result is
         * passed to {@link #sslCertificateS3Path(S3Path)}.
         * 
         * @param sslCertificateS3Path
         *        a consumer that will call methods on {@link S3Path.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sslCertificateS3Path(S3Path)
         */
        default Builder sslCertificateS3Path(Consumer<S3Path.Builder> sslCertificateS3Path) {
            return sslCertificateS3Path(S3Path.builder().applyMutation(sslCertificateS3Path).build());
        }

        /**
         * <p>
         * <code>TRUE</code> to index shared files.
         * </p>
         * 
         * @param crawlSystemFolders
         *        <code>TRUE</code> to index shared files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlSystemFolders(Boolean crawlSystemFolders);

        /**
         * <p>
         * <code>TRUE</code> to index comments of blogs and other content.
         * </p>
         * 
         * @param crawlComments
         *        <code>TRUE</code> to index comments of blogs and other content.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlComments(Boolean crawlComments);

        /**
         * <p>
         * Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these options.
         * </p>
         * 
         * @param entityFilter
         *        Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityFilterWithStrings(Collection<String> entityFilter);

        /**
         * <p>
         * Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these options.
         * </p>
         * 
         * @param entityFilter
         *        Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityFilterWithStrings(String... entityFilter);

        /**
         * <p>
         * Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these options.
         * </p>
         * 
         * @param entityFilter
         *        Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityFilter(Collection<AlfrescoEntity> entityFilter);

        /**
         * <p>
         * Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these options.
         * </p>
         * 
         * @param entityFilter
         *        Specify whether to index document libraries, wikis, or blogs. You can specify one or more of these
         *        options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityFilter(AlfrescoEntity... entityFilter);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco
         * document libraries to Amazon Kendra index field names. To create custom fields, use the
         * <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
         * Alfresco data source field names must exist in your Alfresco custom metadata.
         * </p>
         * 
         * @param documentLibraryFieldMappings
         *        A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of
         *        Alfresco document libraries to Amazon Kendra index field names. To create custom fields, use the
         *        <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Alfresco data source field names must exist in your Alfresco custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentLibraryFieldMappings(Collection<DataSourceToIndexFieldMapping> documentLibraryFieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco
         * document libraries to Amazon Kendra index field names. To create custom fields, use the
         * <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
         * Alfresco data source field names must exist in your Alfresco custom metadata.
         * </p>
         * 
         * @param documentLibraryFieldMappings
         *        A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of
         *        Alfresco document libraries to Amazon Kendra index field names. To create custom fields, use the
         *        <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Alfresco data source field names must exist in your Alfresco custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentLibraryFieldMappings(DataSourceToIndexFieldMapping... documentLibraryFieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco
         * document libraries to Amazon Kendra index field names. To create custom fields, use the
         * <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
         * Alfresco data source field names must exist in your Alfresco custom metadata.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #documentLibraryFieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param documentLibraryFieldMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #documentLibraryFieldMappings(java.util.Collection<DataSourceToIndexFieldMapping>)
         */
        Builder documentLibraryFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... documentLibraryFieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco
         * blogs to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API
         * before you map to Alfresco fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
         * Alfresco data source field names must exist in your Alfresco custom metadata.
         * </p>
         * 
         * @param blogFieldMappings
         *        A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of
         *        Alfresco blogs to Amazon Kendra index field names. To create custom fields, use the
         *        <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Alfresco data source field names must exist in your Alfresco custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blogFieldMappings(Collection<DataSourceToIndexFieldMapping> blogFieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco
         * blogs to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API
         * before you map to Alfresco fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
         * Alfresco data source field names must exist in your Alfresco custom metadata.
         * </p>
         * 
         * @param blogFieldMappings
         *        A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of
         *        Alfresco blogs to Amazon Kendra index field names. To create custom fields, use the
         *        <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Alfresco data source field names must exist in your Alfresco custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blogFieldMappings(DataSourceToIndexFieldMapping... blogFieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco
         * blogs to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API
         * before you map to Alfresco fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
         * Alfresco data source field names must exist in your Alfresco custom metadata.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder#build()} is called
         * immediately and its result is passed to {@link #blogFieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param blogFieldMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #blogFieldMappings(java.util.Collection<DataSourceToIndexFieldMapping>)
         */
        Builder blogFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... blogFieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco
         * wikis to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API
         * before you map to Alfresco fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
         * Alfresco data source field names must exist in your Alfresco custom metadata.
         * </p>
         * 
         * @param wikiFieldMappings
         *        A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of
         *        Alfresco wikis to Amazon Kendra index field names. To create custom fields, use the
         *        <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Alfresco data source field names must exist in your Alfresco custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wikiFieldMappings(Collection<DataSourceToIndexFieldMapping> wikiFieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco
         * wikis to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API
         * before you map to Alfresco fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
         * Alfresco data source field names must exist in your Alfresco custom metadata.
         * </p>
         * 
         * @param wikiFieldMappings
         *        A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of
         *        Alfresco wikis to Amazon Kendra index field names. To create custom fields, use the
         *        <code>UpdateIndex</code> API before you map to Alfresco fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source
         *        fields</a>. The Alfresco data source field names must exist in your Alfresco custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wikiFieldMappings(DataSourceToIndexFieldMapping... wikiFieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map attributes or field names of Alfresco
         * wikis to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API
         * before you map to Alfresco fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html"> Mapping data source fields</a>. The
         * Alfresco data source field names must exist in your Alfresco custom metadata.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder#build()} is called
         * immediately and its result is passed to {@link #wikiFieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param wikiFieldMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #wikiFieldMappings(java.util.Collection<DataSourceToIndexFieldMapping>)
         */
        Builder wikiFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... wikiFieldMappings);

        /**
         * <p>
         * A list of regular expression patterns to include certain files in your Alfresco data source. Files that match
         * the patterns are included in the index. Files that don't match the patterns are excluded from the index. If a
         * file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence and
         * the file isn't included in the index.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns to include certain files in your Alfresco data source. Files
         *        that match the patterns are included in the index. Files that don't match the patterns are excluded
         *        from the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion
         *        pattern takes precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(Collection<String> inclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to include certain files in your Alfresco data source. Files that match
         * the patterns are included in the index. Files that don't match the patterns are excluded from the index. If a
         * file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence and
         * the file isn't included in the index.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns to include certain files in your Alfresco data source. Files
         *        that match the patterns are included in the index. Files that don't match the patterns are excluded
         *        from the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion
         *        pattern takes precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(String... inclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to exclude certain files in your Alfresco data source. Files that match
         * the patterns are excluded from the index. Files that don't match the patterns are included in the index. If a
         * file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence and
         * the file isn't included in the index.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of regular expression patterns to exclude certain files in your Alfresco data source. Files
         *        that match the patterns are excluded from the index. Files that don't match the patterns are included
         *        in the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion
         *        pattern takes precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(Collection<String> exclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to exclude certain files in your Alfresco data source. Files that match
         * the patterns are excluded from the index. Files that don't match the patterns are included in the index. If a
         * file matches both an inclusion pattern and an exclusion pattern, the exclusion pattern takes precedence and
         * the file isn't included in the index.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of regular expression patterns to exclude certain files in your Alfresco data source. Files
         *        that match the patterns are excluded from the index. Files that don't match the patterns are included
         *        in the index. If a file matches both an inclusion pattern and an exclusion pattern, the exclusion
         *        pattern takes precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(String... exclusionPatterns);

        /**
         * <p>
         * Configuration information for an Amazon Virtual Private Cloud to connect to your Alfresco. For more
         * information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/vpc-configuration.html">Configuring a
         * VPC</a>.
         * </p>
         * 
         * @param vpcConfiguration
         *        Configuration information for an Amazon Virtual Private Cloud to connect to your Alfresco. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/vpc-configuration.html">Configuring a VPC</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration);

        /**
         * <p>
         * Configuration information for an Amazon Virtual Private Cloud to connect to your Alfresco. For more
         * information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/vpc-configuration.html">Configuring a
         * VPC</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataSourceVpcConfiguration.Builder}
         * avoiding the need to create one manually via {@link DataSourceVpcConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataSourceVpcConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #vpcConfiguration(DataSourceVpcConfiguration)}.
         * 
         * @param vpcConfiguration
         *        a consumer that will call methods on {@link DataSourceVpcConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfiguration(DataSourceVpcConfiguration)
         */
        default Builder vpcConfiguration(Consumer<DataSourceVpcConfiguration.Builder> vpcConfiguration) {
            return vpcConfiguration(DataSourceVpcConfiguration.builder().applyMutation(vpcConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String siteUrl;

        private String siteId;

        private String secretArn;

        private S3Path sslCertificateS3Path;

        private Boolean crawlSystemFolders;

        private Boolean crawlComments;

        private List<String> entityFilter = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> documentLibraryFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> blogFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> wikiFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private List<String> inclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private List<String> exclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private DataSourceVpcConfiguration vpcConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(AlfrescoConfiguration model) {
            siteUrl(model.siteUrl);
            siteId(model.siteId);
            secretArn(model.secretArn);
            sslCertificateS3Path(model.sslCertificateS3Path);
            crawlSystemFolders(model.crawlSystemFolders);
            crawlComments(model.crawlComments);
            entityFilterWithStrings(model.entityFilter);
            documentLibraryFieldMappings(model.documentLibraryFieldMappings);
            blogFieldMappings(model.blogFieldMappings);
            wikiFieldMappings(model.wikiFieldMappings);
            inclusionPatterns(model.inclusionPatterns);
            exclusionPatterns(model.exclusionPatterns);
            vpcConfiguration(model.vpcConfiguration);
        }

        public final String getSiteUrl() {
            return siteUrl;
        }

        public final void setSiteUrl(String siteUrl) {
            this.siteUrl = siteUrl;
        }

        @Override
        public final Builder siteUrl(String siteUrl) {
            this.siteUrl = siteUrl;
            return this;
        }

        public final String getSiteId() {
            return siteId;
        }

        public final void setSiteId(String siteId) {
            this.siteId = siteId;
        }

        @Override
        public final Builder siteId(String siteId) {
            this.siteId = siteId;
            return this;
        }

        public final String getSecretArn() {
            return secretArn;
        }

        public final void setSecretArn(String secretArn) {
            this.secretArn = secretArn;
        }

        @Override
        public final Builder secretArn(String secretArn) {
            this.secretArn = secretArn;
            return this;
        }

        public final S3Path.Builder getSslCertificateS3Path() {
            return sslCertificateS3Path != null ? sslCertificateS3Path.toBuilder() : null;
        }

        public final void setSslCertificateS3Path(S3Path.BuilderImpl sslCertificateS3Path) {
            this.sslCertificateS3Path = sslCertificateS3Path != null ? sslCertificateS3Path.build() : null;
        }

        @Override
        public final Builder sslCertificateS3Path(S3Path sslCertificateS3Path) {
            this.sslCertificateS3Path = sslCertificateS3Path;
            return this;
        }

        public final Boolean getCrawlSystemFolders() {
            return crawlSystemFolders;
        }

        public final void setCrawlSystemFolders(Boolean crawlSystemFolders) {
            this.crawlSystemFolders = crawlSystemFolders;
        }

        @Override
        public final Builder crawlSystemFolders(Boolean crawlSystemFolders) {
            this.crawlSystemFolders = crawlSystemFolders;
            return this;
        }

        public final Boolean getCrawlComments() {
            return crawlComments;
        }

        public final void setCrawlComments(Boolean crawlComments) {
            this.crawlComments = crawlComments;
        }

        @Override
        public final Builder crawlComments(Boolean crawlComments) {
            this.crawlComments = crawlComments;
            return this;
        }

        public final Collection<String> getEntityFilter() {
            if (entityFilter instanceof SdkAutoConstructList) {
                return null;
            }
            return entityFilter;
        }

        public final void setEntityFilter(Collection<String> entityFilter) {
            this.entityFilter = EntityFilterCopier.copy(entityFilter);
        }

        @Override
        public final Builder entityFilterWithStrings(Collection<String> entityFilter) {
            this.entityFilter = EntityFilterCopier.copy(entityFilter);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entityFilterWithStrings(String... entityFilter) {
            entityFilterWithStrings(Arrays.asList(entityFilter));
            return this;
        }

        @Override
        public final Builder entityFilter(Collection<AlfrescoEntity> entityFilter) {
            this.entityFilter = EntityFilterCopier.copyEnumToString(entityFilter);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entityFilter(AlfrescoEntity... entityFilter) {
            entityFilter(Arrays.asList(entityFilter));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getDocumentLibraryFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.documentLibraryFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDocumentLibraryFieldMappings(
                Collection<DataSourceToIndexFieldMapping.BuilderImpl> documentLibraryFieldMappings) {
            this.documentLibraryFieldMappings = DataSourceToIndexFieldMappingListCopier
                    .copyFromBuilder(documentLibraryFieldMappings);
        }

        @Override
        public final Builder documentLibraryFieldMappings(Collection<DataSourceToIndexFieldMapping> documentLibraryFieldMappings) {
            this.documentLibraryFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(documentLibraryFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder documentLibraryFieldMappings(DataSourceToIndexFieldMapping... documentLibraryFieldMappings) {
            documentLibraryFieldMappings(Arrays.asList(documentLibraryFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder documentLibraryFieldMappings(
                Consumer<DataSourceToIndexFieldMapping.Builder>... documentLibraryFieldMappings) {
            documentLibraryFieldMappings(Stream.of(documentLibraryFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getBlogFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.blogFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBlogFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> blogFieldMappings) {
            this.blogFieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(blogFieldMappings);
        }

        @Override
        public final Builder blogFieldMappings(Collection<DataSourceToIndexFieldMapping> blogFieldMappings) {
            this.blogFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(blogFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder blogFieldMappings(DataSourceToIndexFieldMapping... blogFieldMappings) {
            blogFieldMappings(Arrays.asList(blogFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder blogFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... blogFieldMappings) {
            blogFieldMappings(Stream.of(blogFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getWikiFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.wikiFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setWikiFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> wikiFieldMappings) {
            this.wikiFieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(wikiFieldMappings);
        }

        @Override
        public final Builder wikiFieldMappings(Collection<DataSourceToIndexFieldMapping> wikiFieldMappings) {
            this.wikiFieldMappings = DataSourceToIndexFieldMappingListCopier.copy(wikiFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder wikiFieldMappings(DataSourceToIndexFieldMapping... wikiFieldMappings) {
            wikiFieldMappings(Arrays.asList(wikiFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder wikiFieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... wikiFieldMappings) {
            wikiFieldMappings(Stream.of(wikiFieldMappings)
                    .map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getInclusionPatterns() {
            if (inclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return inclusionPatterns;
        }

        public final void setInclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
        }

        @Override
        public final Builder inclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inclusionPatterns(String... inclusionPatterns) {
            inclusionPatterns(Arrays.asList(inclusionPatterns));
            return this;
        }

        public final Collection<String> getExclusionPatterns() {
            if (exclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return exclusionPatterns;
        }

        public final void setExclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
        }

        @Override
        public final Builder exclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exclusionPatterns(String... exclusionPatterns) {
            exclusionPatterns(Arrays.asList(exclusionPatterns));
            return this;
        }

        public final DataSourceVpcConfiguration.Builder getVpcConfiguration() {
            return vpcConfiguration != null ? vpcConfiguration.toBuilder() : null;
        }

        public final void setVpcConfiguration(DataSourceVpcConfiguration.BuilderImpl vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration != null ? vpcConfiguration.build() : null;
        }

        @Override
        public final Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration;
            return this;
        }

        @Override
        public AlfrescoConfiguration build() {
            return new AlfrescoConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
