/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration information for indexing Confluence spaces.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConfluenceSpaceConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ConfluenceSpaceConfiguration.Builder, ConfluenceSpaceConfiguration> {
    private static final SdkField<Boolean> CRAWL_PERSONAL_SPACES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CrawlPersonalSpaces").getter(getter(ConfluenceSpaceConfiguration::crawlPersonalSpaces))
            .setter(setter(Builder::crawlPersonalSpaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlPersonalSpaces").build())
            .build();

    private static final SdkField<Boolean> CRAWL_ARCHIVED_SPACES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CrawlArchivedSpaces").getter(getter(ConfluenceSpaceConfiguration::crawlArchivedSpaces))
            .setter(setter(Builder::crawlArchivedSpaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlArchivedSpaces").build())
            .build();

    private static final SdkField<List<String>> INCLUDE_SPACES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IncludeSpaces")
            .getter(getter(ConfluenceSpaceConfiguration::includeSpaces))
            .setter(setter(Builder::includeSpaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeSpaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUDE_SPACES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExcludeSpaces")
            .getter(getter(ConfluenceSpaceConfiguration::excludeSpaces))
            .setter(setter(Builder::excludeSpaces))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludeSpaces").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ConfluenceSpaceToIndexFieldMapping>> SPACE_FIELD_MAPPINGS_FIELD = SdkField
            .<List<ConfluenceSpaceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("SpaceFieldMappings")
            .getter(getter(ConfluenceSpaceConfiguration::spaceFieldMappings))
            .setter(setter(Builder::spaceFieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpaceFieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ConfluenceSpaceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(ConfluenceSpaceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CRAWL_PERSONAL_SPACES_FIELD,
            CRAWL_ARCHIVED_SPACES_FIELD, INCLUDE_SPACES_FIELD, EXCLUDE_SPACES_FIELD, SPACE_FIELD_MAPPINGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean crawlPersonalSpaces;

    private final Boolean crawlArchivedSpaces;

    private final List<String> includeSpaces;

    private final List<String> excludeSpaces;

    private final List<ConfluenceSpaceToIndexFieldMapping> spaceFieldMappings;

    private ConfluenceSpaceConfiguration(BuilderImpl builder) {
        this.crawlPersonalSpaces = builder.crawlPersonalSpaces;
        this.crawlArchivedSpaces = builder.crawlArchivedSpaces;
        this.includeSpaces = builder.includeSpaces;
        this.excludeSpaces = builder.excludeSpaces;
        this.spaceFieldMappings = builder.spaceFieldMappings;
    }

    /**
     * <p>
     * <code>TRUE</code> to index personal spaces. You can add restrictions to items in personal spaces. If personal
     * spaces are indexed, queries without user context information may return restricted items from a personal space in
     * their results. For more information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/user-context-filter.html">Filtering on user context</a>.
     * </p>
     * 
     * @return <code>TRUE</code> to index personal spaces. You can add restrictions to items in personal spaces. If
     *         personal spaces are indexed, queries without user context information may return restricted items from a
     *         personal space in their results. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/user-context-filter.html">Filtering on user
     *         context</a>.
     */
    public final Boolean crawlPersonalSpaces() {
        return crawlPersonalSpaces;
    }

    /**
     * <p>
     * <code>TRUE</code> to index archived spaces.
     * </p>
     * 
     * @return <code>TRUE</code> to index archived spaces.
     */
    public final Boolean crawlArchivedSpaces() {
        return crawlArchivedSpaces;
    }

    /**
     * For responses, this returns true if the service returned a value for the IncludeSpaces property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIncludeSpaces() {
        return includeSpaces != null && !(includeSpaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of space keys for Confluence spaces. If you include a key, the blogs, documents, and attachments in the
     * space are indexed. Spaces that aren't in the list aren't indexed. A space in the list must exist. Otherwise,
     * Amazon Kendra logs an error when the data source is synchronized. If a space is in both the
     * <code>IncludeSpaces</code> and the <code>ExcludeSpaces</code> list, the space is excluded.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludeSpaces} method.
     * </p>
     * 
     * @return A list of space keys for Confluence spaces. If you include a key, the blogs, documents, and attachments
     *         in the space are indexed. Spaces that aren't in the list aren't indexed. A space in the list must exist.
     *         Otherwise, Amazon Kendra logs an error when the data source is synchronized. If a space is in both the
     *         <code>IncludeSpaces</code> and the <code>ExcludeSpaces</code> list, the space is excluded.
     */
    public final List<String> includeSpaces() {
        return includeSpaces;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludeSpaces property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExcludeSpaces() {
        return excludeSpaces != null && !(excludeSpaces instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of space keys of Confluence spaces. If you include a key, the blogs, documents, and attachments in the
     * space are not indexed. If a space is in both the <code>ExcludeSpaces</code> and the <code>IncludeSpaces</code>
     * list, the space is excluded.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludeSpaces} method.
     * </p>
     * 
     * @return A list of space keys of Confluence spaces. If you include a key, the blogs, documents, and attachments in
     *         the space are not indexed. If a space is in both the <code>ExcludeSpaces</code> and the
     *         <code>IncludeSpaces</code> list, the space is excluded.
     */
    public final List<String> excludeSpaces() {
        return excludeSpaces;
    }

    /**
     * For responses, this returns true if the service returned a value for the SpaceFieldMappings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSpaceFieldMappings() {
        return spaceFieldMappings != null && !(spaceFieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Maps attributes or field names of Confluence spaces to Amazon Kendra index field names. To create custom fields,
     * use the <code>UpdateIndex</code> API before you map to Confluence fields. For more information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>. The
     * Confluence data source field names must exist in your Confluence custom metadata.
     * </p>
     * <p>
     * If you specify the <code>SpaceFieldMappings</code> parameter, you must specify at least one field mapping.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSpaceFieldMappings} method.
     * </p>
     * 
     * @return Maps attributes or field names of Confluence spaces to Amazon Kendra index field names. To create custom
     *         fields, use the <code>UpdateIndex</code> API before you map to Confluence fields. For more information,
     *         see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source
     *         fields</a>. The Confluence data source field names must exist in your Confluence custom metadata.</p>
     *         <p>
     *         If you specify the <code>SpaceFieldMappings</code> parameter, you must specify at least one field
     *         mapping.
     */
    public final List<ConfluenceSpaceToIndexFieldMapping> spaceFieldMappings() {
        return spaceFieldMappings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(crawlPersonalSpaces());
        hashCode = 31 * hashCode + Objects.hashCode(crawlArchivedSpaces());
        hashCode = 31 * hashCode + Objects.hashCode(hasIncludeSpaces() ? includeSpaces() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludeSpaces() ? excludeSpaces() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSpaceFieldMappings() ? spaceFieldMappings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfluenceSpaceConfiguration)) {
            return false;
        }
        ConfluenceSpaceConfiguration other = (ConfluenceSpaceConfiguration) obj;
        return Objects.equals(crawlPersonalSpaces(), other.crawlPersonalSpaces())
                && Objects.equals(crawlArchivedSpaces(), other.crawlArchivedSpaces())
                && hasIncludeSpaces() == other.hasIncludeSpaces() && Objects.equals(includeSpaces(), other.includeSpaces())
                && hasExcludeSpaces() == other.hasExcludeSpaces() && Objects.equals(excludeSpaces(), other.excludeSpaces())
                && hasSpaceFieldMappings() == other.hasSpaceFieldMappings()
                && Objects.equals(spaceFieldMappings(), other.spaceFieldMappings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConfluenceSpaceConfiguration").add("CrawlPersonalSpaces", crawlPersonalSpaces())
                .add("CrawlArchivedSpaces", crawlArchivedSpaces())
                .add("IncludeSpaces", hasIncludeSpaces() ? includeSpaces() : null)
                .add("ExcludeSpaces", hasExcludeSpaces() ? excludeSpaces() : null)
                .add("SpaceFieldMappings", hasSpaceFieldMappings() ? spaceFieldMappings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CrawlPersonalSpaces":
            return Optional.ofNullable(clazz.cast(crawlPersonalSpaces()));
        case "CrawlArchivedSpaces":
            return Optional.ofNullable(clazz.cast(crawlArchivedSpaces()));
        case "IncludeSpaces":
            return Optional.ofNullable(clazz.cast(includeSpaces()));
        case "ExcludeSpaces":
            return Optional.ofNullable(clazz.cast(excludeSpaces()));
        case "SpaceFieldMappings":
            return Optional.ofNullable(clazz.cast(spaceFieldMappings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CrawlPersonalSpaces", CRAWL_PERSONAL_SPACES_FIELD);
        map.put("CrawlArchivedSpaces", CRAWL_ARCHIVED_SPACES_FIELD);
        map.put("IncludeSpaces", INCLUDE_SPACES_FIELD);
        map.put("ExcludeSpaces", EXCLUDE_SPACES_FIELD);
        map.put("SpaceFieldMappings", SPACE_FIELD_MAPPINGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ConfluenceSpaceConfiguration, T> g) {
        return obj -> g.apply((ConfluenceSpaceConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConfluenceSpaceConfiguration> {
        /**
         * <p>
         * <code>TRUE</code> to index personal spaces. You can add restrictions to items in personal spaces. If personal
         * spaces are indexed, queries without user context information may return restricted items from a personal
         * space in their results. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/user-context-filter.html">Filtering on user context</a>.
         * </p>
         * 
         * @param crawlPersonalSpaces
         *        <code>TRUE</code> to index personal spaces. You can add restrictions to items in personal spaces. If
         *        personal spaces are indexed, queries without user context information may return restricted items from
         *        a personal space in their results. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/user-context-filter.html">Filtering on user
         *        context</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlPersonalSpaces(Boolean crawlPersonalSpaces);

        /**
         * <p>
         * <code>TRUE</code> to index archived spaces.
         * </p>
         * 
         * @param crawlArchivedSpaces
         *        <code>TRUE</code> to index archived spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlArchivedSpaces(Boolean crawlArchivedSpaces);

        /**
         * <p>
         * A list of space keys for Confluence spaces. If you include a key, the blogs, documents, and attachments in
         * the space are indexed. Spaces that aren't in the list aren't indexed. A space in the list must exist.
         * Otherwise, Amazon Kendra logs an error when the data source is synchronized. If a space is in both the
         * <code>IncludeSpaces</code> and the <code>ExcludeSpaces</code> list, the space is excluded.
         * </p>
         * 
         * @param includeSpaces
         *        A list of space keys for Confluence spaces. If you include a key, the blogs, documents, and
         *        attachments in the space are indexed. Spaces that aren't in the list aren't indexed. A space in the
         *        list must exist. Otherwise, Amazon Kendra logs an error when the data source is synchronized. If a
         *        space is in both the <code>IncludeSpaces</code> and the <code>ExcludeSpaces</code> list, the space is
         *        excluded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeSpaces(Collection<String> includeSpaces);

        /**
         * <p>
         * A list of space keys for Confluence spaces. If you include a key, the blogs, documents, and attachments in
         * the space are indexed. Spaces that aren't in the list aren't indexed. A space in the list must exist.
         * Otherwise, Amazon Kendra logs an error when the data source is synchronized. If a space is in both the
         * <code>IncludeSpaces</code> and the <code>ExcludeSpaces</code> list, the space is excluded.
         * </p>
         * 
         * @param includeSpaces
         *        A list of space keys for Confluence spaces. If you include a key, the blogs, documents, and
         *        attachments in the space are indexed. Spaces that aren't in the list aren't indexed. A space in the
         *        list must exist. Otherwise, Amazon Kendra logs an error when the data source is synchronized. If a
         *        space is in both the <code>IncludeSpaces</code> and the <code>ExcludeSpaces</code> list, the space is
         *        excluded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeSpaces(String... includeSpaces);

        /**
         * <p>
         * A list of space keys of Confluence spaces. If you include a key, the blogs, documents, and attachments in the
         * space are not indexed. If a space is in both the <code>ExcludeSpaces</code> and the
         * <code>IncludeSpaces</code> list, the space is excluded.
         * </p>
         * 
         * @param excludeSpaces
         *        A list of space keys of Confluence spaces. If you include a key, the blogs, documents, and attachments
         *        in the space are not indexed. If a space is in both the <code>ExcludeSpaces</code> and the
         *        <code>IncludeSpaces</code> list, the space is excluded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeSpaces(Collection<String> excludeSpaces);

        /**
         * <p>
         * A list of space keys of Confluence spaces. If you include a key, the blogs, documents, and attachments in the
         * space are not indexed. If a space is in both the <code>ExcludeSpaces</code> and the
         * <code>IncludeSpaces</code> list, the space is excluded.
         * </p>
         * 
         * @param excludeSpaces
         *        A list of space keys of Confluence spaces. If you include a key, the blogs, documents, and attachments
         *        in the space are not indexed. If a space is in both the <code>ExcludeSpaces</code> and the
         *        <code>IncludeSpaces</code> list, the space is excluded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeSpaces(String... excludeSpaces);

        /**
         * <p>
         * Maps attributes or field names of Confluence spaces to Amazon Kendra index field names. To create custom
         * fields, use the <code>UpdateIndex</code> API before you map to Confluence fields. For more information, see
         * <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>. The
         * Confluence data source field names must exist in your Confluence custom metadata.
         * </p>
         * <p>
         * If you specify the <code>SpaceFieldMappings</code> parameter, you must specify at least one field mapping.
         * </p>
         * 
         * @param spaceFieldMappings
         *        Maps attributes or field names of Confluence spaces to Amazon Kendra index field names. To create
         *        custom fields, use the <code>UpdateIndex</code> API before you map to Confluence fields. For more
         *        information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping
         *        data source fields</a>. The Confluence data source field names must exist in your Confluence custom
         *        metadata.</p>
         *        <p>
         *        If you specify the <code>SpaceFieldMappings</code> parameter, you must specify at least one field
         *        mapping.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spaceFieldMappings(Collection<ConfluenceSpaceToIndexFieldMapping> spaceFieldMappings);

        /**
         * <p>
         * Maps attributes or field names of Confluence spaces to Amazon Kendra index field names. To create custom
         * fields, use the <code>UpdateIndex</code> API before you map to Confluence fields. For more information, see
         * <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>. The
         * Confluence data source field names must exist in your Confluence custom metadata.
         * </p>
         * <p>
         * If you specify the <code>SpaceFieldMappings</code> parameter, you must specify at least one field mapping.
         * </p>
         * 
         * @param spaceFieldMappings
         *        Maps attributes or field names of Confluence spaces to Amazon Kendra index field names. To create
         *        custom fields, use the <code>UpdateIndex</code> API before you map to Confluence fields. For more
         *        information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping
         *        data source fields</a>. The Confluence data source field names must exist in your Confluence custom
         *        metadata.</p>
         *        <p>
         *        If you specify the <code>SpaceFieldMappings</code> parameter, you must specify at least one field
         *        mapping.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spaceFieldMappings(ConfluenceSpaceToIndexFieldMapping... spaceFieldMappings);

        /**
         * <p>
         * Maps attributes or field names of Confluence spaces to Amazon Kendra index field names. To create custom
         * fields, use the <code>UpdateIndex</code> API before you map to Confluence fields. For more information, see
         * <a href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>. The
         * Confluence data source field names must exist in your Confluence custom metadata.
         * </p>
         * <p>
         * If you specify the <code>SpaceFieldMappings</code> parameter, you must specify at least one field mapping.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.ConfluenceSpaceToIndexFieldMapping.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.ConfluenceSpaceToIndexFieldMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.ConfluenceSpaceToIndexFieldMapping.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #spaceFieldMappings(List<ConfluenceSpaceToIndexFieldMapping>)}.
         * 
         * @param spaceFieldMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.ConfluenceSpaceToIndexFieldMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #spaceFieldMappings(java.util.Collection<ConfluenceSpaceToIndexFieldMapping>)
         */
        Builder spaceFieldMappings(Consumer<ConfluenceSpaceToIndexFieldMapping.Builder>... spaceFieldMappings);
    }

    static final class BuilderImpl implements Builder {
        private Boolean crawlPersonalSpaces;

        private Boolean crawlArchivedSpaces;

        private List<String> includeSpaces = DefaultSdkAutoConstructList.getInstance();

        private List<String> excludeSpaces = DefaultSdkAutoConstructList.getInstance();

        private List<ConfluenceSpaceToIndexFieldMapping> spaceFieldMappings = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ConfluenceSpaceConfiguration model) {
            crawlPersonalSpaces(model.crawlPersonalSpaces);
            crawlArchivedSpaces(model.crawlArchivedSpaces);
            includeSpaces(model.includeSpaces);
            excludeSpaces(model.excludeSpaces);
            spaceFieldMappings(model.spaceFieldMappings);
        }

        public final Boolean getCrawlPersonalSpaces() {
            return crawlPersonalSpaces;
        }

        public final void setCrawlPersonalSpaces(Boolean crawlPersonalSpaces) {
            this.crawlPersonalSpaces = crawlPersonalSpaces;
        }

        @Override
        public final Builder crawlPersonalSpaces(Boolean crawlPersonalSpaces) {
            this.crawlPersonalSpaces = crawlPersonalSpaces;
            return this;
        }

        public final Boolean getCrawlArchivedSpaces() {
            return crawlArchivedSpaces;
        }

        public final void setCrawlArchivedSpaces(Boolean crawlArchivedSpaces) {
            this.crawlArchivedSpaces = crawlArchivedSpaces;
        }

        @Override
        public final Builder crawlArchivedSpaces(Boolean crawlArchivedSpaces) {
            this.crawlArchivedSpaces = crawlArchivedSpaces;
            return this;
        }

        public final Collection<String> getIncludeSpaces() {
            if (includeSpaces instanceof SdkAutoConstructList) {
                return null;
            }
            return includeSpaces;
        }

        public final void setIncludeSpaces(Collection<String> includeSpaces) {
            this.includeSpaces = ConfluenceSpaceListCopier.copy(includeSpaces);
        }

        @Override
        public final Builder includeSpaces(Collection<String> includeSpaces) {
            this.includeSpaces = ConfluenceSpaceListCopier.copy(includeSpaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includeSpaces(String... includeSpaces) {
            includeSpaces(Arrays.asList(includeSpaces));
            return this;
        }

        public final Collection<String> getExcludeSpaces() {
            if (excludeSpaces instanceof SdkAutoConstructList) {
                return null;
            }
            return excludeSpaces;
        }

        public final void setExcludeSpaces(Collection<String> excludeSpaces) {
            this.excludeSpaces = ConfluenceSpaceListCopier.copy(excludeSpaces);
        }

        @Override
        public final Builder excludeSpaces(Collection<String> excludeSpaces) {
            this.excludeSpaces = ConfluenceSpaceListCopier.copy(excludeSpaces);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludeSpaces(String... excludeSpaces) {
            excludeSpaces(Arrays.asList(excludeSpaces));
            return this;
        }

        public final List<ConfluenceSpaceToIndexFieldMapping.Builder> getSpaceFieldMappings() {
            List<ConfluenceSpaceToIndexFieldMapping.Builder> result = ConfluenceSpaceFieldMappingsListCopier
                    .copyToBuilder(this.spaceFieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSpaceFieldMappings(Collection<ConfluenceSpaceToIndexFieldMapping.BuilderImpl> spaceFieldMappings) {
            this.spaceFieldMappings = ConfluenceSpaceFieldMappingsListCopier.copyFromBuilder(spaceFieldMappings);
        }

        @Override
        public final Builder spaceFieldMappings(Collection<ConfluenceSpaceToIndexFieldMapping> spaceFieldMappings) {
            this.spaceFieldMappings = ConfluenceSpaceFieldMappingsListCopier.copy(spaceFieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder spaceFieldMappings(ConfluenceSpaceToIndexFieldMapping... spaceFieldMappings) {
            spaceFieldMappings(Arrays.asList(spaceFieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder spaceFieldMappings(Consumer<ConfluenceSpaceToIndexFieldMapping.Builder>... spaceFieldMappings) {
            spaceFieldMappings(Stream.of(spaceFieldMappings)
                    .map(c -> ConfluenceSpaceToIndexFieldMapping.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ConfluenceSpaceConfiguration build() {
            return new ConfluenceSpaceConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
