/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the configuration information to connect to Amazon FSx as your data source.
 * </p>
 * <note>
 * <p>
 * Amazon Kendra now supports an upgraded Amazon FSx Windows connector.
 * </p>
 * <p>
 * You must now use the <a
 * href="https://docs.aws.amazon.com/kendra/latest/APIReference/API_TemplateConfiguration.html">TemplateConfiguration
 * </a> object instead of the <code>FsxConfiguration</code> object to configure your connector.
 * </p>
 * <p>
 * Connectors configured using the older console and API architecture will continue to function as configured. However,
 * you won't be able to edit or update them. If you want to edit or update your connector configuration, you must create
 * a new connector.
 * </p>
 * <p>
 * We recommended migrating your connector workflow to the upgraded version. Support for connectors configured using the
 * older architecture is scheduled to end by June 2024.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FsxConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<FsxConfiguration.Builder, FsxConfiguration> {
    private static final SdkField<String> FILE_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemId").getter(getter(FsxConfiguration::fileSystemId)).setter(setter(Builder::fileSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemId").build()).build();

    private static final SdkField<String> FILE_SYSTEM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemType").getter(getter(FsxConfiguration::fileSystemTypeAsString))
            .setter(setter(Builder::fileSystemType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemType").build()).build();

    private static final SdkField<DataSourceVpcConfiguration> VPC_CONFIGURATION_FIELD = SdkField
            .<DataSourceVpcConfiguration> builder(MarshallingType.SDK_POJO).memberName("VpcConfiguration")
            .getter(getter(FsxConfiguration::vpcConfiguration)).setter(setter(Builder::vpcConfiguration))
            .constructor(DataSourceVpcConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfiguration").build()).build();

    private static final SdkField<String> SECRET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretArn").getter(getter(FsxConfiguration::secretArn)).setter(setter(Builder::secretArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretArn").build()).build();

    private static final SdkField<List<String>> INCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InclusionPatterns")
            .getter(getter(FsxConfiguration::inclusionPatterns))
            .setter(setter(Builder::inclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUSION_PATTERNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExclusionPatterns")
            .getter(getter(FsxConfiguration::exclusionPatterns))
            .setter(setter(Builder::exclusionPatterns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExclusionPatterns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataSourceToIndexFieldMapping>> FIELD_MAPPINGS_FIELD = SdkField
            .<List<DataSourceToIndexFieldMapping>> builder(MarshallingType.LIST)
            .memberName("FieldMappings")
            .getter(getter(FsxConfiguration::fieldMappings))
            .setter(setter(Builder::fieldMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FieldMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataSourceToIndexFieldMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataSourceToIndexFieldMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FILE_SYSTEM_ID_FIELD,
            FILE_SYSTEM_TYPE_FIELD, VPC_CONFIGURATION_FIELD, SECRET_ARN_FIELD, INCLUSION_PATTERNS_FIELD,
            EXCLUSION_PATTERNS_FIELD, FIELD_MAPPINGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String fileSystemId;

    private final String fileSystemType;

    private final DataSourceVpcConfiguration vpcConfiguration;

    private final String secretArn;

    private final List<String> inclusionPatterns;

    private final List<String> exclusionPatterns;

    private final List<DataSourceToIndexFieldMapping> fieldMappings;

    private FsxConfiguration(BuilderImpl builder) {
        this.fileSystemId = builder.fileSystemId;
        this.fileSystemType = builder.fileSystemType;
        this.vpcConfiguration = builder.vpcConfiguration;
        this.secretArn = builder.secretArn;
        this.inclusionPatterns = builder.inclusionPatterns;
        this.exclusionPatterns = builder.exclusionPatterns;
        this.fieldMappings = builder.fieldMappings;
    }

    /**
     * <p>
     * The identifier of the Amazon FSx file system.
     * </p>
     * <p>
     * You can find your file system ID on the file system dashboard in the Amazon FSx console. For information on how
     * to create a file system in Amazon FSx console, using Windows File Server as an example, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/getting-started-step1.html">Amazon FSx Getting started
     * guide</a>.
     * </p>
     * 
     * @return The identifier of the Amazon FSx file system.</p>
     *         <p>
     *         You can find your file system ID on the file system dashboard in the Amazon FSx console. For information
     *         on how to create a file system in Amazon FSx console, using Windows File Server as an example, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/getting-started-step1.html">Amazon FSx Getting
     *         started guide</a>.
     */
    public final String fileSystemId() {
        return fileSystemId;
    }

    /**
     * <p>
     * The Amazon FSx file system type. Windows is currently the only supported type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileSystemType}
     * will return {@link FsxFileSystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fileSystemTypeAsString}.
     * </p>
     * 
     * @return The Amazon FSx file system type. Windows is currently the only supported type.
     * @see FsxFileSystemType
     */
    public final FsxFileSystemType fileSystemType() {
        return FsxFileSystemType.fromValue(fileSystemType);
    }

    /**
     * <p>
     * The Amazon FSx file system type. Windows is currently the only supported type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileSystemType}
     * will return {@link FsxFileSystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fileSystemTypeAsString}.
     * </p>
     * 
     * @return The Amazon FSx file system type. Windows is currently the only supported type.
     * @see FsxFileSystemType
     */
    public final String fileSystemTypeAsString() {
        return fileSystemType;
    }

    /**
     * <p>
     * Configuration information for an Amazon Virtual Private Cloud to connect to your Amazon FSx. Your Amazon FSx
     * instance must reside inside your VPC.
     * </p>
     * 
     * @return Configuration information for an Amazon Virtual Private Cloud to connect to your Amazon FSx. Your Amazon
     *         FSx instance must reside inside your VPC.
     */
    public final DataSourceVpcConfiguration vpcConfiguration() {
        return vpcConfiguration;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required to connect
     * to your Amazon FSx file system. Windows is currently the only supported type. The secret must contain a JSON
     * structure with the following keys:
     * </p>
     * <ul>
     * <li>
     * <p>
     * username—The Active Directory user name, along with the Domain Name System (DNS) domain name. For example,
     * <i>user@corp.example.com</i>. The Active Directory user account must have read and mounting access to the Amazon
     * FSx file system for Windows.
     * </p>
     * </li>
     * <li>
     * <p>
     * password—The password of the Active Directory user account with read and mounting access to the Amazon FSx
     * Windows file system.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required to
     *         connect to your Amazon FSx file system. Windows is currently the only supported type. The secret must
     *         contain a JSON structure with the following keys:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         username—The Active Directory user name, along with the Domain Name System (DNS) domain name. For
     *         example, <i>user@corp.example.com</i>. The Active Directory user account must have read and mounting
     *         access to the Amazon FSx file system for Windows.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         password—The password of the Active Directory user account with read and mounting access to the Amazon
     *         FSx Windows file system.
     *         </p>
     *         </li>
     */
    public final String secretArn() {
        return secretArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the InclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInclusionPatterns() {
        return inclusionPatterns != null && !(inclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to include certain files in your Amazon FSx file system. Files that match
     * the patterns are included in the index. Files that don't match the patterns are excluded from the index. If a
     * file matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence and the file isn't
     * included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to include certain files in your Amazon FSx file system. Files that
     *         match the patterns are included in the index. Files that don't match the patterns are excluded from the
     *         index. If a file matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence
     *         and the file isn't included in the index.
     */
    public final List<String> inclusionPatterns() {
        return inclusionPatterns;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExclusionPatterns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExclusionPatterns() {
        return exclusionPatterns != null && !(exclusionPatterns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of regular expression patterns to exclude certain files in your Amazon FSx file system. Files that match
     * the patterns are excluded from the index. Files that don't match the patterns are included in the index. If a
     * file matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence and the file isn't
     * included in the index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExclusionPatterns} method.
     * </p>
     * 
     * @return A list of regular expression patterns to exclude certain files in your Amazon FSx file system. Files that
     *         match the patterns are excluded from the index. Files that don't match the patterns are included in the
     *         index. If a file matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence
     *         and the file isn't included in the index.
     */
    public final List<String> exclusionPatterns() {
        return exclusionPatterns;
    }

    /**
     * For responses, this returns true if the service returned a value for the FieldMappings property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFieldMappings() {
        return fieldMappings != null && !(fieldMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>DataSourceToIndexFieldMapping</code> objects that map Amazon FSx data source attributes or field
     * names to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API before
     * you map to Amazon FSx fields. For more information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>. The Amazon
     * FSx data source field names must exist in your Amazon FSx custom metadata.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFieldMappings} method.
     * </p>
     * 
     * @return A list of <code>DataSourceToIndexFieldMapping</code> objects that map Amazon FSx data source attributes
     *         or field names to Amazon Kendra index field names. To create custom fields, use the
     *         <code>UpdateIndex</code> API before you map to Amazon FSx fields. For more information, see <a
     *         href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>.
     *         The Amazon FSx data source field names must exist in your Amazon FSx custom metadata.
     */
    public final List<DataSourceToIndexFieldMapping> fieldMappings() {
        return fieldMappings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(secretArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasInclusionPatterns() ? inclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExclusionPatterns() ? exclusionPatterns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasFieldMappings() ? fieldMappings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FsxConfiguration)) {
            return false;
        }
        FsxConfiguration other = (FsxConfiguration) obj;
        return Objects.equals(fileSystemId(), other.fileSystemId())
                && Objects.equals(fileSystemTypeAsString(), other.fileSystemTypeAsString())
                && Objects.equals(vpcConfiguration(), other.vpcConfiguration()) && Objects.equals(secretArn(), other.secretArn())
                && hasInclusionPatterns() == other.hasInclusionPatterns()
                && Objects.equals(inclusionPatterns(), other.inclusionPatterns())
                && hasExclusionPatterns() == other.hasExclusionPatterns()
                && Objects.equals(exclusionPatterns(), other.exclusionPatterns())
                && hasFieldMappings() == other.hasFieldMappings() && Objects.equals(fieldMappings(), other.fieldMappings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FsxConfiguration").add("FileSystemId", fileSystemId())
                .add("FileSystemType", fileSystemTypeAsString()).add("VpcConfiguration", vpcConfiguration())
                .add("SecretArn", secretArn()).add("InclusionPatterns", hasInclusionPatterns() ? inclusionPatterns() : null)
                .add("ExclusionPatterns", hasExclusionPatterns() ? exclusionPatterns() : null)
                .add("FieldMappings", hasFieldMappings() ? fieldMappings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FileSystemId":
            return Optional.ofNullable(clazz.cast(fileSystemId()));
        case "FileSystemType":
            return Optional.ofNullable(clazz.cast(fileSystemTypeAsString()));
        case "VpcConfiguration":
            return Optional.ofNullable(clazz.cast(vpcConfiguration()));
        case "SecretArn":
            return Optional.ofNullable(clazz.cast(secretArn()));
        case "InclusionPatterns":
            return Optional.ofNullable(clazz.cast(inclusionPatterns()));
        case "ExclusionPatterns":
            return Optional.ofNullable(clazz.cast(exclusionPatterns()));
        case "FieldMappings":
            return Optional.ofNullable(clazz.cast(fieldMappings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("FileSystemId", FILE_SYSTEM_ID_FIELD);
        map.put("FileSystemType", FILE_SYSTEM_TYPE_FIELD);
        map.put("VpcConfiguration", VPC_CONFIGURATION_FIELD);
        map.put("SecretArn", SECRET_ARN_FIELD);
        map.put("InclusionPatterns", INCLUSION_PATTERNS_FIELD);
        map.put("ExclusionPatterns", EXCLUSION_PATTERNS_FIELD);
        map.put("FieldMappings", FIELD_MAPPINGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<FsxConfiguration, T> g) {
        return obj -> g.apply((FsxConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FsxConfiguration> {
        /**
         * <p>
         * The identifier of the Amazon FSx file system.
         * </p>
         * <p>
         * You can find your file system ID on the file system dashboard in the Amazon FSx console. For information on
         * how to create a file system in Amazon FSx console, using Windows File Server as an example, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/getting-started-step1.html">Amazon FSx Getting
         * started guide</a>.
         * </p>
         * 
         * @param fileSystemId
         *        The identifier of the Amazon FSx file system.</p>
         *        <p>
         *        You can find your file system ID on the file system dashboard in the Amazon FSx console. For
         *        information on how to create a file system in Amazon FSx console, using Windows File Server as an
         *        example, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/getting-started-step1.html">Amazon FSx
         *        Getting started guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * The Amazon FSx file system type. Windows is currently the only supported type.
         * </p>
         * 
         * @param fileSystemType
         *        The Amazon FSx file system type. Windows is currently the only supported type.
         * @see FsxFileSystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FsxFileSystemType
         */
        Builder fileSystemType(String fileSystemType);

        /**
         * <p>
         * The Amazon FSx file system type. Windows is currently the only supported type.
         * </p>
         * 
         * @param fileSystemType
         *        The Amazon FSx file system type. Windows is currently the only supported type.
         * @see FsxFileSystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FsxFileSystemType
         */
        Builder fileSystemType(FsxFileSystemType fileSystemType);

        /**
         * <p>
         * Configuration information for an Amazon Virtual Private Cloud to connect to your Amazon FSx. Your Amazon FSx
         * instance must reside inside your VPC.
         * </p>
         * 
         * @param vpcConfiguration
         *        Configuration information for an Amazon Virtual Private Cloud to connect to your Amazon FSx. Your
         *        Amazon FSx instance must reside inside your VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration);

        /**
         * <p>
         * Configuration information for an Amazon Virtual Private Cloud to connect to your Amazon FSx. Your Amazon FSx
         * instance must reside inside your VPC.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataSourceVpcConfiguration.Builder}
         * avoiding the need to create one manually via {@link DataSourceVpcConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataSourceVpcConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #vpcConfiguration(DataSourceVpcConfiguration)}.
         * 
         * @param vpcConfiguration
         *        a consumer that will call methods on {@link DataSourceVpcConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfiguration(DataSourceVpcConfiguration)
         */
        default Builder vpcConfiguration(Consumer<DataSourceVpcConfiguration.Builder> vpcConfiguration) {
            return vpcConfiguration(DataSourceVpcConfiguration.builder().applyMutation(vpcConfiguration).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required to
         * connect to your Amazon FSx file system. Windows is currently the only supported type. The secret must contain
         * a JSON structure with the following keys:
         * </p>
         * <ul>
         * <li>
         * <p>
         * username—The Active Directory user name, along with the Domain Name System (DNS) domain name. For example,
         * <i>user@corp.example.com</i>. The Active Directory user account must have read and mounting access to the
         * Amazon FSx file system for Windows.
         * </p>
         * </li>
         * <li>
         * <p>
         * password—The password of the Active Directory user account with read and mounting access to the Amazon FSx
         * Windows file system.
         * </p>
         * </li>
         * </ul>
         * 
         * @param secretArn
         *        The Amazon Resource Name (ARN) of an Secrets Manager secret that contains the key-value pairs required
         *        to connect to your Amazon FSx file system. Windows is currently the only supported type. The secret
         *        must contain a JSON structure with the following keys:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        username—The Active Directory user name, along with the Domain Name System (DNS) domain name. For
         *        example, <i>user@corp.example.com</i>. The Active Directory user account must have read and mounting
         *        access to the Amazon FSx file system for Windows.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        password—The password of the Active Directory user account with read and mounting access to the Amazon
         *        FSx Windows file system.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretArn(String secretArn);

        /**
         * <p>
         * A list of regular expression patterns to include certain files in your Amazon FSx file system. Files that
         * match the patterns are included in the index. Files that don't match the patterns are excluded from the
         * index. If a file matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence and
         * the file isn't included in the index.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns to include certain files in your Amazon FSx file system. Files
         *        that match the patterns are included in the index. Files that don't match the patterns are excluded
         *        from the index. If a file matches both an inclusion and exclusion pattern, the exclusion pattern takes
         *        precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(Collection<String> inclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to include certain files in your Amazon FSx file system. Files that
         * match the patterns are included in the index. Files that don't match the patterns are excluded from the
         * index. If a file matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence and
         * the file isn't included in the index.
         * </p>
         * 
         * @param inclusionPatterns
         *        A list of regular expression patterns to include certain files in your Amazon FSx file system. Files
         *        that match the patterns are included in the index. Files that don't match the patterns are excluded
         *        from the index. If a file matches both an inclusion and exclusion pattern, the exclusion pattern takes
         *        precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPatterns(String... inclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to exclude certain files in your Amazon FSx file system. Files that
         * match the patterns are excluded from the index. Files that don't match the patterns are included in the
         * index. If a file matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence and
         * the file isn't included in the index.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of regular expression patterns to exclude certain files in your Amazon FSx file system. Files
         *        that match the patterns are excluded from the index. Files that don't match the patterns are included
         *        in the index. If a file matches both an inclusion and exclusion pattern, the exclusion pattern takes
         *        precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(Collection<String> exclusionPatterns);

        /**
         * <p>
         * A list of regular expression patterns to exclude certain files in your Amazon FSx file system. Files that
         * match the patterns are excluded from the index. Files that don't match the patterns are included in the
         * index. If a file matches both an inclusion and exclusion pattern, the exclusion pattern takes precedence and
         * the file isn't included in the index.
         * </p>
         * 
         * @param exclusionPatterns
         *        A list of regular expression patterns to exclude certain files in your Amazon FSx file system. Files
         *        that match the patterns are excluded from the index. Files that don't match the patterns are included
         *        in the index. If a file matches both an inclusion and exclusion pattern, the exclusion pattern takes
         *        precedence and the file isn't included in the index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exclusionPatterns(String... exclusionPatterns);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map Amazon FSx data source attributes or
         * field names to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API
         * before you map to Amazon FSx fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>. The
         * Amazon FSx data source field names must exist in your Amazon FSx custom metadata.
         * </p>
         * 
         * @param fieldMappings
         *        A list of <code>DataSourceToIndexFieldMapping</code> objects that map Amazon FSx data source
         *        attributes or field names to Amazon Kendra index field names. To create custom fields, use the
         *        <code>UpdateIndex</code> API before you map to Amazon FSx fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>.
         *        The Amazon FSx data source field names must exist in your Amazon FSx custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fieldMappings(Collection<DataSourceToIndexFieldMapping> fieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map Amazon FSx data source attributes or
         * field names to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API
         * before you map to Amazon FSx fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>. The
         * Amazon FSx data source field names must exist in your Amazon FSx custom metadata.
         * </p>
         * 
         * @param fieldMappings
         *        A list of <code>DataSourceToIndexFieldMapping</code> objects that map Amazon FSx data source
         *        attributes or field names to Amazon Kendra index field names. To create custom fields, use the
         *        <code>UpdateIndex</code> API before you map to Amazon FSx fields. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>.
         *        The Amazon FSx data source field names must exist in your Amazon FSx custom metadata.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fieldMappings(DataSourceToIndexFieldMapping... fieldMappings);

        /**
         * <p>
         * A list of <code>DataSourceToIndexFieldMapping</code> objects that map Amazon FSx data source attributes or
         * field names to Amazon Kendra index field names. To create custom fields, use the <code>UpdateIndex</code> API
         * before you map to Amazon FSx fields. For more information, see <a
         * href="https://docs.aws.amazon.com/kendra/latest/dg/field-mapping.html">Mapping data source fields</a>. The
         * Amazon FSx data source field names must exist in your Amazon FSx custom metadata.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder#build()} is called
         * immediately and its result is passed to {@link #fieldMappings(List<DataSourceToIndexFieldMapping>)}.
         * 
         * @param fieldMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.kendra.model.DataSourceToIndexFieldMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fieldMappings(java.util.Collection<DataSourceToIndexFieldMapping>)
         */
        Builder fieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... fieldMappings);
    }

    static final class BuilderImpl implements Builder {
        private String fileSystemId;

        private String fileSystemType;

        private DataSourceVpcConfiguration vpcConfiguration;

        private String secretArn;

        private List<String> inclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private List<String> exclusionPatterns = DefaultSdkAutoConstructList.getInstance();

        private List<DataSourceToIndexFieldMapping> fieldMappings = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FsxConfiguration model) {
            fileSystemId(model.fileSystemId);
            fileSystemType(model.fileSystemType);
            vpcConfiguration(model.vpcConfiguration);
            secretArn(model.secretArn);
            inclusionPatterns(model.inclusionPatterns);
            exclusionPatterns(model.exclusionPatterns);
            fieldMappings(model.fieldMappings);
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        @Override
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final String getFileSystemType() {
            return fileSystemType;
        }

        public final void setFileSystemType(String fileSystemType) {
            this.fileSystemType = fileSystemType;
        }

        @Override
        public final Builder fileSystemType(String fileSystemType) {
            this.fileSystemType = fileSystemType;
            return this;
        }

        @Override
        public final Builder fileSystemType(FsxFileSystemType fileSystemType) {
            this.fileSystemType(fileSystemType == null ? null : fileSystemType.toString());
            return this;
        }

        public final DataSourceVpcConfiguration.Builder getVpcConfiguration() {
            return vpcConfiguration != null ? vpcConfiguration.toBuilder() : null;
        }

        public final void setVpcConfiguration(DataSourceVpcConfiguration.BuilderImpl vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration != null ? vpcConfiguration.build() : null;
        }

        @Override
        public final Builder vpcConfiguration(DataSourceVpcConfiguration vpcConfiguration) {
            this.vpcConfiguration = vpcConfiguration;
            return this;
        }

        public final String getSecretArn() {
            return secretArn;
        }

        public final void setSecretArn(String secretArn) {
            this.secretArn = secretArn;
        }

        @Override
        public final Builder secretArn(String secretArn) {
            this.secretArn = secretArn;
            return this;
        }

        public final Collection<String> getInclusionPatterns() {
            if (inclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return inclusionPatterns;
        }

        public final void setInclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
        }

        @Override
        public final Builder inclusionPatterns(Collection<String> inclusionPatterns) {
            this.inclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(inclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inclusionPatterns(String... inclusionPatterns) {
            inclusionPatterns(Arrays.asList(inclusionPatterns));
            return this;
        }

        public final Collection<String> getExclusionPatterns() {
            if (exclusionPatterns instanceof SdkAutoConstructList) {
                return null;
            }
            return exclusionPatterns;
        }

        public final void setExclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
        }

        @Override
        public final Builder exclusionPatterns(Collection<String> exclusionPatterns) {
            this.exclusionPatterns = DataSourceInclusionsExclusionsStringsCopier.copy(exclusionPatterns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder exclusionPatterns(String... exclusionPatterns) {
            exclusionPatterns(Arrays.asList(exclusionPatterns));
            return this;
        }

        public final List<DataSourceToIndexFieldMapping.Builder> getFieldMappings() {
            List<DataSourceToIndexFieldMapping.Builder> result = DataSourceToIndexFieldMappingListCopier
                    .copyToBuilder(this.fieldMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFieldMappings(Collection<DataSourceToIndexFieldMapping.BuilderImpl> fieldMappings) {
            this.fieldMappings = DataSourceToIndexFieldMappingListCopier.copyFromBuilder(fieldMappings);
        }

        @Override
        public final Builder fieldMappings(Collection<DataSourceToIndexFieldMapping> fieldMappings) {
            this.fieldMappings = DataSourceToIndexFieldMappingListCopier.copy(fieldMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fieldMappings(DataSourceToIndexFieldMapping... fieldMappings) {
            fieldMappings(Arrays.asList(fieldMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fieldMappings(Consumer<DataSourceToIndexFieldMapping.Builder>... fieldMappings) {
            fieldMappings(Stream.of(fieldMappings).map(c -> DataSourceToIndexFieldMapping.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public FsxConfiguration build() {
            return new FsxConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
