/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kendra.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the configuration information for invoking a Lambda function in Lambda to alter document metadata and
 * content when ingesting documents into Amazon Kendra. You can configure your Lambda function using <a
 * href="https://docs.aws.amazon.com/kendra/latest/dg/API_CustomDocumentEnrichmentConfiguration.html"
 * >PreExtractionHookConfiguration</a> if you want to apply advanced alterations on the original or raw documents. If
 * you want to apply advanced alterations on the Amazon Kendra structured documents, you must configure your Lambda
 * function using <a href="https://docs.aws.amazon.com/kendra/latest/dg/API_CustomDocumentEnrichmentConfiguration.html">
 * PostExtractionHookConfiguration</a>. You can only invoke one Lambda function. However, this function can invoke other
 * functions it requires.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/kendra/latest/dg/custom-document-enrichment.html">Customizing document metadata
 * during the ingestion process</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HookConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<HookConfiguration.Builder, HookConfiguration> {
    private static final SdkField<DocumentAttributeCondition> INVOCATION_CONDITION_FIELD = SdkField
            .<DocumentAttributeCondition> builder(MarshallingType.SDK_POJO).memberName("InvocationCondition")
            .getter(getter(HookConfiguration::invocationCondition)).setter(setter(Builder::invocationCondition))
            .constructor(DocumentAttributeCondition::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InvocationCondition").build())
            .build();

    private static final SdkField<String> LAMBDA_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LambdaArn").getter(getter(HookConfiguration::lambdaArn)).setter(setter(Builder::lambdaArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LambdaArn").build()).build();

    private static final SdkField<String> S3_BUCKET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3Bucket").getter(getter(HookConfiguration::s3Bucket)).setter(setter(Builder::s3Bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Bucket").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INVOCATION_CONDITION_FIELD,
            LAMBDA_ARN_FIELD, S3_BUCKET_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final DocumentAttributeCondition invocationCondition;

    private final String lambdaArn;

    private final String s3Bucket;

    private HookConfiguration(BuilderImpl builder) {
        this.invocationCondition = builder.invocationCondition;
        this.lambdaArn = builder.lambdaArn;
        this.s3Bucket = builder.s3Bucket;
    }

    /**
     * <p>
     * The condition used for when a Lambda function should be invoked.
     * </p>
     * <p>
     * For example, you can specify a condition that if there are empty date-time values, then Amazon Kendra should
     * invoke a function that inserts the current date-time.
     * </p>
     * 
     * @return The condition used for when a Lambda function should be invoked.</p>
     *         <p>
     *         For example, you can specify a condition that if there are empty date-time values, then Amazon Kendra
     *         should invoke a function that inserts the current date-time.
     */
    public final DocumentAttributeCondition invocationCondition() {
        return invocationCondition;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an IAM role with permission to run a Lambda function during ingestion. For more
     * information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/iam-roles.html">an IAM roles for Amazon
     * Kendra</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an IAM role with permission to run a Lambda function during ingestion.
     *         For more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/iam-roles.html">an IAM
     *         roles for Amazon Kendra</a>.
     */
    public final String lambdaArn() {
        return lambdaArn;
    }

    /**
     * <p>
     * Stores the original, raw documents or the structured, parsed documents before and after altering them. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/kendra/latest/dg/custom-document-enrichment.html#cde-data-contracts-lambda"
     * >Data contracts for Lambda functions</a>.
     * </p>
     * 
     * @return Stores the original, raw documents or the structured, parsed documents before and after altering them.
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/kendra/latest/dg/custom-document-enrichment.html#cde-data-contracts-lambda"
     *         >Data contracts for Lambda functions</a>.
     */
    public final String s3Bucket() {
        return s3Bucket;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(invocationCondition());
        hashCode = 31 * hashCode + Objects.hashCode(lambdaArn());
        hashCode = 31 * hashCode + Objects.hashCode(s3Bucket());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HookConfiguration)) {
            return false;
        }
        HookConfiguration other = (HookConfiguration) obj;
        return Objects.equals(invocationCondition(), other.invocationCondition())
                && Objects.equals(lambdaArn(), other.lambdaArn()) && Objects.equals(s3Bucket(), other.s3Bucket());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("HookConfiguration").add("InvocationCondition", invocationCondition())
                .add("LambdaArn", lambdaArn()).add("S3Bucket", s3Bucket()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InvocationCondition":
            return Optional.ofNullable(clazz.cast(invocationCondition()));
        case "LambdaArn":
            return Optional.ofNullable(clazz.cast(lambdaArn()));
        case "S3Bucket":
            return Optional.ofNullable(clazz.cast(s3Bucket()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InvocationCondition", INVOCATION_CONDITION_FIELD);
        map.put("LambdaArn", LAMBDA_ARN_FIELD);
        map.put("S3Bucket", S3_BUCKET_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<HookConfiguration, T> g) {
        return obj -> g.apply((HookConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HookConfiguration> {
        /**
         * <p>
         * The condition used for when a Lambda function should be invoked.
         * </p>
         * <p>
         * For example, you can specify a condition that if there are empty date-time values, then Amazon Kendra should
         * invoke a function that inserts the current date-time.
         * </p>
         * 
         * @param invocationCondition
         *        The condition used for when a Lambda function should be invoked.</p>
         *        <p>
         *        For example, you can specify a condition that if there are empty date-time values, then Amazon Kendra
         *        should invoke a function that inserts the current date-time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invocationCondition(DocumentAttributeCondition invocationCondition);

        /**
         * <p>
         * The condition used for when a Lambda function should be invoked.
         * </p>
         * <p>
         * For example, you can specify a condition that if there are empty date-time values, then Amazon Kendra should
         * invoke a function that inserts the current date-time.
         * </p>
         * This is a convenience method that creates an instance of the {@link DocumentAttributeCondition.Builder}
         * avoiding the need to create one manually via {@link DocumentAttributeCondition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DocumentAttributeCondition.Builder#build()} is called immediately
         * and its result is passed to {@link #invocationCondition(DocumentAttributeCondition)}.
         * 
         * @param invocationCondition
         *        a consumer that will call methods on {@link DocumentAttributeCondition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #invocationCondition(DocumentAttributeCondition)
         */
        default Builder invocationCondition(Consumer<DocumentAttributeCondition.Builder> invocationCondition) {
            return invocationCondition(DocumentAttributeCondition.builder().applyMutation(invocationCondition).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an IAM role with permission to run a Lambda function during ingestion. For
         * more information, see <a href="https://docs.aws.amazon.com/kendra/latest/dg/iam-roles.html">an IAM roles for
         * Amazon Kendra</a>.
         * </p>
         * 
         * @param lambdaArn
         *        The Amazon Resource Name (ARN) of an IAM role with permission to run a Lambda function during
         *        ingestion. For more information, see <a
         *        href="https://docs.aws.amazon.com/kendra/latest/dg/iam-roles.html">an IAM roles for Amazon Kendra</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lambdaArn(String lambdaArn);

        /**
         * <p>
         * Stores the original, raw documents or the structured, parsed documents before and after altering them. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/kendra/latest/dg/custom-document-enrichment.html#cde-data-contracts-lambda">Data
         * contracts for Lambda functions</a>.
         * </p>
         * 
         * @param s3Bucket
         *        Stores the original, raw documents or the structured, parsed documents before and after altering them.
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/kendra/latest/dg/custom-document-enrichment.html#cde-data-contracts-lambda"
         *        >Data contracts for Lambda functions</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Bucket(String s3Bucket);
    }

    static final class BuilderImpl implements Builder {
        private DocumentAttributeCondition invocationCondition;

        private String lambdaArn;

        private String s3Bucket;

        private BuilderImpl() {
        }

        private BuilderImpl(HookConfiguration model) {
            invocationCondition(model.invocationCondition);
            lambdaArn(model.lambdaArn);
            s3Bucket(model.s3Bucket);
        }

        public final DocumentAttributeCondition.Builder getInvocationCondition() {
            return invocationCondition != null ? invocationCondition.toBuilder() : null;
        }

        public final void setInvocationCondition(DocumentAttributeCondition.BuilderImpl invocationCondition) {
            this.invocationCondition = invocationCondition != null ? invocationCondition.build() : null;
        }

        @Override
        public final Builder invocationCondition(DocumentAttributeCondition invocationCondition) {
            this.invocationCondition = invocationCondition;
            return this;
        }

        public final String getLambdaArn() {
            return lambdaArn;
        }

        public final void setLambdaArn(String lambdaArn) {
            this.lambdaArn = lambdaArn;
        }

        @Override
        public final Builder lambdaArn(String lambdaArn) {
            this.lambdaArn = lambdaArn;
            return this;
        }

        public final String getS3Bucket() {
            return s3Bucket;
        }

        public final void setS3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
        }

        @Override
        public final Builder s3Bucket(String s3Bucket) {
            this.s3Bucket = s3Bucket;
            return this;
        }

        @Override
        public HookConfiguration build() {
            return new HookConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
